// DShowTut2.cpp : Defines the entry point for the application.
//

#include <windows.h>
#include <stdio.h>
#include "resource.h"
#include <streams.h>

#define WM_GRAPHEVENT	WM_USER		// define a custom window message for graph events


// convenient macro for releasing interfaces
#define HELPER_RELEASE(x)   if (x != NULL) \
                            { \
                                x->Release(); \
                                x = NULL; \
                            }



// DirectShow interfaces
IGraphBuilder*	g_pGraphBuilder = NULL;
IMediaControl*	g_pMediaControl = NULL;
IMediaEventEx*	g_pMediaEvent = NULL;
IMediaPosition*	g_pMediaPosition = NULL;


HINSTANCE   g_AppInstance;
HWND        g_AppWindow;


BOOL        g_Looping = FALSE;          // should we loop the media

char        g_ExecuteDir[MAX_PATH];     // directory of execution
char        g_WindowTitle[] = "DShowTut2";              // text for title bar


int InitDirectShow()
{
	HRESULT hr;

	hr = CoCreateInstance(CLSID_FilterGraph, NULL, CLSCTX_INPROC_SERVER, IID_IGraphBuilder,
        (void**)&g_pGraphBuilder);
    if (FAILED(hr))
        return -1;
	g_pGraphBuilder->QueryInterface(IID_IMediaControl, (void**)&g_pMediaControl);
	g_pGraphBuilder->QueryInterface(IID_IMediaEvent, (void**)&g_pMediaEvent);
	g_pGraphBuilder->QueryInterface(IID_IMediaPosition, (void**)&g_pMediaPosition);

	g_pMediaEvent->SetNotifyWindow((OAHWND)g_AppWindow, WM_GRAPHEVENT, 0);
	g_pMediaEvent->SetNotifyFlags(0);	// turn on notifications

    return 0;
}


void CleanUpDirectShow()
{
    HELPER_RELEASE(g_pMediaPosition);
    HELPER_RELEASE(g_pMediaEvent);
    HELPER_RELEASE(g_pMediaControl);
    HELPER_RELEASE(g_pGraphBuilder);
}


int CreateGraph(char* filename)
{
	int	    length;		// length of filename
	WCHAR*	wfilename;	// where we store WCHAR version of filename
    HRESULT hr;
    HANDLE  logfile;                  // log file for graphbuilder
    char    logfilename[MAX_PATH];    // full path and name of log file

    // create graphbuilder's log file;  we must give it a full path or
    // else it will create the log file in the directory the media file is
    // located in
    sprintf(logfilename, "%s\\graph.log", g_ExecuteDir);
    logfile = CreateFile(logfilename, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS,
        FILE_ATTRIBUTE_NORMAL, NULL);
    g_pGraphBuilder->SetLogFile(logfile);

    // attempt to build a graph
	length = strlen(filename)+1;
	wfilename = new WCHAR[length];
	MultiByteToWideChar(CP_ACP, 0, filename, -1, wfilename, length);
    hr = g_pGraphBuilder->RenderFile(wfilename, NULL);
    CloseHandle(logfile);
	if (FAILED(hr))
		return -1;
	else
		return 0;
}


// called when directshow events occur
void OnGraphEvent()
{
	long EventCode, Param1, Param2;
	while (g_pMediaEvent->GetEvent(&EventCode, &Param1, &Param2, 0) != E_ABORT)
	{
		switch (EventCode)
		{
        case EC_COMPLETE:            
            // here when media is completely done playing
            if (!g_Looping)
                g_pMediaControl->Stop();
            g_pMediaPosition->put_CurrentPosition(0);   // reset to beginning
            break;
		default:
			break;
		}	
		g_pMediaEvent->FreeEventParams(EventCode, Param1, Param2);
	}
}





LRESULT CALLBACK WindowProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    switch (uMsg)
    {
    case WM_COMMAND:
        switch (LOWORD(wParam))
        {
        case ID_FILE_OPEN:
            OPENFILENAME    ofn;    // use common 'Open' control
            char            file[MAX_PATH];     // filename with path
            char            filetitle[MAX_PATH];    // thrown in to show file in title bar

            memset(file, 0, MAX_PATH);
            memset(&ofn, 0, sizeof(OPENFILENAME));
            ofn.lStructSize = sizeof(OPENFILENAME);
            ofn.hwndOwner = NULL;
            ofn.lpstrFile = file;
            ofn.nMaxFile = MAX_PATH-1;
            ofn.lpstrFileTitle = filetitle;
            ofn.nMaxFileTitle = MAX_PATH-1;
            ofn.Flags = OFN_FILEMUSTEXIST | OFN_PATHMUSTEXIST;
            if (GetOpenFileName(&ofn))
            {
                CleanUpDirectShow();    // release previous graph

                if (InitDirectShow() != 0)
                    MessageBox(NULL, "Could not initialize DirectShow!", "ERROR", MB_OK);
                else
                {
                    if (CreateGraph(file) != 0)
                        MessageBox(NULL, "Could not create a filter graph for this file!",
                            "ERROR", MB_OK);
                    else
                    {
                        char    NewTitle[256];
                        sprintf(NewTitle, "%s - %s", g_WindowTitle, filetitle);
                        SetWindowText(g_AppWindow, NewTitle);
                    }
                }
            }
            break;
        case ID_FILE_EXIT:
            PostQuitMessage(0);
            break;
        case ID_MEDIA_PLAY:
            if (g_pMediaControl)
                g_pMediaControl->Run();
            else
                MessageBox(NULL, "Select a file to play first!", g_WindowTitle, MB_OK);
            break;
        case ID_MEDIA_STOP:
            if (g_pMediaPosition)
            {
                g_pMediaControl->Stop();
                g_pMediaPosition->put_CurrentPosition(0);   // return to beginning
            }
            break;
        case ID_OPTIONS_LOOPING:
            if (g_Looping)
            {
                CheckMenuItem(GetMenu(g_AppWindow), ID_OPTIONS_LOOPING, MF_BYCOMMAND | MF_UNCHECKED);
                g_Looping = FALSE;
            }
            else
            {
                CheckMenuItem(GetMenu(g_AppWindow), ID_OPTIONS_LOOPING, MF_BYCOMMAND | MF_CHECKED);
                g_Looping = TRUE;
            }
            break;
        default:
            break;
        }
        break;
    case WM_SYSCOMMAND:
        if (wParam == SC_CLOSE)
            PostQuitMessage(0);
        break;
	case WM_GRAPHEVENT:
		OnGraphEvent();		// handles events
		break;    
    default:
        break;
    }
    return DefWindowProc(hwnd, uMsg, wParam, lParam);
}

int InitApp()
{
    WNDCLASSEX wc;

    CoInitialize(NULL);

    memset(&wc,0, sizeof(wc));
    wc.cbSize = sizeof(wc);
    wc.lpfnWndProc = WindowProc;
    wc.hInstance = g_AppInstance;
    wc.hbrBackground = (HBRUSH)GetStockObject(BLACK_BRUSH);
    wc.lpszMenuName = MAKEINTRESOURCE(IDR_MENU1);
    wc.lpszClassName = "DShowTut2 Class";
    RegisterClassEx(&wc);
    
    g_AppWindow = CreateWindowEx(0, "DShowTut2 Class",
                                g_WindowTitle,
                                WS_OVERLAPPEDWINDOW,
                                CW_USEDEFAULT,
                                CW_USEDEFAULT,
                                300,
                                100,
                                NULL,
                                NULL,
                                g_AppInstance,
                                NULL);
    if (g_AppWindow == NULL)
        return -1;

    CheckMenuItem(GetMenu(g_AppWindow), ID_OPTIONS_LOOPING, MF_BYCOMMAND | MF_UNCHECKED);
    ShowWindow(g_AppWindow, SW_SHOW);

    // save directory for log file later
    GetCurrentDirectory(MAX_PATH, g_ExecuteDir);
    return 0;
}




int APIENTRY WinMain(HINSTANCE hInstance,
                     HINSTANCE hPrevInstance,
                     LPSTR     lpCmdLine,
                     int       nCmdShow)
{
    MSG msg;

    g_AppInstance = hInstance;

    if (InitApp() != 0)
        return -1;

    while (GetMessage(&msg, NULL, 0, 0) != 0)
    {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }

    CleanUpDirectShow();

    CoUninitialize();

	return 0;
}



