/*
Name: Sprite Explode
Author: Sean Howe
Date: 2/10/2001
Description: This program is an example of a simple method of doing 
sprite explosions. It's based on the lesson five code. Several
ways to improve it would be: alpha maps and different shapes than just squares
 All it does right now is divide up a sprite then put a random value for
its rotation on all axis, generates a random velocity then calculates 
the x and y velocities from that and the piece's position in the ship. */
// Includes for Windows and DirectX
#include <windows.h>
#include <d3d8.h>
#include <d3dx8.h>


// Our vertex class
class CVertex
{
public:
	// Its coordinates
	float fX, fY, fZ;

	// Its color
	DWORD dwColor;

	// Its texture coordinates
	float fU, fV;

	// A function to simplify initialization
	void Create(float fp_fX,float fp_fY,float fp_fZ, float fp_fR, float fp_fG, float fp_fB, float fp_fU, float fp_fV)
	{
		fX=fp_fX;
		fY=fp_fY;
		fZ=fp_fZ;

		dwColor=D3DCOLOR_XRGB((BYTE)(fp_fR*255), (BYTE)(fp_fG*255), (BYTE)(fp_fB*255));

		fU=fp_fU;
		fV=fp_fV;
	}
};
// The CVertex FVF
const DWORD D3DFVF_CVertex=D3DFVF_XYZ | D3DFVF_TEX1 | D3DFVF_DIFFUSE;

class CExplosionPart
{
public:
	CVertex vtx[4];
	float fX, fY;
	float fXVel, fYVel;
	float fRotationX, fRotationY, fRotationZ;
	float fRotationModX, fRotationModY, fRotationModZ;
	D3DXMATRIX mat;

	void Update()
	{
		D3DXMATRIX matTrans;
		D3DXMATRIX matRot;

		fX+=fXVel;
		fY+=fYVel;
		fRotationX+=fRotationModX;
		fRotationY+=fRotationModY;
		fRotationZ+=fRotationModZ;

		D3DXMatrixTranslation(&matTrans, fX,fY, 0);
		D3DXMatrixRotationYawPitchRoll(&matRot, fRotationX, fRotationY, fRotationZ);
		D3DXMatrixMultiply(&mat, &matRot, &matTrans);
		
	}


	void Draw(IDirect3DDevice8 *fp_pDevice)
	{
		fp_pDevice->SetTransform(D3DTS_WORLD, &((D3DMATRIX)mat));
		fp_pDevice->DrawPrimitiveUP(D3DPT_TRIANGLESTRIP, 2, vtx, sizeof(CVertex));
	}
};

// Constants to make changing stuff in initialization easier
const int WINDOW_WIDTH = 640; 
const int WINDOW_HEIGHT = 480;
const int WINDOW_X = 0; 
const int WINDOW_Y = 0; 
const char *WINDOW_TITLE="Sprite Explosion Demo";
const char *WINDOW_CLASS_NAME="NeXe Tutorial"; 


// Important Windows variables
HINSTANCE g_hInst;
HWND g_hWnd;	

// Whether or not it's fullscreen
bool g_bFullscreen;

// Important Direct3D variables
IDirect3D8 *g_pDirect3D;
IDirect3DDevice8 *g_pDevice;
IDirect3DTexture8 *g_pTexture;

// Results from Direct3D function calls are stored here
HRESULT g_hr;

// Explosion stuff
CExplosionPart *g_pExplosion;
int g_iNumParts;
int g_iNumPerSide; // g_iNumPerSide * g_iNumPerSide=g_iNumParts

// We use this variable to keep track of what hitting enter should do, 
// reset it or make it go boom
bool g_bGoneBoom;

void Boom()
{
	int i;
	D3DXVECTOR3 vec;
	float totalVel;

	g_bGoneBoom=true;

	for (i=0; i<g_iNumParts; i++)
	{
		g_pExplosion[i].fRotationModX=D3DX_PI/(rand()%2000 + 50);
		g_pExplosion[i].fRotationModY=D3DX_PI/(rand()%2000 + 50);
		g_pExplosion[i].fRotationModZ=D3DX_PI/(rand()%2000 + 50);

		totalVel=(rand()%1000) / 25000.0;
		g_pExplosion[i].fXVel=(i%g_iNumPerSide - g_iNumPerSide/2)/(float)(g_iNumPerSide/2) * totalVel;
		g_pExplosion[i].fYVel=(i/g_iNumPerSide - g_iNumPerSide/2)/(float)(g_iNumPerSide/2) * totalVel;

	}
}

void Reset()
{
	int i, j;
	float fXBase, fYBase;
	float fAdd=1.0/g_iNumPerSide;

	g_bGoneBoom=false;


	for (i=0; i<g_iNumPerSide; i++)
	{
		fYBase=i * fAdd;
		for (j=0; j<g_iNumPerSide; j++)
		{
			fXBase=j * fAdd;
			g_pExplosion[i*g_iNumPerSide + j].vtx[0].Create(-fAdd/2, -fAdd/2, 0, 1, 1, 1, fXBase, (1.0-fYBase));
			g_pExplosion[i*g_iNumPerSide + j].vtx[1].Create(fAdd/2, -fAdd/2, 0, 1, 1, 1, fXBase+fAdd, (1.0-fYBase));
			g_pExplosion[i*g_iNumPerSide + j].vtx[2].Create(-fAdd/2, fAdd/2, 0, 1, 1, 1, fXBase, (1.0-fYBase)-fAdd);
			g_pExplosion[i*g_iNumPerSide + j].vtx[3].Create(fAdd/2, fAdd/2, 0, 1, 1, 1, fXBase+fAdd, (1.0-fYBase)-fAdd);
			g_pExplosion[i*g_iNumPerSide + j].fRotationX=0;
			g_pExplosion[i*g_iNumPerSide + j].fRotationY=0;
			g_pExplosion[i*g_iNumPerSide + j].fRotationZ=0;
			g_pExplosion[i*g_iNumPerSide + j].fX=fXBase+fAdd/2;
			g_pExplosion[i*g_iNumPerSide + j].fY=fYBase+fAdd/2;
			g_pExplosion[i*g_iNumPerSide + j].fRotationModX=0;
			g_pExplosion[i*g_iNumPerSide + j].fRotationModY=0;
			g_pExplosion[i*g_iNumPerSide + j].fRotationModZ=0;
			g_pExplosion[i*g_iNumPerSide + j].fXVel=0;
			g_pExplosion[i*g_iNumPerSide + j].fYVel=0;

		}
	}
}

bool Direct3DInit()
{
	// Seed the random number generator
	srand(GetTickCount());

	// Create the IDirect3D object
	g_pDirect3D=Direct3DCreate8(D3D_SDK_VERSION);
	if (g_pDirect3D==NULL)
		return 0;
	
	// Set up a structure with either the current display mode for windowed mode or the desired display mode for fullscreen
	D3DDISPLAYMODE displayMode;
	if (g_bFullscreen==false)
	{
		g_hr=g_pDirect3D->GetAdapterDisplayMode( D3DADAPTER_DEFAULT, &displayMode);
		if (FAILED(g_hr))
			return 0;
	}
	else
	{
		displayMode.Width=WINDOW_WIDTH;
		displayMode.Height=WINDOW_HEIGHT;
		displayMode.RefreshRate=0;
		displayMode.Format=D3DFMT_R5G6B5;

	}

	// Setup the present parameters
	D3DPRESENT_PARAMETERS presentParameters;
	memset(&presentParameters, 0, sizeof(D3DPRESENT_PARAMETERS));
	if (g_bFullscreen==false)
	{
		presentParameters.Windowed   = TRUE;
	}
	else
	{
		presentParameters.Windowed   = FALSE;
	}

	presentParameters.SwapEffect = D3DSWAPEFFECT_DISCARD;
	presentParameters.BackBufferFormat = displayMode.Format;
	presentParameters.BackBufferWidth = displayMode.Width;
	presentParameters.BackBufferHeight = displayMode.Height;

	// Create the device
	g_hr=g_pDirect3D->CreateDevice( D3DADAPTER_DEFAULT, D3DDEVTYPE_HAL, g_hWnd,
                                  D3DCREATE_SOFTWARE_VERTEXPROCESSING,
                                  &presentParameters, &g_pDevice ); 

	if (FAILED(g_hr))
		return false;

	// Matrix code
	D3DXMATRIX mat;
	
	D3DXMatrixPerspectiveFovLH(&mat, D3DX_PI/6, 1.0, 1.0, 100.0);
	g_pDevice->SetTransform(D3DTS_PROJECTION, &(D3DMATRIX)mat);

	D3DXMatrixIdentity(&mat);
	g_pDevice->SetTransform(D3DTS_WORLD, &(D3DMATRIX)mat);

	D3DXMatrixTranslation(&mat, 0, 0, 10.0);
	g_pDevice->SetTransform(D3DTS_VIEW, &(D3DMATRIX)mat);

	// Setup the explosion
	g_iNumPerSide=16;
	g_iNumParts=g_iNumPerSide*g_iNumPerSide;
	g_pExplosion=new CExplosionPart[g_iNumParts];
	Reset();
	

	// Make it so that it'll draw triangles facing either towards or away from the camera
    g_pDevice->SetRenderState( D3DRS_CULLMODE, D3DCULL_NONE );
	// Turn off Direct3D's lighting since we user our own vertex color
	g_pDevice->SetRenderState( D3DRS_LIGHTING, FALSE );

	// Set it up to blend between the vertex color and the texture color
	g_pDevice->SetTextureStageState(0, D3DTSS_COLOROP, D3DTOP_MODULATE);
	g_pDevice->SetTextureStageState(0, D3DTSS_COLORARG1, D3DTA_TEXTURE);
	g_pDevice->SetTextureStageState(0, D3DTSS_COLORARG2, D3DTA_DIFFUSE);
	// Turn off the alpha channel since the texture doesn't have one
	g_pDevice->SetTextureStageState(0, D3DTSS_ALPHAOP, D3DTOP_DISABLE);

	// Create the texture
	D3DXCreateTextureFromFile(g_pDevice, "Images\\Ship.bmp", &g_pTexture);

	return true;
}

void UpdateScene()
{
	int i;
	static bool wasEnterPressed=false; // Was enter pressed last time
	bool enterPressed;

	for (i=0; i<g_iNumParts; i++)
	{
		g_pExplosion[i].Update();
	}

	if (GetAsyncKeyState(VK_RETURN))
		enterPressed=true;
	else
		enterPressed=false;

	if (enterPressed==false && wasEnterPressed==true)
	{
		// Enter hit
		if (g_bGoneBoom==true)
			Reset();
		else
			Boom();
	}

	wasEnterPressed=enterPressed;
}

void DrawScene()
{
	// Clear the screen
	g_pDevice->Clear( 0, NULL, D3DCLEAR_TARGET, D3DCOLOR_XRGB(0, 0, 0), 1.0f, 0 );
	// Make it so we can draw stuff
	g_pDevice->BeginScene();

	// Tell it what kind of vertices we're using
	g_pDevice->SetVertexShader(D3DFVF_CVertex);
	// Tell it to use our texture
	g_pDevice->SetTexture(0, g_pTexture);

	// Draw the pieces of the explosion
	int i;
	for (i=0; i<g_iNumParts; i++)
	{
		g_pExplosion[i].Draw(g_pDevice);
	}
	
	// Make it so we can't draw stuff
	g_pDevice->EndScene();
	// Put the back buffer onto the screen
	g_pDevice->Present(NULL, NULL, NULL, NULL);
}

void Direct3DRelease()
{
	delete []g_pExplosion;

	// Release the texture
	if (g_pTexture)
		g_pTexture->Release();
	g_pTexture=NULL;

	// Release the device
	if (g_pDevice)
		g_pDevice->Release();
	g_pDevice=NULL;

	// Release the Direct3D object
	if (g_pDirect3D)
		g_pDirect3D->Release();
	g_pDirect3D=NULL;

}

// Windows message processing function
LRESULT CALLBACK WndProc(HWND wpHWnd, 
						    UINT msg, 
                            WPARAM wParam, 
                            LPARAM lParam)
{


	switch(msg)
	{	
		case WM_DESTROY: 
		{ 
			PostQuitMessage(0);
			return 0;
		} break;


		default:break; 
	} 


	return DefWindowProc(wpHWnd, msg, wParam, lParam);
}
 

int WINAPI WinMain(	HINSTANCE hInstance, 
					HINSTANCE hPrevInstance, 
					LPSTR lpCmdLine, 
					int nCmdShow) 
{

	WNDCLASSEX winClass; 
	MSG msg;	

	// Set our global HINSTANCE to the one we get passed
	g_hInst=hInstance;

	// Setup and register the window class
	winClass.cbSize         = sizeof(WNDCLASSEX); 
	winClass.style			= CS_OWNDC | CS_HREDRAW | CS_VREDRAW;
	winClass.lpfnWndProc	= WndProc; 
	winClass.cbClsExtra		= 0;
	winClass.cbWndExtra		= 0; 
	winClass.hInstance		= g_hInst; 
	winClass.hIcon			= LoadIcon(NULL, IDI_APPLICATION); 
	winClass.hCursor		= LoadCursor(NULL, IDC_ARROW); 
	winClass.hbrBackground	= (HBRUSH)GetStockObject(BLACK_BRUSH); 
	winClass.lpszMenuName	= NULL; 
	winClass.lpszClassName	= WINDOW_CLASS_NAME; 
	winClass.hIconSm        = LoadIcon(NULL, IDI_APPLICATION);

	if (!RegisterClassEx(&winClass)) 
		return 0;

	// Ask the user whether or not they want to run in fullscreen
	if (MessageBox(g_hWnd, "Would you like to run in fullscreen?", "Fullscreen?", MB_YESNO)==IDYES)
		g_bFullscreen=true;
	else 
		g_bFullscreen=false;

	if (g_bFullscreen==false)
	{
		// Create a normal window with a border, a caption, and an X button
		g_hWnd = CreateWindowEx(WS_EX_CLIENTEDGE,  
							 WINDOW_CLASS_NAME,     
							 WINDOW_TITLE, 
							 WS_SYSMENU | WS_BORDER | WS_CAPTION | WS_VISIBLE, 
					 		 WINDOW_X, WINDOW_Y ,
							 WINDOW_WIDTH, WINDOW_HEIGHT,
							 NULL,
							 NULL,
							 g_hInst,  
							 NULL);	
	}
	else
	{
		// Create a fullscreen window, one that doesn't have anything in it
		g_hWnd = CreateWindowEx(NULL,  
							 WINDOW_CLASS_NAME,     
							 WINDOW_TITLE, 
							 WS_POPUP | WS_VISIBLE, 
					 		 WINDOW_X, WINDOW_Y ,
							 WINDOW_WIDTH, WINDOW_HEIGHT,
							 NULL,
							 NULL,
							 g_hInst,  
							 NULL);		
		

	}
	
	// Make sure the window was created properly. If it wasn't, quit.
	if (!g_hWnd) 
		return 0; 
	
	

	// Try to initialize the Direct3D parts of our program. If it fails, make sure everything they might have been created gets released.
	if (Direct3DInit()==false)
	{
		Direct3DRelease();
		return 0;
	}

	while(1)
	{
		// Windows message stuff
	    while (PeekMessage(&msg,NULL,0,0,PM_REMOVE))
		{ 
			if (msg.message == WM_QUIT)
				break;
		    TranslateMessage(&msg);
		    DispatchMessage(&msg); 
		}
		
		if (msg.message==WM_QUIT)
			break;

		// Quit if the users presses escape
		if (GetAsyncKeyState(VK_ESCAPE))
			PostQuitMessage(0);

		// Update the scene each frame
		UpdateScene();

		// Draw the scene each frame
		DrawScene();
	} 

	// Release Direct3D
	Direct3DRelease();


	return 0;
} 