// 3dInpDev.h
//
// Copyright (c) Nigel Thompson 1996
//
// This file defines the input device classes
// It is automatically included in 3dPlus.h
//

#ifndef _3DINPDEV_H_
#define _3DINPDEV_H_

// Controller notification flags
#define CONTROL_X_CHANGE		0x00000001
#define CONTROL_Y_CHANGE		0x00000002
#define CONTROL_Z_CHANGE		0x00000004
#define CONTROL_R_CHANGE		0x00000008
#define CONTROL_U_CHANGE		0x00000010
#define CONTROL_V_CHANGE		0x00000020
#define CONTROL_POV_CHANGE		0x00000040
#define CONTROL_BTN_CHANGE		0x00000080
#define CONTROL_NOTIFY_ALL		0xFFFFFFFF

//////////////////////////////////////////////////////////////////
// Input device state structure

typedef struct __3DINPUTSTATE {
	double dX;		// -1 <= value <= 1
	double dY;		// -1 <= value <= 1
	double dZ;		// -1 <= value <= 1
	double dR;		// -1 <= value <= 1
	double dU;		// -1 <= value <= 1
	double dV;		// -1 <= value <= 1
	double dPov;	//  0 <= value <= 359
	                // (< 0 indicates not valid)
	DWORD dwButtons;// 1 = active (pressed)
} _3DINPUTSTATE;

// Called to request a pointer to a 3dFrame object to manipulate
typedef C3dFrame* (CALLBACK * GET3DFRAMEFN)(void *pArg);

// Controller notification function
typedef void (CALLBACK * CONTROLNOTIFYFN)(void* pArg,
										  DWORD dwFlags,
										  const _3DINPUTSTATE* pState);

//////////////////////////////////////////////////////////////////
// C3dInputDevice
//
// All input devices are derived from this class
//
// Note: the ShowConfigDlg() fn is rather pointless unless
// the class also implements serialization so you can save
// and restore the configuration.
//

class C3dInputDevice : public CObject
{
public:
	DECLARE_SERIAL(C3dInputDevice);
    C3dInputDevice();
    virtual ~C3dInputDevice();
	virtual const char* GetName() {return (const char*) m_strName;}
	virtual BOOL ShowConfigDlg();
	virtual BOOL GetState(_3DINPUTSTATE& st);
	virtual void OnUserEvent(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam);

protected:
	CString m_strName;
};
	
//////////////////////////////////////////////////////////////////
// C3dKeyInDev
//

class C3dKeyInDev : public C3dInputDevice
{
public:
	DECLARE_SERIAL(C3dKeyInDev);
    C3dKeyInDev();
    virtual ~C3dKeyInDev();
	virtual BOOL GetState(_3DINPUTSTATE& st);
	virtual void OnUserEvent(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam);
	virtual void OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags);
	virtual void OnKeyUp(UINT nChar, UINT nRepCnt, UINT nFlags);

protected:
	_3DINPUTSTATE m_st;
	BOOL m_bShift;
	BOOL m_bControl;
};

//////////////////////////////////////////////////////////////////
// C3dMouseInDev
//

class C3dMouseInDev : public C3dInputDevice
{
public:
	DECLARE_SERIAL(C3dMouseInDev);
    C3dMouseInDev();
    virtual ~C3dMouseInDev();
	virtual void OnUserEvent(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam);
	virtual BOOL GetState(_3DINPUTSTATE& st);

protected:
	_3DINPUTSTATE m_st;
	CPoint m_ptPrev;
	CPoint m_ptCur;
	DWORD m_dwFlags;
	BOOL m_bCaptured;
};

//////////////////////////////////////////////////////////////////
// C3dJoyInDev
//

// data structures for joystick configuration
typedef struct _JOYAXIS {
	int iInput;				// input chanel 0=X, 1=Y, 2=Z, 3=R, 4=U, 5=V
	int iButton;			// non-zero if button required
	int iDead;				// dead zone as percentage of range
	double dSens;			// sensitivity. 1.0 is norm.
} JOYAXIS;

typedef struct _JOYCONFIG {
	JOYAXIS axis[6];
} JOYCONFIG;

class C3dJoyInDev : public C3dInputDevice
{
public:
	DECLARE_SERIAL(C3dJoyInDev);
    C3dJoyInDev();
    virtual ~C3dJoyInDev();
	virtual BOOL GetState(_3DINPUTSTATE& st);
	virtual BOOL ShowConfigDlg();
	JOYCAPS* GetCaps() {return &m_caps;}
	JOYINFOEX* GetInfo();
	JOYCONFIG* GetConfig() {return &m_cfg;}
	BOOL IsPresent() {return m_bDevPresent;}

protected:
	BOOL m_bDevPresent;
	JOYCAPS m_caps;
	JOYINFOEX m_inf;
	JOYCONFIG m_cfg;
};

//////////////////////////////////////////////////////////////////
// C3dController
//

class C3dController : public CObject
{
public:
	DECLARE_SERIAL(C3dController);
	C3dController(); // used for serialization only
    BOOL Create(C3dWnd* pWnd,
				GET3DFRAMEFN pGet3dFrameFn,
				void* pArg = NULL,
				CONTROLNOTIFYFN pNotifyFn = NULL,
				DWORD dwNotify = CONTROL_NOTIFY_ALL);
    virtual ~C3dController();
	virtual BOOL ShowConfigDlg();
	virtual void Update();
	virtual void OnUserEvent(HWND hWnd, UINT uiMsg, WPARAM wParam, LPARAM lParam);
	virtual int GetNumDevices() {return m_iDevs;}
	virtual BOOL SelectDevice(int i);
	virtual const char* GetDeviceName(int i);
	int GetCurDevice() {return m_iCurDevice;}
	C3dInputDevice* GetInputDevice() {return m_pInpDev;}

protected:
	GET3DFRAMEFN m_pGet3dFrameFn;
	void* m_pArg;
	CONTROLNOTIFYFN m_pNotifyFn;
	DWORD m_dwNotify;
	C3dInputDevice* m_pInpDev;
	_3DINPUTSTATE m_PrevState;
	C3dWnd* m_pWnd;
	int m_iCurDevice;

	virtual void OnUpdate(_3DINPUTSTATE& st, C3dFrame* pFrame);
	BOOL _InRange(double x, double min, double max)
	{return ((x >= min) && (x <= max));}

private:
	int m_iDevs;
	C3dJoyInDev m_JoyDev;
	C3dMouseInDev m_MouseDev;
	C3dKeyInDev m_KeyDev;
};

//////////////////////////////////////////////////////////////////
// C3dPosCtlr - positioning controller
//

class C3dPosCtlr : public C3dController
{
public:
	DECLARE_SERIAL(C3dPosCtlr);

protected:
	virtual void OnUpdate(_3DINPUTSTATE& st, C3dFrame* pFrame);
};

//////////////////////////////////////////////////////////////////
// C3dFlyCtlr - flying controller
//

class C3dFlyCtlr : public C3dController
{
public:
	DECLARE_SERIAL(C3dFlyCtlr);

protected:
	virtual void OnUpdate(_3DINPUTSTATE& st, C3dFrame* pFrame);
};

#endif // _3DINPDEV_H_

