/*
 * Copyright (c) 2008-2022 Haulmont. All rights reserved.
 * Use is subject to license terms, see http://www.cuba-platform.com/commercial-software-license for details.
 */

function Connector(modeler) {
    const connector = this;

    const eventBus = modeler['_eventBus'];

    function on(eventType, handler) {
        modeler.on(eventType, (event) => {
            handler(event);
        });
    }

    connector.selectElement = function (elementId) {
        var activeView = modeler.getActiveView();

        if (activeView.type === 'drd') {
            var activeEditor = modeler.getActiveViewer();

            if (connector.importing || !activeEditor) {
                connector.selectionId = elementId;
                return;
            }

            // access active editor components
            var canvas = activeEditor.get('canvas');
            let elementRegistry = activeEditor.get('elementRegistry');
            let selection = activeEditor.get('selection');

            let element = elementRegistry.get(elementId);
            if (element) selection.select(element);
        }
    }

    let firstUpdate = true; // todo move to separate command
    connector.setDmnXml = async function (dmnXml) {
        try {
            let oldViewbox = connector.viewbox;
            let oldSelection = connector.selection;
            let oldSelectionId = connector.selectionId;
            let oldActiveViewType = connector.activeViewType;

            const result = await modeler.importXML(dmnXml);
            const { warnings } = result;
            if (warnings.length) console.log(JSON.stringify(warnings));

            if (oldActiveViewType) {
                try {
                    modeler.open(oldActiveViewType);
                } catch(e) {}
            }

            // fetch currently active view
            var activeView = modeler.getActiveView();

            // apply initial logic in DRD view
            if (activeView.type === 'drd') {
                var activeEditor = modeler.getActiveViewer();

                // access active editor components
                var canvas = activeEditor.get('canvas');

                if (firstUpdate) {
                    // zoom to fit full viewport
                    canvas.zoom('fit-viewport');
                }

                if (oldViewbox) {
                    canvas.viewbox(oldViewbox);
                }

                if (oldSelection)
                    connector.selectElement(oldSelection.id);
                else if (oldSelectionId)
                    connector.selectElement(oldSelectionId);
            }
        } catch (err) {
            return console.error('could not import DMN 1.1 diagram', err);
        }
    };

    on(
        'viewer.created',
        ({viewer}) => {
            viewer.on(
                'elements.changed',
                ({elements}) => {
                    modeler.saveXML({format: true})
                        .then(result => {
                                sendCefQuery("SCHEMA_UPDATE", result.xml);
                                let selection = connector.selection;
                                sendCefQuery(
                                    "ELEMENT_SELECTED",
                                    {
                                        'newSelection': selection && selection.id,
                                        'oldSelection': selection && selection.id,
                                        'requestor': 'dmnJs'
                                    }
                                );
                            }
                        )
                        .catch(err => console.error('could not save BPMN 2.0 diagram', err));
                }
            );

            viewer.on("selection.changed", function (e) {
                let newSelection = e.newSelection && e.newSelection[0] && e.newSelection[0].businessObject;
                let oldSelection = e.oldSelection && e.oldSelection[0] && e.oldSelection[0].businessObject;
                connector.selection = newSelection;
                let attrs = newSelection && newSelection['$attrs'];
                if (attrs) {
                    for (let key in attrs) {
                        newSelection[key] = attrs[key];
                    }
                }
                sendCefQuery(
                    "ELEMENT_SELECTED",
                     {
                        'newSelection': newSelection && newSelection.id,
                        'oldSelection': oldSelection && oldSelection.id,
                        'requestor': 'dmnJs'
                     }
                 );
            });

            viewer.on("canvas.viewbox.changed", function (e) {
                connector.viewbox = e.viewbox;
            });
        }
    );

    on("changed", function (e) {
        sendCefQuery("SCHEMA_UPDATE", modeler.saveSchema());
    });

    on("import.parse.start", function (e) {
        connector.importing = true;
    });

    on("import.done", function (e) {
        connector.importing = false;
        sendCefQuery("import.done", '');
    });

    on("views.changed", function (e) {
        connector.activeViewType = e.activeView.type;
    });

    (async function () {
        sendCefQuery("INITIALIZATION");
    })();
}

function sendCefQuery(id, payload) {
    if (payload == null)
    payload = "";

    let query = {
        request: JSON.stringify(
            {
                "id": id,
                "payload": (typeof payload === "string" || payload instanceof String) ? payload : JSON.stringify(payload)
            }
        ),
        onSuccess: function (response) {
            console.log(response);
        },
        onFailure: function (error_code, error_message) {
            console.log("Error: (" + error_code + ") " + error_message);
        }
    };

    window.cefQueryBpm(query);
}