/*
 * Copyright (c) 2008-2022 Haulmont. All rights reserved.
 * Use is subject to license terms, see http://www.cuba-platform.com/commercial-software-license for details.
 */

function Connector(modeler) {
    const connector = this;

    const eventBus = modeler.get('eventBus');
    const canvas = modeler.get('canvas');
    const selection = modeler.get('selection');
    const bpmModeling = modeler.get('bpmModeling');
    const elementRegistry = modeler.get('elementRegistry');
    const dragging = modeler.get('dragging');
    const tasks = new Set();

    let silentMode = false;

    function doSilently(action) {
        silentMode = true;
        try {
            action();
        } finally {
            silentMode = false;
        }
    }

    function on(eventType, handler) {
        eventBus.on(eventType, (event) => {
            if (!silentMode)
                handler(event);
        });
    }

    let firstUpdate = true; // todo move to separate command
    connector.setBpmnXml = function (bpmnXml) {
        doSilently(() => {
            let selectedIds = selection.get().map(element => element.id);
            modeler.importXML(bpmnXml)
                .then(function () {
                    selection.select(elementRegistry.filter(element => selectedIds.includes(element.id)));
                    if (firstUpdate) {
                        canvas.zoom('fit-viewport');
                        firstUpdate = false;
                    }
                })
                .catch(function (err) {
                    console.error('could not import BPMN 2.0 diagram', err);
                })
        })
    };

    connector.selectElement = function (elementId) {
        doSilently(() => {
            selection.deselect(selection.get());
            selection.select(elementRegistry.filter(element => elementId === element.id));
        });
        notifySelection(selection.get()[0], "connector")
    }

    connector.reselectElement = function() {
        let element = selection.get()[0];
        doSilently(() => {
            selection.deselect(selection.get());
            selection.select(element);
        });
        notifySelection(selection.get()[0], "connector")
    }

    connector.updateElementProperties = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.updateElementProperties(cmd.elementId, cmd.elementProperties);
    };

    connector.createAndSetListProperty = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.createAndSetListProperty(cmd);
    };

    connector.createOrUpdateNestedObject = function (cmdJson) {
        const {elementId, nestedPropertyName, nestedElementType, objectProperties} = JSON.parse(cmdJson)
        bpmModeling.createOrUpdateNestedObject(elementId, nestedPropertyName, nestedElementType, objectProperties)
    }

    connector.callModeling = function (methodName, cmdJson) {
        const cmd = JSON.parse(cmdJson);
        bpmModeling[methodName](cmd);
    }

    connector.createOrUpdateExtensionElement = function (cmdJson) {
        const cmd = JSON.parse(cmdJson)
        bpmModeling.createOrUpdateExtensionElement(cmd)
    }

    connector.createOrUpdateEventDefinitionExtensionElement = function (cmdJson) {
        const cmd = JSON.parse(cmdJson)
        bpmModeling.createOrUpdateEventDefinitionExtensionElement(cmd)
    }

    connector.createOrUpdateExtensionProperty = function (cmdJson) {
        const cmd = JSON.parse(cmdJson)
        bpmModeling.createOrUpdateExtensionProperty(cmd)
    }

    connector.removeExtensionProperty = function (cmdJson) {
        const cmd = JSON.parse(cmdJson)
        bpmModeling.removeExtensionProperty(cmd)
    }

    connector.createOrUpdateProcessVariable = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.createOrUpdateProcessVariable(cmd.elementId, cmd.variableName, cmd.processVariableDTO);
    }

    connector.createOrUpdateInputParameterData = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.createOrUpdateInputParameterData(cmd.elementId, cmd.variableName, cmd.parameterDataDTO);
    }

    connector.createOrUpdateOutputParameterData = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.createOrUpdateOutputParameterData(cmd.elementId, cmd.variableName, cmd.parameterDataDTO);
    }

    connector.createOrUpdateEventDefinitionInputParameterData = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.createOrUpdateEventDefinitionInputParameterData(cmd.elementId, cmd.variableName, cmd.parameterDataDTO, cmd.eventDefinitionType);
    }

    connector.createOrUpdateEventDefinitionOutputParameterData = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.createOrUpdateEventDefinitionOutputParameterData(cmd.elementId, cmd.variableName, cmd.parameterDataDTO, cmd.eventDefinitionType);
    }

    connector.createOrUpdateGeneratedFormData = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.createOrUpdateGeneratedFormData(cmd.elementId, cmd.formFieldId, cmd.formFieldDataDTO);
    }

    connector.updateServiceTaskConnectorData = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.updateServiceTaskConnectorData(cmd.elementId, cmd.connectorId);
    }

    connector.updateEventDefinitionConnectorData = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.updateEventDefinitionConnectorData(cmd);
    }

    connector.removeEventDefinitionConnectorData = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.removeEventDefinitionConnectorData(cmd);
    }

    connector.removeExtensionElements = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.removeExtensionElements(cmd);
    }

    connector.removeEventDefinitionExtensionElements = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.removeEventDefinitionExtensionElements(cmd);
    }

    connector.removeProcessVariable = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.removeProcessVariable(cmd);
    };

    connector.removeInputParameterData = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.removeInputParameterData(cmd);
    };

    connector.removeOutputParameterData = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.removeOutputParameterData(cmd);
    };

    connector.removeEventDefinitionInputParameterData = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.removeEventDefinitionInputParameterData(cmd);
    };

    connector.removeEventDefinitionOutputParameterData = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.removeEventDefinitionOutputParameterData(cmd);
    };

    connector.removeGeneratedFormFieldData = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.removeGeneratedFormFieldData(cmd);
    };

        connector.removeConnectorData = function (cmdJson) {
            let cmd = JSON.parse(cmdJson);
            bpmModeling.removeConnectorData(cmd);
        };

    connector.setMultiInstanceFormalExpression = function (cmdJson) {
        const {elementId, propertyName, propertyValue} = JSON.parse(cmdJson);
        bpmModeling.setMultiInstanceFormalExpression(elementId, propertyName, propertyValue)
    }

    connector.createOrUpdateMultiInstanceExtensionElement = function (cmdJson) {
        const cmd = JSON.parse(cmdJson);
        bpmModeling.createOrUpdateMultiInstanceExtensionElement(cmd)
    }

    connector.removeMultiInstanceExtensionElements = function (cmdJson) {
        const cmd = JSON.parse(cmdJson);
        bpmModeling.removeMultiInstanceExtensionElements(cmd.elementId, cmd.extensionElementType)
    }

    connector.createOrUpdateSpringBean = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.createOrUpdateSpringBean(cmd);
    }

    connector.removeSpringBean = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.removeSpringBean(cmd);
    };

    connector.createOrUpdateFormData = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.createOrUpdateFormData(cmd.elementId, cmd.formData);
    };

    connector.createOrUpdateFormDefinition = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.createOrUpdateFormDefinition(cmd.elementId, cmd.formDefinition);
    };

    connector.createListener = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.createListener(cmd);
    };

    connector.removeExtensionElements = function (cmdJson) {
        let cmd = JSON.parse(cmdJson);
        bpmModeling.removeExtensionElements(cmd);
    };

    connector.createOrUpdateRootElement = function (cmdJson) {
        const {elementId, elementType, properties} = JSON.parse(cmdJson);
        bpmModeling.createOrUpdateRootElement(modeler.getDefinitions(), elementId, elementType, properties);
    };

    connector.removeRootElement = function (cmdJson) {
        const {elementId, elementType} = JSON.parse(cmdJson);
        bpmModeling.removeRootElement(modeler.getDefinitions(), elementId, elementType);
    };

    connector.updateEventDefinition = function (cmdJson) {
        const cmd = JSON.parse(cmdJson);
        bpmModeling.updateEventDefinition(cmd)
    }

    connector.updateEventDefinitionProperties = function (cmdJson) {
        const cmd = JSON.parse(cmdJson);
        bpmModeling.updateEventDefinitionProperties(cmd)
    }

    connector.updateTimerEventDefinition = function (cmdJson) {
        const {elementId, timerDefinitionType, value} = JSON.parse(cmdJson);
        bpmModeling.updateTimerEventDefinition(elementId, timerDefinitionType, value)
    }

    connector.updateConditionalEventDefinition = function (cmdJson) {
        const {elementId, language, resource, body} = JSON.parse(cmdJson);
        bpmModeling.updateConditionalEventDefinition(elementId, language, resource, body)
    }

    connector.updateParticipantProcessProperties = function (cmdJson) {
        const cmd = JSON.parse(cmdJson);
        bpmModeling.updateParticipantProcessProperties(cmd.participantElementId, cmd.processProperties)
    }

    connector.createOrUpdateParticipantProcessExtensionElement = function (cmdJson) {
        const cmd = JSON.parse(cmdJson);
        bpmModeling.createOrUpdateParticipantProcessExtensionElement(cmd)
    }

    connector.removeParticipantProcessExtensionElements = function (cmdJson) {
        const cmd = JSON.parse(cmdJson);
        bpmModeling.removeParticipantProcessExtensionElements(cmd.elementId, cmd.extensionElementType)
    }

    connector.updateProcessDocumentation = function (cmdJson) {
        const cmd = JSON.parse(cmdJson);
        bpmModeling.updateProcessDocumentation(cmd)
    }

    on('element.changed', function (e) {
        modeler.saveXML({format: true})
            .then(function (result) {
                sendCefQuery("SCHEMA_UPDATE", result.xml);
            })
            .catch(function (err) {
                console.error('could not save BPMN 2.0 diagram', err);
            })
    });

    /**
     * @param requestor it is a hack to prevent cycle callbacks
     */
    function notifySelection(newElement, requestor) {
        if (typeof newElement === 'undefined') {
            newElement = dragging.context()?.data?.shape;
            if (typeof newElement === 'undefined') {
                newElement = canvas.getRootElement();
            }
        }
        var bo = newElement.businessObject;

        //some properties must be marked as enumerable, otherwise they are not serialized to JSON
        function fixBusinessObject(bo) {
            function fixDefinitions(process) {
                if (process) {
                    const definitions = modeler.getDefinitions()['rootElements'];
                    if (definitions) {
                        process.signalDefinitions = definitions.filter(x => x.$type === 'bpmn:Signal')
                        process.messageDefinitions = definitions.filter(x => x.$type === 'bpmn:Message')
                        process.errorDefinitions = definitions.filter(x => x.$type === 'bpmn:Error')
                        process.escalationDefinitions = definitions.filter(x => x.$type === 'bpmn:Escalation')
                    }
                }
            }

            if (bo.eventDefinitions) {
                var eventDefinition = bo.eventDefinitions[0];
                if (eventDefinition) {
                    var propertiesThatMustBeEnumerable = ["signalRef", "messageRef", "errorRef", "escalationRef", "activityRef"];
                    for (var i = 0; i < propertiesThatMustBeEnumerable.length; i++) {
                        var propertyName = propertiesThatMustBeEnumerable[i];
                        if (eventDefinition[propertyName]) {
                            Object.defineProperty(eventDefinition, propertyName, {enumerable: true});
                        }
                    }
                }
            } else if (bo.messageRef) {
                Object.defineProperty(bo, 'messageRef', {enumerable: true});
            } else if (bo.extensionElements && bo.extensionElements.values) {
                for (let extensionElement of bo.extensionElements.values) {
                    if (extensionElement.errorRef)
                        Object.defineProperty(extensionElement, "errorRef", {enumerable: true});
                }
            }

            if (bo.$type === 'bpmn:BoundaryEvent') {
                Object.defineProperty(bo, 'attachedToRef', {enumerable: true})
            } else if (bo.$type === 'bpmn:Participant') {
                Object.defineProperty(bo, 'processRef', {enumerable: true})
                Object.defineProperty(bo, 'di', {enumerable: false})
                fixDefinitions(bo['processRef']);
            } else if (bo.$type === 'bpmn:Process') {
                fixDefinitions(bo);
            } else if (bo.$type === 'bpmn:SequenceFlow') {
                bo.sourceReference = bo["sourceRef"].id
                bo.targetReference = bo["targetRef"].id
                bo.isNewConnection = elementRegistry.getAll().find(e => e.id === bo.id).isNewConnection
            } else if(bo.$type === 'bpmn:UserTask') {
                bo.isNewTask = elementRegistry.getAll().find(e => e.id === bo.id).isNewTask
            }
            return bo;
        }

        if (bo) {
            fixBusinessObject(bo);
            sendCefQuery("ELEMENT_SELECTED", {
                "bo": bo,
                "requestor": requestor,
                "allElements": elementRegistry.getAll().map((element) => fixBusinessObject(element.businessObject))
            });
        }
    }

    on('connection.added', function (event) {
        elementRegistry.getAll().forEach(e => {
            delete e.isNewConnection
        })
        const connection = elementRegistry.getAll().find(e => e.id === event.element.id);
        if (connection) {
            connection.isNewConnection = true
        }
    });

    on('connection.changed', function (event) {
        const connection = elementRegistry.getAll().find(e => e.isNewConnection && e.condition);
        if (connection) {
            delete connection.isNewConnection
        }
    });

    on('shape.added', function (event) {
        if (event.element.type !== "bpmn:UserTask") return
        const taskId = event.element.id
        tasks.add(taskId)
    });

    // Storing ids into set handles
    // Receive double changed events
    on('shape.changed', function (event) {
        if (event.element.type !== "bpmn:UserTask") return

        const taskId = event.element.id

        const taskElement = elementRegistry.getAll().find(e => e.id === taskId)

        if (tasks.has(taskId) && taskElement) {
            delete taskElement.isNewTask
            return
        }

        if (!taskElement) throw new Error(`Element with id ${taskId} not found`)

        taskElement.isNewTask = true
        tasks.add(taskId)
    });

    on("selection.changed", function (e) {
        var newElement = e.newSelection[0];
        notifySelection(newElement, "bpmnjs");
    });

    (async function () {
        sendCefQuery("INITIALIZATION");
    })();
}

function sendCefQuery(id, payload) {
    if (payload == null)
        payload = "";

    let query = {
        request: JSON.stringify(
            {
                "id": id,
                "payload": (typeof payload === "string" || payload instanceof String) ? payload : JSON.stringify(payload)
            }
        ),
        onSuccess: function (response) {
            console.log(response);
        },
        onFailure: function (error_code, error_message) {
            console.log("Error: (" + error_code + ") " + error_message);
        }
    };

    window.cefQueryBpm(query);
}