/*!
 * dmn-js - dmn-modeler v17.3.0
 *
 * Copyright (c) 2014-present, camunda Services GmbH
 *
 * Released under the bpmn.io license
 * http://bpmn.io/license
 *
 * Source Code: https://github.com/bpmn-io/dmn-js
 *
 * Date: 2025-06-30
 */
(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory() :
  typeof define === 'function' && define.amd ? define(factory) :
  (global = typeof globalThis !== 'undefined' ? globalThis : global || self, global.DmnJS = factory());
})(this, (function () { 'use strict';

  /**
   * Flatten array, one level deep.
   *
   * @template T
   *
   * @param {T[][] | T[] | null} [arr]
   *
   * @return {T[]}
   */
  function flatten$1(arr) {
    return Array.prototype.concat.apply([], arr);
  }
  const nativeToString$4 = Object.prototype.toString;
  const nativeHasOwnProperty$3 = Object.prototype.hasOwnProperty;
  function isUndefined$5(obj) {
    return obj === undefined;
  }
  function isDefined(obj) {
    return obj !== undefined;
  }
  function isNil(obj) {
    return obj == null;
  }
  function isArray$6(obj) {
    return nativeToString$4.call(obj) === '[object Array]';
  }
  function isObject$2(obj) {
    return nativeToString$4.call(obj) === '[object Object]';
  }
  function isNumber$3(obj) {
    return nativeToString$4.call(obj) === '[object Number]';
  }

  /**
   * @param {any} obj
   *
   * @return {boolean}
   */
  function isFunction$2(obj) {
    const tag = nativeToString$4.call(obj);
    return tag === '[object Function]' || tag === '[object AsyncFunction]' || tag === '[object GeneratorFunction]' || tag === '[object AsyncGeneratorFunction]' || tag === '[object Proxy]';
  }
  function isString$4(obj) {
    return nativeToString$4.call(obj) === '[object String]';
  }

  /**
   * Return true, if target owns a property with the given key.
   *
   * @param {Object} target
   * @param {String} key
   *
   * @return {Boolean}
   */
  function has$3(target, key) {
    return !isNil(target) && nativeHasOwnProperty$3.call(target, key);
  }

  /**
   * @template T
   * @typedef { (
   *   ((e: T) => boolean) |
   *   ((e: T, idx: number) => boolean) |
   *   ((e: T, key: string) => boolean) |
   *   string |
   *   number
   * ) } Matcher
   */

  /**
   * @template T
   * @template U
   *
   * @typedef { (
   *   ((e: T) => U) | string | number
   * ) } Extractor
   */

  /**
   * @template T
   * @typedef { (val: T, key: any) => boolean } MatchFn
   */

  /**
   * @template T
   * @typedef { T[] } ArrayCollection
   */

  /**
   * @template T
   * @typedef { { [key: string]: T } } StringKeyValueCollection
   */

  /**
   * @template T
   * @typedef { { [key: number]: T } } NumberKeyValueCollection
   */

  /**
   * @template T
   * @typedef { StringKeyValueCollection<T> | NumberKeyValueCollection<T> } KeyValueCollection
   */

  /**
   * @template T
   * @typedef { KeyValueCollection<T> | ArrayCollection<T> } Collection
   */

  /**
   * Find element in collection.
   *
   * @template T
   * @param {Collection<T>} collection
   * @param {Matcher<T>} matcher
   *
   * @return {Object}
   */
  function find$2(collection, matcher) {
    const matchFn = toMatcher$1(matcher);
    let match;
    forEach$3(collection, function (val, key) {
      if (matchFn(val, key)) {
        match = val;
        return false;
      }
    });
    return match;
  }

  /**
   * Filter elements in collection.
   *
   * @template T
   * @param {Collection<T>} collection
   * @param {Matcher<T>} matcher
   *
   * @return {T[]} result
   */
  function filter$2(collection, matcher) {
    const matchFn = toMatcher$1(matcher);
    let result = [];
    forEach$3(collection, function (val, key) {
      if (matchFn(val, key)) {
        result.push(val);
      }
    });
    return result;
  }

  /**
   * Iterate over collection; returning something
   * (non-undefined) will stop iteration.
   *
   * @template T
   * @param {Collection<T>} collection
   * @param { ((item: T, idx: number) => (boolean|void)) | ((item: T, key: string) => (boolean|void)) } iterator
   *
   * @return {T} return result that stopped the iteration
   */
  function forEach$3(collection, iterator) {
    let val, result;
    if (isUndefined$5(collection)) {
      return;
    }
    const convertKey = isArray$6(collection) ? toNum$3 : identity$3;
    for (let key in collection) {
      if (has$3(collection, key)) {
        val = collection[key];
        result = iterator(val, convertKey(key));
        if (result === false) {
          return val;
        }
      }
    }
  }

  /**
   * Reduce collection, returning a single result.
   *
   * @template T
   * @template V
   *
   * @param {Collection<T>} collection
   * @param {(result: V, entry: T, index: any) => V} iterator
   * @param {V} result
   *
   * @return {V} result returned from last iterator
   */
  function reduce(collection, iterator, result) {
    forEach$3(collection, function (value, idx) {
      result = iterator(result, value, idx);
    });
    return result;
  }

  /**
   * Return true if every element in the collection
   * matches the criteria.
   *
   * @param  {Object|Array} collection
   * @param  {Function} matcher
   *
   * @return {Boolean}
   */
  function every(collection, matcher) {
    return !!reduce(collection, function (matches, val, key) {
      return matches && matcher(val, key);
    }, true);
  }

  /**
   * Return true if some elements in the collection
   * match the criteria.
   *
   * @param  {Object|Array} collection
   * @param  {Function} matcher
   *
   * @return {Boolean}
   */
  function some(collection, matcher) {
    return !!find$2(collection, matcher);
  }

  /**
   * Transform a collection into another collection
   * by piping each member through the given fn.
   *
   * @param  {Object|Array}   collection
   * @param  {Function} fn
   *
   * @return {Array} transformed collection
   */
  function map$3(collection, fn) {
    let result = [];
    forEach$3(collection, function (val, key) {
      result.push(fn(val, key));
    });
    return result;
  }

  /**
   * Get the collections keys.
   *
   * @param  {Object|Array} collection
   *
   * @return {Array}
   */
  function keys(collection) {
    return collection && Object.keys(collection) || [];
  }

  /**
   * Shorthand for `keys(o).length`.
   *
   * @param  {Object|Array} collection
   *
   * @return {Number}
   */
  function size(collection) {
    return keys(collection).length;
  }

  /**
   * Get the values in the collection.
   *
   * @param  {Object|Array} collection
   *
   * @return {Array}
   */
  function values$1(collection) {
    return map$3(collection, val => val);
  }

  /**
   * Group collection members by attribute.
   *
   * @param {Object|Array} collection
   * @param {Extractor} extractor
   *
   * @return {Object} map with { attrValue => [ a, b, c ] }
   */
  function groupBy(collection, extractor, grouped = {}) {
    extractor = toExtractor(extractor);
    forEach$3(collection, function (val) {
      let discriminator = extractor(val) || '_';
      let group = grouped[discriminator];
      if (!group) {
        group = grouped[discriminator] = [];
      }
      group.push(val);
    });
    return grouped;
  }
  function uniqueBy(extractor, ...collections) {
    extractor = toExtractor(extractor);
    let grouped = {};
    forEach$3(collections, c => groupBy(c, extractor, grouped));
    let result = map$3(grouped, function (val, key) {
      return val[0];
    });
    return result;
  }

  /**
   * Sort collection by criteria.
   *
   * @template T
   *
   * @param {Collection<T>} collection
   * @param {Extractor<T, number | string>} extractor
   *
   * @return {Array}
   */
  function sortBy(collection, extractor) {
    extractor = toExtractor(extractor);
    let sorted = [];
    forEach$3(collection, function (value, key) {
      let disc = extractor(value, key);
      let entry = {
        d: disc,
        v: value
      };
      for (var idx = 0; idx < sorted.length; idx++) {
        let {
          d
        } = sorted[idx];
        if (disc < d) {
          sorted.splice(idx, 0, entry);
          return;
        }
      }

      // not inserted, append (!)
      sorted.push(entry);
    });
    return map$3(sorted, e => e.v);
  }

  /**
   * Create an object pattern matcher.
   *
   * @example
   *
   * ```javascript
   * const matcher = matchPattern({ id: 1 });
   *
   * let element = find(elements, matcher);
   * ```
   *
   * @template T
   *
   * @param {T} pattern
   *
   * @return { (el: any) =>  boolean } matcherFn
   */
  function matchPattern(pattern) {
    return function (el) {
      return every(pattern, function (val, key) {
        return el[key] === val;
      });
    };
  }

  /**
   * @param {string | ((e: any) => any) } extractor
   *
   * @return { (e: any) => any }
   */
  function toExtractor(extractor) {
    /**
     * @satisfies { (e: any) => any }
     */
    return isFunction$2(extractor) ? extractor : e => {
      // @ts-ignore: just works
      return e[extractor];
    };
  }

  /**
   * @template T
   * @param {Matcher<T>} matcher
   *
   * @return {MatchFn<T>}
   */
  function toMatcher$1(matcher) {
    return isFunction$2(matcher) ? matcher : e => {
      return e === matcher;
    };
  }
  function identity$3(arg) {
    return arg;
  }
  function toNum$3(arg) {
    return Number(arg);
  }

  /* global setTimeout clearTimeout */

  /**
   * @typedef { {
   *   (...args: any[]): any;
   *   flush: () => void;
   *   cancel: () => void;
   * } } DebouncedFunction
   */

  /**
   * Debounce fn, calling it only once if the given time
   * elapsed between calls.
   *
   * Lodash-style the function exposes methods to `#clear`
   * and `#flush` to control internal behavior.
   *
   * @param  {Function} fn
   * @param  {Number} timeout
   *
   * @return {DebouncedFunction} debounced function
   */
  function debounce(fn, timeout) {
    let timer;
    let lastArgs;
    let lastThis;
    let lastNow;
    function fire(force) {
      let now = Date.now();
      let scheduledDiff = force ? 0 : lastNow + timeout - now;
      if (scheduledDiff > 0) {
        return schedule(scheduledDiff);
      }
      fn.apply(lastThis, lastArgs);
      clear();
    }
    function schedule(timeout) {
      timer = setTimeout(fire, timeout);
    }
    function clear() {
      if (timer) {
        clearTimeout(timer);
      }
      timer = lastNow = lastArgs = lastThis = undefined;
    }
    function flush() {
      if (timer) {
        fire(true);
      }
      clear();
    }

    /**
     * @type { DebouncedFunction }
     */
    function callback(...args) {
      lastNow = Date.now();
      lastArgs = args;
      lastThis = this;

      // ensure an execution is scheduled
      if (!timer) {
        schedule(timeout);
      }
    }
    callback.flush = flush;
    callback.cancel = clear;
    return callback;
  }

  /**
   * Throttle fn, calling at most once
   * in the given interval.
   *
   * @param  {Function} fn
   * @param  {Number} interval
   *
   * @return {Function} throttled function
   */
  function throttle(fn, interval) {
    let throttling = false;
    return function (...args) {
      if (throttling) {
        return;
      }
      fn(...args);
      throttling = true;
      setTimeout(() => {
        throttling = false;
      }, interval);
    };
  }

  /**
   * Bind function against target <this>.
   *
   * @param  {Function} fn
   * @param  {Object}   target
   *
   * @return {Function} bound function
   */
  function bind$3(fn, target) {
    return fn.bind(target);
  }

  /**
   * Convenience wrapper for `Object.assign`.
   *
   * @param {Object} target
   * @param {...Object} others
   *
   * @return {Object} the target
   */
  function assign$4(target, ...others) {
    return Object.assign(target, ...others);
  }

  /**
   * Pick properties from the given target.
   *
   * @template T
   * @template {any[]} V
   *
   * @param {T} target
   * @param {V} properties
   *
   * @return Pick<T, V>
   */
  function pick$1(target, properties) {
    let result = {};
    let obj = Object(target);
    forEach$3(properties, function (prop) {
      if (prop in obj) {
        result[prop] = target[prop];
      }
    });
    return result;
  }

  /**
   * Pick all target properties, excluding the given ones.
   *
   * @template T
   * @template {any[]} V
   *
   * @param {T} target
   * @param {V} properties
   *
   * @return {Omit<T, V>} target
   */
  function omit(target, properties) {
    let result = {};
    let obj = Object(target);
    forEach$3(obj, function (prop, key) {
      if (properties.indexOf(key) === -1) {
        result[key] = prop;
      }
    });
    return result;
  }

  var FN_REF = '__fn';
  var DEFAULT_PRIORITY$8 = 1000;
  var slice = Array.prototype.slice;

  /**
   * @typedef { {
   *   stopPropagation(): void;
   *   preventDefault(): void;
   *   cancelBubble: boolean;
   *   defaultPrevented: boolean;
   *   returnValue: any;
   * } } Event
   */

  /**
   * @template E
   *
   * @typedef { (event: E & Event, ...any) => any } EventBusEventCallback
   */

  /**
   * @typedef { {
   *  priority: number;
   *  next: EventBusListener | null;
   *  callback: EventBusEventCallback<any>;
   * } } EventBusListener
   */

  /**
   * A general purpose event bus.
   *
   * This component is used to communicate across a diagram instance.
   * Other parts of a diagram can use it to listen to and broadcast events.
   *
   *
   * ## Registering for Events
   *
   * The event bus provides the {@link EventBus#on} and {@link EventBus#once}
   * methods to register for events. {@link EventBus#off} can be used to
   * remove event registrations. Listeners receive an instance of {@link Event}
   * as the first argument. It allows them to hook into the event execution.
   *
   * ```javascript
   *
   * // listen for event
   * eventBus.on('foo', function(event) {
   *
   *   // access event type
   *   event.type; // 'foo'
   *
   *   // stop propagation to other listeners
   *   event.stopPropagation();
   *
   *   // prevent event default
   *   event.preventDefault();
   * });
   *
   * // listen for event with custom payload
   * eventBus.on('bar', function(event, payload) {
   *   console.log(payload);
   * });
   *
   * // listen for event returning value
   * eventBus.on('foobar', function(event) {
   *
   *   // stop event propagation + prevent default
   *   return false;
   *
   *   // stop event propagation + return custom result
   *   return {
   *     complex: 'listening result'
   *   };
   * });
   *
   *
   * // listen with custom priority (default=1000, higher is better)
   * eventBus.on('priorityfoo', 1500, function(event) {
   *   console.log('invoked first!');
   * });
   *
   *
   * // listen for event and pass the context (`this`)
   * eventBus.on('foobar', function(event) {
   *   this.foo();
   * }, this);
   * ```
   *
   *
   * ## Emitting Events
   *
   * Events can be emitted via the event bus using {@link EventBus#fire}.
   *
   * ```javascript
   *
   * // false indicates that the default action
   * // was prevented by listeners
   * if (eventBus.fire('foo') === false) {
   *   console.log('default has been prevented!');
   * };
   *
   *
   * // custom args + return value listener
   * eventBus.on('sum', function(event, a, b) {
   *   return a + b;
   * });
   *
   * // you can pass custom arguments + retrieve result values.
   * var sum = eventBus.fire('sum', 1, 2);
   * console.log(sum); // 3
   * ```
   *
   * @template [EventMap=null]
   */
  function EventBus() {
    /**
     * @type { Record<string, EventBusListener> }
     */
    this._listeners = {};

    // cleanup on destroy on lowest priority to allow
    // message passing until the bitter end
    this.on('diagram.destroy', 1, this._destroy, this);
  }

  /**
   * @overlord
   *
   * Register an event listener for events with the given name.
   *
   * The callback will be invoked with `event, ...additionalArguments`
   * that have been passed to {@link EventBus#fire}.
   *
   * Returning false from a listener will prevent the events default action
   * (if any is specified). To stop an event from being processed further in
   * other listeners execute {@link Event#stopPropagation}.
   *
   * Returning anything but `undefined` from a listener will stop the listener propagation.
   *
   * @template T
   *
   * @param {string|string[]} events to subscribe to
   * @param {number} [priority=1000] listen priority
   * @param {EventBusEventCallback<T>} callback
   * @param {any} [that] callback context
   */
  /**
   * Register an event listener for events with the given name.
   *
   * The callback will be invoked with `event, ...additionalArguments`
   * that have been passed to {@link EventBus#fire}.
   *
   * Returning false from a listener will prevent the events default action
   * (if any is specified). To stop an event from being processed further in
   * other listeners execute {@link Event#stopPropagation}.
   *
   * Returning anything but `undefined` from a listener will stop the listener propagation.
   *
   * @template {keyof EventMap} EventName
   *
   * @param {EventName} events to subscribe to
   * @param {number} [priority=1000] listen priority
   * @param {EventBusEventCallback<EventMap[EventName]>} callback
   * @param {any} [that] callback context
   */
  EventBus.prototype.on = function (events, priority, callback, that) {
    events = isArray$6(events) ? events : [events];
    if (isFunction$2(priority)) {
      that = callback;
      callback = priority;
      priority = DEFAULT_PRIORITY$8;
    }
    if (!isNumber$3(priority)) {
      throw new Error('priority must be a number');
    }
    var actualCallback = callback;
    if (that) {
      actualCallback = bind$3(callback, that);

      // make sure we remember and are able to remove
      // bound callbacks via {@link #off} using the original
      // callback
      actualCallback[FN_REF] = callback[FN_REF] || callback;
    }
    var self = this;
    events.forEach(function (e) {
      self._addListener(e, {
        priority: priority,
        callback: actualCallback,
        next: null
      });
    });
  };

  /**
   * @overlord
   *
   * Register an event listener that is called only once.
   *
   * @template T
   *
   * @param {string|string[]} events to subscribe to
   * @param {number} [priority=1000] the listen priority
   * @param {EventBusEventCallback<T>} callback
   * @param {any} [that] callback context
   */
  /**
   * Register an event listener that is called only once.
   *
   * @template {keyof EventMap} EventName
   *
   * @param {EventName} events to subscribe to
   * @param {number} [priority=1000] listen priority
   * @param {EventBusEventCallback<EventMap[EventName]>} callback
   * @param {any} [that] callback context
   */
  EventBus.prototype.once = function (events, priority, callback, that) {
    var self = this;
    if (isFunction$2(priority)) {
      that = callback;
      callback = priority;
      priority = DEFAULT_PRIORITY$8;
    }
    if (!isNumber$3(priority)) {
      throw new Error('priority must be a number');
    }
    function wrappedCallback() {
      wrappedCallback.__isTomb = true;
      var result = callback.apply(that, arguments);
      self.off(events, wrappedCallback);
      return result;
    }

    // make sure we remember and are able to remove
    // bound callbacks via {@link #off} using the original
    // callback
    wrappedCallback[FN_REF] = callback;
    this.on(events, priority, wrappedCallback);
  };

  /**
   * Removes event listeners by event and callback.
   *
   * If no callback is given, all listeners for a given event name are being removed.
   *
   * @param {string|string[]} events
   * @param {EventBusEventCallback} [callback]
   */
  EventBus.prototype.off = function (events, callback) {
    events = isArray$6(events) ? events : [events];
    var self = this;
    events.forEach(function (event) {
      self._removeListener(event, callback);
    });
  };

  /**
   * Create an event recognized be the event bus.
   *
   * @param {Object} data Event data.
   *
   * @return {Event} An event that will be recognized by the event bus.
   */
  EventBus.prototype.createEvent = function (data) {
    var event = new InternalEvent();
    event.init(data);
    return event;
  };

  /**
   * Fires an event.
   *
   * @example
   *
   * ```javascript
   * // fire event by name
   * events.fire('foo');
   *
   * // fire event object with nested type
   * var event = { type: 'foo' };
   * events.fire(event);
   *
   * // fire event with explicit type
   * var event = { x: 10, y: 20 };
   * events.fire('element.moved', event);
   *
   * // pass additional arguments to the event
   * events.on('foo', function(event, bar) {
   *   alert(bar);
   * });
   *
   * events.fire({ type: 'foo' }, 'I am bar!');
   * ```
   *
   * @param {string} [type] event type
   * @param {Object} [data] event or event data
   * @param {...any} [args] additional arguments the callback will be called with.
   *
   * @return {any} The return value. Will be set to `false` if the default was prevented.
   */
  EventBus.prototype.fire = function (type, data) {
    var event, firstListener, returnValue, args;
    args = slice.call(arguments);
    if (typeof type === 'object') {
      data = type;
      type = data.type;
    }
    if (!type) {
      throw new Error('no event type specified');
    }
    firstListener = this._listeners[type];
    if (!firstListener) {
      return;
    }

    // we make sure we fire instances of our home made
    // events here. We wrap them only once, though
    if (data instanceof InternalEvent) {
      // we are fine, we alread have an event
      event = data;
    } else {
      event = this.createEvent(data);
    }

    // ensure we pass the event as the first parameter
    args[0] = event;

    // original event type (in case we delegate)
    var originalType = event.type;

    // update event type before delegation
    if (type !== originalType) {
      event.type = type;
    }
    try {
      returnValue = this._invokeListeners(event, args, firstListener);
    } finally {
      // reset event type after delegation
      if (type !== originalType) {
        event.type = originalType;
      }
    }

    // set the return value to false if the event default
    // got prevented and no other return value exists
    if (returnValue === undefined && event.defaultPrevented) {
      returnValue = false;
    }
    return returnValue;
  };

  /**
   * Handle an error by firing an event.
   *
   * @param {Error} error The error to be handled.
   *
   * @return {boolean} Whether the error was handled.
   */
  EventBus.prototype.handleError = function (error) {
    return this.fire('error', {
      error: error
    }) === false;
  };
  EventBus.prototype._destroy = function () {
    this._listeners = {};
  };

  /**
   * @param {Event} event
   * @param {any[]} args
   * @param {EventBusListener} listener
   *
   * @return {any}
   */
  EventBus.prototype._invokeListeners = function (event, args, listener) {
    var returnValue;
    while (listener) {
      // handle stopped propagation
      if (event.cancelBubble) {
        break;
      }
      returnValue = this._invokeListener(event, args, listener);
      listener = listener.next;
    }
    return returnValue;
  };

  /**
   * @param {Event} event
   * @param {any[]} args
   * @param {EventBusListener} listener
   *
   * @return {any}
   */
  EventBus.prototype._invokeListener = function (event, args, listener) {
    var returnValue;
    if (listener.callback.__isTomb) {
      return returnValue;
    }
    try {
      // returning false prevents the default action
      returnValue = invokeFunction(listener.callback, args);

      // stop propagation on return value
      if (returnValue !== undefined) {
        event.returnValue = returnValue;
        event.stopPropagation();
      }

      // prevent default on return false
      if (returnValue === false) {
        event.preventDefault();
      }
    } catch (error) {
      if (!this.handleError(error)) {
        console.error('unhandled error in event listener', error);
        throw error;
      }
    }
    return returnValue;
  };

  /**
   * Add new listener with a certain priority to the list
   * of listeners (for the given event).
   *
   * The semantics of listener registration / listener execution are
   * first register, first serve: New listeners will always be inserted
   * after existing listeners with the same priority.
   *
   * Example: Inserting two listeners with priority 1000 and 1300
   *
   *    * before: [ 1500, 1500, 1000, 1000 ]
   *    * after: [ 1500, 1500, (new=1300), 1000, 1000, (new=1000) ]
   *
   * @param {string} event
   * @param {EventBusListener} newListener
   */
  EventBus.prototype._addListener = function (event, newListener) {
    var listener = this._getListeners(event),
      previousListener;

    // no prior listeners
    if (!listener) {
      this._setListeners(event, newListener);
      return;
    }

    // ensure we order listeners by priority from
    // 0 (high) to n > 0 (low)
    while (listener) {
      if (listener.priority < newListener.priority) {
        newListener.next = listener;
        if (previousListener) {
          previousListener.next = newListener;
        } else {
          this._setListeners(event, newListener);
        }
        return;
      }
      previousListener = listener;
      listener = listener.next;
    }

    // add new listener to back
    previousListener.next = newListener;
  };

  /**
   * @param {string} name
   *
   * @return {EventBusListener}
   */
  EventBus.prototype._getListeners = function (name) {
    return this._listeners[name];
  };

  /**
   * @param {string} name
   * @param {EventBusListener} listener
   */
  EventBus.prototype._setListeners = function (name, listener) {
    this._listeners[name] = listener;
  };
  EventBus.prototype._removeListener = function (event, callback) {
    var listener = this._getListeners(event),
      nextListener,
      previousListener,
      listenerCallback;
    if (!callback) {
      // clear listeners
      this._setListeners(event, null);
      return;
    }
    while (listener) {
      nextListener = listener.next;
      listenerCallback = listener.callback;
      if (listenerCallback === callback || listenerCallback[FN_REF] === callback) {
        if (previousListener) {
          previousListener.next = nextListener;
        } else {
          // new first listener
          this._setListeners(event, nextListener);
        }
      }
      previousListener = listener;
      listener = nextListener;
    }
  };

  /**
   * A event that is emitted via the event bus.
   */
  function InternalEvent() {}
  InternalEvent.prototype.stopPropagation = function () {
    this.cancelBubble = true;
  };
  InternalEvent.prototype.preventDefault = function () {
    this.defaultPrevented = true;
  };
  InternalEvent.prototype.init = function (data) {
    assign$4(this, data || {});
  };

  /**
   * Invoke function. Be fast...
   *
   * @param {Function} fn
   * @param {any[]} args
   *
   * @return {any}
   */
  function invokeFunction(fn, args) {
    return fn.apply(null, args);
  }

  /**
   * Flatten array, one level deep.
   *
   * @param {Array<?>} arr
   *
   * @return {Array<?>}
   */
  var nativeToString$3 = Object.prototype.toString;
  function isString$3(obj) {
    return nativeToString$3.call(obj) === '[object String]';
  }
  function _extends$3() {
    _extends$3 = Object.assign || function (target) {
      for (var i = 1; i < arguments.length; i++) {
        var source = arguments[i];
        for (var key in source) {
          if (Object.prototype.hasOwnProperty.call(source, key)) {
            target[key] = source[key];
          }
        }
      }
      return target;
    };
    return _extends$3.apply(this, arguments);
  }

  /**
   * Convenience wrapper for `Object.assign`.
   *
   * @param {Object} target
   * @param {...Object} others
   *
   * @return {Object} the target
   */

  function assign$3(target) {
    for (var _len = arguments.length, others = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
      others[_key - 1] = arguments[_key];
    }
    return _extends$3.apply(void 0, [target].concat(others));
  }

  /**
   * Flatten array, one level deep.
   *
   * @param {Array<?>} arr
   *
   * @return {Array<?>}
   */
  var nativeToString$2 = Object.prototype.toString;
  var nativeHasOwnProperty$2 = Object.prototype.hasOwnProperty;
  function isUndefined$4(obj) {
    return obj === undefined;
  }
  function isArray$5(obj) {
    return nativeToString$2.call(obj) === '[object Array]';
  }
  function isObject$1(obj) {
    return nativeToString$2.call(obj) === '[object Object]';
  }
  function isString$2(obj) {
    return nativeToString$2.call(obj) === '[object String]';
  }
  /**
   * Return true, if target owns a property with the given key.
   *
   * @param {Object} target
   * @param {String} key
   *
   * @return {Boolean}
   */

  function has$2(target, key) {
    return nativeHasOwnProperty$2.call(target, key);
  }
  /**
   * Iterate over collection; returning something
   * (non-undefined) will stop iteration.
   *
   * @param  {Array|Object} collection
   * @param  {Function} iterator
   *
   * @return {Object} return result that stopped the iteration
   */

  function forEach$2(collection, iterator) {
    var val, result;
    if (isUndefined$4(collection)) {
      return;
    }
    var convertKey = isArray$5(collection) ? toNum$2 : identity$2;
    for (var key in collection) {
      if (has$2(collection, key)) {
        val = collection[key];
        result = iterator(val, convertKey(key));
        if (result === false) {
          return val;
        }
      }
    }
  }
  function identity$2(arg) {
    return arg;
  }
  function toNum$2(arg) {
    return Number(arg);
  }
  /**
   * Bind function against target <this>.
   *
   * @param  {Function} fn
   * @param  {Object}   target
   *
   * @return {Function} bound function
   */

  function bind$2(fn, target) {
    return fn.bind(target);
  }
  function _extends$2() {
    _extends$2 = Object.assign || function (target) {
      for (var i = 1; i < arguments.length; i++) {
        var source = arguments[i];
        for (var key in source) {
          if (Object.prototype.hasOwnProperty.call(source, key)) {
            target[key] = source[key];
          }
        }
      }
      return target;
    };
    return _extends$2.apply(this, arguments);
  }

  /**
   * Convenience wrapper for `Object.assign`.
   *
   * @param {Object} target
   * @param {...Object} others
   *
   * @return {Object} the target
   */

  function assign$2(target) {
    for (var _len = arguments.length, others = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
      others[_key - 1] = arguments[_key];
    }
    return _extends$2.apply(void 0, [target].concat(others));
  }
  /**
   * Pick given properties from the target object.
   *
   * @param {Object} target
   * @param {Array} properties
   *
   * @return {Object} target
   */

  function pick(target, properties) {
    var result = {};
    var obj = Object(target);
    forEach$2(properties, function (prop) {
      if (prop in obj) {
        result[prop] = target[prop];
      }
    });
    return result;
  }

  /**
   * Moddle base element.
   */
  function Base$1() {}
  Base$1.prototype.get = function (name) {
    return this.$model.properties.get(this, name);
  };
  Base$1.prototype.set = function (name, value) {
    this.$model.properties.set(this, name, value);
  };

  /**
   * A model element factory.
   *
   * @param {Moddle} model
   * @param {Properties} properties
   */
  function Factory(model, properties) {
    this.model = model;
    this.properties = properties;
  }
  Factory.prototype.createType = function (descriptor) {
    var model = this.model;
    var props = this.properties,
      prototype = Object.create(Base$1.prototype);

    // initialize default values
    forEach$2(descriptor.properties, function (p) {
      if (!p.isMany && p.default !== undefined) {
        prototype[p.name] = p.default;
      }
    });
    props.defineModel(prototype, model);
    props.defineDescriptor(prototype, descriptor);
    var name = descriptor.ns.name;

    /**
     * The new type constructor
     */
    function ModdleElement(attrs) {
      props.define(this, '$type', {
        value: name,
        enumerable: true
      });
      props.define(this, '$attrs', {
        value: {}
      });
      props.define(this, '$parent', {
        writable: true
      });
      forEach$2(attrs, bind$2(function (val, key) {
        this.set(key, val);
      }, this));
    }
    ModdleElement.prototype = prototype;
    ModdleElement.hasType = prototype.$instanceOf = this.model.hasType;

    // static links
    props.defineModel(ModdleElement, model);
    props.defineDescriptor(ModdleElement, descriptor);
    return ModdleElement;
  };

  /**
   * Built-in moddle types
   */
  var BUILTINS = {
    String: true,
    Boolean: true,
    Integer: true,
    Real: true,
    Element: true
  };

  /**
   * Converters for built in types from string representations
   */
  var TYPE_CONVERTERS = {
    String: function (s) {
      return s;
    },
    Boolean: function (s) {
      return s === 'true';
    },
    Integer: function (s) {
      return parseInt(s, 10);
    },
    Real: function (s) {
      return parseFloat(s);
    }
  };

  /**
   * Convert a type to its real representation
   */
  function coerceType(type, value) {
    var converter = TYPE_CONVERTERS[type];
    if (converter) {
      return converter(value);
    } else {
      return value;
    }
  }

  /**
   * Return whether the given type is built-in
   */
  function isBuiltIn(type) {
    return !!BUILTINS[type];
  }

  /**
   * Return whether the given type is simple
   */
  function isSimple(type) {
    return !!TYPE_CONVERTERS[type];
  }

  /**
   * Parses a namespaced attribute name of the form (ns:)localName to an object,
   * given a default prefix to assume in case no explicit namespace is given.
   *
   * @param {String} name
   * @param {String} [defaultPrefix] the default prefix to take, if none is present.
   *
   * @return {Object} the parsed name
   */
  function parseName$1(name, defaultPrefix) {
    var parts = name.split(/:/),
      localName,
      prefix;

    // no prefix (i.e. only local name)
    if (parts.length === 1) {
      localName = name;
      prefix = defaultPrefix;
    } else
      // prefix + local name
      if (parts.length === 2) {
        localName = parts[1];
        prefix = parts[0];
      } else {
        throw new Error('expected <prefix:localName> or <localName>, got ' + name);
      }
    name = (prefix ? prefix + ':' : '') + localName;
    return {
      name: name,
      prefix: prefix,
      localName: localName
    };
  }

  /**
   * A utility to build element descriptors.
   */
  function DescriptorBuilder(nameNs) {
    this.ns = nameNs;
    this.name = nameNs.name;
    this.allTypes = [];
    this.allTypesByName = {};
    this.properties = [];
    this.propertiesByName = {};
  }
  DescriptorBuilder.prototype.build = function () {
    return pick(this, ['ns', 'name', 'allTypes', 'allTypesByName', 'properties', 'propertiesByName', 'bodyProperty', 'idProperty']);
  };

  /**
   * Add property at given index.
   *
   * @param {Object} p
   * @param {Number} [idx]
   * @param {Boolean} [validate=true]
   */
  DescriptorBuilder.prototype.addProperty = function (p, idx, validate) {
    if (typeof idx === 'boolean') {
      validate = idx;
      idx = undefined;
    }
    this.addNamedProperty(p, validate !== false);
    var properties = this.properties;
    if (idx !== undefined) {
      properties.splice(idx, 0, p);
    } else {
      properties.push(p);
    }
  };
  DescriptorBuilder.prototype.replaceProperty = function (oldProperty, newProperty, replace) {
    var oldNameNs = oldProperty.ns;
    var props = this.properties,
      propertiesByName = this.propertiesByName,
      rename = oldProperty.name !== newProperty.name;
    if (oldProperty.isId) {
      if (!newProperty.isId) {
        throw new Error('property <' + newProperty.ns.name + '> must be id property ' + 'to refine <' + oldProperty.ns.name + '>');
      }
      this.setIdProperty(newProperty, false);
    }
    if (oldProperty.isBody) {
      if (!newProperty.isBody) {
        throw new Error('property <' + newProperty.ns.name + '> must be body property ' + 'to refine <' + oldProperty.ns.name + '>');
      }

      // TODO: Check compatibility
      this.setBodyProperty(newProperty, false);
    }

    // validate existence and get location of old property
    var idx = props.indexOf(oldProperty);
    if (idx === -1) {
      throw new Error('property <' + oldNameNs.name + '> not found in property list');
    }

    // remove old property
    props.splice(idx, 1);

    // replacing the named property is intentional
    //
    //  * validate only if this is a "rename" operation
    //  * add at specific index unless we "replace"
    //
    this.addProperty(newProperty, replace ? undefined : idx, rename);

    // make new property available under old name
    propertiesByName[oldNameNs.name] = propertiesByName[oldNameNs.localName] = newProperty;
  };
  DescriptorBuilder.prototype.redefineProperty = function (p, targetPropertyName, replace) {
    var nsPrefix = p.ns.prefix;
    var parts = targetPropertyName.split('#');
    var name = parseName$1(parts[0], nsPrefix);
    var attrName = parseName$1(parts[1], name.prefix).name;
    var redefinedProperty = this.propertiesByName[attrName];
    if (!redefinedProperty) {
      throw new Error('refined property <' + attrName + '> not found');
    } else {
      this.replaceProperty(redefinedProperty, p, replace);
    }
    delete p.redefines;
  };
  DescriptorBuilder.prototype.addNamedProperty = function (p, validate) {
    var ns = p.ns,
      propsByName = this.propertiesByName;
    if (validate) {
      this.assertNotDefined(p, ns.name);
      this.assertNotDefined(p, ns.localName);
    }
    propsByName[ns.name] = propsByName[ns.localName] = p;
  };
  DescriptorBuilder.prototype.removeNamedProperty = function (p) {
    var ns = p.ns,
      propsByName = this.propertiesByName;
    delete propsByName[ns.name];
    delete propsByName[ns.localName];
  };
  DescriptorBuilder.prototype.setBodyProperty = function (p, validate) {
    if (validate && this.bodyProperty) {
      throw new Error('body property defined multiple times ' + '(<' + this.bodyProperty.ns.name + '>, <' + p.ns.name + '>)');
    }
    this.bodyProperty = p;
  };
  DescriptorBuilder.prototype.setIdProperty = function (p, validate) {
    if (validate && this.idProperty) {
      throw new Error('id property defined multiple times ' + '(<' + this.idProperty.ns.name + '>, <' + p.ns.name + '>)');
    }
    this.idProperty = p;
  };
  DescriptorBuilder.prototype.assertNotDefined = function (p, name) {
    var propertyName = p.name,
      definedProperty = this.propertiesByName[propertyName];
    if (definedProperty) {
      throw new Error('property <' + propertyName + '> already defined; ' + 'override of <' + definedProperty.definedBy.ns.name + '#' + definedProperty.ns.name + '> by ' + '<' + p.definedBy.ns.name + '#' + p.ns.name + '> not allowed without redefines');
    }
  };
  DescriptorBuilder.prototype.hasProperty = function (name) {
    return this.propertiesByName[name];
  };
  DescriptorBuilder.prototype.addTrait = function (t, inherited) {
    var typesByName = this.allTypesByName,
      types = this.allTypes;
    var typeName = t.name;
    if (typeName in typesByName) {
      return;
    }
    forEach$2(t.properties, bind$2(function (p) {
      // clone property to allow extensions
      p = assign$2({}, p, {
        name: p.ns.localName,
        inherited: inherited
      });
      Object.defineProperty(p, 'definedBy', {
        value: t
      });
      var replaces = p.replaces,
        redefines = p.redefines;

      // add replace/redefine support
      if (replaces || redefines) {
        this.redefineProperty(p, replaces || redefines, replaces);
      } else {
        if (p.isBody) {
          this.setBodyProperty(p);
        }
        if (p.isId) {
          this.setIdProperty(p);
        }
        this.addProperty(p);
      }
    }, this));
    types.push(t);
    typesByName[typeName] = t;
  };

  /**
   * A registry of Moddle packages.
   *
   * @param {Array<Package>} packages
   * @param {Properties} properties
   */
  function Registry(packages, properties) {
    this.packageMap = {};
    this.typeMap = {};
    this.packages = [];
    this.properties = properties;
    forEach$2(packages, bind$2(this.registerPackage, this));
  }
  Registry.prototype.getPackage = function (uriOrPrefix) {
    return this.packageMap[uriOrPrefix];
  };
  Registry.prototype.getPackages = function () {
    return this.packages;
  };
  Registry.prototype.registerPackage = function (pkg) {
    // copy package
    pkg = assign$2({}, pkg);
    var pkgMap = this.packageMap;
    ensureAvailable(pkgMap, pkg, 'prefix');
    ensureAvailable(pkgMap, pkg, 'uri');

    // register types
    forEach$2(pkg.types, bind$2(function (descriptor) {
      this.registerType(descriptor, pkg);
    }, this));
    pkgMap[pkg.uri] = pkgMap[pkg.prefix] = pkg;
    this.packages.push(pkg);
  };

  /**
   * Register a type from a specific package with us
   */
  Registry.prototype.registerType = function (type, pkg) {
    type = assign$2({}, type, {
      superClass: (type.superClass || []).slice(),
      extends: (type.extends || []).slice(),
      properties: (type.properties || []).slice(),
      meta: assign$2(type.meta || {})
    });
    var ns = parseName$1(type.name, pkg.prefix),
      name = ns.name,
      propertiesByName = {};

    // parse properties
    forEach$2(type.properties, bind$2(function (p) {
      // namespace property names
      var propertyNs = parseName$1(p.name, ns.prefix),
        propertyName = propertyNs.name;

      // namespace property types
      if (!isBuiltIn(p.type)) {
        p.type = parseName$1(p.type, propertyNs.prefix).name;
      }
      assign$2(p, {
        ns: propertyNs,
        name: propertyName
      });
      propertiesByName[propertyName] = p;
    }, this));

    // update ns + name
    assign$2(type, {
      ns: ns,
      name: name,
      propertiesByName: propertiesByName
    });
    forEach$2(type.extends, bind$2(function (extendsName) {
      var extended = this.typeMap[extendsName];
      extended.traits = extended.traits || [];
      extended.traits.push(name);
    }, this));

    // link to package
    this.definePackage(type, pkg);

    // register
    this.typeMap[name] = type;
  };

  /**
   * Traverse the type hierarchy from bottom to top,
   * calling iterator with (type, inherited) for all elements in
   * the inheritance chain.
   *
   * @param {Object} nsName
   * @param {Function} iterator
   * @param {Boolean} [trait=false]
   */
  Registry.prototype.mapTypes = function (nsName, iterator, trait) {
    var type = isBuiltIn(nsName.name) ? {
      name: nsName.name
    } : this.typeMap[nsName.name];
    var self = this;

    /**
     * Traverse the selected trait.
     *
     * @param {String} cls
     */
    function traverseTrait(cls) {
      return traverseSuper(cls, true);
    }

    /**
     * Traverse the selected super type or trait
     *
     * @param {String} cls
     * @param {Boolean} [trait=false]
     */
    function traverseSuper(cls, trait) {
      var parentNs = parseName$1(cls, isBuiltIn(cls) ? '' : nsName.prefix);
      self.mapTypes(parentNs, iterator, trait);
    }
    if (!type) {
      throw new Error('unknown type <' + nsName.name + '>');
    }
    forEach$2(type.superClass, trait ? traverseTrait : traverseSuper);

    // call iterator with (type, inherited=!trait)
    iterator(type, !trait);
    forEach$2(type.traits, traverseTrait);
  };

  /**
   * Returns the effective descriptor for a type.
   *
   * @param  {String} type the namespaced name (ns:localName) of the type
   *
   * @return {Descriptor} the resulting effective descriptor
   */
  Registry.prototype.getEffectiveDescriptor = function (name) {
    var nsName = parseName$1(name);
    var builder = new DescriptorBuilder(nsName);
    this.mapTypes(nsName, function (type, inherited) {
      builder.addTrait(type, inherited);
    });
    var descriptor = builder.build();

    // define package link
    this.definePackage(descriptor, descriptor.allTypes[descriptor.allTypes.length - 1].$pkg);
    return descriptor;
  };
  Registry.prototype.definePackage = function (target, pkg) {
    this.properties.define(target, '$pkg', {
      value: pkg
    });
  };

  ///////// helpers ////////////////////////////

  function ensureAvailable(packageMap, pkg, identifierKey) {
    var value = pkg[identifierKey];
    if (value in packageMap) {
      throw new Error('package with ' + identifierKey + ' <' + value + '> already defined');
    }
  }

  /**
   * A utility that gets and sets properties of model elements.
   *
   * @param {Model} model
   */
  function Properties(model) {
    this.model = model;
  }

  /**
   * Sets a named property on the target element.
   * If the value is undefined, the property gets deleted.
   *
   * @param {Object} target
   * @param {String} name
   * @param {Object} value
   */
  Properties.prototype.set = function (target, name, value) {
    if (!isString$2(name) || !name.length) {
      throw new TypeError('property name must be a non-empty string');
    }
    var property = this.model.getPropertyDescriptor(target, name);
    var propertyName = property && property.name;
    if (isUndefined$3(value)) {
      // unset the property, if the specified value is undefined;
      // delete from $attrs (for extensions) or the target itself
      if (property) {
        delete target[propertyName];
      } else {
        delete target.$attrs[name];
      }
    } else {
      // set the property, defining well defined properties on the fly
      // or simply updating them in target.$attrs (for extensions)
      if (property) {
        if (propertyName in target) {
          target[propertyName] = value;
        } else {
          defineProperty$2(target, property, value);
        }
      } else {
        target.$attrs[name] = value;
      }
    }
  };

  /**
   * Returns the named property of the given element
   *
   * @param  {Object} target
   * @param  {String} name
   *
   * @return {Object}
   */
  Properties.prototype.get = function (target, name) {
    var property = this.model.getPropertyDescriptor(target, name);
    if (!property) {
      return target.$attrs[name];
    }
    var propertyName = property.name;

    // check if access to collection property and lazily initialize it
    if (!target[propertyName] && property.isMany) {
      defineProperty$2(target, property, []);
    }
    return target[propertyName];
  };

  /**
   * Define a property on the target element
   *
   * @param  {Object} target
   * @param  {String} name
   * @param  {Object} options
   */
  Properties.prototype.define = function (target, name, options) {
    if (!options.writable) {
      var value = options.value;

      // use getters for read-only variables to support ES6 proxies
      // cf. https://github.com/bpmn-io/internal-docs/issues/386
      options = assign$2({}, options, {
        get: function () {
          return value;
        }
      });
      delete options.value;
    }
    Object.defineProperty(target, name, options);
  };

  /**
   * Define the descriptor for an element
   */
  Properties.prototype.defineDescriptor = function (target, descriptor) {
    this.define(target, '$descriptor', {
      value: descriptor
    });
  };

  /**
   * Define the model for an element
   */
  Properties.prototype.defineModel = function (target, model) {
    this.define(target, '$model', {
      value: model
    });
  };
  function isUndefined$3(val) {
    return typeof val === 'undefined';
  }
  function defineProperty$2(target, property, value) {
    Object.defineProperty(target, property.name, {
      enumerable: !property.isReference,
      writable: true,
      value: value,
      configurable: true
    });
  }

  //// Moddle implementation /////////////////////////////////////////////////

  /**
   * @class Moddle
   *
   * A model that can be used to create elements of a specific type.
   *
   * @example
   *
   * var Moddle = require('moddle');
   *
   * var pkg = {
   *   name: 'mypackage',
   *   prefix: 'my',
   *   types: [
   *     { name: 'Root' }
   *   ]
   * };
   *
   * var moddle = new Moddle([pkg]);
   *
   * @param {Array<Package>} packages the packages to contain
   */
  function Moddle(packages) {
    this.properties = new Properties(this);
    this.factory = new Factory(this, this.properties);
    this.registry = new Registry(packages, this.properties);
    this.typeCache = {};
  }

  /**
   * Create an instance of the specified type.
   *
   * @method Moddle#create
   *
   * @example
   *
   * var foo = moddle.create('my:Foo');
   * var bar = moddle.create('my:Bar', { id: 'BAR_1' });
   *
   * @param  {String|Object} descriptor the type descriptor or name know to the model
   * @param  {Object} attrs   a number of attributes to initialize the model instance with
   * @return {Object}         model instance
   */
  Moddle.prototype.create = function (descriptor, attrs) {
    var Type = this.getType(descriptor);
    if (!Type) {
      throw new Error('unknown type <' + descriptor + '>');
    }
    return new Type(attrs);
  };

  /**
   * Returns the type representing a given descriptor
   *
   * @method Moddle#getType
   *
   * @example
   *
   * var Foo = moddle.getType('my:Foo');
   * var foo = new Foo({ 'id' : 'FOO_1' });
   *
   * @param  {String|Object} descriptor the type descriptor or name know to the model
   * @return {Object}         the type representing the descriptor
   */
  Moddle.prototype.getType = function (descriptor) {
    var cache = this.typeCache;
    var name = isString$2(descriptor) ? descriptor : descriptor.ns.name;
    var type = cache[name];
    if (!type) {
      descriptor = this.registry.getEffectiveDescriptor(name);
      type = cache[name] = this.factory.createType(descriptor);
    }
    return type;
  };

  /**
   * Creates an any-element type to be used within model instances.
   *
   * This can be used to create custom elements that lie outside the meta-model.
   * The created element contains all the meta-data required to serialize it
   * as part of meta-model elements.
   *
   * @method Moddle#createAny
   *
   * @example
   *
   * var foo = moddle.createAny('vendor:Foo', 'http://vendor', {
   *   value: 'bar'
   * });
   *
   * var container = moddle.create('my:Container', 'http://my', {
   *   any: [ foo ]
   * });
   *
   * // go ahead and serialize the stuff
   *
   *
   * @param  {String} name  the name of the element
   * @param  {String} nsUri the namespace uri of the element
   * @param  {Object} [properties] a map of properties to initialize the instance with
   * @return {Object} the any type instance
   */
  Moddle.prototype.createAny = function (name, nsUri, properties) {
    var nameNs = parseName$1(name);
    var element = {
      $type: name,
      $instanceOf: function (type) {
        return type === this.$type;
      }
    };
    var descriptor = {
      name: name,
      isGeneric: true,
      ns: {
        prefix: nameNs.prefix,
        localName: nameNs.localName,
        uri: nsUri
      }
    };
    this.properties.defineDescriptor(element, descriptor);
    this.properties.defineModel(element, this);
    this.properties.define(element, '$parent', {
      enumerable: false,
      writable: true
    });
    this.properties.define(element, '$instanceOf', {
      enumerable: false,
      writable: true
    });
    forEach$2(properties, function (a, key) {
      if (isObject$1(a) && a.value !== undefined) {
        element[a.name] = a.value;
      } else {
        element[key] = a;
      }
    });
    return element;
  };

  /**
   * Returns a registered package by uri or prefix
   *
   * @return {Object} the package
   */
  Moddle.prototype.getPackage = function (uriOrPrefix) {
    return this.registry.getPackage(uriOrPrefix);
  };

  /**
   * Returns a snapshot of all known packages
   *
   * @return {Object} the package
   */
  Moddle.prototype.getPackages = function () {
    return this.registry.getPackages();
  };

  /**
   * Returns the descriptor for an element
   */
  Moddle.prototype.getElementDescriptor = function (element) {
    return element.$descriptor;
  };

  /**
   * Returns true if the given descriptor or instance
   * represents the given type.
   *
   * May be applied to this, if element is omitted.
   */
  Moddle.prototype.hasType = function (element, type) {
    if (type === undefined) {
      type = element;
      element = this;
    }
    var descriptor = element.$model.getElementDescriptor(element);
    return type in descriptor.allTypesByName;
  };

  /**
   * Returns the descriptor of an elements named property
   */
  Moddle.prototype.getPropertyDescriptor = function (element, property) {
    return this.getElementDescriptor(element).propertiesByName[property];
  };

  /**
   * Returns a mapped type's descriptor
   */
  Moddle.prototype.getTypeDescriptor = function (type) {
    return this.registry.typeMap[type];
  };

  /**
   * Flatten array, one level deep.
   *
   * @param {Array<?>} arr
   *
   * @return {Array<?>}
   */
  var nativeToString$1 = Object.prototype.toString;
  var nativeHasOwnProperty$1 = Object.prototype.hasOwnProperty;
  function isUndefined$2(obj) {
    return obj === undefined;
  }
  function isArray$4(obj) {
    return nativeToString$1.call(obj) === '[object Array]';
  }
  function isFunction$1(obj) {
    var tag = nativeToString$1.call(obj);
    return tag === '[object Function]' || tag === '[object AsyncFunction]' || tag === '[object GeneratorFunction]' || tag === '[object AsyncGeneratorFunction]' || tag === '[object Proxy]';
  }
  function isString$1(obj) {
    return nativeToString$1.call(obj) === '[object String]';
  }
  /**
   * Return true, if target owns a property with the given key.
   *
   * @param {Object} target
   * @param {String} key
   *
   * @return {Boolean}
   */

  function has$1(target, key) {
    return nativeHasOwnProperty$1.call(target, key);
  }

  /**
   * Find element in collection.
   *
   * @param  {Array|Object} collection
   * @param  {Function|Object} matcher
   *
   * @return {Object}
   */

  function find$1(collection, matcher) {
    matcher = toMatcher(matcher);
    var match;
    forEach$1(collection, function (val, key) {
      if (matcher(val, key)) {
        match = val;
        return false;
      }
    });
    return match;
  }
  /**
   * Find element index in collection.
   *
   * @param  {Array|Object} collection
   * @param  {Function} matcher
   *
   * @return {Object}
   */

  function findIndex(collection, matcher) {
    matcher = toMatcher(matcher);
    var idx = isArray$4(collection) ? -1 : undefined;
    forEach$1(collection, function (val, key) {
      if (matcher(val, key)) {
        idx = key;
        return false;
      }
    });
    return idx;
  }
  /**
   * Find element in collection.
   *
   * @param  {Array|Object} collection
   * @param  {Function} matcher
   *
   * @return {Array} result
   */

  function filter$1(collection, matcher) {
    var result = [];
    forEach$1(collection, function (val, key) {
      if (matcher(val, key)) {
        result.push(val);
      }
    });
    return result;
  }
  /**
   * Iterate over collection; returning something
   * (non-undefined) will stop iteration.
   *
   * @param  {Array|Object} collection
   * @param  {Function} iterator
   *
   * @return {Object} return result that stopped the iteration
   */

  function forEach$1(collection, iterator) {
    var val, result;
    if (isUndefined$2(collection)) {
      return;
    }
    var convertKey = isArray$4(collection) ? toNum$1 : identity$1;
    for (var key in collection) {
      if (has$1(collection, key)) {
        val = collection[key];
        result = iterator(val, convertKey(key));
        if (result === false) {
          return val;
        }
      }
    }
  }
  function toMatcher(matcher) {
    return isFunction$1(matcher) ? matcher : function (e) {
      return e === matcher;
    };
  }
  function identity$1(arg) {
    return arg;
  }
  function toNum$1(arg) {
    return Number(arg);
  }
  function _extends$1() {
    _extends$1 = Object.assign || function (target) {
      for (var i = 1; i < arguments.length; i++) {
        var source = arguments[i];
        for (var key in source) {
          if (Object.prototype.hasOwnProperty.call(source, key)) {
            target[key] = source[key];
          }
        }
      }
      return target;
    };
    return _extends$1.apply(this, arguments);
  }

  /**
   * Convenience wrapper for `Object.assign`.
   *
   * @param {Object} target
   * @param {...Object} others
   *
   * @return {Object} the target
   */

  function assign$1(target) {
    for (var _len = arguments.length, others = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
      others[_key - 1] = arguments[_key];
    }
    return _extends$1.apply(void 0, [target].concat(others));
  }

  var fromCharCode = String.fromCharCode;
  var hasOwnProperty$1 = Object.prototype.hasOwnProperty;
  var ENTITY_PATTERN = /&#(\d+);|&#x([0-9a-f]+);|&(\w+);/ig;
  var ENTITY_MAPPING = {
    'amp': '&',
    'apos': '\'',
    'gt': '>',
    'lt': '<',
    'quot': '"'
  };

  // map UPPERCASE variants of supported special chars
  Object.keys(ENTITY_MAPPING).forEach(function (k) {
    ENTITY_MAPPING[k.toUpperCase()] = ENTITY_MAPPING[k];
  });
  function replaceEntities(_, d, x, z) {
    // reserved names, i.e. &nbsp;
    if (z) {
      if (hasOwnProperty$1.call(ENTITY_MAPPING, z)) {
        return ENTITY_MAPPING[z];
      } else {
        // fall back to original value
        return '&' + z + ';';
      }
    }

    // decimal encoded char
    if (d) {
      return fromCharCode(d);
    }

    // hex encoded char
    return fromCharCode(parseInt(x, 16));
  }

  /**
   * A basic entity decoder that can decode a minimal
   * sub-set of reserved names (&amp;) as well as
   * hex (&#xaaf;) and decimal (&#1231;) encoded characters.
   *
   * @param {string} str
   *
   * @return {string} decoded string
   */
  function decodeEntities(s) {
    if (s.length > 3 && s.indexOf('&') !== -1) {
      return s.replace(ENTITY_PATTERN, replaceEntities);
    }
    return s;
  }
  var XSI_URI = 'http://www.w3.org/2001/XMLSchema-instance';
  var XSI_PREFIX = 'xsi';
  var XSI_TYPE$1 = 'xsi:type';
  var NON_WHITESPACE_OUTSIDE_ROOT_NODE = 'non-whitespace outside of root node';
  function error$5(msg) {
    return new Error(msg);
  }
  function missingNamespaceForPrefix(prefix) {
    return 'missing namespace for prefix <' + prefix + '>';
  }
  function getter(getFn) {
    return {
      'get': getFn,
      'enumerable': true
    };
  }
  function cloneNsMatrix(nsMatrix) {
    var clone = {},
      key;
    for (key in nsMatrix) {
      clone[key] = nsMatrix[key];
    }
    return clone;
  }
  function uriPrefix(prefix) {
    return prefix + '$uri';
  }
  function buildNsMatrix(nsUriToPrefix) {
    var nsMatrix = {},
      uri,
      prefix;
    for (uri in nsUriToPrefix) {
      prefix = nsUriToPrefix[uri];
      nsMatrix[prefix] = prefix;
      nsMatrix[uriPrefix(prefix)] = uri;
    }
    return nsMatrix;
  }
  function noopGetContext() {
    return {
      'line': 0,
      'column': 0
    };
  }
  function throwFunc(err) {
    throw err;
  }

  /**
   * Creates a new parser with the given options.
   *
   * @constructor
   *
   * @param  {!Object<string, ?>=} options
   */
  function Parser$1(options) {
    if (!this) {
      return new Parser$1(options);
    }
    var proxy = options && options['proxy'];
    var onText,
      onOpenTag,
      onCloseTag,
      onCDATA,
      onError = throwFunc,
      onWarning,
      onComment,
      onQuestion,
      onAttention;
    var getContext = noopGetContext;

    /**
     * Do we need to parse the current elements attributes for namespaces?
     *
     * @type {boolean}
     */
    var maybeNS = false;

    /**
     * Do we process namespaces at all?
     *
     * @type {boolean}
     */
    var isNamespace = false;

    /**
     * The caught error returned on parse end
     *
     * @type {Error}
     */
    var returnError = null;

    /**
     * Should we stop parsing?
     *
     * @type {boolean}
     */
    var parseStop = false;

    /**
     * A map of { uri: prefix } used by the parser.
     *
     * This map will ensure we can normalize prefixes during processing;
     * for each uri, only one prefix will be exposed to the handlers.
     *
     * @type {!Object<string, string>}}
     */
    var nsUriToPrefix;

    /**
     * Handle parse error.
     *
     * @param  {string|Error} err
     */
    function handleError(err) {
      if (!(err instanceof Error)) {
        err = error$5(err);
      }
      returnError = err;
      onError(err, getContext);
    }

    /**
     * Handle parse error.
     *
     * @param  {string|Error} err
     */
    function handleWarning(err) {
      if (!onWarning) {
        return;
      }
      if (!(err instanceof Error)) {
        err = error$5(err);
      }
      onWarning(err, getContext);
    }

    /**
     * Register parse listener.
     *
     * @param  {string}   name
     * @param  {Function} cb
     *
     * @return {Parser}
     */
    this['on'] = function (name, cb) {
      if (typeof cb !== 'function') {
        throw error$5('required args <name, cb>');
      }
      switch (name) {
        case 'openTag':
          onOpenTag = cb;
          break;
        case 'text':
          onText = cb;
          break;
        case 'closeTag':
          onCloseTag = cb;
          break;
        case 'error':
          onError = cb;
          break;
        case 'warn':
          onWarning = cb;
          break;
        case 'cdata':
          onCDATA = cb;
          break;
        case 'attention':
          onAttention = cb;
          break;
        // <!XXXXX zzzz="eeee">
        case 'question':
          onQuestion = cb;
          break;
        // <? ....  ?>
        case 'comment':
          onComment = cb;
          break;
        default:
          throw error$5('unsupported event: ' + name);
      }
      return this;
    };

    /**
     * Set the namespace to prefix mapping.
     *
     * @example
     *
     * parser.ns({
     *   'http://foo': 'foo',
     *   'http://bar': 'bar'
     * });
     *
     * @param  {!Object<string, string>} nsMap
     *
     * @return {Parser}
     */
    this['ns'] = function (nsMap) {
      if (typeof nsMap === 'undefined') {
        nsMap = {};
      }
      if (typeof nsMap !== 'object') {
        throw error$5('required args <nsMap={}>');
      }
      var _nsUriToPrefix = {},
        k;
      for (k in nsMap) {
        _nsUriToPrefix[k] = nsMap[k];
      }

      // FORCE default mapping for schema instance
      _nsUriToPrefix[XSI_URI] = XSI_PREFIX;
      isNamespace = true;
      nsUriToPrefix = _nsUriToPrefix;
      return this;
    };

    /**
     * Parse xml string.
     *
     * @param  {string} xml
     *
     * @return {Error} returnError, if not thrown
     */
    this['parse'] = function (xml) {
      if (typeof xml !== 'string') {
        throw error$5('required args <xml=string>');
      }
      returnError = null;
      parse(xml);
      getContext = noopGetContext;
      parseStop = false;
      return returnError;
    };

    /**
     * Stop parsing.
     */
    this['stop'] = function () {
      parseStop = true;
    };

    /**
     * Parse string, invoking configured listeners on element.
     *
     * @param  {string} xml
     */
    function parse(xml) {
      var nsMatrixStack = isNamespace ? [] : null,
        nsMatrix = isNamespace ? buildNsMatrix(nsUriToPrefix) : null,
        _nsMatrix,
        nodeStack = [],
        anonymousNsCount = 0,
        tagStart = false,
        tagEnd = false,
        i = 0,
        j = 0,
        x,
        y,
        q,
        w,
        v,
        xmlns,
        elementName,
        _elementName,
        elementProxy;
      var attrsString = '',
        attrsStart = 0,
        cachedAttrs // false = parsed with errors, null = needs parsing
      ;

      /**
       * Parse attributes on demand and returns the parsed attributes.
       *
       * Return semantics: (1) `false` on attribute parse error,
       * (2) object hash on extracted attrs.
       *
       * @return {boolean|Object}
       */
      function getAttrs() {
        if (cachedAttrs !== null) {
          return cachedAttrs;
        }
        var nsUri,
          nsUriPrefix,
          nsName,
          defaultAlias = isNamespace && nsMatrix['xmlns'],
          attrList = isNamespace && maybeNS ? [] : null,
          i = attrsStart,
          s = attrsString,
          l = s.length,
          hasNewMatrix,
          newalias,
          value,
          alias,
          name,
          attrs = {},
          seenAttrs = {},
          skipAttr,
          w,
          j;
        parseAttr: for (; i < l; i++) {
          skipAttr = false;
          w = s.charCodeAt(i);
          if (w === 32 || w < 14 && w > 8) {
            // WHITESPACE={ \f\n\r\t\v}
            continue;
          }

          // wait for non whitespace character
          if (w < 65 || w > 122 || w > 90 && w < 97) {
            if (w !== 95 && w !== 58) {
              // char 95"_" 58":"
              handleWarning('illegal first char attribute name');
              skipAttr = true;
            }
          }

          // parse attribute name
          for (j = i + 1; j < l; j++) {
            w = s.charCodeAt(j);
            if (w > 96 && w < 123 || w > 64 && w < 91 || w > 47 && w < 59 || w === 46 ||
            // '.'
            w === 45 ||
            // '-'
            w === 95 // '_'
            ) {
              continue;
            }

            // unexpected whitespace
            if (w === 32 || w < 14 && w > 8) {
              // WHITESPACE
              handleWarning('missing attribute value');
              i = j;
              continue parseAttr;
            }

            // expected "="
            if (w === 61) {
              // "=" == 61
              break;
            }
            handleWarning('illegal attribute name char');
            skipAttr = true;
          }
          name = s.substring(i, j);
          if (name === 'xmlns:xmlns') {
            handleWarning('illegal declaration of xmlns');
            skipAttr = true;
          }
          w = s.charCodeAt(j + 1);
          if (w === 34) {
            // '"'
            j = s.indexOf('"', i = j + 2);
            if (j === -1) {
              j = s.indexOf('\'', i);
              if (j !== -1) {
                handleWarning('attribute value quote missmatch');
                skipAttr = true;
              }
            }
          } else if (w === 39) {
            // "'"
            j = s.indexOf('\'', i = j + 2);
            if (j === -1) {
              j = s.indexOf('"', i);
              if (j !== -1) {
                handleWarning('attribute value quote missmatch');
                skipAttr = true;
              }
            }
          } else {
            handleWarning('missing attribute value quotes');
            skipAttr = true;

            // skip to next space
            for (j = j + 1; j < l; j++) {
              w = s.charCodeAt(j + 1);
              if (w === 32 || w < 14 && w > 8) {
                // WHITESPACE
                break;
              }
            }
          }
          if (j === -1) {
            handleWarning('missing closing quotes');
            j = l;
            skipAttr = true;
          }
          if (!skipAttr) {
            value = s.substring(i, j);
          }
          i = j;

          // ensure SPACE follows attribute
          // skip illegal content otherwise
          // example a="b"c
          for (; j + 1 < l; j++) {
            w = s.charCodeAt(j + 1);
            if (w === 32 || w < 14 && w > 8) {
              // WHITESPACE
              break;
            }

            // FIRST ILLEGAL CHAR
            if (i === j) {
              handleWarning('illegal character after attribute end');
              skipAttr = true;
            }
          }

          // advance cursor to next attribute
          i = j + 1;
          if (skipAttr) {
            continue parseAttr;
          }

          // check attribute re-declaration
          if (name in seenAttrs) {
            handleWarning('attribute <' + name + '> already defined');
            continue;
          }
          seenAttrs[name] = true;
          if (!isNamespace) {
            attrs[name] = value;
            continue;
          }

          // try to extract namespace information
          if (maybeNS) {
            newalias = name === 'xmlns' ? 'xmlns' : name.charCodeAt(0) === 120 && name.substr(0, 6) === 'xmlns:' ? name.substr(6) : null;

            // handle xmlns(:alias) assignment
            if (newalias !== null) {
              nsUri = decodeEntities(value);
              nsUriPrefix = uriPrefix(newalias);
              alias = nsUriToPrefix[nsUri];
              if (!alias) {
                // no prefix defined or prefix collision
                if (newalias === 'xmlns' || nsUriPrefix in nsMatrix && nsMatrix[nsUriPrefix] !== nsUri) {
                  // alocate free ns prefix
                  do {
                    alias = 'ns' + anonymousNsCount++;
                  } while (typeof nsMatrix[alias] !== 'undefined');
                } else {
                  alias = newalias;
                }
                nsUriToPrefix[nsUri] = alias;
              }
              if (nsMatrix[newalias] !== alias) {
                if (!hasNewMatrix) {
                  nsMatrix = cloneNsMatrix(nsMatrix);
                  hasNewMatrix = true;
                }
                nsMatrix[newalias] = alias;
                if (newalias === 'xmlns') {
                  nsMatrix[uriPrefix(alias)] = nsUri;
                  defaultAlias = alias;
                }
                nsMatrix[nsUriPrefix] = nsUri;
              }

              // expose xmlns(:asd)="..." in attributes
              attrs[name] = value;
              continue;
            }

            // collect attributes until all namespace
            // declarations are processed
            attrList.push(name, value);
            continue;
          } /** end if (maybeNs) */

          // handle attributes on element without
          // namespace declarations
          w = name.indexOf(':');
          if (w === -1) {
            attrs[name] = value;
            continue;
          }

          // normalize ns attribute name
          if (!(nsName = nsMatrix[name.substring(0, w)])) {
            handleWarning(missingNamespaceForPrefix(name.substring(0, w)));
            continue;
          }
          name = defaultAlias === nsName ? name.substr(w + 1) : nsName + name.substr(w);

          // end: normalize ns attribute name

          // normalize xsi:type ns attribute value
          if (name === XSI_TYPE$1) {
            w = value.indexOf(':');
            if (w !== -1) {
              nsName = value.substring(0, w);

              // handle default prefixes, i.e. xs:String gracefully
              nsName = nsMatrix[nsName] || nsName;
              value = nsName + value.substring(w);
            } else {
              value = defaultAlias + ':' + value;
            }
          }

          // end: normalize xsi:type ns attribute value

          attrs[name] = value;
        }

        // handle deferred, possibly namespaced attributes
        if (maybeNS) {
          // normalize captured attributes
          for (i = 0, l = attrList.length; i < l; i++) {
            name = attrList[i++];
            value = attrList[i];
            w = name.indexOf(':');
            if (w !== -1) {
              // normalize ns attribute name
              if (!(nsName = nsMatrix[name.substring(0, w)])) {
                handleWarning(missingNamespaceForPrefix(name.substring(0, w)));
                continue;
              }
              name = defaultAlias === nsName ? name.substr(w + 1) : nsName + name.substr(w);

              // end: normalize ns attribute name

              // normalize xsi:type ns attribute value
              if (name === XSI_TYPE$1) {
                w = value.indexOf(':');
                if (w !== -1) {
                  nsName = value.substring(0, w);

                  // handle default prefixes, i.e. xs:String gracefully
                  nsName = nsMatrix[nsName] || nsName;
                  value = nsName + value.substring(w);
                } else {
                  value = defaultAlias + ':' + value;
                }
              }

              // end: normalize xsi:type ns attribute value
            }
            attrs[name] = value;
          }

          // end: normalize captured attributes
        }
        return cachedAttrs = attrs;
      }

      /**
       * Extract the parse context { line, column, part }
       * from the current parser position.
       *
       * @return {Object} parse context
       */
      function getParseContext() {
        var splitsRe = /(\r\n|\r|\n)/g;
        var line = 0;
        var column = 0;
        var startOfLine = 0;
        var endOfLine = j;
        var match;
        var data;
        while (i >= startOfLine) {
          match = splitsRe.exec(xml);
          if (!match) {
            break;
          }

          // end of line = (break idx + break chars)
          endOfLine = match[0].length + match.index;
          if (endOfLine > i) {
            break;
          }

          // advance to next line
          line += 1;
          startOfLine = endOfLine;
        }

        // EOF errors
        if (i == -1) {
          column = endOfLine;
          data = xml.substring(j);
        } else
          // start errors
          if (j === 0) {
            data = xml.substring(j, i);
          }

          // other errors
          else {
            column = i - startOfLine;
            data = j == -1 ? xml.substring(i) : xml.substring(i, j + 1);
          }
        return {
          'data': data,
          'line': line,
          'column': column
        };
      }
      getContext = getParseContext;
      if (proxy) {
        elementProxy = Object.create({}, {
          'name': getter(function () {
            return elementName;
          }),
          'originalName': getter(function () {
            return _elementName;
          }),
          'attrs': getter(getAttrs),
          'ns': getter(function () {
            return nsMatrix;
          })
        });
      }

      // actual parse logic
      while (j !== -1) {
        if (xml.charCodeAt(j) === 60) {
          // "<"
          i = j;
        } else {
          i = xml.indexOf('<', j);
        }

        // parse end
        if (i === -1) {
          if (nodeStack.length) {
            return handleError('unexpected end of file');
          }
          if (j === 0) {
            return handleError('missing start tag');
          }
          if (j < xml.length) {
            if (xml.substring(j).trim()) {
              handleWarning(NON_WHITESPACE_OUTSIDE_ROOT_NODE);
            }
          }
          return;
        }

        // parse text
        if (j !== i) {
          if (nodeStack.length) {
            if (onText) {
              onText(xml.substring(j, i), decodeEntities, getContext);
              if (parseStop) {
                return;
              }
            }
          } else {
            if (xml.substring(j, i).trim()) {
              handleWarning(NON_WHITESPACE_OUTSIDE_ROOT_NODE);
              if (parseStop) {
                return;
              }
            }
          }
        }
        w = xml.charCodeAt(i + 1);

        // parse comments + CDATA
        if (w === 33) {
          // "!"
          q = xml.charCodeAt(i + 2);

          // CDATA section
          if (q === 91 && xml.substr(i + 3, 6) === 'CDATA[') {
            // 91 == "["
            j = xml.indexOf(']]>', i);
            if (j === -1) {
              return handleError('unclosed cdata');
            }
            if (onCDATA) {
              onCDATA(xml.substring(i + 9, j), getContext);
              if (parseStop) {
                return;
              }
            }
            j += 3;
            continue;
          }

          // comment
          if (q === 45 && xml.charCodeAt(i + 3) === 45) {
            // 45 == "-"
            j = xml.indexOf('-->', i);
            if (j === -1) {
              return handleError('unclosed comment');
            }
            if (onComment) {
              onComment(xml.substring(i + 4, j), decodeEntities, getContext);
              if (parseStop) {
                return;
              }
            }
            j += 3;
            continue;
          }
        }

        // parse question <? ... ?>
        if (w === 63) {
          // "?"
          j = xml.indexOf('?>', i);
          if (j === -1) {
            return handleError('unclosed question');
          }
          if (onQuestion) {
            onQuestion(xml.substring(i, j + 2), getContext);
            if (parseStop) {
              return;
            }
          }
          j += 2;
          continue;
        }

        // find matching closing tag for attention or standard tags
        // for that we must skip through attribute values
        // (enclosed in single or double quotes)
        for (x = i + 1;; x++) {
          v = xml.charCodeAt(x);
          if (isNaN(v)) {
            j = -1;
            return handleError('unclosed tag');
          }

          // [10] AttValue ::= '"' ([^<&"] | Reference)* '"' | "'" ([^<&'] | Reference)* "'"
          // skips the quoted string
          // (double quotes) does not appear in a literal enclosed by (double quotes)
          // (single quote) does not appear in a literal enclosed by (single quote)
          if (v === 34) {
            //  '"'
            q = xml.indexOf('"', x + 1);
            x = q !== -1 ? q : x;
          } else if (v === 39) {
            // "'"
            q = xml.indexOf("'", x + 1);
            x = q !== -1 ? q : x;
          } else if (v === 62) {
            // '>'
            j = x;
            break;
          }
        }

        // parse attention <! ...>
        // previously comment and CDATA have already been parsed
        if (w === 33) {
          // "!"

          if (onAttention) {
            onAttention(xml.substring(i, j + 1), decodeEntities, getContext);
            if (parseStop) {
              return;
            }
          }
          j += 1;
          continue;
        }

        // don't process attributes;
        // there are none
        cachedAttrs = {};

        // if (xml.charCodeAt(i+1) === 47) { // </...
        if (w === 47) {
          // </...
          tagStart = false;
          tagEnd = true;
          if (!nodeStack.length) {
            return handleError('missing open tag');
          }

          // verify open <-> close tag match
          x = elementName = nodeStack.pop();
          q = i + 2 + x.length;
          if (xml.substring(i + 2, q) !== x) {
            return handleError('closing tag mismatch');
          }

          // verify chars in close tag
          for (; q < j; q++) {
            w = xml.charCodeAt(q);
            if (w === 32 || w > 8 && w < 14) {
              // \f\n\r\t\v space
              continue;
            }
            return handleError('close tag');
          }
        } else {
          if (xml.charCodeAt(j - 1) === 47) {
            // .../>
            x = elementName = xml.substring(i + 1, j - 1);
            tagStart = true;
            tagEnd = true;
          } else {
            x = elementName = xml.substring(i + 1, j);
            tagStart = true;
            tagEnd = false;
          }
          if (!(w > 96 && w < 123 || w > 64 && w < 91 || w === 95 || w === 58)) {
            // char 95"_" 58":"
            return handleError('illegal first char nodeName');
          }
          for (q = 1, y = x.length; q < y; q++) {
            w = x.charCodeAt(q);
            if (w > 96 && w < 123 || w > 64 && w < 91 || w > 47 && w < 59 || w === 45 || w === 95 || w == 46) {
              continue;
            }
            if (w === 32 || w < 14 && w > 8) {
              // \f\n\r\t\v space
              elementName = x.substring(0, q);

              // maybe there are attributes
              cachedAttrs = null;
              break;
            }
            return handleError('invalid nodeName');
          }
          if (!tagEnd) {
            nodeStack.push(elementName);
          }
        }
        if (isNamespace) {
          _nsMatrix = nsMatrix;
          if (tagStart) {
            // remember old namespace
            // unless we're self-closing
            if (!tagEnd) {
              nsMatrixStack.push(_nsMatrix);
            }
            if (cachedAttrs === null) {
              // quick check, whether there may be namespace
              // declarations on the node; if that is the case
              // we need to eagerly parse the node attributes
              if (maybeNS = x.indexOf('xmlns', q) !== -1) {
                attrsStart = q;
                attrsString = x;
                getAttrs();
                maybeNS = false;
              }
            }
          }
          _elementName = elementName;
          w = elementName.indexOf(':');
          if (w !== -1) {
            xmlns = nsMatrix[elementName.substring(0, w)];

            // prefix given; namespace must exist
            if (!xmlns) {
              return handleError('missing namespace on <' + _elementName + '>');
            }
            elementName = elementName.substr(w + 1);
          } else {
            xmlns = nsMatrix['xmlns'];

            // if no default namespace is defined,
            // we'll import the element as anonymous.
            //
            // it is up to users to correct that to the document defined
            // targetNamespace, or whatever their undersanding of the
            // XML spec mandates.
          }

          // adjust namespace prefixs as configured
          if (xmlns) {
            elementName = xmlns + ':' + elementName;
          }
        }
        if (tagStart) {
          attrsStart = q;
          attrsString = x;
          if (onOpenTag) {
            if (proxy) {
              onOpenTag(elementProxy, decodeEntities, tagEnd, getContext);
            } else {
              onOpenTag(elementName, getAttrs, decodeEntities, tagEnd, getContext);
            }
            if (parseStop) {
              return;
            }
          }
        }
        if (tagEnd) {
          if (onCloseTag) {
            onCloseTag(proxy ? elementProxy : elementName, decodeEntities, tagStart, getContext);
            if (parseStop) {
              return;
            }
          }

          // restore old namespace
          if (isNamespace) {
            if (!tagStart) {
              nsMatrix = nsMatrixStack.pop();
            } else {
              nsMatrix = _nsMatrix;
            }
          }
        }
        j += 1;
      }
    } /** end parse */
  }

  function hasLowerCaseAlias(pkg) {
    return pkg.xml && pkg.xml.tagAlias === 'lowerCase';
  }
  var DEFAULT_NS_MAP = {
    'xsi': 'http://www.w3.org/2001/XMLSchema-instance',
    'xml': 'http://www.w3.org/XML/1998/namespace'
  };
  var XSI_TYPE = 'xsi:type';
  function serializeFormat(element) {
    return element.xml && element.xml.serialize;
  }
  function serializeAsType(element) {
    return serializeFormat(element) === XSI_TYPE;
  }
  function serializeAsProperty(element) {
    return serializeFormat(element) === 'property';
  }
  function capitalize(str) {
    return str.charAt(0).toUpperCase() + str.slice(1);
  }
  function aliasToName(aliasNs, pkg) {
    if (!hasLowerCaseAlias(pkg)) {
      return aliasNs.name;
    }
    return aliasNs.prefix + ':' + capitalize(aliasNs.localName);
  }
  function prefixedToName(nameNs, pkg) {
    var name = nameNs.name,
      localName = nameNs.localName;
    var typePrefix = pkg.xml && pkg.xml.typePrefix;
    if (typePrefix && localName.indexOf(typePrefix) === 0) {
      return nameNs.prefix + ':' + localName.slice(typePrefix.length);
    } else {
      return name;
    }
  }
  function normalizeXsiTypeName(name, model) {
    var nameNs = parseName$1(name);
    var pkg = model.getPackage(nameNs.prefix);
    return prefixedToName(nameNs, pkg);
  }
  function error$4(message) {
    return new Error(message);
  }

  /**
   * Get the moddle descriptor for a given instance or type.
   *
   * @param  {ModdleElement|Function} element
   *
   * @return {Object} the moddle descriptor
   */
  function getModdleDescriptor(element) {
    return element.$descriptor;
  }

  /**
   * A parse context.
   *
   * @class
   *
   * @param {Object} options
   * @param {ElementHandler} options.rootHandler the root handler for parsing a document
   * @param {boolean} [options.lax=false] whether or not to ignore invalid elements
   */
  function Context$1(options) {
    /**
     * @property {ElementHandler} rootHandler
     */

    /**
     * @property {Boolean} lax
     */

    assign$1(this, options);
    this.elementsById = {};
    this.references = [];
    this.warnings = [];

    /**
     * Add an unresolved reference.
     *
     * @param {Object} reference
     */
    this.addReference = function (reference) {
      this.references.push(reference);
    };

    /**
     * Add a processed element.
     *
     * @param {ModdleElement} element
     */
    this.addElement = function (element) {
      if (!element) {
        throw error$4('expected element');
      }
      var elementsById = this.elementsById;
      var descriptor = getModdleDescriptor(element);
      var idProperty = descriptor.idProperty,
        id;
      if (idProperty) {
        id = element.get(idProperty.name);
        if (id) {
          // for QName validation as per http://www.w3.org/TR/REC-xml/#NT-NameChar
          if (!/^([a-z][\w-.]*:)?[a-z_][\w-.]*$/i.test(id)) {
            throw new Error('illegal ID <' + id + '>');
          }
          if (elementsById[id]) {
            throw error$4('duplicate ID <' + id + '>');
          }
          elementsById[id] = element;
        }
      }
    };

    /**
     * Add an import warning.
     *
     * @param {Object} warning
     * @param {String} warning.message
     * @param {Error} [warning.error]
     */
    this.addWarning = function (warning) {
      this.warnings.push(warning);
    };
  }
  function BaseHandler() {}
  BaseHandler.prototype.handleEnd = function () {};
  BaseHandler.prototype.handleText = function () {};
  BaseHandler.prototype.handleNode = function () {};

  /**
   * A simple pass through handler that does nothing except for
   * ignoring all input it receives.
   *
   * This is used to ignore unknown elements and
   * attributes.
   */
  function NoopHandler() {}
  NoopHandler.prototype = Object.create(BaseHandler.prototype);
  NoopHandler.prototype.handleNode = function () {
    return this;
  };
  function BodyHandler() {}
  BodyHandler.prototype = Object.create(BaseHandler.prototype);
  BodyHandler.prototype.handleText = function (text) {
    this.body = (this.body || '') + text;
  };
  function ReferenceHandler(property, context) {
    this.property = property;
    this.context = context;
  }
  ReferenceHandler.prototype = Object.create(BodyHandler.prototype);
  ReferenceHandler.prototype.handleNode = function (node) {
    if (this.element) {
      throw error$4('expected no sub nodes');
    } else {
      this.element = this.createReference(node);
    }
    return this;
  };
  ReferenceHandler.prototype.handleEnd = function () {
    this.element.id = this.body;
  };
  ReferenceHandler.prototype.createReference = function (node) {
    return {
      property: this.property.ns.name,
      id: ''
    };
  };
  function ValueHandler(propertyDesc, element) {
    this.element = element;
    this.propertyDesc = propertyDesc;
  }
  ValueHandler.prototype = Object.create(BodyHandler.prototype);
  ValueHandler.prototype.handleEnd = function () {
    var value = this.body || '',
      element = this.element,
      propertyDesc = this.propertyDesc;
    value = coerceType(propertyDesc.type, value);
    if (propertyDesc.isMany) {
      element.get(propertyDesc.name).push(value);
    } else {
      element.set(propertyDesc.name, value);
    }
  };
  function BaseElementHandler() {}
  BaseElementHandler.prototype = Object.create(BodyHandler.prototype);
  BaseElementHandler.prototype.handleNode = function (node) {
    var parser = this,
      element = this.element;
    if (!element) {
      element = this.element = this.createElement(node);
      this.context.addElement(element);
    } else {
      parser = this.handleChild(node);
    }
    return parser;
  };

  /**
   * @class Reader.ElementHandler
   *
   */
  function ElementHandler(model, typeName, context) {
    this.model = model;
    this.type = model.getType(typeName);
    this.context = context;
  }
  ElementHandler.prototype = Object.create(BaseElementHandler.prototype);
  ElementHandler.prototype.addReference = function (reference) {
    this.context.addReference(reference);
  };
  ElementHandler.prototype.handleText = function (text) {
    var element = this.element,
      descriptor = getModdleDescriptor(element),
      bodyProperty = descriptor.bodyProperty;
    if (!bodyProperty) {
      throw error$4('unexpected body text <' + text + '>');
    }
    BodyHandler.prototype.handleText.call(this, text);
  };
  ElementHandler.prototype.handleEnd = function () {
    var value = this.body,
      element = this.element,
      descriptor = getModdleDescriptor(element),
      bodyProperty = descriptor.bodyProperty;
    if (bodyProperty && value !== undefined) {
      value = coerceType(bodyProperty.type, value);
      element.set(bodyProperty.name, value);
    }
  };

  /**
   * Create an instance of the model from the given node.
   *
   * @param  {Element} node the xml node
   */
  ElementHandler.prototype.createElement = function (node) {
    var attributes = node.attributes,
      Type = this.type,
      descriptor = getModdleDescriptor(Type),
      context = this.context,
      instance = new Type({}),
      model = this.model,
      propNameNs;
    forEach$1(attributes, function (value, name) {
      var prop = descriptor.propertiesByName[name],
        values;
      if (prop && prop.isReference) {
        if (!prop.isMany) {
          context.addReference({
            element: instance,
            property: prop.ns.name,
            id: value
          });
        } else {
          // IDREFS: parse references as whitespace-separated list
          values = value.split(' ');
          forEach$1(values, function (v) {
            context.addReference({
              element: instance,
              property: prop.ns.name,
              id: v
            });
          });
        }
      } else {
        if (prop) {
          value = coerceType(prop.type, value);
        } else if (name !== 'xmlns') {
          propNameNs = parseName$1(name, descriptor.ns.prefix);

          // check whether attribute is defined in a well-known namespace
          // if that is the case we emit a warning to indicate potential misuse
          if (model.getPackage(propNameNs.prefix)) {
            context.addWarning({
              message: 'unknown attribute <' + name + '>',
              element: instance,
              property: name,
              value: value
            });
          }
        }
        instance.set(name, value);
      }
    });
    return instance;
  };
  ElementHandler.prototype.getPropertyForNode = function (node) {
    var name = node.name;
    var nameNs = parseName$1(name);
    var type = this.type,
      model = this.model,
      descriptor = getModdleDescriptor(type);
    var propertyName = nameNs.name,
      property = descriptor.propertiesByName[propertyName],
      elementTypeName,
      elementType;

    // search for properties by name first

    if (property && !property.isAttr) {
      if (serializeAsType(property)) {
        elementTypeName = node.attributes[XSI_TYPE];

        // xsi type is optional, if it does not exists the
        // default type is assumed
        if (elementTypeName) {
          // take possible type prefixes from XML
          // into account, i.e.: xsi:type="t{ActualType}"
          elementTypeName = normalizeXsiTypeName(elementTypeName, model);
          elementType = model.getType(elementTypeName);
          return assign$1({}, property, {
            effectiveType: getModdleDescriptor(elementType).name
          });
        }
      }

      // search for properties by name first
      return property;
    }
    var pkg = model.getPackage(nameNs.prefix);
    if (pkg) {
      elementTypeName = aliasToName(nameNs, pkg);
      elementType = model.getType(elementTypeName);

      // search for collection members later
      property = find$1(descriptor.properties, function (p) {
        return !p.isVirtual && !p.isReference && !p.isAttribute && elementType.hasType(p.type);
      });
      if (property) {
        return assign$1({}, property, {
          effectiveType: getModdleDescriptor(elementType).name
        });
      }
    } else {
      // parse unknown element (maybe extension)
      property = find$1(descriptor.properties, function (p) {
        return !p.isReference && !p.isAttribute && p.type === 'Element';
      });
      if (property) {
        return property;
      }
    }
    throw error$4('unrecognized element <' + nameNs.name + '>');
  };
  ElementHandler.prototype.toString = function () {
    return 'ElementDescriptor[' + getModdleDescriptor(this.type).name + ']';
  };
  ElementHandler.prototype.valueHandler = function (propertyDesc, element) {
    return new ValueHandler(propertyDesc, element);
  };
  ElementHandler.prototype.referenceHandler = function (propertyDesc) {
    return new ReferenceHandler(propertyDesc, this.context);
  };
  ElementHandler.prototype.handler = function (type) {
    if (type === 'Element') {
      return new GenericElementHandler(this.model, type, this.context);
    } else {
      return new ElementHandler(this.model, type, this.context);
    }
  };

  /**
   * Handle the child element parsing
   *
   * @param  {Element} node the xml node
   */
  ElementHandler.prototype.handleChild = function (node) {
    var propertyDesc, type, element, childHandler;
    propertyDesc = this.getPropertyForNode(node);
    element = this.element;
    type = propertyDesc.effectiveType || propertyDesc.type;
    if (isSimple(type)) {
      return this.valueHandler(propertyDesc, element);
    }
    if (propertyDesc.isReference) {
      childHandler = this.referenceHandler(propertyDesc).handleNode(node);
    } else {
      childHandler = this.handler(type).handleNode(node);
    }
    var newElement = childHandler.element;

    // child handles may decide to skip elements
    // by not returning anything
    if (newElement !== undefined) {
      if (propertyDesc.isMany) {
        element.get(propertyDesc.name).push(newElement);
      } else {
        element.set(propertyDesc.name, newElement);
      }
      if (propertyDesc.isReference) {
        assign$1(newElement, {
          element: element
        });
        this.context.addReference(newElement);
      } else {
        // establish child -> parent relationship
        newElement.$parent = element;
      }
    }
    return childHandler;
  };

  /**
   * An element handler that performs special validation
   * to ensure the node it gets initialized with matches
   * the handlers type (namespace wise).
   *
   * @param {Moddle} model
   * @param {String} typeName
   * @param {Context} context
   */
  function RootElementHandler(model, typeName, context) {
    ElementHandler.call(this, model, typeName, context);
  }
  RootElementHandler.prototype = Object.create(ElementHandler.prototype);
  RootElementHandler.prototype.createElement = function (node) {
    var name = node.name,
      nameNs = parseName$1(name),
      model = this.model,
      type = this.type,
      pkg = model.getPackage(nameNs.prefix),
      typeName = pkg && aliasToName(nameNs, pkg) || name;

    // verify the correct namespace if we parse
    // the first element in the handler tree
    //
    // this ensures we don't mistakenly import wrong namespace elements
    if (!type.hasType(typeName)) {
      throw error$4('unexpected element <' + node.originalName + '>');
    }
    return ElementHandler.prototype.createElement.call(this, node);
  };
  function GenericElementHandler(model, typeName, context) {
    this.model = model;
    this.context = context;
  }
  GenericElementHandler.prototype = Object.create(BaseElementHandler.prototype);
  GenericElementHandler.prototype.createElement = function (node) {
    var name = node.name,
      ns = parseName$1(name),
      prefix = ns.prefix,
      uri = node.ns[prefix + '$uri'],
      attributes = node.attributes;
    return this.model.createAny(name, uri, attributes);
  };
  GenericElementHandler.prototype.handleChild = function (node) {
    var handler = new GenericElementHandler(this.model, 'Element', this.context).handleNode(node),
      element = this.element;
    var newElement = handler.element,
      children;
    if (newElement !== undefined) {
      children = element.$children = element.$children || [];
      children.push(newElement);

      // establish child -> parent relationship
      newElement.$parent = element;
    }
    return handler;
  };
  GenericElementHandler.prototype.handleEnd = function () {
    if (this.body) {
      this.element.$body = this.body;
    }
  };

  /**
   * A reader for a meta-model
   *
   * @param {Object} options
   * @param {Model} options.model used to read xml files
   * @param {Boolean} options.lax whether to make parse errors warnings
   */
  function Reader(options) {
    if (options instanceof Moddle) {
      options = {
        model: options
      };
    }
    assign$1(this, {
      lax: false
    }, options);
  }

  /**
   * The fromXML result.
   *
   * @typedef {Object} ParseResult
   *
   * @property {ModdleElement} rootElement
   * @property {Array<Object>} references
   * @property {Array<Error>} warnings
   * @property {Object} elementsById - a mapping containing each ID -> ModdleElement
   */

  /**
   * The fromXML result.
   *
   * @typedef {Error} ParseError
   *
   * @property {Array<Error>} warnings
   */

  /**
   * Parse the given XML into a moddle document tree.
   *
   * @param {String} xml
   * @param {ElementHandler|Object} options or rootHandler
   *
   * @returns {Promise<ParseResult, ParseError>}
   */
  Reader.prototype.fromXML = function (xml, options, done) {
    var rootHandler = options.rootHandler;
    if (options instanceof ElementHandler) {
      // root handler passed via (xml, { rootHandler: ElementHandler }, ...)
      rootHandler = options;
      options = {};
    } else {
      if (typeof options === 'string') {
        // rootHandler passed via (xml, 'someString', ...)
        rootHandler = this.handler(options);
        options = {};
      } else if (typeof rootHandler === 'string') {
        // rootHandler passed via (xml, { rootHandler: 'someString' }, ...)
        rootHandler = this.handler(rootHandler);
      }
    }
    var model = this.model,
      lax = this.lax;
    var context = new Context$1(assign$1({}, options, {
        rootHandler: rootHandler
      })),
      parser = new Parser$1({
        proxy: true
      }),
      stack = createStack();
    rootHandler.context = context;

    // push root handler
    stack.push(rootHandler);

    /**
     * Handle error.
     *
     * @param  {Error} err
     * @param  {Function} getContext
     * @param  {boolean} lax
     *
     * @return {boolean} true if handled
     */
    function handleError(err, getContext, lax) {
      var ctx = getContext();
      var line = ctx.line,
        column = ctx.column,
        data = ctx.data;

      // we receive the full context data here,
      // for elements trim down the information
      // to the tag name, only
      if (data.charAt(0) === '<' && data.indexOf(' ') !== -1) {
        data = data.slice(0, data.indexOf(' ')) + '>';
      }
      var message = 'unparsable content ' + (data ? data + ' ' : '') + 'detected\n\t' + 'line: ' + line + '\n\t' + 'column: ' + column + '\n\t' + 'nested error: ' + err.message;
      if (lax) {
        context.addWarning({
          message: message,
          error: err
        });
        return true;
      } else {
        throw error$4(message);
      }
    }
    function handleWarning(err, getContext) {
      // just like handling errors in <lax=true> mode
      return handleError(err, getContext, true);
    }

    /**
     * Resolve collected references on parse end.
     */
    function resolveReferences() {
      var elementsById = context.elementsById;
      var references = context.references;
      var i, r;
      for (i = 0; r = references[i]; i++) {
        var element = r.element;
        var reference = elementsById[r.id];
        var property = getModdleDescriptor(element).propertiesByName[r.property];
        if (!reference) {
          context.addWarning({
            message: 'unresolved reference <' + r.id + '>',
            element: r.element,
            property: r.property,
            value: r.id
          });
        }
        if (property.isMany) {
          var collection = element.get(property.name),
            idx = collection.indexOf(r);

          // we replace an existing place holder (idx != -1) or
          // append to the collection instead
          if (idx === -1) {
            idx = collection.length;
          }
          if (!reference) {
            // remove unresolvable reference
            collection.splice(idx, 1);
          } else {
            // add or update reference in collection
            collection[idx] = reference;
          }
        } else {
          element.set(property.name, reference);
        }
      }
    }
    function handleClose() {
      stack.pop().handleEnd();
    }
    var PREAMBLE_START_PATTERN = /^<\?xml /i;
    var ENCODING_PATTERN = / encoding="([^"]+)"/i;
    var UTF_8_PATTERN = /^utf-8$/i;
    function handleQuestion(question) {
      if (!PREAMBLE_START_PATTERN.test(question)) {
        return;
      }
      var match = ENCODING_PATTERN.exec(question);
      var encoding = match && match[1];
      if (!encoding || UTF_8_PATTERN.test(encoding)) {
        return;
      }
      context.addWarning({
        message: 'unsupported document encoding <' + encoding + '>, ' + 'falling back to UTF-8'
      });
    }
    function handleOpen(node, getContext) {
      var handler = stack.peek();
      try {
        stack.push(handler.handleNode(node));
      } catch (err) {
        if (handleError(err, getContext, lax)) {
          stack.push(new NoopHandler());
        }
      }
    }
    function handleCData(text, getContext) {
      try {
        stack.peek().handleText(text);
      } catch (err) {
        handleWarning(err, getContext);
      }
    }
    function handleText(text, getContext) {
      // strip whitespace only nodes, i.e. before
      // <!CDATA[ ... ]> sections and in between tags

      if (!text.trim()) {
        return;
      }
      handleCData(text, getContext);
    }
    var uriMap = model.getPackages().reduce(function (uriMap, p) {
      uriMap[p.uri] = p.prefix;
      return uriMap;
    }, {
      'http://www.w3.org/XML/1998/namespace': 'xml' // add default xml ns
    });
    parser.ns(uriMap).on('openTag', function (obj, decodeStr, selfClosing, getContext) {
      // gracefully handle unparsable attributes (attrs=false)
      var attrs = obj.attrs || {};
      var decodedAttrs = Object.keys(attrs).reduce(function (d, key) {
        var value = decodeStr(attrs[key]);
        d[key] = value;
        return d;
      }, {});
      var node = {
        name: obj.name,
        originalName: obj.originalName,
        attributes: decodedAttrs,
        ns: obj.ns
      };
      handleOpen(node, getContext);
    }).on('question', handleQuestion).on('closeTag', handleClose).on('cdata', handleCData).on('text', function (text, decodeEntities, getContext) {
      handleText(decodeEntities(text), getContext);
    }).on('error', handleError).on('warn', handleWarning);

    // async XML parsing to make sure the execution environment
    // (node or brower) is kept responsive and that certain optimization
    // strategies can kick in.
    return new Promise(function (resolve, reject) {
      var err;
      try {
        parser.parse(xml);
        resolveReferences();
      } catch (e) {
        err = e;
      }
      var rootElement = rootHandler.element;
      if (!err && !rootElement) {
        err = error$4('failed to parse document as <' + rootHandler.type.$descriptor.name + '>');
      }
      var warnings = context.warnings;
      var references = context.references;
      var elementsById = context.elementsById;
      if (err) {
        err.warnings = warnings;
        return reject(err);
      } else {
        return resolve({
          rootElement: rootElement,
          elementsById: elementsById,
          references: references,
          warnings: warnings
        });
      }
    });
  };
  Reader.prototype.handler = function (name) {
    return new RootElementHandler(this.model, name);
  };

  // helpers //////////////////////////

  function createStack() {
    var stack = [];
    Object.defineProperty(stack, 'peek', {
      value: function () {
        return this[this.length - 1];
      }
    });
    return stack;
  }
  var XML_PREAMBLE = '<?xml version="1.0" encoding="UTF-8"?>\n';
  var ESCAPE_ATTR_CHARS = /<|>|'|"|&|\n\r|\n/g;
  var ESCAPE_CHARS = /<|>|&/g;
  function Namespaces(parent) {
    var prefixMap = {};
    var uriMap = {};
    var used = {};
    var wellknown = [];
    var custom = [];

    // API

    this.byUri = function (uri) {
      return uriMap[uri] || parent && parent.byUri(uri);
    };
    this.add = function (ns, isWellknown) {
      uriMap[ns.uri] = ns;
      if (isWellknown) {
        wellknown.push(ns);
      } else {
        custom.push(ns);
      }
      this.mapPrefix(ns.prefix, ns.uri);
    };
    this.uriByPrefix = function (prefix) {
      return prefixMap[prefix || 'xmlns'];
    };
    this.mapPrefix = function (prefix, uri) {
      prefixMap[prefix || 'xmlns'] = uri;
    };
    this.getNSKey = function (ns) {
      return ns.prefix !== undefined ? ns.uri + '|' + ns.prefix : ns.uri;
    };
    this.logUsed = function (ns) {
      var uri = ns.uri;
      var nsKey = this.getNSKey(ns);
      used[nsKey] = this.byUri(uri);

      // Inform parent recursively about the usage of this NS
      if (parent) {
        parent.logUsed(ns);
      }
    };
    this.getUsed = function (ns) {
      function isUsed(ns) {
        var nsKey = self.getNSKey(ns);
        return used[nsKey];
      }
      var self = this;
      var allNs = [].concat(wellknown, custom);
      return allNs.filter(isUsed);
    };
  }
  function lower(string) {
    return string.charAt(0).toLowerCase() + string.slice(1);
  }
  function nameToAlias(name, pkg) {
    if (hasLowerCaseAlias(pkg)) {
      return lower(name);
    } else {
      return name;
    }
  }
  function inherits(ctor, superCtor) {
    ctor.super_ = superCtor;
    ctor.prototype = Object.create(superCtor.prototype, {
      constructor: {
        value: ctor,
        enumerable: false,
        writable: true,
        configurable: true
      }
    });
  }
  function nsName(ns) {
    if (isString$1(ns)) {
      return ns;
    } else {
      return (ns.prefix ? ns.prefix + ':' : '') + ns.localName;
    }
  }
  function getNsAttrs(namespaces) {
    return namespaces.getUsed().filter(function (ns) {
      // do not serialize built in <xml> namespace
      return ns.prefix !== 'xml';
    }).map(function (ns) {
      var name = 'xmlns' + (ns.prefix ? ':' + ns.prefix : '');
      return {
        name: name,
        value: ns.uri
      };
    });
  }
  function getElementNs(ns, descriptor) {
    if (descriptor.isGeneric) {
      return assign$1({
        localName: descriptor.ns.localName
      }, ns);
    } else {
      return assign$1({
        localName: nameToAlias(descriptor.ns.localName, descriptor.$pkg)
      }, ns);
    }
  }
  function getPropertyNs(ns, descriptor) {
    return assign$1({
      localName: descriptor.ns.localName
    }, ns);
  }
  function getSerializableProperties(element) {
    var descriptor = element.$descriptor;
    return filter$1(descriptor.properties, function (p) {
      var name = p.name;
      if (p.isVirtual) {
        return false;
      }

      // do not serialize defaults
      if (!has$1(element, name)) {
        return false;
      }
      var value = element[name];

      // do not serialize default equals
      if (value === p.default) {
        return false;
      }

      // do not serialize null properties
      if (value === null) {
        return false;
      }
      return p.isMany ? value.length : true;
    });
  }
  var ESCAPE_ATTR_MAP = {
    '\n': '#10',
    '\n\r': '#10',
    '"': '#34',
    '\'': '#39',
    '<': '#60',
    '>': '#62',
    '&': '#38'
  };
  var ESCAPE_MAP = {
    '<': 'lt',
    '>': 'gt',
    '&': 'amp'
  };
  function escape$1(str, charPattern, replaceMap) {
    // ensure we are handling strings here
    str = isString$1(str) ? str : '' + str;
    return str.replace(charPattern, function (s) {
      return '&' + replaceMap[s] + ';';
    });
  }

  /**
   * Escape a string attribute to not contain any bad values (line breaks, '"', ...)
   *
   * @param {String} str the string to escape
   * @return {String} the escaped string
   */
  function escapeAttr(str) {
    return escape$1(str, ESCAPE_ATTR_CHARS, ESCAPE_ATTR_MAP);
  }
  function escapeBody(str) {
    return escape$1(str, ESCAPE_CHARS, ESCAPE_MAP);
  }
  function filterAttributes(props) {
    return filter$1(props, function (p) {
      return p.isAttr;
    });
  }
  function filterContained(props) {
    return filter$1(props, function (p) {
      return !p.isAttr;
    });
  }
  function ReferenceSerializer(tagName) {
    this.tagName = tagName;
  }
  ReferenceSerializer.prototype.build = function (element) {
    this.element = element;
    return this;
  };
  ReferenceSerializer.prototype.serializeTo = function (writer) {
    writer.appendIndent().append('<' + this.tagName + '>' + this.element.id + '</' + this.tagName + '>').appendNewLine();
  };
  function BodySerializer() {}
  BodySerializer.prototype.serializeValue = BodySerializer.prototype.serializeTo = function (writer) {
    writer.append(this.escape ? escapeBody(this.value) : this.value);
  };
  BodySerializer.prototype.build = function (prop, value) {
    this.value = value;
    if (prop.type === 'String' && value.search(ESCAPE_CHARS) !== -1) {
      this.escape = true;
    }
    return this;
  };
  function ValueSerializer(tagName) {
    this.tagName = tagName;
  }
  inherits(ValueSerializer, BodySerializer);
  ValueSerializer.prototype.serializeTo = function (writer) {
    writer.appendIndent().append('<' + this.tagName + '>');
    this.serializeValue(writer);
    writer.append('</' + this.tagName + '>').appendNewLine();
  };
  function ElementSerializer(parent, propertyDescriptor) {
    this.body = [];
    this.attrs = [];
    this.parent = parent;
    this.propertyDescriptor = propertyDescriptor;
  }
  ElementSerializer.prototype.build = function (element) {
    this.element = element;
    var elementDescriptor = element.$descriptor,
      propertyDescriptor = this.propertyDescriptor;
    var otherAttrs, properties;
    var isGeneric = elementDescriptor.isGeneric;
    if (isGeneric) {
      otherAttrs = this.parseGeneric(element);
    } else {
      otherAttrs = this.parseNsAttributes(element);
    }
    if (propertyDescriptor) {
      this.ns = this.nsPropertyTagName(propertyDescriptor);
    } else {
      this.ns = this.nsTagName(elementDescriptor);
    }

    // compute tag name
    this.tagName = this.addTagName(this.ns);
    if (!isGeneric) {
      properties = getSerializableProperties(element);
      this.parseAttributes(filterAttributes(properties));
      this.parseContainments(filterContained(properties));
    }
    this.parseGenericAttributes(element, otherAttrs);
    return this;
  };
  ElementSerializer.prototype.nsTagName = function (descriptor) {
    var effectiveNs = this.logNamespaceUsed(descriptor.ns);
    return getElementNs(effectiveNs, descriptor);
  };
  ElementSerializer.prototype.nsPropertyTagName = function (descriptor) {
    var effectiveNs = this.logNamespaceUsed(descriptor.ns);
    return getPropertyNs(effectiveNs, descriptor);
  };
  ElementSerializer.prototype.isLocalNs = function (ns) {
    return ns.uri === this.ns.uri;
  };

  /**
   * Get the actual ns attribute name for the given element.
   *
   * @param {Object} element
   * @param {Boolean} [element.inherited=false]
   *
   * @return {Object} nsName
   */
  ElementSerializer.prototype.nsAttributeName = function (element) {
    var ns;
    if (isString$1(element)) {
      ns = parseName$1(element);
    } else {
      ns = element.ns;
    }

    // return just local name for inherited attributes
    if (element.inherited) {
      return {
        localName: ns.localName
      };
    }

    // parse + log effective ns
    var effectiveNs = this.logNamespaceUsed(ns);

    // LOG ACTUAL namespace use
    this.getNamespaces().logUsed(effectiveNs);

    // strip prefix if same namespace like parent
    if (this.isLocalNs(effectiveNs)) {
      return {
        localName: ns.localName
      };
    } else {
      return assign$1({
        localName: ns.localName
      }, effectiveNs);
    }
  };
  ElementSerializer.prototype.parseGeneric = function (element) {
    var self = this,
      body = this.body;
    var attributes = [];
    forEach$1(element, function (val, key) {
      var nonNsAttr;
      if (key === '$body') {
        body.push(new BodySerializer().build({
          type: 'String'
        }, val));
      } else if (key === '$children') {
        forEach$1(val, function (child) {
          body.push(new ElementSerializer(self).build(child));
        });
      } else if (key.indexOf('$') !== 0) {
        nonNsAttr = self.parseNsAttribute(element, key, val);
        if (nonNsAttr) {
          attributes.push({
            name: key,
            value: val
          });
        }
      }
    });
    return attributes;
  };
  ElementSerializer.prototype.parseNsAttribute = function (element, name, value) {
    var model = element.$model;
    var nameNs = parseName$1(name);
    var ns;

    // parse xmlns:foo="http://foo.bar"
    if (nameNs.prefix === 'xmlns') {
      ns = {
        prefix: nameNs.localName,
        uri: value
      };
    }

    // parse xmlns="http://foo.bar"
    if (!nameNs.prefix && nameNs.localName === 'xmlns') {
      ns = {
        uri: value
      };
    }
    if (!ns) {
      return {
        name: name,
        value: value
      };
    }
    if (model && model.getPackage(value)) {
      // register well known namespace
      this.logNamespace(ns, true, true);
    } else {
      // log custom namespace directly as used
      var actualNs = this.logNamespaceUsed(ns, true);
      this.getNamespaces().logUsed(actualNs);
    }
  };

  /**
   * Parse namespaces and return a list of left over generic attributes
   *
   * @param  {Object} element
   * @return {Array<Object>}
   */
  ElementSerializer.prototype.parseNsAttributes = function (element, attrs) {
    var self = this;
    var genericAttrs = element.$attrs;
    var attributes = [];

    // parse namespace attributes first
    // and log them. push non namespace attributes to a list
    // and process them later
    forEach$1(genericAttrs, function (value, name) {
      var nonNsAttr = self.parseNsAttribute(element, name, value);
      if (nonNsAttr) {
        attributes.push(nonNsAttr);
      }
    });
    return attributes;
  };
  ElementSerializer.prototype.parseGenericAttributes = function (element, attributes) {
    var self = this;
    forEach$1(attributes, function (attr) {
      // do not serialize xsi:type attribute
      // it is set manually based on the actual implementation type
      if (attr.name === XSI_TYPE) {
        return;
      }
      try {
        self.addAttribute(self.nsAttributeName(attr.name), attr.value);
      } catch (e) {
        console.warn('missing namespace information for ', attr.name, '=', attr.value, 'on', element, e);
      }
    });
  };
  ElementSerializer.prototype.parseContainments = function (properties) {
    var self = this,
      body = this.body,
      element = this.element;
    forEach$1(properties, function (p) {
      var value = element.get(p.name),
        isReference = p.isReference,
        isMany = p.isMany;
      if (!isMany) {
        value = [value];
      }
      if (p.isBody) {
        body.push(new BodySerializer().build(p, value[0]));
      } else if (isSimple(p.type)) {
        forEach$1(value, function (v) {
          body.push(new ValueSerializer(self.addTagName(self.nsPropertyTagName(p))).build(p, v));
        });
      } else if (isReference) {
        forEach$1(value, function (v) {
          body.push(new ReferenceSerializer(self.addTagName(self.nsPropertyTagName(p))).build(v));
        });
      } else {
        // allow serialization via type
        // rather than element name
        var asType = serializeAsType(p),
          asProperty = serializeAsProperty(p);
        forEach$1(value, function (v) {
          var serializer;
          if (asType) {
            serializer = new TypeSerializer(self, p);
          } else if (asProperty) {
            serializer = new ElementSerializer(self, p);
          } else {
            serializer = new ElementSerializer(self);
          }
          body.push(serializer.build(v));
        });
      }
    });
  };
  ElementSerializer.prototype.getNamespaces = function (local) {
    var namespaces = this.namespaces,
      parent = this.parent,
      parentNamespaces;
    if (!namespaces) {
      parentNamespaces = parent && parent.getNamespaces();
      if (local || !parentNamespaces) {
        this.namespaces = namespaces = new Namespaces(parentNamespaces);
      } else {
        namespaces = parentNamespaces;
      }
    }
    return namespaces;
  };
  ElementSerializer.prototype.logNamespace = function (ns, wellknown, local) {
    var namespaces = this.getNamespaces(local);
    var nsUri = ns.uri,
      nsPrefix = ns.prefix;
    var existing = namespaces.byUri(nsUri);
    if (!existing || local) {
      namespaces.add(ns, wellknown);
    }
    namespaces.mapPrefix(nsPrefix, nsUri);
    return ns;
  };
  ElementSerializer.prototype.logNamespaceUsed = function (ns, local) {
    var element = this.element,
      model = element.$model,
      namespaces = this.getNamespaces(local);

    // ns may be
    //
    //   * prefix only
    //   * prefix:uri
    //   * localName only

    var prefix = ns.prefix,
      uri = ns.uri,
      newPrefix,
      idx,
      wellknownUri;

    // handle anonymous namespaces (elementForm=unqualified), cf. #23
    if (!prefix && !uri) {
      return {
        localName: ns.localName
      };
    }
    wellknownUri = DEFAULT_NS_MAP[prefix] || model && (model.getPackage(prefix) || {}).uri;
    uri = uri || wellknownUri || namespaces.uriByPrefix(prefix);
    if (!uri) {
      throw new Error('no namespace uri given for prefix <' + prefix + '>');
    }
    ns = namespaces.byUri(uri);
    if (!ns) {
      newPrefix = prefix;
      idx = 1;

      // find a prefix that is not mapped yet
      while (namespaces.uriByPrefix(newPrefix)) {
        newPrefix = prefix + '_' + idx++;
      }
      ns = this.logNamespace({
        prefix: newPrefix,
        uri: uri
      }, wellknownUri === uri);
    }
    if (prefix) {
      namespaces.mapPrefix(prefix, uri);
    }
    return ns;
  };
  ElementSerializer.prototype.parseAttributes = function (properties) {
    var self = this,
      element = this.element;
    forEach$1(properties, function (p) {
      var value = element.get(p.name);
      if (p.isReference) {
        if (!p.isMany) {
          value = value.id;
        } else {
          var values = [];
          forEach$1(value, function (v) {
            values.push(v.id);
          });

          // IDREFS is a whitespace-separated list of references.
          value = values.join(' ');
        }
      }
      self.addAttribute(self.nsAttributeName(p), value);
    });
  };
  ElementSerializer.prototype.addTagName = function (nsTagName) {
    var actualNs = this.logNamespaceUsed(nsTagName);
    this.getNamespaces().logUsed(actualNs);
    return nsName(nsTagName);
  };
  ElementSerializer.prototype.addAttribute = function (name, value) {
    var attrs = this.attrs;
    if (isString$1(value)) {
      value = escapeAttr(value);
    }

    // de-duplicate attributes
    // https://github.com/bpmn-io/moddle-xml/issues/66
    var idx = findIndex(attrs, function (element) {
      return element.name.localName === name.localName && element.name.uri === name.uri && element.name.prefix === name.prefix;
    });
    var attr = {
      name: name,
      value: value
    };
    if (idx !== -1) {
      attrs.splice(idx, 1, attr);
    } else {
      attrs.push(attr);
    }
  };
  ElementSerializer.prototype.serializeAttributes = function (writer) {
    var attrs = this.attrs,
      namespaces = this.namespaces;
    if (namespaces) {
      attrs = getNsAttrs(namespaces).concat(attrs);
    }
    forEach$1(attrs, function (a) {
      writer.append(' ').append(nsName(a.name)).append('="').append(a.value).append('"');
    });
  };
  ElementSerializer.prototype.serializeTo = function (writer) {
    var firstBody = this.body[0],
      indent = firstBody && firstBody.constructor !== BodySerializer;
    writer.appendIndent().append('<' + this.tagName);
    this.serializeAttributes(writer);
    writer.append(firstBody ? '>' : ' />');
    if (firstBody) {
      if (indent) {
        writer.appendNewLine().indent();
      }
      forEach$1(this.body, function (b) {
        b.serializeTo(writer);
      });
      if (indent) {
        writer.unindent().appendIndent();
      }
      writer.append('</' + this.tagName + '>');
    }
    writer.appendNewLine();
  };

  /**
   * A serializer for types that handles serialization of data types
   */
  function TypeSerializer(parent, propertyDescriptor) {
    ElementSerializer.call(this, parent, propertyDescriptor);
  }
  inherits(TypeSerializer, ElementSerializer);
  TypeSerializer.prototype.parseNsAttributes = function (element) {
    // extracted attributes
    var attributes = ElementSerializer.prototype.parseNsAttributes.call(this, element);
    var descriptor = element.$descriptor;

    // only serialize xsi:type if necessary
    if (descriptor.name === this.propertyDescriptor.type) {
      return attributes;
    }
    var typeNs = this.typeNs = this.nsTagName(descriptor);
    this.getNamespaces().logUsed(this.typeNs);

    // add xsi:type attribute to represent the elements
    // actual type

    var pkg = element.$model.getPackage(typeNs.uri),
      typePrefix = pkg.xml && pkg.xml.typePrefix || '';
    this.addAttribute(this.nsAttributeName(XSI_TYPE), (typeNs.prefix ? typeNs.prefix + ':' : '') + typePrefix + descriptor.ns.localName);
    return attributes;
  };
  TypeSerializer.prototype.isLocalNs = function (ns) {
    return ns.uri === (this.typeNs || this.ns).uri;
  };
  function SavingWriter() {
    this.value = '';
    this.write = function (str) {
      this.value += str;
    };
  }
  function FormatingWriter(out, format) {
    var indent = [''];
    this.append = function (str) {
      out.write(str);
      return this;
    };
    this.appendNewLine = function () {
      if (format) {
        out.write('\n');
      }
      return this;
    };
    this.appendIndent = function () {
      if (format) {
        out.write(indent.join('  '));
      }
      return this;
    };
    this.indent = function () {
      indent.push('');
      return this;
    };
    this.unindent = function () {
      indent.pop();
      return this;
    };
  }

  /**
   * A writer for meta-model backed document trees
   *
   * @param {Object} options output options to pass into the writer
   */
  function Writer(options) {
    options = assign$1({
      format: false,
      preamble: true
    }, options || {});
    function toXML(tree, writer) {
      var internalWriter = writer || new SavingWriter();
      var formatingWriter = new FormatingWriter(internalWriter, options.format);
      if (options.preamble) {
        formatingWriter.append(XML_PREAMBLE);
      }
      new ElementSerializer().build(tree).serializeTo(formatingWriter);
      if (!writer) {
        return internalWriter.value;
      }
    }
    return {
      toXML: toXML
    };
  }

  /**
   * A sub class of {@link Moddle} with support for import and export of DMN xml files.
   *
   * @class DmnModdle
   * @extends Moddle
   *
   * @param {Object|Array} packages to use for instantiating the model
   * @param {Object} [options] additional options to pass over
   */
  function DmnModdle(packages, options) {
    Moddle.call(this, packages, options);
  }
  DmnModdle.prototype = Object.create(Moddle.prototype);

  /**
   * The fromXML result.
   *
   * @typedef {Object} ParseResult
   *
   * @property {ModdleElement} rootElement
   * @property {Array<Object>} references
   * @property {Array<Error>} warnings
   * @property {Object} elementsById - a mapping containing each ID -> ModdleElement
   */

  /**
   * The fromXML error.
   *
   * @typedef {Error} ParseError
   *
   * @property {Array<Error>} warnings
   */

  /**
   * Instantiates a DMN model tree from a given xml string.
   *
   * @param {String}   xmlStr
   * @param {String}   [typeName='dmn:Definitions'] name of the root element
   * @param {Object}   [options]  options to pass to the underlying reader
   *
   * @returns {Promise<ParseResult, ParseError>}
   */
  DmnModdle.prototype.fromXML = function (xmlStr, typeName, options) {
    if (!isString$3(typeName)) {
      options = typeName;
      typeName = 'dmn:Definitions';
    }
    var reader = new Reader(assign$3({
      model: this,
      lax: true
    }, options));
    var rootHandler = reader.handler(typeName);
    return reader.fromXML(xmlStr, rootHandler);
  };

  /**
   * The toXML result.
   *
   * @typedef {Object} SerializationResult
   *
   * @property {String} xml
   */

  /**
   * Serializes a DMN object tree to XML.
   *
   * @param {String}   element    the root element, typically an instance of `Definitions`
   * @param {Object}   [options]  to pass to the underlying writer
   *
   * @returns {Promise<SerializationResult, Error>}
   */
  DmnModdle.prototype.toXML = function (element, options) {
    var writer = new Writer(options);
    return new Promise(function (resolve, reject) {
      try {
        var result = writer.toXML(element);
        return resolve({
          xml: result
        });
      } catch (err) {
        return reject(err);
      }
    });
  };
  var name$4 = "DC";
  var prefix$4 = "dc";
  var uri$4 = "http://www.omg.org/spec/DMN/20180521/DC/";
  var types$4 = [{
    name: "Dimension",
    properties: [{
      name: "width",
      isAttr: true,
      type: "Real"
    }, {
      name: "height",
      isAttr: true,
      type: "Real"
    }]
  }, {
    name: "Bounds",
    properties: [{
      name: "height",
      isAttr: true,
      type: "Real"
    }, {
      name: "width",
      isAttr: true,
      type: "Real"
    }, {
      name: "x",
      isAttr: true,
      type: "Real"
    }, {
      name: "y",
      isAttr: true,
      type: "Real"
    }]
  }, {
    name: "Point",
    properties: [{
      name: "x",
      isAttr: true,
      type: "Real"
    }, {
      name: "y",
      isAttr: true,
      type: "Real"
    }]
  }, {
    name: "Color",
    properties: [{
      name: "red",
      type: "UML_Standard_Profile.mdzip:eee_1045467100323_917313_65"
    }, {
      name: "green",
      type: "UML_Standard_Profile.mdzip:eee_1045467100323_917313_65"
    }, {
      name: "blue",
      type: "UML_Standard_Profile.mdzip:eee_1045467100323_917313_65"
    }]
  }];
  var associations$3 = [];
  var enumerations$3 = [{
    name: "AlignmentKind",
    literalValues: [{
      name: "start"
    }, {
      name: "center"
    }, {
      name: "end"
    }]
  }];
  var DcPackage = {
    name: name$4,
    prefix: prefix$4,
    uri: uri$4,
    types: types$4,
    associations: associations$3,
    enumerations: enumerations$3
  };
  var name$3 = "DI";
  var prefix$3 = "di";
  var uri$3 = "http://www.omg.org/spec/DMN/20180521/DI/";
  var types$3 = [{
    name: "DiagramElement",
    isAbstract: true,
    properties: [{
      name: "extension",
      type: "Extension"
    }, {
      name: "id",
      isAttr: true,
      isId: true,
      type: "String"
    }, {
      name: "style",
      isReference: true,
      type: "Style",
      xml: {
        serialize: "property"
      }
    }, {
      name: "sharedStyle",
      isReference: true,
      isVirtual: true,
      type: "Style"
    }]
  }, {
    name: "Diagram",
    superClass: ["DiagramElement"],
    properties: [{
      name: "name",
      isAttr: true,
      type: "String"
    }, {
      name: "documentation",
      isAttr: true,
      type: "String"
    }, {
      name: "resolution",
      isAttr: true,
      type: "Real"
    }]
  }, {
    name: "Shape",
    isAbstract: true,
    properties: [{
      name: "bounds",
      type: "dc:Bounds"
    }],
    superClass: ["DiagramElement"]
  }, {
    name: "Edge",
    isAbstract: true,
    properties: [{
      name: "waypoint",
      type: "dc:Point",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }],
    superClass: ["DiagramElement"]
  }, {
    name: "Style",
    isAbstract: true,
    properties: [{
      name: "id",
      isAttr: true,
      isId: true,
      type: "String"
    }]
  }, {
    name: "Extension",
    properties: [{
      name: "values",
      isMany: true,
      type: "Element"
    }]
  }];
  var associations$2 = [];
  var enumerations$2 = [];
  var xml$2 = {
    tagAlias: "lowerCase"
  };
  var DiPackage = {
    name: name$3,
    prefix: prefix$3,
    uri: uri$3,
    types: types$3,
    associations: associations$2,
    enumerations: enumerations$2,
    xml: xml$2
  };
  var name$2 = "DMN";
  var prefix$2 = "dmn";
  var uri$2 = "https://www.omg.org/spec/DMN/20191111/MODEL/";
  var types$2 = [{
    name: "AuthorityRequirement",
    superClass: ["DMNElement"],
    properties: [{
      name: "requiredAuthority",
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }, {
      name: "requiredDecision",
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }, {
      name: "requiredInput",
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "ItemDefinition",
    superClass: ["NamedElement"],
    properties: [{
      name: "typeRef",
      type: "String"
    }, {
      name: "allowedValues",
      type: "UnaryTests",
      xml: {
        serialize: "property"
      }
    }, {
      name: "typeLanguage",
      type: "String",
      isAttr: true
    }, {
      name: "itemComponent",
      type: "ItemDefinition",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "functionItem",
      type: "FunctionItem"
    }, {
      name: "isCollection",
      isAttr: true,
      type: "Boolean"
    }]
  }, {
    name: "Definitions",
    superClass: ["NamedElement"],
    properties: [{
      name: "import",
      type: "Import",
      isMany: true
    }, {
      name: "itemDefinition",
      type: "ItemDefinition",
      isMany: true
    }, {
      name: "drgElement",
      type: "DRGElement",
      isMany: true
    }, {
      name: "artifact",
      type: "Artifact",
      isMany: true
    }, {
      name: "elementCollection",
      type: "ElementCollection",
      isMany: true
    }, {
      name: "businessContextElement",
      type: "BusinessContextElement",
      isMany: true
    }, {
      name: "namespace",
      type: "String",
      isAttr: true
    }, {
      name: "expressionLanguage",
      type: "String",
      isAttr: true
    }, {
      name: "typeLanguage",
      type: "String",
      isAttr: true
    }, {
      name: "exporter",
      isAttr: true,
      type: "String"
    }, {
      name: "exporterVersion",
      isAttr: true,
      type: "String"
    }, {
      name: "dmnDI",
      type: "dmndi:DMNDI"
    }]
  }, {
    name: "KnowledgeSource",
    superClass: ["DRGElement"],
    properties: [{
      name: "authorityRequirement",
      type: "AuthorityRequirement",
      isMany: true
    }, {
      name: "type",
      type: "String"
    }, {
      name: "owner",
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }, {
      name: "locationURI",
      type: "String",
      isAttr: true
    }]
  }, {
    name: "DecisionRule",
    superClass: ["DMNElement"],
    properties: [{
      name: "inputEntry",
      type: "UnaryTests",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "outputEntry",
      type: "LiteralExpression",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "annotationEntry",
      type: "RuleAnnotation",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "Expression",
    isAbstract: true,
    superClass: ["DMNElement"],
    properties: [{
      name: "typeRef",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "InformationItem",
    superClass: ["NamedElement"],
    properties: [{
      name: "typeRef",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "Decision",
    superClass: ["DRGElement"],
    properties: [{
      name: "question",
      type: "String",
      xml: {
        serialize: "property"
      }
    }, {
      name: "allowedAnswers",
      type: "String",
      xml: {
        serialize: "property"
      }
    }, {
      name: "variable",
      type: "InformationItem",
      xml: {
        serialize: "property"
      }
    }, {
      name: "informationRequirement",
      type: "InformationRequirement",
      isMany: true
    }, {
      name: "knowledgeRequirement",
      type: "KnowledgeRequirement",
      isMany: true
    }, {
      name: "authorityRequirement",
      type: "AuthorityRequirement",
      isMany: true
    }, {
      name: "supportedObjective",
      isMany: true,
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }, {
      name: "impactedPerformanceIndicator",
      type: "DMNElementReference",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "decisionMaker",
      type: "DMNElementReference",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "decisionOwner",
      type: "DMNElementReference",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "usingProcess",
      isMany: true,
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }, {
      name: "usingTask",
      isMany: true,
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }, {
      name: "decisionLogic",
      type: "Expression"
    }]
  }, {
    name: "Invocation",
    superClass: ["Expression"],
    properties: [{
      name: "calledFunction",
      type: "Expression"
    }, {
      name: "binding",
      type: "Binding",
      isMany: true
    }]
  }, {
    name: "OrganisationalUnit",
    superClass: ["BusinessContextElement"],
    properties: [{
      name: "decisionMade",
      type: "Decision",
      isReference: true,
      isMany: true
    }, {
      name: "decisionOwned",
      type: "Decision",
      isReference: true,
      isMany: true
    }]
  }, {
    name: "Import",
    superClass: ["NamedElement"],
    properties: [{
      name: "importType",
      type: "String",
      isAttr: true
    }, {
      name: "locationURI",
      type: "String",
      isAttr: true
    }, {
      name: "namespace",
      type: "String",
      isAttr: true
    }]
  }, {
    name: "InformationRequirement",
    superClass: ["DMNElement"],
    properties: [{
      name: "requiredDecision",
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }, {
      name: "requiredInput",
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "ElementCollection",
    superClass: ["NamedElement"],
    properties: [{
      name: "drgElement",
      type: "DMNElementReference",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "DRGElement",
    isAbstract: true,
    superClass: ["NamedElement"],
    properties: []
  }, {
    name: "InputData",
    superClass: ["DRGElement"],
    properties: [{
      name: "variable",
      type: "InformationItem",
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "DMNElement",
    isAbstract: true,
    properties: [{
      name: "description",
      type: "String"
    }, {
      name: "extensionElements",
      type: "ExtensionElements"
    }, {
      name: "id",
      type: "String",
      isAttr: true,
      isId: true
    }, {
      name: "extensionAttribute",
      type: "ExtensionAttribute",
      isMany: true
    }, {
      name: "label",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "InputClause",
    superClass: ["DMNElement"],
    properties: [{
      name: "inputExpression",
      type: "LiteralExpression",
      xml: {
        serialize: "property"
      }
    }, {
      name: "inputValues",
      type: "UnaryTests",
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "DecisionTable",
    superClass: ["Expression"],
    properties: [{
      name: "input",
      type: "InputClause",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "output",
      type: "OutputClause",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "annotation",
      type: "RuleAnnotationClause",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "rule",
      type: "DecisionRule",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "hitPolicy",
      type: "HitPolicy",
      isAttr: true,
      "default": "UNIQUE"
    }, {
      name: "aggregation",
      type: "BuiltinAggregator",
      isAttr: true
    }, {
      name: "preferredOrientation",
      type: "DecisionTableOrientation",
      isAttr: true
    }, {
      name: "outputLabel",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "LiteralExpression",
    superClass: ["Expression"],
    properties: [{
      name: "expressionLanguage",
      type: "String",
      isAttr: true
    }, {
      name: "text",
      type: "String"
    }, {
      name: "importedValues",
      type: "ImportedValues"
    }]
  }, {
    name: "Binding",
    properties: [{
      name: "parameter",
      type: "InformationItem",
      xml: {
        serialize: "property"
      }
    }, {
      name: "bindingFormula",
      type: "Expression"
    }]
  }, {
    name: "KnowledgeRequirement",
    superClass: ["DMNElement"],
    properties: [{
      name: "requiredKnowledge",
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "BusinessKnowledgeModel",
    superClass: ["Invocable"],
    properties: [{
      name: "encapsulatedLogic",
      type: "FunctionDefinition",
      xml: {
        serialize: "property"
      }
    }, {
      name: "knowledgeRequirement",
      type: "KnowledgeRequirement",
      isMany: true
    }, {
      name: "authorityRequirement",
      type: "AuthorityRequirement",
      isMany: true
    }]
  }, {
    name: "BusinessContextElement",
    isAbstract: true,
    superClass: ["NamedElement"],
    properties: [{
      name: "URI",
      type: "String",
      isAttr: true
    }]
  }, {
    name: "PerformanceIndicator",
    superClass: ["BusinessContextElement"],
    properties: [{
      name: "impactingDecision",
      type: "DMNElementReference",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "FunctionDefinition",
    superClass: ["Expression"],
    properties: [{
      name: "formalParameter",
      type: "InformationItem",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "body",
      type: "Expression"
    }, {
      name: "kind",
      type: "FunctionKind",
      isAttr: true
    }]
  }, {
    name: "Context",
    superClass: ["Expression"],
    properties: [{
      name: "contextEntry",
      type: "ContextEntry",
      isMany: true
    }]
  }, {
    name: "ContextEntry",
    superClass: ["DMNElement"],
    properties: [{
      name: "variable",
      type: "InformationItem",
      xml: {
        serialize: "property"
      }
    }, {
      name: "value",
      type: "Expression"
    }]
  }, {
    name: "List",
    superClass: ["Expression"],
    properties: [{
      name: "elements",
      isMany: true,
      type: "Expression"
    }]
  }, {
    name: "Relation",
    superClass: ["Expression"],
    properties: [{
      name: "column",
      type: "InformationItem",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "row",
      type: "List",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "OutputClause",
    superClass: ["DMNElement"],
    properties: [{
      name: "outputValues",
      type: "UnaryTests",
      xml: {
        serialize: "property"
      }
    }, {
      name: "defaultOutputEntry",
      type: "LiteralExpression",
      xml: {
        serialize: "property"
      }
    }, {
      name: "name",
      isAttr: true,
      type: "String"
    }, {
      name: "typeRef",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "UnaryTests",
    superClass: ["Expression"],
    properties: [{
      name: "text",
      type: "String"
    }, {
      name: "expressionLanguage",
      type: "String",
      isAttr: true
    }]
  }, {
    name: "NamedElement",
    isAbstract: true,
    superClass: ["DMNElement"],
    properties: [{
      name: "name",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "ImportedValues",
    superClass: ["Import"],
    properties: [{
      name: "importedElement",
      type: "String"
    }, {
      name: "expressionLanguage",
      type: "String",
      isAttr: true
    }]
  }, {
    name: "DecisionService",
    superClass: ["Invocable"],
    properties: [{
      name: "outputDecision",
      type: "DMNElementReference",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "encapsulatedDecision",
      type: "DMNElementReference",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "inputDecision",
      type: "DMNElementReference",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }, {
      name: "inputData",
      type: "DMNElementReference",
      isMany: true,
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "ExtensionElements",
    properties: [{
      name: "values",
      type: "Element",
      isMany: true
    }]
  }, {
    name: "ExtensionAttribute",
    properties: [{
      name: "value",
      type: "Element"
    }, {
      name: "valueRef",
      type: "Element",
      isAttr: true,
      isReference: true
    }, {
      name: "name",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "Element",
    isAbstract: true,
    properties: [{
      name: "extensionAttribute",
      type: "ExtensionAttribute",
      isAttr: true,
      isReference: true
    }, {
      name: "elements",
      type: "ExtensionElements",
      isAttr: true,
      isReference: true
    }]
  }, {
    name: "Artifact",
    isAbstract: true,
    superClass: ["DMNElement"],
    properties: []
  }, {
    name: "Association",
    superClass: ["Artifact"],
    properties: [{
      name: "sourceRef",
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }, {
      name: "targetRef",
      type: "DMNElementReference",
      xml: {
        serialize: "property"
      }
    }, {
      name: "associationDirection",
      type: "AssociationDirection",
      isAttr: true
    }]
  }, {
    name: "TextAnnotation",
    superClass: ["Artifact"],
    properties: [{
      name: "text",
      type: "String"
    }, {
      name: "textFormat",
      isAttr: true,
      type: "String",
      "default": "text/plain"
    }]
  }, {
    name: "RuleAnnotationClause",
    properties: [{
      name: "name",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "RuleAnnotation",
    properties: [{
      name: "text",
      type: "String"
    }]
  }, {
    name: "Invocable",
    isAbstract: true,
    superClass: ["DRGElement"],
    properties: [{
      name: "variable",
      type: "InformationItem",
      xml: {
        serialize: "property"
      }
    }]
  }, {
    name: "Group",
    superClass: ["Artifact"],
    properties: [{
      name: "name",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "FunctionItem",
    superClass: ["DMNElement"],
    properties: [{
      name: "parameters",
      isMany: true,
      type: "InformationItem",
      xml: {
        serialize: "property"
      }
    }, {
      name: "outputTypeRef",
      isAttr: true,
      type: "String"
    }]
  }, {
    name: "DMNElementReference",
    properties: [{
      isAttr: true,
      name: "href",
      type: "String"
    }]
  }];
  var enumerations$1 = [{
    name: "HitPolicy",
    literalValues: [{
      name: "UNIQUE"
    }, {
      name: "FIRST"
    }, {
      name: "PRIORITY"
    }, {
      name: "ANY"
    }, {
      name: "COLLECT"
    }, {
      name: "RULE ORDER"
    }, {
      name: "OUTPUT ORDER"
    }]
  }, {
    name: "BuiltinAggregator",
    literalValues: [{
      name: "SUM"
    }, {
      name: "COUNT"
    }, {
      name: "MIN"
    }, {
      name: "MAX"
    }]
  }, {
    name: "DecisionTableOrientation",
    literalValues: [{
      name: "Rule-as-Row"
    }, {
      name: "Rule-as-Column"
    }, {
      name: "CrossTable"
    }]
  }, {
    name: "AssociationDirection",
    literalValues: [{
      name: "None"
    }, {
      name: "One"
    }, {
      name: "Both"
    }]
  }, {
    name: "FunctionKind",
    literalValues: [{
      name: "FEEL"
    }, {
      name: "Java"
    }, {
      name: "PMML"
    }]
  }];
  var associations$1 = [];
  var xml$1 = {
    tagAlias: "lowerCase"
  };
  var DmnPackage = {
    name: name$2,
    prefix: prefix$2,
    uri: uri$2,
    types: types$2,
    enumerations: enumerations$1,
    associations: associations$1,
    xml: xml$1
  };
  var name$1 = "DMNDI";
  var prefix$1 = "dmndi";
  var uri$1 = "https://www.omg.org/spec/DMN/20191111/DMNDI/";
  var types$1$1 = [{
    name: "DMNDI",
    properties: [{
      name: "diagrams",
      type: "DMNDiagram",
      isMany: true
    }, {
      name: "styles",
      type: "DMNStyle",
      isMany: true
    }]
  }, {
    name: "DMNStyle",
    superClass: ["di:Style"],
    properties: [{
      name: "fillColor",
      type: "dc:Color",
      isAttr: true
    }, {
      name: "strokeColor",
      type: "dc:Color",
      isAttr: true
    }, {
      name: "fontColor",
      type: "dc:Color",
      isAttr: true
    }, {
      name: "fontSize",
      isAttr: true,
      type: "Real"
    }, {
      name: "fontFamily",
      isAttr: true,
      type: "String"
    }, {
      name: "fontItalic",
      isAttr: true,
      type: "Boolean"
    }, {
      name: "fontBold",
      isAttr: true,
      type: "Boolean"
    }, {
      name: "fontUnderline",
      isAttr: true,
      type: "Boolean"
    }, {
      name: "fontStrikeThrough",
      isAttr: true,
      type: "Boolean"
    }, {
      name: "labelHorizontalAlignment",
      type: "dc:AlignmentKind",
      isAttr: true
    }, {
      name: "labelVerticalAlignment",
      type: "dc:AlignmentKind",
      isAttr: true
    }]
  }, {
    name: "DMNDiagram",
    superClass: ["di:Diagram"],
    properties: [{
      name: "dmnElementRef",
      type: "dmn:DMNElement",
      isAttr: true,
      isReference: true
    }, {
      name: "size",
      type: "Size"
    }, {
      name: "localStyle",
      type: "DMNStyle",
      isVirtual: true
    }, {
      name: "sharedStyle",
      type: "DMNStyle",
      isVirtual: true,
      isReference: true,
      redefines: "di:DiagramElement#sharedStyle"
    }, {
      name: "diagramElements",
      type: "DMNDiagramElement",
      isMany: true
    }]
  }, {
    name: "DMNDiagramElement",
    isAbstract: true,
    superClass: ["di:DiagramElement"],
    properties: [{
      name: "dmnElementRef",
      type: "dmn:DMNElement",
      isAttr: true,
      isReference: true
    }, {
      name: "sharedStyle",
      type: "DMNStyle",
      isVirtual: true,
      isReference: true,
      redefines: "di:DiagramElement#sharedStyle"
    }, {
      name: "localStyle",
      type: "DMNStyle",
      isVirtual: true
    }, {
      name: "label",
      type: "DMNLabel"
    }]
  }, {
    name: "DMNLabel",
    superClass: ["di:Shape"],
    properties: [{
      name: "text",
      type: "Text"
    }]
  }, {
    name: "DMNShape",
    superClass: ["di:Shape", "DMNDiagramElement"],
    properties: [{
      name: "isListedInputData",
      isAttr: true,
      type: "Boolean"
    }, {
      name: "decisionServiceDividerLine",
      type: "DMNDecisionServiceDividerLine"
    }, {
      name: "isCollapsed",
      isAttr: true,
      type: "Boolean"
    }]
  }, {
    name: "DMNEdge",
    superClass: ["di:Edge", "DMNDiagramElement"],
    properties: [{
      name: "sourceElement",
      type: "DMNDiagramElement",
      isAttr: true,
      isReference: true
    }, {
      name: "targetElement",
      type: "DMNDiagramElement",
      isAttr: true,
      isReference: true
    }]
  }, {
    name: "DMNDecisionServiceDividerLine",
    superClass: ["di:Edge"]
  }, {
    name: "Text",
    properties: [{
      name: "text",
      isBody: true,
      type: "String"
    }]
  }, {
    name: "Size",
    superClass: ["dc:Dimension"]
  }];
  var associations = [];
  var enumerations = [];
  var DmnDiPackage = {
    name: name$1,
    prefix: prefix$1,
    uri: uri$1,
    types: types$1$1,
    associations: associations,
    enumerations: enumerations
  };
  var name$5 = "bpmn.io DI for DMN";
  var uri = "http://bpmn.io/schema/dmn/biodi/2.0";
  var prefix$5 = "biodi";
  var xml = {
    tagAlias: "lowerCase"
  };
  var types$5 = [{
    name: "DecisionTable",
    isAbstract: true,
    "extends": ["dmn:DecisionTable"],
    properties: [{
      name: "annotationsWidth",
      isAttr: true,
      type: "Integer"
    }]
  }, {
    name: "OutputClause",
    isAbstract: true,
    "extends": ["dmn:OutputClause"],
    properties: [{
      name: "width",
      isAttr: true,
      type: "Integer"
    }]
  }, {
    name: "InputClause",
    isAbstract: true,
    "extends": ["dmn:InputClause"],
    properties: [{
      name: "width",
      isAttr: true,
      type: "Integer"
    }]
  }];
  var BioDiPackage = {
    name: name$5,
    uri: uri,
    prefix: prefix$5,
    xml: xml,
    types: types$5
  };
  var packages = {
    dc: DcPackage,
    di: DiPackage,
    dmn: DmnPackage,
    dmndi: DmnDiPackage,
    biodi: BioDiPackage
  };
  function simple(additionalPackages, options) {
    var pks = assign$3({}, packages, additionalPackages);
    return new DmnModdle(pks, options);
  }

  function _mergeNamespaces$1(n, m) {
    m.forEach(function (e) {
      e && typeof e !== 'string' && !Array.isArray(e) && Object.keys(e).forEach(function (k) {
        if (k !== 'default' && !(k in n)) {
          var d = Object.getOwnPropertyDescriptor(e, k);
          Object.defineProperty(n, k, d.get ? d : {
            enumerable: true,
            get: function () {
              return e[k];
            }
          });
        }
      });
    });
    return Object.freeze(n);
  }

  /**
   * Flatten array, one level deep.
   *
   * @template T
   *
   * @param {T[][] | T[] | null} [arr]
   *
   * @return {T[]}
   */

  const nativeToString = Object.prototype.toString;
  const nativeHasOwnProperty = Object.prototype.hasOwnProperty;
  function isUndefined$1(obj) {
    return obj === undefined;
  }
  function isArray$3(obj) {
    return nativeToString.call(obj) === '[object Array]';
  }

  /**
   * Return true, if target owns a property with the given key.
   *
   * @param {Object} target
   * @param {String} key
   *
   * @return {Boolean}
   */
  function has(target, key) {
    return nativeHasOwnProperty.call(target, key);
  }

  /**
   * Iterate over collection; returning something
   * (non-undefined) will stop iteration.
   *
   * @template T
   * @param {Collection<T>} collection
   * @param { ((item: T, idx: number) => (boolean|void)) | ((item: T, key: string) => (boolean|void)) } iterator
   *
   * @return {T} return result that stopped the iteration
   */
  function forEach(collection, iterator) {
    let val, result;
    if (isUndefined$1(collection)) {
      return;
    }
    const convertKey = isArray$3(collection) ? toNum : identity;
    for (let key in collection) {
      if (has(collection, key)) {
        val = collection[key];
        result = iterator(val, convertKey(key));
        if (result === false) {
          return val;
        }
      }
    }
  }
  function identity(arg) {
    return arg;
  }
  function toNum(arg) {
    return Number(arg);
  }

  /**
   * Assigns style attributes in a style-src compliant way.
   *
   * @param {Element} element
   * @param {...Object} styleSources
   *
   * @return {Element} the element
   */
  function assign(element, ...styleSources) {
    const target = element.style;
    forEach(styleSources, function (style) {
      if (!style) {
        return;
      }
      forEach(style, function (value, key) {
        target[key] = value;
      });
    });
    return element;
  }

  /**
   * Set attribute `name` to `val`, or get attr `name`.
   *
   * @param {Element} el
   * @param {String} name
   * @param {String} [val]
   * @api public
   */
  function attr$1(el, name, val) {
    // get
    if (arguments.length == 2) {
      return el.getAttribute(name);
    }

    // remove
    if (val === null) {
      return el.removeAttribute(name);
    }

    // set
    el.setAttribute(name, val);
    return el;
  }

  /**
   * Taken from https://github.com/component/classes
   *
   * Without the component bits.
   */

  /**
   * toString reference.
   */

  const toString$1 = Object.prototype.toString;

  /**
   * Wrap `el` in a `ClassList`.
   *
   * @param {Element} el
   * @return {ClassList}
   * @api public
   */

  function classes$1(el) {
    return new ClassList$1(el);
  }

  /**
   * Initialize a new ClassList for `el`.
   *
   * @param {Element} el
   * @api private
   */

  function ClassList$1(el) {
    if (!el || !el.nodeType) {
      throw new Error('A DOM element reference is required');
    }
    this.el = el;
    this.list = el.classList;
  }

  /**
   * Add class `name` if not already present.
   *
   * @param {String} name
   * @return {ClassList}
   * @api public
   */

  ClassList$1.prototype.add = function (name) {
    this.list.add(name);
    return this;
  };

  /**
   * Remove class `name` when present, or
   * pass a regular expression to remove
   * any which match.
   *
   * @param {String|RegExp} name
   * @return {ClassList}
   * @api public
   */

  ClassList$1.prototype.remove = function (name) {
    if ('[object RegExp]' == toString$1.call(name)) {
      return this.removeMatching(name);
    }
    this.list.remove(name);
    return this;
  };

  /**
   * Remove all classes matching `re`.
   *
   * @param {RegExp} re
   * @return {ClassList}
   * @api private
   */

  ClassList$1.prototype.removeMatching = function (re) {
    const arr = this.array();
    for (let i = 0; i < arr.length; i++) {
      if (re.test(arr[i])) {
        this.remove(arr[i]);
      }
    }
    return this;
  };

  /**
   * Toggle class `name`, can force state via `force`.
   *
   * For browsers that support classList, but do not support `force` yet,
   * the mistake will be detected and corrected.
   *
   * @param {String} name
   * @param {Boolean} force
   * @return {ClassList}
   * @api public
   */

  ClassList$1.prototype.toggle = function (name, force) {
    if ('undefined' !== typeof force) {
      if (force !== this.list.toggle(name, force)) {
        this.list.toggle(name); // toggle again to correct
      }
    } else {
      this.list.toggle(name);
    }
    return this;
  };

  /**
   * Return an array of classes.
   *
   * @return {Array}
   * @api public
   */

  ClassList$1.prototype.array = function () {
    return Array.from(this.list);
  };

  /**
   * Check if class `name` is present.
   *
   * @param {String} name
   * @return {ClassList}
   * @api public
   */

  ClassList$1.prototype.has = ClassList$1.prototype.contains = function (name) {
    return this.list.contains(name);
  };

  /**
   * Clear utility
   */

  /**
   * Removes all children from the given element
   *
   * @param {Element} element
   *
   * @return {Element} the element (for chaining)
   */
  function clear$1(element) {
    var child;
    while (child = element.firstChild) {
      element.removeChild(child);
    }
    return element;
  }

  /**
   * Closest
   *
   * @param {Element} el
   * @param {string} selector
   * @param {boolean} checkYourSelf (optional)
   */
  function closest(element, selector, checkYourSelf) {
    var actualElement = checkYourSelf ? element : element.parentNode;
    return actualElement && typeof actualElement.closest === 'function' && actualElement.closest(selector) || null;
  }
  var componentEvent = {};
  var bind$1, unbind$1, prefix;
  function detect() {
    bind$1 = window.addEventListener ? 'addEventListener' : 'attachEvent';
    unbind$1 = window.removeEventListener ? 'removeEventListener' : 'detachEvent';
    prefix = bind$1 !== 'addEventListener' ? 'on' : '';
  }

  /**
   * Bind `el` event `type` to `fn`.
   *
   * @param {Element} el
   * @param {String} type
   * @param {Function} fn
   * @param {Boolean} capture
   * @return {Function}
   * @api public
   */

  var bind_1 = componentEvent.bind = function (el, type, fn, capture) {
    if (!bind$1) detect();
    el[bind$1](prefix + type, fn, capture || false);
    return fn;
  };

  /**
   * Unbind `el` event `type`'s callback `fn`.
   *
   * @param {Element} el
   * @param {String} type
   * @param {Function} fn
   * @param {Boolean} capture
   * @return {Function}
   * @api public
   */

  var unbind_1 = componentEvent.unbind = function (el, type, fn, capture) {
    if (!unbind$1) detect();
    el[unbind$1](prefix + type, fn, capture || false);
    return fn;
  };
  var event = /*#__PURE__*/_mergeNamespaces$1({
    __proto__: null,
    bind: bind_1,
    unbind: unbind_1,
    'default': componentEvent
  }, [componentEvent]);

  /**
   * Module dependencies.
   */

  /**
   * Delegate event `type` to `selector`
   * and invoke `fn(e)`. A callback function
   * is returned which may be passed to `.unbind()`.
   *
   * @param {Element} el
   * @param {String} selector
   * @param {String} type
   * @param {Function} fn
   * @param {Boolean} capture
   * @return {Function}
   * @api public
   */

  // Some events don't bubble, so we want to bind to the capture phase instead
  // when delegating.
  var forceCaptureEvents = ['focus', 'blur'];
  function bind(el, selector, type, fn, capture) {
    if (forceCaptureEvents.indexOf(type) !== -1) {
      capture = true;
    }
    return event.bind(el, type, function (e) {
      var target = e.target || e.srcElement;
      e.delegateTarget = closest(target, selector, true);
      if (e.delegateTarget) {
        fn.call(el, e);
      }
    }, capture);
  }

  /**
   * Unbind event `type`'s callback `fn`.
   *
   * @param {Element} el
   * @param {String} type
   * @param {Function} fn
   * @param {Boolean} capture
   * @api public
   */
  function unbind(el, type, fn, capture) {
    if (forceCaptureEvents.indexOf(type) !== -1) {
      capture = true;
    }
    return event.unbind(el, type, fn, capture);
  }
  var delegate = {
    bind,
    unbind
  };

  /**
   * Expose `parse`.
   */

  var domify = parse$1;

  /**
   * Tests for browser support.
   */

  var innerHTMLBug = false;
  var bugTestDiv;
  if (typeof document !== 'undefined') {
    bugTestDiv = document.createElement('div');
    // Setup
    bugTestDiv.innerHTML = '  <link/><table></table><a href="/a">a</a><input type="checkbox"/>';
    // Make sure that link elements get serialized correctly by innerHTML
    // This requires a wrapper element in IE
    innerHTMLBug = !bugTestDiv.getElementsByTagName('link').length;
    bugTestDiv = undefined;
  }

  /**
   * Wrap map from jquery.
   */

  var map$2 = {
    legend: [1, '<fieldset>', '</fieldset>'],
    tr: [2, '<table><tbody>', '</tbody></table>'],
    col: [2, '<table><tbody></tbody><colgroup>', '</colgroup></table>'],
    // for script/link/style tags to work in IE6-8, you have to wrap
    // in a div with a non-whitespace character in front, ha!
    _default: innerHTMLBug ? [1, 'X<div>', '</div>'] : [0, '', '']
  };
  map$2.td = map$2.th = [3, '<table><tbody><tr>', '</tr></tbody></table>'];
  map$2.option = map$2.optgroup = [1, '<select multiple="multiple">', '</select>'];
  map$2.thead = map$2.tbody = map$2.colgroup = map$2.caption = map$2.tfoot = [1, '<table>', '</table>'];
  map$2.polyline = map$2.ellipse = map$2.polygon = map$2.circle = map$2.text = map$2.line = map$2.path = map$2.rect = map$2.g = [1, '<svg xmlns="http://www.w3.org/2000/svg" version="1.1">', '</svg>'];

  /**
   * Parse `html` and return a DOM Node instance, which could be a TextNode,
   * HTML DOM Node of some kind (<div> for example), or a DocumentFragment
   * instance, depending on the contents of the `html` string.
   *
   * @param {String} html - HTML string to "domify"
   * @param {Document} doc - The `document` instance to create the Node for
   * @return {DOMNode} the TextNode, DOM Node, or DocumentFragment instance
   * @api private
   */

  function parse$1(html, doc) {
    if ('string' != typeof html) throw new TypeError('String expected');

    // default to the global `document` object
    if (!doc) doc = document;

    // tag name
    var m = /<([\w:]+)/.exec(html);
    if (!m) return doc.createTextNode(html);
    html = html.replace(/^\s+|\s+$/g, ''); // Remove leading/trailing whitespace

    var tag = m[1];

    // body support
    if (tag == 'body') {
      var el = doc.createElement('html');
      el.innerHTML = html;
      return el.removeChild(el.lastChild);
    }

    // wrap map
    var wrap = Object.prototype.hasOwnProperty.call(map$2, tag) ? map$2[tag] : map$2._default;
    var depth = wrap[0];
    var prefix = wrap[1];
    var suffix = wrap[2];
    var el = doc.createElement('div');
    el.innerHTML = prefix + html + suffix;
    while (depth--) el = el.lastChild;

    // one element
    if (el.firstChild == el.lastChild) {
      return el.removeChild(el.firstChild);
    }

    // several elements
    var fragment = doc.createDocumentFragment();
    while (el.firstChild) {
      fragment.appendChild(el.removeChild(el.firstChild));
    }
    return fragment;
  }
  var domify$1 = domify;

  /**
   * @param { HTMLElement } element
   * @param { String } selector
   *
   * @return { boolean }
   */
  function matches(element, selector) {
    return element && typeof element.matches === 'function' && element.matches(selector) || false;
  }
  function query(selector, el) {
    el = el || document;
    return el.querySelector(selector);
  }
  function all(selector, el) {
    el = el || document;
    return el.querySelectorAll(selector);
  }
  function remove$4(el) {
    el.parentNode && el.parentNode.removeChild(el);
  }

  // TODO: remove with future dmn-js version

  /**
   * Wraps APIs to check:
   *
   * 1) If a callback is passed -> Warn users about callback deprecation.
   * 2) If Promise class is implemented in current environment.
   *
   * @private
   */
  function wrapForCompatibility(api) {
    return function () {
      if (!window.Promise) {
        throw new Error('Promises is not supported in this environment.' + ' Please polyfill Promise.');
      }
      var argLen = arguments.length;
      if (argLen >= 1 && isFunction$2(arguments[argLen - 1])) {
        var callback = arguments[argLen - 1];
        console.warn(new Error('Passing callbacks to ' + replaceBoundPrefix(api.name) + ' is deprecated and will be removed in a future major release. ' + 'Please switch to promises: https://bpmn.io/l/moving-to-promises.html'));
        var argsWithoutCallback = Array.prototype.slice.call(arguments, 0, -1);
        api.apply(this, argsWithoutCallback).then(function (result) {
          var firstKey = Object.keys(result)[0];

          // The APIs we are wrapping all resolve a single item depending on the API.
          // For instance, importXML resolves { warnings } and saveXML returns { xml }.
          // That's why we can call the callback with the first item of result.
          return callback(null, result[firstKey]);

          // Passing a second paramter instead of catch because we don't want to
          // catch errors thrown by callback().
        }, function (err) {
          return callback(err, err.warnings);
        });
      } else {
        return api.apply(this, arguments);
      }
    };
  }

  // helper ////////

  /**
   * replaceBoundPrefix - replace the <bound > prefix from a string. Can be used
   * when logging the name of an API, not being sure whether is was bound or not.
   *
   * @param  {string} string
   * @return {string} the string without the <bound > prefix. If no <bound > prefix
   * was present, the same string will be returned.
   */
  function replaceBoundPrefix(string) {
    return string.replace('bound ', '');
  }

  /**
   * @typedef {import('./View').OpenResult} OpenResult
   */

  /**
   * @typedef {import('./View').OpenError} OpenError
   */

  const DEFAULT_CONTAINER_OPTIONS = {
    width: '100%',
    height: '100%',
    position: 'relative'
  };

  /**
   * The base class for DMN viewers and editors.
   *
   * @abstract
   */
  class Manager {
    /**
     * Create a new instance with the given options.
     *
     * @param  {Object} options
     *
     * @return {Manager}
     */
    constructor(options = {}) {
      this._eventBus = new EventBus();
      this._viewsChanged = debounce(this._viewsChanged, 0);
      this._views = [];
      this._viewers = {};

      // keep support for callbacks
      this.open = wrapForCompatibility(this.open.bind(this));
      this.importXML = wrapForCompatibility(this.importXML.bind(this));
      this.saveXML = wrapForCompatibility(this.saveXML.bind(this));
      this._init(options);
    }

    /**
    * The importXML result.
    *
    * @typedef {Object} ImportXMLResult
    *
    * @property {Array<string>} warnings
    */

    /**
    * The importXML error.
    *
    * @typedef {Error} ImportXMLError
    *
    * @property {Array<string>} warnings
    */

    /**
     * Parse and render a DMN diagram.
     *
     * Once finished the viewer reports back the result to the
     * provided callback function with (err, warnings).
     *
     * ## Life-Cycle Events
     *
     * During import the viewer will fire life-cycle events:
     *
     *   * import.parse.start (about to read model from xml)
     *   * import.parse.complete (model read; may have worked or not)
     *   * import.render.start (graphical import start)
     *   * import.render.complete (graphical import finished)
     *   * import.done (everything done)
     *
     * You can use these events to hook into the life-cycle.
     *
     * @param {string} xml the DMN xml
     * @param {Object} [options]
     * @param {boolean} [options.open=true]
     *
     * @return {Promise<ImportXMLResult, ImportXMLError>}
     */
    importXML(xml, options) {
      var self = this;
      options = options || {
        open: true
      };
      return new Promise(function (resolve, reject) {
        var previousActiveView = self._activeView;

        // clean up previously rendered diagram before new import
        self._clear().then(() => {
          // hook in pre-parse listeners +
          // allow xml manipulation
          xml = self._emit('import.parse.start', {
            xml: xml
          }) || xml;
          var parseWarnings;
          self._moddle.fromXML(xml, 'dmn:Definitions').then(parseResult => {
            var definitions = parseResult.rootElement;
            var references = parseResult.references;
            var elementsById = parseResult.elementsById;
            parseWarnings = parseResult.warnings;

            // hook in post parse listeners +
            // allow definitions manipulation
            definitions = self._emit('import.parse.complete', ParseCompleteEvent({
              error: null,
              definitions: definitions,
              elementsById: elementsById,
              references: references,
              warnings: parseWarnings
            })) || definitions;
            self._setDefinitions(definitions);
            if (!options.open) {
              self._emit('import.done', {
                error: null,
                warnings: parseWarnings
              });
              resolve({
                warnings: parseWarnings
              });
              return;
            }

            // open either previously active view or view of the same type if available
            var view = self._getInitialView(self._views, previousActiveView);
            if (!view) {
              var noDisplayableContentsErr = new Error('no displayable contents');
              self._emit('import.done', {
                error: noDisplayableContentsErr,
                warnings: parseWarnings
              });
              noDisplayableContentsErr.warnings = parseWarnings;
              return reject(noDisplayableContentsErr);
            }
            self.open(view).then(result => ({
              warnings: result.warnings
            })).catch(error => ({
              error: error,
              warnings: error.warnings
            })).then(result => {
              var allWarnings = [].concat(parseWarnings, result.warnings);
              self._emit('import.done', {
                error: result.error,
                warnings: allWarnings
              });
              if (result.error) {
                result.error.warnings = allWarnings;
                reject(result.error);
              } else {
                resolve({
                  warnings: allWarnings
                });
              }
            });
          }).catch(parseError => {
            parseWarnings = parseError.warnings;
            parseError = checkDMNCompatibilityError(parseError, xml) || checkValidationError(parseError) || parseError;
            self._emit('import.parse.complete', ParseCompleteEvent({
              error: parseError,
              warnings: parseWarnings
            }));
            self._emit('import.done', {
              error: parseError,
              warnings: parseWarnings
            });
            parseError.warnings = parseWarnings;
            return reject(parseError);
          });
        }).catch(clearError => {
          self._emit('import.done', {
            error: clearError,
            warnings: []
          });
          clearError.warnings = [];
          return reject(clearError);
        });
      });

      // TODO: remove with future dmn-js version
      function ParseCompleteEvent(data) {
        var event = self._eventBus.createEvent(data);
        Object.defineProperty(event, 'context', {
          enumerable: true,
          get: function () {
            console.warn(new Error('import.parse.complete <context> is deprecated ' + 'and will be removed in future library versions'));
            return {
              warnings: data.warnings,
              references: data.references,
              elementsById: data.elementsById
            };
          }
        });
        return event;
      }
    }
    getDefinitions() {
      return this._definitions;
    }

    /**
     * Return active view.
     *
     * @return {View}
     */
    getActiveView() {
      return this._activeView;
    }

    /**
     * Get the currently active viewer instance.
     *
     * @return {View}
     */
    getActiveViewer() {
      var activeView = this.getActiveView();
      return activeView && this._getViewer(activeView);
    }
    getView(element) {
      return this._views.filter(function (v) {
        return v.element === element;
      })[0];
    }
    getViews() {
      return this._views;
    }

    /**
     * The saveXML result.
     *
     * @typedef {Object} SaveXMLResult
     *
     * @property {string} xml
     */

    /**
     * Export the currently displayed DMN diagram as
     * a DMN XML document.
     *
     * ## Life-Cycle Events
     *
     * During XML saving the viewer will fire life-cycle events:
     *
     *   * saveXML.start (before serialization)
     *   * saveXML.serialized (after xml generation)
     *   * saveXML.done (everything done)
     *
     * You can use these events to hook into the life-cycle.
     *
     * @param {Object} [options] export options
     * @param {boolean} [options.format=false] output formated XML
     * @param {boolean} [options.preamble=true] output preamble
     *
     * @return {Promise<SaveXMLResult, Error>}
     */
    saveXML(options) {
      var self = this;
      options = options || {};
      var definitions = this._definitions;
      return new Promise(function (resolve, reject) {
        if (!definitions) {
          reject(new Error('no definitions loaded'));
          return;
        }

        // allow to fiddle around with definitions
        definitions = self._emit('saveXML.start', {
          definitions: definitions
        }) || definitions;
        self._moddle.toXML(definitions, options).then(function (result) {
          var xml = result.xml;
          xml = self._emit('saveXML.serialized', {
            xml: xml
          }) || xml;
          return {
            xml
          };
        }).catch(error => ({
          error
        })).then(result => {
          self._emit('saveXML.done', result);
          if (result.error) {
            reject(result.error);
          } else {
            resolve({
              xml: result.xml
            });
          }
        });
      });
    }

    /**
     * Register an event listener
     *
     * Remove a previously added listener via {@link #off(event, callback)}.
     *
     * @param {string} event
     * @param {number} [priority]
     * @param {Function} callback
     * @param {Object} [that]
     */
    on(...args) {
      this._eventBus.on(...args);
    }

    /**
     * De-register an event listener
     *
     * @param {string} event
     * @param {Function} callback
     */
    off(...args) {
      this._eventBus.off(...args);
    }

    /**
     * Register a listener to be invoked once only.
     *
     * @param {string} event
     * @param {number} [priority]
     * @param {Function} callback
     * @param {Object} [that]
     */
    once(...args) {
      this._eventBus.once(...args);
    }
    attachTo(parentNode) {
      // unwrap jQuery if provided
      if (parentNode.get && parentNode.constructor.prototype.jquery) {
        parentNode = parentNode.get(0);
      }
      if (typeof parentNode === 'string') {
        parentNode = query(parentNode);
      }
      parentNode.appendChild(this._container);
      this._emit('attach', {});
    }
    detach() {
      this._emit('detach', {});
      remove$4(this._container);
    }
    destroy() {
      Object.keys(this._viewers).forEach(viewerId => {
        var viewer = this._viewers[viewerId];
        safeExecute(viewer, 'destroy');
      });
      remove$4(this._container);
    }
    _init(options) {
      this._options = options;
      this._moddle = this._createModdle(options);
      this._viewers = {};
      this._views = [];
      const container = domify$1('<div class="dmn-js-parent"></div>');
      const containerOptions = assign$4({}, DEFAULT_CONTAINER_OPTIONS, options);
      assign$4(container.style, {
        width: ensureUnit(containerOptions.width),
        height: ensureUnit(containerOptions.height),
        position: containerOptions.position
      });
      this._container = container;
      if (options.container) {
        this.attachTo(options.container);
      }
    }
    _clear() {
      return this._switchView(null);
    }

    /**
     * Open diagram view.
     *
     * @param  {View} view
     * @returns {Promise} Resolves with {OpenResult} when successful
     * or rejects with {OpenError}
     */
    open(view) {
      return this._switchView(view);
    }
    _setDefinitions(definitions) {
      this._definitions = definitions;
      this._updateViews();
    }
    _viewsChanged = () => {
      this._emit('views.changed', {
        views: this._views,
        activeView: this._activeView
      });
    };

    /**
     * Recompute changed views after elements in
     * the DMN diagram have changed.
     */
    _updateViews() {
      var definitions = this._definitions;
      if (!definitions) {
        this._views = [];
        this._switchView(null);
        return;
      }
      var viewProviders = this._getViewProviders();
      var displayableElements = [definitions, ...(definitions.drgElement || [])];

      // compute list of available views
      var views = this._views,
        newViews = [];
      for (var element of displayableElements) {
        var provider = find$2(viewProviders, function (provider) {
          if (typeof provider.opens === 'string') {
            return provider.opens === element.$type;
          } else {
            return provider.opens(element);
          }
        });
        if (!provider) {
          continue;
        }
        var view = {
          element,
          id: element.id,
          name: element.name,
          type: provider.id
        };
        newViews.push(view);
      }
      var activeView = this._activeView,
        newActiveView;
      if (activeView) {
        // check the new active view
        newActiveView = find$2(newViews, function (view) {
          return viewsEqual(activeView, view);
        }) || this._getInitialView(newViews);
        if (!newActiveView) {
          this._switchView(null);
          return;
        }
      }

      // Views have changed if
      // active view has changed OR
      // number of views has changed OR
      // not all views equal
      var activeViewChanged = !viewsEqual(activeView, newActiveView) || viewNameChanged(activeView, newActiveView);
      var viewsChanged = views.length !== newViews.length || !every(newViews, function (newView) {
        return find$2(views, function (view) {
          return viewsEqual(view, newView) && !viewNameChanged(view, newView);
        });
      });
      this._activeView = newActiveView;
      this._views = newViews;
      if (activeViewChanged || viewsChanged) {
        this._viewsChanged();
      }
    }
    _getInitialView(views, preferredView) {
      var initialView;
      if (preferredView) {
        initialView = find$2(views, function (view) {
          return viewsEqual(view, preferredView);
        }) || find$2(views, function (view) {
          return view.type === preferredView;
        });
      }
      return initialView || views[0];
    }

    /**
     * Switch to another view.
     *
     * @param  {View} newView
     * @returns {Promise} Resolves with {OpenResult} when successful
     * or rejects with {OpenError}
     */
    _switchView(newView) {
      var self = this;
      return new Promise(function (resolve, reject) {
        var complete = (openError, openResult) => {
          self._viewsChanged();
          if (openError) {
            reject(openError);
          } else {
            resolve(openResult);
          }
        };
        var activeView = self.getActiveView(),
          activeViewer;
        var newViewer = newView && self._getViewer(newView),
          element = newView && newView.element;
        if (activeView) {
          activeViewer = self._getViewer(activeView);
          if (activeViewer !== newViewer) {
            safeExecute(activeViewer, 'clear');
            activeViewer.detach();
          }
        }
        self._activeView = newView;
        if (newViewer) {
          if (activeViewer !== newViewer) {
            newViewer.attachTo(self._container);
          }
          self._emit('import.render.start', {
            view: newView,
            element: element
          });
          newViewer.open(element).then(result => {
            self._emit('import.render.complete', {
              view: newView,
              error: null,
              warnings: result.warnings
            });
            complete(null, result);
          }).catch(error => {
            self._emit('import.render.complete', {
              view: newView,
              error: error,
              warnings: error.warnings
            });
            complete(error, null);
          });
          return;
        }

        // no active view
        complete();
      });
    }
    _getViewer(view) {
      var type = view.type;
      var viewer = this._viewers[type];
      if (!viewer) {
        viewer = this._viewers[type] = this._createViewer(view.type);
        this._emit('viewer.created', {
          type: type,
          viewer: viewer
        });
      }
      return viewer;
    }
    _createViewer(id) {
      var provider = find$2(this._getViewProviders(), function (provider) {
        return provider.id === id;
      });
      if (!provider) {
        throw new Error('no provider for view type <' + id + '>');
      }
      var Viewer = provider.constructor;
      var providerOptions = this._options[id] || {};
      var commonOptions = this._options.common || {};
      return new Viewer({
        ...commonOptions,
        ...providerOptions,
        additionalModules: [...(providerOptions.additionalModules || []), {
          _parent: ['value', this],
          moddle: ['value', this._moddle]
        }]
      });
    }

    /**
     * Emit an event.
     */
    _emit(...args) {
      return this._eventBus.fire(...args);
    }
    _createModdle(options) {
      return new simple(options.moddleExtensions);
    }

    /**
     * Return the list of available view providers.
     *
     * @abstract
     *
     * @return {Array<ViewProvider>}
     */
    _getViewProviders() {
      return [];
    }
  }

  // helpers //////////////////////

  /**
   * Ensure the passed argument is a proper unit (defaulting to px)
   */
  function ensureUnit(val) {
    return val + (isNumber$3(val) ? 'px' : '');
  }
  function checkDMNCompatibilityError(err, xml) {
    // check if we can indicate opening of old DMN 1.1 or DMN 1.2 diagrams

    if (err.message !== 'failed to parse document as <dmn:Definitions>') {
      return null;
    }
    var olderDMNVersion = xml.indexOf('"http://www.omg.org/spec/DMN/20151101/dmn.xsd"') !== -1 && '1.1' || xml.indexOf('"http://www.omg.org/spec/DMN/20180521/MODEL/"') !== -1 && '1.2';
    if (!olderDMNVersion) {
      return null;
    }
    err = new Error('unsupported DMN ' + olderDMNVersion + ' file detected; ' + 'only DMN 1.3 files can be opened');
    console.error('Cannot open what looks like a DMN ' + olderDMNVersion + ' diagram. ' + 'Please refer to https://bpmn.io/l/dmn-compatibility.html ' + 'to learn how to make the toolkit compatible with older DMN files', err);
    return err;
  }
  function checkValidationError(err) {
    // check if we can help the user by indicating wrong DMN 1.3 xml
    // (in case he or the exporting tool did not get that right)

    var pattern = /unparsable content <([^>]+)> detected([\s\S]*)$/,
      match = pattern.exec(err.message);
    if (!match) {
      return null;
    }
    err.message = 'unparsable content <' + match[1] + '> detected; ' + 'this may indicate an invalid DMN 1.3 diagram file' + match[2];
    return err;
  }
  function viewsEqual(a, b) {
    if (!isDefined(a)) {
      if (!isDefined(b)) {
        return true;
      } else {
        return false;
      }
    }
    if (!isDefined(b)) {
      return false;
    }

    // compare by element OR element ID equality
    return a.element === b.element || a.id === b.id;
  }
  function viewNameChanged(a, b) {
    return !a || !b || a.name !== b.name;
  }
  function safeExecute(viewer, method) {
    if (isFunction$2(viewer[method])) {
      viewer[method]();
    }
  }

  function createCommonjsModule(fn, module) {
    return module = {
      exports: {}
    }, fn(module, module.exports), module.exports;
  }
  var hat_1 = createCommonjsModule(function (module) {
    var hat = module.exports = function (bits, base) {
      if (!base) base = 16;
      if (bits === undefined) bits = 128;
      if (bits <= 0) return '0';
      var digits = Math.log(Math.pow(2, bits)) / Math.log(base);
      for (var i = 2; digits === Infinity; i *= 2) {
        digits = Math.log(Math.pow(2, bits / i)) / Math.log(base) * i;
      }
      var rem = digits - Math.floor(digits);
      var res = '';
      for (var i = 0; i < Math.floor(digits); i++) {
        var x = Math.floor(Math.random() * base).toString(base);
        res = x + res;
      }
      if (rem) {
        var b = Math.pow(base, rem);
        var x = Math.floor(Math.random() * b).toString(base);
        res = x + res;
      }
      var parsed = parseInt(res, base);
      if (parsed !== Infinity && parsed >= Math.pow(2, bits)) {
        return hat(bits, base);
      } else return res;
    };
    hat.rack = function (bits, base, expandBy) {
      var fn = function (data) {
        var iters = 0;
        do {
          if (iters++ > 10) {
            if (expandBy) bits += expandBy;else throw new Error('too many ID collisions, use more bits');
          }
          var id = hat(bits, base);
        } while (Object.hasOwnProperty.call(hats, id));
        hats[id] = data;
        return id;
      };
      var hats = fn.hats = {};
      fn.get = function (id) {
        return fn.hats[id];
      };
      fn.set = function (id, value) {
        fn.hats[id] = value;
        return fn;
      };
      fn.bits = bits || 128;
      fn.base = base || 16;
      return fn;
    };
  });

  /**
   * Create a new id generator / cache instance.
   *
   * You may optionally provide a seed that is used internally.
   *
   * @param {Seed} seed
   */
  function Ids$1(seed) {
    if (!(this instanceof Ids$1)) {
      return new Ids$1(seed);
    }
    seed = seed || [128, 36, 1];
    this._seed = seed.length ? hat_1.rack(seed[0], seed[1], seed[2]) : seed;
  }

  /**
   * Generate a next id.
   *
   * @param {Object} [element] element to bind the id to
   *
   * @return {String} id
   */
  Ids$1.prototype.next = function (element) {
    return this._seed(element || true);
  };

  /**
   * Generate a next id with a given prefix.
   *
   * @param {Object} [element] element to bind the id to
   *
   * @return {String} id
   */
  Ids$1.prototype.nextPrefixed = function (prefix, element) {
    var id;
    do {
      id = prefix + this.next(true);
    } while (this.assigned(id));

    // claim {prefix}{random}
    this.claim(id, element);

    // return
    return id;
  };

  /**
   * Manually claim an existing id.
   *
   * @param {String} id
   * @param {String} [element] element the id is claimed by
   */
  Ids$1.prototype.claim = function (id, element) {
    this._seed.set(id, element || true);
  };

  /**
   * Returns true if the given id has already been assigned.
   *
   * @param  {String} id
   * @return {Boolean}
   */
  Ids$1.prototype.assigned = function (id) {
    return this._seed.get(id) || false;
  };

  /**
   * Unclaim an id.
   *
   * @param  {String} id the id to unclaim
   */
  Ids$1.prototype.unclaim = function (id) {
    delete this._seed.hats[id];
  };

  /**
   * Clear all claimed ids.
   */
  Ids$1.prototype.clear = function () {
    var hats = this._seed.hats,
      id;
    for (id in hats) {
      this.unclaim(id);
    }
  };

  /**
   * Is an element of the given DMN type?
   *
   * @param  {tjs.model.Base|ModdleElement} element
   * @param  {string} type
   *
   * @return {boolean}
   */
  function is$1(element, type) {
    var bo = getBusinessObject(element);
    return bo && typeof bo.$instanceOf === 'function' && bo.$instanceOf(type);
  }
  function isInput(element) {
    return is$1(element, 'dmn:InputClause');
  }
  function isOutput(element) {
    return is$1(element, 'dmn:OutputClause');
  }

  /**
   * Return the business object for a given element.
   *
   * @param  {tjs.model.Base|ModdleElement} element
   *
   * @return {ModdleElement}
   */
  function getBusinessObject(element) {
    return element && element.businessObject || element;
  }
  function getName(element) {
    return getBusinessObject(element).name;
  }

  /**
   * Return true if element has any of the given types.
   *
   * @param {djs.model.Base} element
   * @param {Array<string>} types
   *
   * @return {boolean}
   */
  function isAny(element, types) {
    return some(types, function (t) {
      return is$1(element, t);
    });
  }

  /**
   * Return main boxed expression of a given decision or BKM.
   *
   * @param {ModdleElement} decisionOrBkm - the decision or business knowledge model
   * @returns {ModdleElement|undefined}
   */
  function getBoxedExpression(decisionOrBkm) {
    var bo = getBusinessObject(decisionOrBkm);
    if (is$1(bo, 'dmn:Decision')) {
      return bo.get('decisionLogic');
    } else if (is$1(bo, 'dmn:BusinessKnowledgeModel')) {
      var encapsulatedLogic = bo.get('encapsulatedLogic');
      return encapsulatedLogic && encapsulatedLogic.get('body');
    }
  }
  const FEEL_NAMESPACE = 'https://www.omg.org/spec/DMN/20191111/FEEL/';

  /**
   * Return `true` if the expression language for a given element is FEEL.
   *
   * @param {ModdleElement} element
   * @returns {string}
   */
  function isFeel(element) {
    for (let current = element; current; current = current.$parent) {
      const expressionLanguage = current.get('expressionLanguage');
      if (expressionLanguage) {
        if (expressionLanguage === FEEL_NAMESPACE || /feel/i.test(expressionLanguage)) {
          return true;
        } else {
          return false;
        }
      }
      if (is$1(current, 'dmn:FunctionDefinition')) {
        // TODO(@barmac): remove fallback when moddle assures the default value
        const kind = current.get('kind') || 'FEEL';
        return kind === 'FEEL';
      }
    }
    return true;
  }

  class EditingManager extends Manager {
    _init(options) {
      super._init(options);

      // hook ID collection into the modeler
      this.on('import.parse.complete', event => {
        if (!event.error) {
          this._collectIds(event.definitions, event.elementsById);
        }
      });
      this.on('destroy', () => {
        this._moddle.ids.clear();
      });
      this.on('viewer.created', ({
        viewer
      }) => {
        viewer.on('elements.changed', ({
          elements
        }) => {
          var viewsChanged = elements.some(function (e) {
            return isAny(e, ['dmn:Decision', 'dmn:Definitions', 'dmn:BusinessKnowledgeModel']);
          });
          if (viewsChanged) {
            this._updateViews();
          }
        });
      });
    }

    /**
     * Collect ids processed during parsing of the
     * definitions object.
     *
     * @param {ModdleElement} definitions
     * @param {Array<ModdleElement>} elementsById
     */
    _collectIds(definitions, elementsById) {
      var moddle = definitions.$model,
        ids = moddle.ids,
        id;

      // remove references from previous import
      ids.clear();
      for (id in elementsById) {
        ids.claim(id, elementsById[id]);
      }
    }
    _createModdle(options) {
      var moddle = super._createModdle(options);

      // attach ids to moddle to be able to track
      // and validated ids in the DMN XML document
      // tree
      moddle.ids = new Ids$1([32, 36, 1]);
      return moddle;
    }
  }

  function e$3(e, t) {
    t && (e.super_ = t, e.prototype = Object.create(t.prototype, {
      constructor: {
        value: e,
        enumerable: !1,
        writable: !0,
        configurable: !0
      }
    }));
  }

  const CLASS_PATTERN = /^class[ {]/;

  /**
   * @param {function} fn
   *
   * @return {boolean}
   */
  function isClass(fn) {
    return CLASS_PATTERN.test(fn.toString());
  }

  /**
   * @param {any} obj
   *
   * @return {boolean}
   */
  function isArray$2(obj) {
    return Array.isArray(obj);
  }

  /**
   * @param {any} obj
   * @param {string} prop
   *
   * @return {boolean}
   */
  function hasOwnProp(obj, prop) {
    return Object.prototype.hasOwnProperty.call(obj, prop);
  }

  /**
   * @typedef {import('./index.js').InjectAnnotated } InjectAnnotated
   */

  /**
   * @template T
   *
   * @params {[...string[], T] | ...string[], T} args
   *
   * @return {T & InjectAnnotated}
   */
  function annotate(...args) {
    if (args.length === 1 && isArray$2(args[0])) {
      args = args[0];
    }
    args = [...args];
    const fn = args.pop();
    fn.$inject = args;
    return fn;
  }

  // Current limitations:
  // - can't put into "function arg" comments
  // function /* (no parenthesis like this) */ (){}
  // function abc( /* xx (no parenthesis like this) */ a, b) {}
  //
  // Just put the comment before function or inside:
  // /* (((this is fine))) */ function(a, b) {}
  // function abc(a) { /* (((this is fine))) */}
  //
  // - can't reliably auto-annotate constructor; we'll match the
  // first constructor(...) pattern found which may be the one
  // of a nested class, too.

  const CONSTRUCTOR_ARGS = /constructor\s*[^(]*\(\s*([^)]*)\)/m;
  const FN_ARGS = /^(?:async\s+)?(?:function\s*[^(]*)?(?:\(\s*([^)]*)\)|(\w+))/m;
  const FN_ARG = /\/\*([^*]*)\*\//m;

  /**
   * @param {unknown} fn
   *
   * @return {string[]}
   */
  function parseAnnotations(fn) {
    if (typeof fn !== 'function') {
      throw new Error(`Cannot annotate "${fn}". Expected a function!`);
    }
    const match = fn.toString().match(isClass(fn) ? CONSTRUCTOR_ARGS : FN_ARGS);

    // may parse class without constructor
    if (!match) {
      return [];
    }
    const args = match[1] || match[2];
    return args && args.split(',').map(arg => {
      const argMatch = arg.match(FN_ARG);
      return (argMatch && argMatch[1] || arg).trim();
    }) || [];
  }

  /**
   * @typedef { import('./index.js').ModuleDeclaration } ModuleDeclaration
   * @typedef { import('./index.js').ModuleDefinition } ModuleDefinition
   * @typedef { import('./index.js').InjectorContext } InjectorContext
   *
   * @typedef { import('./index.js').TypedDeclaration<any, any> } TypedDeclaration
   */

  /**
   * Create a new injector with the given modules.
   *
   * @param {ModuleDefinition[]} modules
   * @param {InjectorContext} [_parent]
   */
  function Injector(modules, _parent) {
    const parent = _parent || (/** @type InjectorContext */{
      get: function (name, strict) {
        currentlyResolving.push(name);
        if (strict === false) {
          return null;
        } else {
          throw error(`No provider for "${name}"!`);
        }
      }
    });
    const currentlyResolving = [];
    const providers = this._providers = Object.create(parent._providers || null);
    const instances = this._instances = Object.create(null);
    const self = instances.injector = this;
    const error = function (msg) {
      const stack = currentlyResolving.join(' -> ');
      currentlyResolving.length = 0;
      return new Error(stack ? `${msg} (Resolving: ${stack})` : msg);
    };

    /**
     * Return a named service.
     *
     * @param {string} name
     * @param {boolean} [strict=true] if false, resolve missing services to null
     *
     * @return {any}
     */
    function get(name, strict) {
      if (!providers[name] && name.includes('.')) {
        const parts = name.split('.');
        let pivot = get(/** @type { string } */parts.shift());
        while (parts.length) {
          pivot = pivot[(/** @type { string } */parts.shift())];
        }
        return pivot;
      }
      if (hasOwnProp(instances, name)) {
        return instances[name];
      }
      if (hasOwnProp(providers, name)) {
        if (currentlyResolving.indexOf(name) !== -1) {
          currentlyResolving.push(name);
          throw error('Cannot resolve circular dependency!');
        }
        currentlyResolving.push(name);
        instances[name] = providers[name][0](providers[name][1]);
        currentlyResolving.pop();
        return instances[name];
      }
      return parent.get(name, strict);
    }
    function fnDef(fn, locals) {
      if (typeof locals === 'undefined') {
        locals = {};
      }
      if (typeof fn !== 'function') {
        if (isArray$2(fn)) {
          fn = annotate(fn.slice());
        } else {
          throw error(`Cannot invoke "${fn}". Expected a function!`);
        }
      }

      /**
       * @type {string[]}
       */
      const inject = fn.$inject || parseAnnotations(fn);
      const dependencies = inject.map(dep => {
        if (hasOwnProp(locals, dep)) {
          return locals[dep];
        } else {
          return get(dep);
        }
      });
      return {
        fn: fn,
        dependencies
      };
    }

    /**
     * Instantiate the given type, injecting dependencies.
     *
     * @template T
     *
     * @param { Function | [...string[], Function ]} type
     *
     * @return T
     */
    function instantiate(type) {
      const {
        fn,
        dependencies
      } = fnDef(type);

      // instantiate var args constructor
      const Constructor = Function.prototype.bind.call(fn, null, ...dependencies);
      return new Constructor();
    }

    /**
     * Invoke the given function, injecting dependencies. Return the result.
     *
     * @template T
     *
     * @param { Function | [...string[], Function ]} func
     * @param { Object } [context]
     * @param { Object } [locals]
     *
     * @return {T} invocation result
     */
    function invoke(func, context, locals) {
      const {
        fn,
        dependencies
      } = fnDef(func, locals);
      return fn.apply(context, dependencies);
    }

    /**
     * @param {Injector} childInjector
     *
     * @return {Function}
     */
    function createPrivateInjectorFactory(childInjector) {
      return annotate(key => childInjector.get(key));
    }

    /**
     * @param {ModuleDefinition[]} modules
     * @param {string[]} [forceNewInstances]
     *
     * @return {Injector}
     */
    function createChild(modules, forceNewInstances) {
      if (forceNewInstances && forceNewInstances.length) {
        const fromParentModule = Object.create(null);
        const matchedScopes = Object.create(null);
        const privateInjectorsCache = [];
        const privateChildInjectors = [];
        const privateChildFactories = [];
        let provider;
        let cacheIdx;
        let privateChildInjector;
        let privateChildInjectorFactory;
        for (let name in providers) {
          provider = providers[name];
          if (forceNewInstances.indexOf(name) !== -1) {
            if (provider[2] === 'private') {
              cacheIdx = privateInjectorsCache.indexOf(provider[3]);
              if (cacheIdx === -1) {
                privateChildInjector = provider[3].createChild([], forceNewInstances);
                privateChildInjectorFactory = createPrivateInjectorFactory(privateChildInjector);
                privateInjectorsCache.push(provider[3]);
                privateChildInjectors.push(privateChildInjector);
                privateChildFactories.push(privateChildInjectorFactory);
                fromParentModule[name] = [privateChildInjectorFactory, name, 'private', privateChildInjector];
              } else {
                fromParentModule[name] = [privateChildFactories[cacheIdx], name, 'private', privateChildInjectors[cacheIdx]];
              }
            } else {
              fromParentModule[name] = [provider[2], provider[1]];
            }
            matchedScopes[name] = true;
          }
          if ((provider[2] === 'factory' || provider[2] === 'type') && provider[1].$scope) {
            /* jshint -W083 */
            forceNewInstances.forEach(scope => {
              if (provider[1].$scope.indexOf(scope) !== -1) {
                fromParentModule[name] = [provider[2], provider[1]];
                matchedScopes[scope] = true;
              }
            });
          }
        }
        forceNewInstances.forEach(scope => {
          if (!matchedScopes[scope]) {
            throw new Error('No provider for "' + scope + '". Cannot use provider from the parent!');
          }
        });
        modules.unshift(fromParentModule);
      }
      return new Injector(modules, self);
    }
    const factoryMap = {
      factory: invoke,
      type: instantiate,
      value: function (value) {
        return value;
      }
    };

    /**
     * @param {ModuleDefinition} moduleDefinition
     * @param {Injector} injector
     */
    function createInitializer(moduleDefinition, injector) {
      const initializers = moduleDefinition.__init__ || [];
      return function () {
        initializers.forEach(initializer => {
          // eagerly resolve component (fn or string)
          if (typeof initializer === 'string') {
            injector.get(initializer);
          } else {
            injector.invoke(initializer);
          }
        });
      };
    }

    /**
     * @param {ModuleDefinition} moduleDefinition
     */
    function loadModule(moduleDefinition) {
      const moduleExports = moduleDefinition.__exports__;

      // private module
      if (moduleExports) {
        const nestedModules = moduleDefinition.__modules__;
        const clonedModule = Object.keys(moduleDefinition).reduce((clonedModule, key) => {
          if (key !== '__exports__' && key !== '__modules__' && key !== '__init__' && key !== '__depends__') {
            clonedModule[key] = moduleDefinition[key];
          }
          return clonedModule;
        }, Object.create(null));
        const childModules = (nestedModules || []).concat(clonedModule);
        const privateInjector = createChild(childModules);
        const getFromPrivateInjector = annotate(function (key) {
          return privateInjector.get(key);
        });
        moduleExports.forEach(function (key) {
          providers[key] = [getFromPrivateInjector, key, 'private', privateInjector];
        });

        // ensure child injector initializes
        const initializers = (moduleDefinition.__init__ || []).slice();
        initializers.unshift(function () {
          privateInjector.init();
        });
        moduleDefinition = Object.assign({}, moduleDefinition, {
          __init__: initializers
        });
        return createInitializer(moduleDefinition, privateInjector);
      }

      // normal module
      Object.keys(moduleDefinition).forEach(function (key) {
        if (key === '__init__' || key === '__depends__') {
          return;
        }
        const typeDeclaration = /** @type { TypedDeclaration } */
        moduleDefinition[key];
        if (typeDeclaration[2] === 'private') {
          providers[key] = typeDeclaration;
          return;
        }
        const type = typeDeclaration[0];
        const value = typeDeclaration[1];
        providers[key] = [factoryMap[type], arrayUnwrap(type, value), type];
      });
      return createInitializer(moduleDefinition, self);
    }

    /**
     * @param {ModuleDefinition[]} moduleDefinitions
     * @param {ModuleDefinition} moduleDefinition
     *
     * @return {ModuleDefinition[]}
     */
    function resolveDependencies(moduleDefinitions, moduleDefinition) {
      if (moduleDefinitions.indexOf(moduleDefinition) !== -1) {
        return moduleDefinitions;
      }
      moduleDefinitions = (moduleDefinition.__depends__ || []).reduce(resolveDependencies, moduleDefinitions);
      if (moduleDefinitions.indexOf(moduleDefinition) !== -1) {
        return moduleDefinitions;
      }
      return moduleDefinitions.concat(moduleDefinition);
    }

    /**
     * @param {ModuleDefinition[]} moduleDefinitions
     *
     * @return { () => void } initializerFn
     */
    function bootstrap(moduleDefinitions) {
      const initializers = moduleDefinitions.reduce(resolveDependencies, []).map(loadModule);
      let initialized = false;
      return function () {
        if (initialized) {
          return;
        }
        initialized = true;
        initializers.forEach(initializer => initializer());
      };
    }

    // public API
    this.get = get;
    this.invoke = invoke;
    this.instantiate = instantiate;
    this.createChild = createChild;

    // setup
    this.init = bootstrap(modules);
  }

  // helpers ///////////////

  function arrayUnwrap(type, value) {
    if (type !== 'value' && isArray$2(value)) {
      value = annotate(value.slice());
    }
    return value;
  }

  var DEFAULT_RENDER_PRIORITY$1 = 1000;

  /**
   * @typedef {import('../core/Types').ElementLike} Element
   * @typedef {import('../core/Types').ConnectionLike} Connection
   * @typedef {import('../core/Types').ShapeLike} Shape
   *
   * @typedef {import('../core/EventBus').default} EventBus
   */

  /**
   * The base implementation of shape and connection renderers.
   *
   * @param {EventBus} eventBus
   * @param {number} [renderPriority=1000]
   */
  function BaseRenderer(eventBus, renderPriority) {
    var self = this;
    renderPriority = renderPriority || DEFAULT_RENDER_PRIORITY$1;
    eventBus.on(['render.shape', 'render.connection'], renderPriority, function (evt, context) {
      var type = evt.type,
        element = context.element,
        visuals = context.gfx,
        attrs = context.attrs;
      if (self.canRender(element)) {
        if (type === 'render.shape') {
          return self.drawShape(visuals, element, attrs);
        } else {
          return self.drawConnection(visuals, element, attrs);
        }
      }
    });
    eventBus.on(['render.getShapePath', 'render.getConnectionPath'], renderPriority, function (evt, element) {
      if (self.canRender(element)) {
        if (evt.type === 'render.getShapePath') {
          return self.getShapePath(element);
        } else {
          return self.getConnectionPath(element);
        }
      }
    });
  }

  /**
   * Checks whether an element can be rendered.
   *
   * @param {Element} element The element to be rendered.
   *
   * @return {boolean} Whether the element can be rendered.
   */
  BaseRenderer.prototype.canRender = function (element) {};

  /**
   * Draws a shape.
   *
   * @param {SVGElement} visuals The SVG element to draw the shape into.
   * @param {Shape} shape The shape to be drawn.
   *
   * @return {SVGElement} The SVG element of the shape drawn.
   */
  BaseRenderer.prototype.drawShape = function (visuals, shape) {};

  /**
   * Draws a connection.
   *
   * @param {SVGElement} visuals The SVG element to draw the connection into.
   * @param {Connection} connection The connection to be drawn.
   *
   * @return {SVGElement} The SVG element of the connection drawn.
   */
  BaseRenderer.prototype.drawConnection = function (visuals, connection) {};

  /**
   * Gets the SVG path of the graphical representation of a shape.
   *
   * @param {Shape} shape The shape.
   *
   * @return {string} The SVG path of the shape.
   */
  BaseRenderer.prototype.getShapePath = function (shape) {};

  /**
   * Gets the SVG path of the graphical representation of a connection.
   *
   * @param {Connection} connection The connection.
   *
   * @return {string} The SVG path of the connection.
   */
  BaseRenderer.prototype.getConnectionPath = function (connection) {};

  function ensureImported(element, target) {
    if (element.ownerDocument !== target.ownerDocument) {
      try {
        // may fail on webkit
        return target.ownerDocument.importNode(element, true);
      } catch (e) {

        // ignore
      }
    }
    return element;
  }

  /**
   * appendTo utility
   */

  /**
   * Append a node to a target element and return the appended node.
   *
   * @param  {SVGElement} element
   * @param  {SVGElement} target
   *
   * @return {SVGElement} the appended node
   */
  function appendTo(element, target) {
    return target.appendChild(ensureImported(element, target));
  }

  /**
   * append utility
   */

  /**
   * Append a node to an element
   *
   * @param  {SVGElement} element
   * @param  {SVGElement} node
   *
   * @return {SVGElement} the element
   */
  function append(target, node) {
    appendTo(node, target);
    return target;
  }

  /**
   * attribute accessor utility
   */

  var LENGTH_ATTR = 2;
  var CSS_PROPERTIES = {
    'alignment-baseline': 1,
    'baseline-shift': 1,
    'clip': 1,
    'clip-path': 1,
    'clip-rule': 1,
    'color': 1,
    'color-interpolation': 1,
    'color-interpolation-filters': 1,
    'color-profile': 1,
    'color-rendering': 1,
    'cursor': 1,
    'direction': 1,
    'display': 1,
    'dominant-baseline': 1,
    'enable-background': 1,
    'fill': 1,
    'fill-opacity': 1,
    'fill-rule': 1,
    'filter': 1,
    'flood-color': 1,
    'flood-opacity': 1,
    'font': 1,
    'font-family': 1,
    'font-size': LENGTH_ATTR,
    'font-size-adjust': 1,
    'font-stretch': 1,
    'font-style': 1,
    'font-variant': 1,
    'font-weight': 1,
    'glyph-orientation-horizontal': 1,
    'glyph-orientation-vertical': 1,
    'image-rendering': 1,
    'kerning': 1,
    'letter-spacing': 1,
    'lighting-color': 1,
    'marker': 1,
    'marker-end': 1,
    'marker-mid': 1,
    'marker-start': 1,
    'mask': 1,
    'opacity': 1,
    'overflow': 1,
    'pointer-events': 1,
    'shape-rendering': 1,
    'stop-color': 1,
    'stop-opacity': 1,
    'stroke': 1,
    'stroke-dasharray': 1,
    'stroke-dashoffset': 1,
    'stroke-linecap': 1,
    'stroke-linejoin': 1,
    'stroke-miterlimit': 1,
    'stroke-opacity': 1,
    'stroke-width': LENGTH_ATTR,
    'text-anchor': 1,
    'text-decoration': 1,
    'text-rendering': 1,
    'unicode-bidi': 1,
    'visibility': 1,
    'word-spacing': 1,
    'writing-mode': 1
  };
  function getAttribute(node, name) {
    if (CSS_PROPERTIES[name]) {
      return node.style[name];
    } else {
      return node.getAttributeNS(null, name);
    }
  }
  function setAttribute(node, name, value) {
    var hyphenated = name.replace(/([a-z])([A-Z])/g, '$1-$2').toLowerCase();
    var type = CSS_PROPERTIES[hyphenated];
    if (type) {
      // append pixel unit, unless present
      if (type === LENGTH_ATTR && typeof value === 'number') {
        value = String(value) + 'px';
      }
      node.style[hyphenated] = value;
    } else {
      node.setAttributeNS(null, name, value);
    }
  }
  function setAttributes(node, attrs) {
    var names = Object.keys(attrs),
      i,
      name;
    for (i = 0, name; name = names[i]; i++) {
      setAttribute(node, name, attrs[name]);
    }
  }

  /**
   * Gets or sets raw attributes on a node.
   *
   * @param  {SVGElement} node
   * @param  {Object} [attrs]
   * @param  {String} [name]
   * @param  {String} [value]
   *
   * @return {String}
   */
  function attr(node, name, value) {
    if (typeof name === 'string') {
      if (value !== undefined) {
        setAttribute(node, name, value);
      } else {
        return getAttribute(node, name);
      }
    } else {
      setAttributes(node, name);
    }
    return node;
  }

  /**
   * Taken from https://github.com/component/classes
   *
   * Without the component bits.
   */

  /**
   * toString reference.
   */

  const toString = Object.prototype.toString;

  /**
    * Wrap `el` in a `ClassList`.
    *
    * @param {Element} el
    * @return {ClassList}
    * @api public
    */

  function classes(el) {
    return new ClassList(el);
  }
  function ClassList(el) {
    if (!el || !el.nodeType) {
      throw new Error('A DOM element reference is required');
    }
    this.el = el;
    this.list = el.classList;
  }

  /**
    * Add class `name` if not already present.
    *
    * @param {String} name
    * @return {ClassList}
    * @api public
    */

  ClassList.prototype.add = function (name) {
    this.list.add(name);
    return this;
  };

  /**
    * Remove class `name` when present, or
    * pass a regular expression to remove
    * any which match.
    *
    * @param {String|RegExp} name
    * @return {ClassList}
    * @api public
    */

  ClassList.prototype.remove = function (name) {
    if ('[object RegExp]' == toString.call(name)) {
      return this.removeMatching(name);
    }
    this.list.remove(name);
    return this;
  };

  /**
    * Remove all classes matching `re`.
    *
    * @param {RegExp} re
    * @return {ClassList}
    * @api private
    */

  ClassList.prototype.removeMatching = function (re) {
    const arr = this.array();
    for (let i = 0; i < arr.length; i++) {
      if (re.test(arr[i])) {
        this.remove(arr[i]);
      }
    }
    return this;
  };

  /**
    * Toggle class `name`, can force state via `force`.
    *
    * For browsers that support classList, but do not support `force` yet,
    * the mistake will be detected and corrected.
    *
    * @param {String} name
    * @param {Boolean} force
    * @return {ClassList}
    * @api public
    */

  ClassList.prototype.toggle = function (name, force) {
    if ('undefined' !== typeof force) {
      if (force !== this.list.toggle(name, force)) {
        this.list.toggle(name); // toggle again to correct
      }
    } else {
      this.list.toggle(name);
    }
    return this;
  };

  /**
    * Return an array of classes.
    *
    * @return {Array}
    * @api public
    */

  ClassList.prototype.array = function () {
    return Array.from(this.list);
  };

  /**
    * Check if class `name` is present.
    *
    * @param {String} name
    * @return {ClassList}
    * @api public
    */

  ClassList.prototype.has = ClassList.prototype.contains = function (name) {
    return this.list.contains(name);
  };

  /**
   * Clear utility
   */

  /**
   * Removes all children from the given element
   *
   * @param  {SVGElement} element
   * @return {Element} the element (for chaining)
   */
  function clear(element) {
    var child;
    while (child = element.firstChild) {
      element.removeChild(child);
    }
    return element;
  }
  function clone$1(element) {
    return element.cloneNode(true);
  }
  var ns = {
    svg: 'http://www.w3.org/2000/svg'
  };

  /**
   * DOM parsing utility
   */

  var SVG_START = '<svg xmlns="' + ns.svg + '"';
  function parse(svg) {
    var unwrap = false;

    // ensure we import a valid svg document
    if (svg.substring(0, 4) === '<svg') {
      if (svg.indexOf(ns.svg) === -1) {
        svg = SVG_START + svg.substring(4);
      }
    } else {
      // namespace svg
      svg = SVG_START + '>' + svg + '</svg>';
      unwrap = true;
    }
    var parsed = parseDocument(svg);
    if (!unwrap) {
      return parsed;
    }
    var fragment = document.createDocumentFragment();
    var parent = parsed.firstChild;
    while (parent.firstChild) {
      fragment.appendChild(parent.firstChild);
    }
    return fragment;
  }
  function parseDocument(svg) {
    var parser;

    // parse
    parser = new DOMParser();
    parser.async = false;
    return parser.parseFromString(svg, 'text/xml');
  }

  /**
   * Create utility for SVG elements
   */

  /**
   * Create a specific type from name or SVG markup.
   *
   * @param {String} name the name or markup of the element
   * @param {Object} [attrs] attributes to set on the element
   *
   * @returns {SVGElement}
   */
  function create$2(name, attrs) {
    var element;
    name = name.trim();
    if (name.charAt(0) === '<') {
      element = parse(name).firstChild;
      element = document.importNode(element, true);
    } else {
      element = document.createElementNS(ns.svg, name);
    }
    if (attrs) {
      attr(element, attrs);
    }
    return element;
  }

  /**
   * Geometry helpers
   */

  // fake node used to instantiate svg geometry elements
  var node = null;
  function getNode() {
    if (node === null) {
      node = create$2('svg');
    }
    return node;
  }
  function extend$1(object, props) {
    var i,
      k,
      keys = Object.keys(props);
    for (i = 0; k = keys[i]; i++) {
      object[k] = props[k];
    }
    return object;
  }

  /**
   * Create matrix via args.
   *
   * @example
   *
   * createMatrix({ a: 1, b: 1 });
   * createMatrix();
   * createMatrix(1, 2, 0, 0, 30, 20);
   *
   * @return {SVGMatrix}
   */
  function createMatrix(a, b, c, d, e, f) {
    var matrix = getNode().createSVGMatrix();
    switch (arguments.length) {
      case 0:
        return matrix;
      case 1:
        return extend$1(matrix, a);
      case 6:
        return extend$1(matrix, {
          a: a,
          b: b,
          c: c,
          d: d,
          e: e,
          f: f
        });
    }
  }
  function createTransform(matrix) {
    {
      return getNode().createSVGTransform();
    }
  }

  /**
   * Serialization util
   */

  var TEXT_ENTITIES = /([&<>]{1})/g;
  var ATTR_ENTITIES = /([&<>\n\r"]{1})/g;
  var ENTITY_REPLACEMENT = {
    '&': '&amp;',
    '<': '&lt;',
    '>': '&gt;',
    '"': '\''
  };
  function escape(str, pattern) {
    function replaceFn(match, entity) {
      return ENTITY_REPLACEMENT[entity] || entity;
    }
    return str.replace(pattern, replaceFn);
  }
  function serialize(node, output) {
    var i, len, attrMap, attrNode, childNodes;
    switch (node.nodeType) {
      // TEXT
      case 3:
        // replace special XML characters
        output.push(escape(node.textContent, TEXT_ENTITIES));
        break;

      // ELEMENT
      case 1:
        output.push('<', node.tagName);
        if (node.hasAttributes()) {
          attrMap = node.attributes;
          for (i = 0, len = attrMap.length; i < len; ++i) {
            attrNode = attrMap.item(i);
            output.push(' ', attrNode.name, '="', escape(attrNode.value, ATTR_ENTITIES), '"');
          }
        }
        if (node.hasChildNodes()) {
          output.push('>');
          childNodes = node.childNodes;
          for (i = 0, len = childNodes.length; i < len; ++i) {
            serialize(childNodes.item(i), output);
          }
          output.push('</', node.tagName, '>');
        } else {
          output.push('/>');
        }
        break;

      // COMMENT
      case 8:
        output.push('<!--', escape(node.nodeValue, TEXT_ENTITIES), '-->');
        break;

      // CDATA
      case 4:
        output.push('<![CDATA[', node.nodeValue, ']]>');
        break;
      default:
        throw new Error('unable to handle node ' + node.nodeType);
    }
    return output;
  }
  function get$1(element) {
    var child = element.firstChild,
      output = [];
    while (child) {
      serialize(child, output);
      child = child.nextSibling;
    }
    return output.join('');
  }
  function innerSVG(element, svg) {
    {
      return get$1(element);
    }
  }
  function remove$3(element) {
    var parent = element.parentNode;
    if (parent) {
      parent.removeChild(element);
    }
    return element;
  }

  /**
   * transform accessor utility
   */

  function wrapMatrix(transformList, transform) {
    if (transform instanceof SVGMatrix) {
      return transformList.createSVGTransformFromMatrix(transform);
    }
    return transform;
  }
  function setTransforms(transformList, transforms) {
    var i, t;
    transformList.clear();
    for (i = 0; t = transforms[i]; i++) {
      transformList.appendItem(wrapMatrix(transformList, t));
    }
  }

  /**
   * Get or set the transforms on the given node.
   *
   * @param {SVGElement} node
   * @param  {SVGTransform|SVGMatrix|Array<SVGTransform|SVGMatrix>} [transforms]
   *
   * @return {SVGTransform} the consolidated transform
   */
  function transform$1(node, transforms) {
    var transformList = node.transform.baseVal;
    if (transforms) {
      if (!Array.isArray(transforms)) {
        transforms = [transforms];
      }
      setTransforms(transformList, transforms);
    }
    return transformList.consolidate();
  }

  /**
   * @typedef {(string|number)[]} Component
   *
   * @typedef {import('../util/Types').Point} Point
   */

  /**
   * @param {Component[] | Component[][]} elements
   *
   * @return {string}
   */
  function componentsToPath(elements) {
    return elements.flat().join(',').replace(/,?([A-z]),?/g, '$1');
  }

  /**
   * @param {Point} point
   *
   * @return {Component[]}
   */
  function move(point) {
    return ['M', point.x, point.y];
  }

  /**
   * @param {Point} point
   *
   * @return {Component[]}
   */
  function lineTo(point) {
    return ['L', point.x, point.y];
  }

  /**
   * @param {Point} p1
   * @param {Point} p2
   * @param {Point} p3
   *
   * @return {Component[]}
   */
  function curveTo(p1, p2, p3) {
    return ['C', p1.x, p1.y, p2.x, p2.y, p3.x, p3.y];
  }

  /**
   * @param {Point[]} waypoints
   * @param {number} [cornerRadius]
   * @return {Component[][]}
   */
  function drawPath(waypoints, cornerRadius) {
    const pointCount = waypoints.length;
    const path = [move(waypoints[0])];
    for (let i = 1; i < pointCount; i++) {
      const pointBefore = waypoints[i - 1];
      const point = waypoints[i];
      const pointAfter = waypoints[i + 1];
      if (!pointAfter || !cornerRadius) {
        path.push(lineTo(point));
        continue;
      }
      const effectiveRadius = Math.min(cornerRadius, vectorLength$1(point.x - pointBefore.x, point.y - pointBefore.y), vectorLength$1(pointAfter.x - point.x, pointAfter.y - point.y));
      if (!effectiveRadius) {
        path.push(lineTo(point));
        continue;
      }
      const beforePoint = getPointAtLength(point, pointBefore, effectiveRadius);
      const beforePoint2 = getPointAtLength(point, pointBefore, effectiveRadius * .5);
      const afterPoint = getPointAtLength(point, pointAfter, effectiveRadius);
      const afterPoint2 = getPointAtLength(point, pointAfter, effectiveRadius * .5);
      path.push(lineTo(beforePoint));
      path.push(curveTo(beforePoint2, afterPoint2, afterPoint));
    }
    return path;
  }
  function getPointAtLength(start, end, length) {
    const deltaX = end.x - start.x;
    const deltaY = end.y - start.y;
    const totalLength = vectorLength$1(deltaX, deltaY);
    const percent = length / totalLength;
    return {
      x: start.x + deltaX * percent,
      y: start.y + deltaY * percent
    };
  }
  function vectorLength$1(x, y) {
    return Math.sqrt(Math.pow(x, 2) + Math.pow(y, 2));
  }

  /**
   * @param {Point[]} points
   * @param {number|Object} [attrs]
   * @param {number} [radius]
   *
   * @return {SVGElement}
   */
  function createLine(points, attrs, radius) {
    if (isNumber$3(attrs)) {
      radius = attrs;
      attrs = null;
    }
    if (!attrs) {
      attrs = {};
    }
    const line = create$2('path', attrs);
    if (isNumber$3(radius)) {
      line.dataset.cornerRadius = String(radius);
    }
    return updateLine(line, points);
  }

  /**
   * @param {SVGElement} gfx
   * @param {Point[]} points
   *
   * @return {SVGElement}
   */
  function updateLine(gfx, points) {
    const cornerRadius = parseInt(gfx.dataset.cornerRadius, 10) || 0;
    attr(gfx, {
      d: componentsToPath(drawPath(points, cornerRadius))
    });
    return gfx;
  }

  /**
   * @typedef {import('../model/Types').Connection} Connection
   * @typedef {import('../model/Types').Element} Element
   * @typedef {import('../model/Types').Shape} Shape
   *
   * @typedef {import('../util/Types').Rect} Rect
   *
   * @typedef { {
   *   allShapes: Record<string, Shape>,
   *   allConnections: Record<string, Connection>,
   *   topLevel: Record<string, Element>,
   *   enclosedConnections: Record<string, Connection>,
   *   enclosedElements: Record<string, Element>
   * } } Closure
   */

  /**
   * Get parent elements.
   *
   * @param {Element[]} elements
   *
   * @return {Element[]}
   */
  function getParents(elements) {
    // find elements that are not children of any other elements
    return filter$2(elements, function (element) {
      return !find$2(elements, function (e) {
        return e !== element && getParent$2(element, e);
      });
    });
  }
  function getParent$2(element, parent) {
    if (!parent) {
      return;
    }
    if (element === parent) {
      return parent;
    }
    if (!element.parent) {
      return;
    }
    return getParent$2(element.parent, parent);
  }

  /**
   * Adds an element to a collection and returns true if the
   * element was added.
   *
   * @param {Object[]} elements
   * @param {Object} element
   * @param {boolean} [unique]
   */
  function add$3(elements, element, unique) {
    var canAdd = !unique;
    {
      elements.push(element);
    }
    return canAdd;
  }

  /**
   * Iterate over each element in a collection, calling the iterator function `fn`
   * with (element, index, recursionDepth).
   *
   * Recurse into all elements that are returned by `fn`.
   *
   * @param {Element|Element[]} elements
   * @param {(element: Element, index: number, depth: number) => Element[] | boolean | undefined} fn
   * @param {number} [depth] maximum recursion depth
   */
  function eachElement(elements, fn, depth) {
    depth = depth || 0;
    if (!isArray$6(elements)) {
      elements = [elements];
    }
    forEach$3(elements, function (s, i) {
      var filter = fn(s, i, depth);
      if (isArray$6(filter) && filter.length) {
        eachElement(filter, fn, depth + 1);
      }
    });
  }

  /**
   * Collects self + child elements up to a given depth from a list of elements.
   *
   * @param {Element|Element[]} elements the elements to select the children from
   * @param {boolean} unique whether to return a unique result set (no duplicates)
   * @param {number} maxDepth the depth to search through or -1 for infinite
   *
   * @return {Element[]} found elements
   */
  function selfAndChildren(elements, unique, maxDepth) {
    var result = [],
      processedChildren = [];
    eachElement(elements, function (element, i, depth) {
      add$3(result, element, unique);
      var children = element.children;

      // max traversal depth not reached yet
      {
        // children exist && children not yet processed
        if (children && add$3(processedChildren, children, unique)) {
          return children;
        }
      }
    });
    return result;
  }

  /**
   * Return self + ALL children for a number of elements
   *
   * @param {Element[]} elements to query
   * @param {boolean} [allowDuplicates] to allow duplicates in the result set
   *
   * @return {Element[]} the collected elements
   */
  function selfAndAllChildren(elements, allowDuplicates) {
    return selfAndChildren(elements, !allowDuplicates);
  }

  /**
   * Gets the the closure for all selected elements,
   * their enclosed children and connections.
   *
   * @param {Element[]} elements
   * @param {boolean} [isTopLevel=true]
   * @param {Closure} [closure]
   *
   * @return {Closure} newClosure
   */
  function getClosure(elements, isTopLevel, closure) {
    if (isUndefined$5(isTopLevel)) {
      isTopLevel = true;
    }
    if (isObject$2(isTopLevel)) {
      closure = isTopLevel;
      isTopLevel = true;
    }
    closure = closure || {};
    var allShapes = copyObject(closure.allShapes),
      allConnections = copyObject(closure.allConnections),
      enclosedElements = copyObject(closure.enclosedElements),
      enclosedConnections = copyObject(closure.enclosedConnections);
    var topLevel = copyObject(closure.topLevel, isTopLevel && groupBy(elements, function (e) {
      return e.id;
    }));
    function handleConnection(c) {
      if (topLevel[c.source.id] && topLevel[c.target.id]) {
        topLevel[c.id] = [c];
      }

      // not enclosed as a child, but maybe logically
      // (connecting two moved elements?)
      if (allShapes[c.source.id] && allShapes[c.target.id]) {
        enclosedConnections[c.id] = enclosedElements[c.id] = c;
      }
      allConnections[c.id] = c;
    }
    function handleElement(element) {
      enclosedElements[element.id] = element;
      if (element.waypoints) {
        // remember connection
        enclosedConnections[element.id] = allConnections[element.id] = element;
      } else {
        // remember shape
        allShapes[element.id] = element;

        // remember all connections
        forEach$3(element.incoming, handleConnection);
        forEach$3(element.outgoing, handleConnection);

        // recurse into children
        return element.children;
      }
    }
    eachElement(elements, handleElement);
    return {
      allShapes: allShapes,
      allConnections: allConnections,
      topLevel: topLevel,
      enclosedConnections: enclosedConnections,
      enclosedElements: enclosedElements
    };
  }

  /**
   * Returns the surrounding bbox for all elements in
   * the array or the element primitive.
   *
   * @param {Element|Element[]} elements
   * @param {boolean} [stopRecursion=false]
   *
   * @return {Rect}
   */
  function getBBox(elements, stopRecursion) {
    stopRecursion = !!stopRecursion;
    if (!isArray$6(elements)) {
      elements = [elements];
    }
    var minX, minY, maxX, maxY;
    forEach$3(elements, function (element) {
      // If element is a connection the bbox must be computed first
      var bbox = element;
      if (element.waypoints && !stopRecursion) {
        bbox = getBBox(element.waypoints, true);
      }
      var x = bbox.x,
        y = bbox.y,
        height = bbox.height || 0,
        width = bbox.width || 0;
      if (x < minX || minX === undefined) {
        minX = x;
      }
      if (y < minY || minY === undefined) {
        minY = y;
      }
      if (x + width > maxX || maxX === undefined) {
        maxX = x + width;
      }
      if (y + height > maxY || maxY === undefined) {
        maxY = y + height;
      }
    });
    return {
      x: minX,
      y: minY,
      height: maxY - minY,
      width: maxX - minX
    };
  }

  /**
   * Returns all elements that are enclosed from the bounding box.
   *
   *   * If bbox.(width|height) is not specified the method returns
   *     all elements with element.x/y > bbox.x/y
   *   * If only bbox.x or bbox.y is specified, method return all elements with
   *     e.x > bbox.x or e.y > bbox.y
   *
   * @param {Element[]} elements List of Elements to search through
   * @param {Rect} bbox the enclosing bbox.
   *
   * @return {Element[]} enclosed elements
   */
  function getEnclosedElements(elements, bbox) {
    var filteredElements = {};
    forEach$3(elements, function (element) {
      var e = element;
      if (e.waypoints) {
        e = getBBox(e);
      }
      if (!isNumber$3(bbox.y) && e.x > bbox.x) {
        filteredElements[element.id] = element;
      }
      if (!isNumber$3(bbox.x) && e.y > bbox.y) {
        filteredElements[element.id] = element;
      }
      if (e.x > bbox.x && e.y > bbox.y) {
        if (isNumber$3(bbox.width) && isNumber$3(bbox.height) && e.width + e.x < bbox.width + bbox.x && e.height + e.y < bbox.height + bbox.y) {
          filteredElements[element.id] = element;
        } else if (!isNumber$3(bbox.width) || !isNumber$3(bbox.height)) {
          filteredElements[element.id] = element;
        }
      }
    });
    return filteredElements;
  }

  /**
   * Get the element's type
   *
   * @param {Element} element
   *
   * @return {'connection' | 'shape' | 'root'}
   */
  function getType$1(element) {
    if ('waypoints' in element) {
      return 'connection';
    }
    if ('x' in element) {
      return 'shape';
    }
    return 'root';
  }

  /**
   * @param {Element} element
   *
   * @return {boolean}
   */
  function isFrameElement(element) {
    return !!(element && element.isFrame);
  }

  // helpers ///////////////////////////////

  function copyObject(src1, src2) {
    return assign$4({}, src1 || {}, src2 || {});
  }

  /**
   * @typedef {import('../core/EventBus').default} EventBus
   * @typedef {import('./Styles').default} Styles
   */

  // apply default renderer with lowest possible priority
  // so that it only kicks in if noone else could render
  var DEFAULT_RENDER_PRIORITY = 1;

  /**
   * The default renderer used for shapes and connections.
   *
   * @param {EventBus} eventBus
   * @param {Styles} styles
   */
  function DefaultRenderer(eventBus, styles) {
    BaseRenderer.call(this, eventBus, DEFAULT_RENDER_PRIORITY);
    this.CONNECTION_STYLE = styles.style(['no-fill'], {
      strokeWidth: 5,
      stroke: 'fuchsia'
    });
    this.SHAPE_STYLE = styles.style({
      fill: 'white',
      stroke: 'fuchsia',
      strokeWidth: 2
    });
    this.FRAME_STYLE = styles.style(['no-fill'], {
      stroke: 'fuchsia',
      strokeDasharray: 4,
      strokeWidth: 2
    });
  }
  e$3(DefaultRenderer, BaseRenderer);

  /**
   * @private
   */
  DefaultRenderer.prototype.canRender = function () {
    return true;
  };

  /**
   * @private
   */
  DefaultRenderer.prototype.drawShape = function drawShape(visuals, element, attrs) {
    var rect = create$2('rect');
    attr(rect, {
      x: 0,
      y: 0,
      width: element.width || 0,
      height: element.height || 0
    });
    if (isFrameElement(element)) {
      attr(rect, assign$4({}, this.FRAME_STYLE, attrs || {}));
    } else {
      attr(rect, assign$4({}, this.SHAPE_STYLE, attrs || {}));
    }
    append(visuals, rect);
    return rect;
  };

  /**
   * @private
   */
  DefaultRenderer.prototype.drawConnection = function drawConnection(visuals, connection, attrs) {
    var line = createLine(connection.waypoints, assign$4({}, this.CONNECTION_STYLE, attrs || {}));
    append(visuals, line);
    return line;
  };

  /**
   * @private
   */
  DefaultRenderer.prototype.getShapePath = function getShapePath(shape) {
    var x = shape.x,
      y = shape.y,
      width = shape.width,
      height = shape.height;
    var shapePath = [['M', x, y], ['l', width, 0], ['l', 0, height], ['l', -width, 0], ['z']];
    return componentsToPath(shapePath);
  };

  /**
   * @private
   */
  DefaultRenderer.prototype.getConnectionPath = function getConnectionPath(connection) {
    var waypoints = connection.waypoints;
    var idx,
      point,
      connectionPath = [];
    for (idx = 0; point = waypoints[idx]; idx++) {
      // take invisible docking into account
      // when creating the path
      point = point.original || point;
      connectionPath.push([idx === 0 ? 'M' : 'L', point.x, point.y]);
    }
    return componentsToPath(connectionPath);
  };
  DefaultRenderer.$inject = ['eventBus', 'styles'];

  /**
   * A component that manages shape styles
   */
  function Styles() {
    var defaultTraits = {
      'no-fill': {
        fill: 'none'
      },
      'no-border': {
        strokeOpacity: 0.0
      },
      'no-events': {
        pointerEvents: 'none'
      }
    };
    var self = this;

    /**
     * Builds a style definition from a className, a list of traits and an object
     * of additional attributes.
     *
     * @param {string} className
     * @param {string[]} [traits]
     * @param {Object} [additionalAttrs]
     *
     * @return {Object} the style definition
     */
    this.cls = function (className, traits, additionalAttrs) {
      var attrs = this.style(traits, additionalAttrs);
      return assign$4(attrs, {
        'class': className
      });
    };

    /**
     * Builds a style definition from a list of traits and an object of additional
     * attributes.
     *
     * @param {string[]} [traits]
     * @param {Object} additionalAttrs
     *
     * @return {Object} the style definition
     */
    this.style = function (traits, additionalAttrs) {
      if (!isArray$6(traits) && !additionalAttrs) {
        additionalAttrs = traits;
        traits = [];
      }
      var attrs = reduce(traits, function (attrs, t) {
        return assign$4(attrs, defaultTraits[t] || {});
      }, {});
      return additionalAttrs ? assign$4(attrs, additionalAttrs) : attrs;
    };

    /**
     * Computes a style definition from a list of traits and an object of
     * additional attributes, with custom style definition object.
     *
     * @param {Object} custom
     * @param {string[]} [traits]
     * @param {Object} defaultStyles
     *
     * @return {Object} the style definition
     */
    this.computeStyle = function (custom, traits, defaultStyles) {
      if (!isArray$6(traits)) {
        defaultStyles = traits;
        traits = [];
      }
      return self.style(traits || [], assign$4({}, defaultStyles, custom || {}));
    };
  }

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var DrawModule$1 = {
    __init__: ['defaultRenderer'],
    defaultRenderer: ['type', DefaultRenderer],
    styles: ['type', Styles]
  };

  /**
   * Failsafe remove an element from a collection
   *
   * @param {Array<Object>} [collection]
   * @param {Object} [element]
   *
   * @return {number} the previous index of the element
   */
  function remove$2(collection, element) {
    if (!collection || !element) {
      return -1;
    }
    var idx = collection.indexOf(element);
    if (idx !== -1) {
      collection.splice(idx, 1);
    }
    return idx;
  }

  /**
   * Fail save add an element to the given connection, ensuring
   * it does not yet exist.
   *
   * @param {Array<Object>} collection
   * @param {Object} element
   * @param {number} [idx]
   */
  function add$2(collection, element, idx) {
    if (!collection || !element) {
      return;
    }
    if (typeof idx !== 'number') {
      idx = -1;
    }
    var currentIdx = collection.indexOf(element);
    if (currentIdx !== -1) {
      if (currentIdx === idx) {
        // nothing to do, position has not changed
        return;
      } else {
        if (idx !== -1) {
          // remove from current position
          collection.splice(currentIdx, 1);
        } else {
          // already exists in collection
          return;
        }
      }
    }
    if (idx !== -1) {
      // insert at specified position
      collection.splice(idx, 0, element);
    } else {
      // push to end
      collection.push(element);
    }
  }

  /**
   * Fail save get the index of an element in a collection.
   *
   * @param {Array<Object>} collection
   * @param {Object} element
   *
   * @return {number} the index or -1 if collection or element do
   *                  not exist or the element is not contained.
   */
  function indexOf(collection, element) {
    if (!collection || !element) {
      return -1;
    }
    return collection.indexOf(element);
  }

  /**
   * @typedef {import('../util/Types').Axis} Axis
   * @typedef {import('../util/Types').Point} Point
   * @typedef {import('../util/Types').Rect} Rect
   */

  /**
   * Computes the distance between two points.
   *
   * @param {Point} a
   * @param {Point} b
   *
   * @return {number} The distance between the two points.
   */
  function pointDistance(a, b) {
    if (!a || !b) {
      return -1;
    }
    return Math.sqrt(Math.pow(a.x - b.x, 2) + Math.pow(a.y - b.y, 2));
  }

  /**
   * Returns true if the point r is on the line between p and q.
   *
   * @param {Point} p
   * @param {Point} q
   * @param {Point} r
   * @param {number} [accuracy=5] The accuracy with which to check (lower is better).
   *
   * @return {boolean}
   */
  function pointsOnLine(p, q, r, accuracy) {
    if (typeof accuracy === 'undefined') {
      accuracy = 5;
    }
    if (!p || !q || !r) {
      return false;
    }
    var val = (q.x - p.x) * (r.y - p.y) - (q.y - p.y) * (r.x - p.x),
      dist = pointDistance(p, q);

    // @see http://stackoverflow.com/a/907491/412190
    return Math.abs(val / dist) <= accuracy;
  }
  var ALIGNED_THRESHOLD = 2;

  /**
   * Check whether two points are horizontally or vertically aligned.
   *
   * @param {Point[]|Point} a
   * @param {Point} [b]
   *
   * @return {string|boolean} If and how the two points are aligned ('h', 'v' or `false`).
   */
  function pointsAligned(a, b) {
    var points = Array.from(arguments).flat();
    const axisMap = {
      'x': 'v',
      'y': 'h'
    };
    for (const [axis, orientation] of Object.entries(axisMap)) {
      if (pointsAlignedOnAxis(axis, points)) {
        return orientation;
      }
    }
    return false;
  }

  /**
   * @param {Axis} axis
   * @param {Point[]} points
   *
   * @return {boolean}
   */
  function pointsAlignedOnAxis(axis, points) {
    const referencePoint = points[0];
    return every(points, function (point) {
      return Math.abs(referencePoint[axis] - point[axis]) <= ALIGNED_THRESHOLD;
    });
  }

  /**
   * Returns a point in the middle of points p and q
   *
   * @param {Point} p
   * @param {Point} q
   *
   * @return {Point} The mid point between the two points.
   */
  function getMidPoint(p, q) {
    return {
      x: Math.round(p.x + (q.x - p.x) / 2.0),
      y: Math.round(p.y + (q.y - p.y) / 2.0)
    };
  }

  /**
   * This file contains source code adapted from Snap.svg (licensed Apache-2.0).
   *
   * @see https://github.com/adobe-webplatform/Snap.svg/blob/master/src/path.js
   */

  /* eslint no-fallthrough: "off" */

  var p2s = /,?([a-z]),?/gi,
    toFloat = parseFloat,
    math = Math,
    PI = math.PI,
    mmin = math.min,
    mmax = math.max,
    pow = math.pow,
    abs$2 = math.abs,
    pathCommand = /([a-z])[\s,]*((-?\d*\.?\d*(?:e[-+]?\d+)?[\s]*,?[\s]*)+)/ig,
    pathValues = /(-?\d*\.?\d*(?:e[-+]?\d+)?)[\s]*,?[\s]*/ig;
  var isArray$1 = Array.isArray || function (o) {
    return o instanceof Array;
  };
  function hasProperty(obj, property) {
    return Object.prototype.hasOwnProperty.call(obj, property);
  }
  function clone(obj) {
    if (typeof obj == 'function' || Object(obj) !== obj) {
      return obj;
    }
    var res = new obj.constructor();
    for (var key in obj) {
      if (hasProperty(obj, key)) {
        res[key] = clone(obj[key]);
      }
    }
    return res;
  }
  function repush(array, item) {
    for (var i = 0, ii = array.length; i < ii; i++) if (array[i] === item) {
      return array.push(array.splice(i, 1)[0]);
    }
  }
  function cacher(f) {
    function newf() {
      var arg = Array.prototype.slice.call(arguments, 0),
        args = arg.join('\u2400'),
        cache = newf.cache = newf.cache || {},
        count = newf.count = newf.count || [];
      if (hasProperty(cache, args)) {
        repush(count, args);
        return cache[args];
      }
      count.length >= 1e3 && delete cache[count.shift()];
      count.push(args);
      cache[args] = f(...arguments);
      return cache[args];
    }
    return newf;
  }
  function parsePathString(pathString) {
    if (!pathString) {
      return null;
    }
    var pth = paths(pathString);
    if (pth.arr) {
      return clone(pth.arr);
    }
    var paramCounts = {
        a: 7,
        c: 6,
        h: 1,
        l: 2,
        m: 2,
        q: 4,
        s: 4,
        t: 2,
        v: 1,
        z: 0
      },
      data = [];
    if (isArray$1(pathString) && isArray$1(pathString[0])) {
      // rough assumption
      data = clone(pathString);
    }
    if (!data.length) {
      String(pathString).replace(pathCommand, function (a, b, c) {
        var params = [],
          name = b.toLowerCase();
        c.replace(pathValues, function (a, b) {
          b && params.push(+b);
        });
        if (name == 'm' && params.length > 2) {
          data.push([b, ...params.splice(0, 2)]);
          name = 'l';
          b = b == 'm' ? 'l' : 'L';
        }
        while (params.length >= paramCounts[name]) {
          data.push([b, ...params.splice(0, paramCounts[name])]);
          if (!paramCounts[name]) {
            break;
          }
        }
      });
    }
    data.toString = paths.toString;
    pth.arr = clone(data);
    return data;
  }
  function paths(ps) {
    var p = paths.ps = paths.ps || {};
    if (p[ps]) {
      p[ps].sleep = 100;
    } else {
      p[ps] = {
        sleep: 100
      };
    }
    setTimeout(function () {
      for (var key in p) {
        if (hasProperty(p, key) && key != ps) {
          p[key].sleep--;
          !p[key].sleep && delete p[key];
        }
      }
    });
    return p[ps];
  }
  function rectBBox(x, y, width, height) {
    if (arguments.length === 1) {
      y = x.y;
      width = x.width;
      height = x.height;
      x = x.x;
    }
    return {
      x: x,
      y: y,
      width: width,
      height: height,
      x2: x + width,
      y2: y + height
    };
  }
  function pathToString() {
    return this.join(',').replace(p2s, '$1');
  }
  function pathClone(pathArray) {
    var res = clone(pathArray);
    res.toString = pathToString;
    return res;
  }
  function findDotsAtSegment(p1x, p1y, c1x, c1y, c2x, c2y, p2x, p2y, t) {
    var t1 = 1 - t,
      t13 = pow(t1, 3),
      t12 = pow(t1, 2),
      t2 = t * t,
      t3 = t2 * t,
      x = t13 * p1x + t12 * 3 * t * c1x + t1 * 3 * t * t * c2x + t3 * p2x,
      y = t13 * p1y + t12 * 3 * t * c1y + t1 * 3 * t * t * c2y + t3 * p2y;
    return {
      x: fixError(x),
      y: fixError(y)
    };
  }
  function bezierBBox(points) {
    var bbox = curveBBox(...points);
    return rectBBox(bbox.x0, bbox.y0, bbox.x1 - bbox.x0, bbox.y1 - bbox.y0);
  }
  function isPointInsideBBox(bbox, x, y) {
    return x >= bbox.x && x <= bbox.x + bbox.width && y >= bbox.y && y <= bbox.y + bbox.height;
  }
  function isBBoxIntersect(bbox1, bbox2) {
    bbox1 = rectBBox(bbox1);
    bbox2 = rectBBox(bbox2);
    return isPointInsideBBox(bbox2, bbox1.x, bbox1.y) || isPointInsideBBox(bbox2, bbox1.x2, bbox1.y) || isPointInsideBBox(bbox2, bbox1.x, bbox1.y2) || isPointInsideBBox(bbox2, bbox1.x2, bbox1.y2) || isPointInsideBBox(bbox1, bbox2.x, bbox2.y) || isPointInsideBBox(bbox1, bbox2.x2, bbox2.y) || isPointInsideBBox(bbox1, bbox2.x, bbox2.y2) || isPointInsideBBox(bbox1, bbox2.x2, bbox2.y2) || (bbox1.x < bbox2.x2 && bbox1.x > bbox2.x || bbox2.x < bbox1.x2 && bbox2.x > bbox1.x) && (bbox1.y < bbox2.y2 && bbox1.y > bbox2.y || bbox2.y < bbox1.y2 && bbox2.y > bbox1.y);
  }
  function base3(t, p1, p2, p3, p4) {
    var t1 = -3 * p1 + 9 * p2 - 9 * p3 + 3 * p4,
      t2 = t * t1 + 6 * p1 - 12 * p2 + 6 * p3;
    return t * t2 - 3 * p1 + 3 * p2;
  }
  function bezlen(x1, y1, x2, y2, x3, y3, x4, y4, z) {
    if (z == null) {
      z = 1;
    }
    z = z > 1 ? 1 : z < 0 ? 0 : z;
    var z2 = z / 2,
      n = 12,
      Tvalues = [-.1252, .1252, -.3678, .3678, -.5873, .5873, -.7699, .7699, -.9041, .9041, -.9816, .9816],
      Cvalues = [0.2491, 0.2491, 0.2335, 0.2335, 0.2032, 0.2032, 0.1601, 0.1601, 0.1069, 0.1069, 0.0472, 0.0472],
      sum = 0;
    for (var i = 0; i < n; i++) {
      var ct = z2 * Tvalues[i] + z2,
        xbase = base3(ct, x1, x2, x3, x4),
        ybase = base3(ct, y1, y2, y3, y4),
        comb = xbase * xbase + ybase * ybase;
      sum += Cvalues[i] * math.sqrt(comb);
    }
    return z2 * sum;
  }
  function intersectLines(x1, y1, x2, y2, x3, y3, x4, y4) {
    if (mmax(x1, x2) < mmin(x3, x4) || mmin(x1, x2) > mmax(x3, x4) || mmax(y1, y2) < mmin(y3, y4) || mmin(y1, y2) > mmax(y3, y4)) {
      return;
    }
    var nx = (x1 * y2 - y1 * x2) * (x3 - x4) - (x1 - x2) * (x3 * y4 - y3 * x4),
      ny = (x1 * y2 - y1 * x2) * (y3 - y4) - (y1 - y2) * (x3 * y4 - y3 * x4),
      denominator = (x1 - x2) * (y3 - y4) - (y1 - y2) * (x3 - x4);
    if (!denominator) {
      return;
    }
    var px = fixError(nx / denominator),
      py = fixError(ny / denominator),
      px2 = +px.toFixed(2),
      py2 = +py.toFixed(2);
    if (px2 < +mmin(x1, x2).toFixed(2) || px2 > +mmax(x1, x2).toFixed(2) || px2 < +mmin(x3, x4).toFixed(2) || px2 > +mmax(x3, x4).toFixed(2) || py2 < +mmin(y1, y2).toFixed(2) || py2 > +mmax(y1, y2).toFixed(2) || py2 < +mmin(y3, y4).toFixed(2) || py2 > +mmax(y3, y4).toFixed(2)) {
      return;
    }
    return {
      x: px,
      y: py
    };
  }
  function fixError(number) {
    return Math.round(number * 100000000000) / 100000000000;
  }
  function findBezierIntersections(bez1, bez2, justCount) {
    var bbox1 = bezierBBox(bez1),
      bbox2 = bezierBBox(bez2);
    if (!isBBoxIntersect(bbox1, bbox2)) {
      return justCount ? 0 : [];
    }

    // As an optimization, lines will have only 1 segment

    var l1 = bezlen(...bez1),
      l2 = bezlen(...bez2),
      n1 = isLine(bez1) ? 1 : ~~(l1 / 5) || 1,
      n2 = isLine(bez2) ? 1 : ~~(l2 / 5) || 1,
      dots1 = [],
      dots2 = [],
      xy = {},
      res = justCount ? 0 : [];
    for (var i = 0; i < n1 + 1; i++) {
      var p = findDotsAtSegment(...bez1, i / n1);
      dots1.push({
        x: p.x,
        y: p.y,
        t: i / n1
      });
    }
    for (i = 0; i < n2 + 1; i++) {
      p = findDotsAtSegment(...bez2, i / n2);
      dots2.push({
        x: p.x,
        y: p.y,
        t: i / n2
      });
    }
    for (i = 0; i < n1; i++) {
      for (var j = 0; j < n2; j++) {
        var di = dots1[i],
          di1 = dots1[i + 1],
          dj = dots2[j],
          dj1 = dots2[j + 1],
          ci = abs$2(di1.x - di.x) < .01 ? 'y' : 'x',
          cj = abs$2(dj1.x - dj.x) < .01 ? 'y' : 'x',
          is = intersectLines(di.x, di.y, di1.x, di1.y, dj.x, dj.y, dj1.x, dj1.y),
          key;
        if (is) {
          key = is.x.toFixed(9) + '#' + is.y.toFixed(9);
          if (xy[key]) {
            continue;
          }
          xy[key] = true;
          var t1 = di.t + abs$2((is[ci] - di[ci]) / (di1[ci] - di[ci])) * (di1.t - di.t),
            t2 = dj.t + abs$2((is[cj] - dj[cj]) / (dj1[cj] - dj[cj])) * (dj1.t - dj.t);
          if (t1 >= 0 && t1 <= 1 && t2 >= 0 && t2 <= 1) {
            if (justCount) {
              res++;
            } else {
              res.push({
                x: is.x,
                y: is.y,
                t1: t1,
                t2: t2
              });
            }
          }
        }
      }
    }
    return res;
  }

  /**
   * Find or counts the intersections between two SVG paths.
   *
   * Returns a number in counting mode and a list of intersections otherwise.
   *
   * A single intersection entry contains the intersection coordinates (x, y)
   * as well as additional information regarding the intersecting segments
   * on each path (segment1, segment2) and the relative location of the
   * intersection on these segments (t1, t2).
   *
   * The path may be an SVG path string or a list of path components
   * such as `[ [ 'M', 0, 10 ], [ 'L', 20, 0 ] ]`.
   *
   * @example
   *
   * var intersections = findPathIntersections(
   *   'M0,0L100,100',
   *   [ [ 'M', 0, 100 ], [ 'L', 100, 0 ] ]
   * );
   *
   * // intersections = [
   * //   { x: 50, y: 50, segment1: 1, segment2: 1, t1: 0.5, t2: 0.5 }
   * // ]
   *
   * @param {String|Array<PathDef>} path1
   * @param {String|Array<PathDef>} path2
   * @param {Boolean} [justCount=false]
   *
   * @return {Array<Intersection>|Number}
   */
  function findPathIntersections(path1, path2, justCount) {
    path1 = pathToCurve(path1);
    path2 = pathToCurve(path2);
    var x1,
      y1,
      x2,
      y2,
      x1m,
      y1m,
      x2m,
      y2m,
      bez1,
      bez2,
      res = justCount ? 0 : [];
    for (var i = 0, ii = path1.length; i < ii; i++) {
      var pi = path1[i];
      if (pi[0] == 'M') {
        x1 = x1m = pi[1];
        y1 = y1m = pi[2];
      } else {
        if (pi[0] == 'C') {
          bez1 = [x1, y1, ...pi.slice(1)];
          x1 = bez1[6];
          y1 = bez1[7];
        } else {
          bez1 = [x1, y1, x1, y1, x1m, y1m, x1m, y1m];
          x1 = x1m;
          y1 = y1m;
        }
        for (var j = 0, jj = path2.length; j < jj; j++) {
          var pj = path2[j];
          if (pj[0] == 'M') {
            x2 = x2m = pj[1];
            y2 = y2m = pj[2];
          } else {
            if (pj[0] == 'C') {
              bez2 = [x2, y2, ...pj.slice(1)];
              x2 = bez2[6];
              y2 = bez2[7];
            } else {
              bez2 = [x2, y2, x2, y2, x2m, y2m, x2m, y2m];
              x2 = x2m;
              y2 = y2m;
            }
            var intr = findBezierIntersections(bez1, bez2, justCount);
            if (justCount) {
              res += intr;
            } else {
              for (var k = 0, kk = intr.length; k < kk; k++) {
                intr[k].segment1 = i;
                intr[k].segment2 = j;
                intr[k].bez1 = bez1;
                intr[k].bez2 = bez2;
              }
              res = res.concat(intr);
            }
          }
        }
      }
    }
    return res;
  }
  function pathToAbsolute(pathArray) {
    var pth = paths(pathArray);
    if (pth.abs) {
      return pathClone(pth.abs);
    }
    if (!isArray$1(pathArray) || !isArray$1(pathArray && pathArray[0])) {
      // rough assumption
      pathArray = parsePathString(pathArray);
    }
    if (!pathArray || !pathArray.length) {
      return [['M', 0, 0]];
    }
    var res = [],
      x = 0,
      y = 0,
      mx = 0,
      my = 0,
      start = 0,
      pa0;
    if (pathArray[0][0] == 'M') {
      x = +pathArray[0][1];
      y = +pathArray[0][2];
      mx = x;
      my = y;
      start++;
      res[0] = ['M', x, y];
    }
    for (var r, pa, i = start, ii = pathArray.length; i < ii; i++) {
      res.push(r = []);
      pa = pathArray[i];
      pa0 = pa[0];
      if (pa0 != pa0.toUpperCase()) {
        r[0] = pa0.toUpperCase();
        switch (r[0]) {
          case 'A':
            r[1] = pa[1];
            r[2] = pa[2];
            r[3] = pa[3];
            r[4] = pa[4];
            r[5] = pa[5];
            r[6] = +pa[6] + x;
            r[7] = +pa[7] + y;
            break;
          case 'V':
            r[1] = +pa[1] + y;
            break;
          case 'H':
            r[1] = +pa[1] + x;
            break;
          case 'M':
            mx = +pa[1] + x;
            my = +pa[2] + y;
          default:
            for (var j = 1, jj = pa.length; j < jj; j++) {
              r[j] = +pa[j] + (j % 2 ? x : y);
            }
        }
      } else {
        for (var k = 0, kk = pa.length; k < kk; k++) {
          r[k] = pa[k];
        }
      }
      pa0 = pa0.toUpperCase();
      switch (r[0]) {
        case 'Z':
          x = +mx;
          y = +my;
          break;
        case 'H':
          x = r[1];
          break;
        case 'V':
          y = r[1];
          break;
        case 'M':
          mx = r[r.length - 2];
          my = r[r.length - 1];
        default:
          x = r[r.length - 2];
          y = r[r.length - 1];
      }
    }
    res.toString = pathToString;
    pth.abs = pathClone(res);
    return res;
  }
  function isLine(bez) {
    return bez[0] === bez[2] && bez[1] === bez[3] && bez[4] === bez[6] && bez[5] === bez[7];
  }
  function lineToCurve(x1, y1, x2, y2) {
    return [x1, y1, x2, y2, x2, y2];
  }
  function qubicToCurve(x1, y1, ax, ay, x2, y2) {
    var _13 = 1 / 3,
      _23 = 2 / 3;
    return [_13 * x1 + _23 * ax, _13 * y1 + _23 * ay, _13 * x2 + _23 * ax, _13 * y2 + _23 * ay, x2, y2];
  }
  function arcToCurve(x1, y1, rx, ry, angle, large_arc_flag, sweep_flag, x2, y2, recursive) {
    // for more information of where this math came from visit:
    // http://www.w3.org/TR/SVG11/implnote.html#ArcImplementationNotes
    var _120 = PI * 120 / 180,
      rad = PI / 180 * (+angle || 0),
      res = [],
      xy,
      rotate = cacher(function (x, y, rad) {
        var X = x * math.cos(rad) - y * math.sin(rad),
          Y = x * math.sin(rad) + y * math.cos(rad);
        return {
          x: X,
          y: Y
        };
      });
    if (!recursive) {
      xy = rotate(x1, y1, -rad);
      x1 = xy.x;
      y1 = xy.y;
      xy = rotate(x2, y2, -rad);
      x2 = xy.x;
      y2 = xy.y;
      var x = (x1 - x2) / 2,
        y = (y1 - y2) / 2;
      var h = x * x / (rx * rx) + y * y / (ry * ry);
      if (h > 1) {
        h = math.sqrt(h);
        rx = h * rx;
        ry = h * ry;
      }
      var rx2 = rx * rx,
        ry2 = ry * ry,
        k = (large_arc_flag == sweep_flag ? -1 : 1) * math.sqrt(abs$2((rx2 * ry2 - rx2 * y * y - ry2 * x * x) / (rx2 * y * y + ry2 * x * x))),
        cx = k * rx * y / ry + (x1 + x2) / 2,
        cy = k * -ry * x / rx + (y1 + y2) / 2,
        f1 = math.asin(((y1 - cy) / ry).toFixed(9)),
        f2 = math.asin(((y2 - cy) / ry).toFixed(9));
      f1 = x1 < cx ? PI - f1 : f1;
      f2 = x2 < cx ? PI - f2 : f2;
      f1 < 0 && (f1 = PI * 2 + f1);
      f2 < 0 && (f2 = PI * 2 + f2);
      if (sweep_flag && f1 > f2) {
        f1 = f1 - PI * 2;
      }
      if (!sweep_flag && f2 > f1) {
        f2 = f2 - PI * 2;
      }
    } else {
      f1 = recursive[0];
      f2 = recursive[1];
      cx = recursive[2];
      cy = recursive[3];
    }
    var df = f2 - f1;
    if (abs$2(df) > _120) {
      var f2old = f2,
        x2old = x2,
        y2old = y2;
      f2 = f1 + _120 * (sweep_flag && f2 > f1 ? 1 : -1);
      x2 = cx + rx * math.cos(f2);
      y2 = cy + ry * math.sin(f2);
      res = arcToCurve(x2, y2, rx, ry, angle, 0, sweep_flag, x2old, y2old, [f2, f2old, cx, cy]);
    }
    df = f2 - f1;
    var c1 = math.cos(f1),
      s1 = math.sin(f1),
      c2 = math.cos(f2),
      s2 = math.sin(f2),
      t = math.tan(df / 4),
      hx = 4 / 3 * rx * t,
      hy = 4 / 3 * ry * t,
      m1 = [x1, y1],
      m2 = [x1 + hx * s1, y1 - hy * c1],
      m3 = [x2 + hx * s2, y2 - hy * c2],
      m4 = [x2, y2];
    m2[0] = 2 * m1[0] - m2[0];
    m2[1] = 2 * m1[1] - m2[1];
    if (recursive) {
      return [m2, m3, m4].concat(res);
    } else {
      res = [m2, m3, m4].concat(res).join().split(',');
      var newres = [];
      for (var i = 0, ii = res.length; i < ii; i++) {
        newres[i] = i % 2 ? rotate(res[i - 1], res[i], rad).y : rotate(res[i], res[i + 1], rad).x;
      }
      return newres;
    }
  }

  // Returns bounding box of cubic bezier curve.
  // Source: http://blog.hackers-cafe.net/2009/06/how-to-calculate-bezier-curves-bounding.html
  // Original version: NISHIO Hirokazu
  // Modifications: https://github.com/timo22345
  function curveBBox(x0, y0, x1, y1, x2, y2, x3, y3) {
    var tvalues = [],
      bounds = [[], []],
      a,
      b,
      c,
      t,
      t1,
      t2,
      b2ac,
      sqrtb2ac;
    for (var i = 0; i < 2; ++i) {
      if (i == 0) {
        b = 6 * x0 - 12 * x1 + 6 * x2;
        a = -3 * x0 + 9 * x1 - 9 * x2 + 3 * x3;
        c = 3 * x1 - 3 * x0;
      } else {
        b = 6 * y0 - 12 * y1 + 6 * y2;
        a = -3 * y0 + 9 * y1 - 9 * y2 + 3 * y3;
        c = 3 * y1 - 3 * y0;
      }
      if (abs$2(a) < 1e-12) {
        if (abs$2(b) < 1e-12) {
          continue;
        }
        t = -c / b;
        if (0 < t && t < 1) {
          tvalues.push(t);
        }
        continue;
      }
      b2ac = b * b - 4 * c * a;
      sqrtb2ac = math.sqrt(b2ac);
      if (b2ac < 0) {
        continue;
      }
      t1 = (-b + sqrtb2ac) / (2 * a);
      if (0 < t1 && t1 < 1) {
        tvalues.push(t1);
      }
      t2 = (-b - sqrtb2ac) / (2 * a);
      if (0 < t2 && t2 < 1) {
        tvalues.push(t2);
      }
    }
    var j = tvalues.length,
      jlen = j,
      mt;
    while (j--) {
      t = tvalues[j];
      mt = 1 - t;
      bounds[0][j] = mt * mt * mt * x0 + 3 * mt * mt * t * x1 + 3 * mt * t * t * x2 + t * t * t * x3;
      bounds[1][j] = mt * mt * mt * y0 + 3 * mt * mt * t * y1 + 3 * mt * t * t * y2 + t * t * t * y3;
    }
    bounds[0][jlen] = x0;
    bounds[1][jlen] = y0;
    bounds[0][jlen + 1] = x3;
    bounds[1][jlen + 1] = y3;
    bounds[0].length = bounds[1].length = jlen + 2;
    return {
      x0: mmin(...bounds[0]),
      y0: mmin(...bounds[1]),
      x1: mmax(...bounds[0]),
      y1: mmax(...bounds[1])
    };
  }
  function pathToCurve(path) {
    var pth = paths(path);

    // return cached curve, if existing
    if (pth.curve) {
      return pathClone(pth.curve);
    }
    var curvedPath = pathToAbsolute(path),
      attrs = {
        x: 0,
        y: 0,
        bx: 0,
        by: 0,
        X: 0,
        Y: 0,
        qx: null,
        qy: null
      },
      processPath = function (path, d, pathCommand) {
        var nx, ny;
        if (!path) {
          return ['C', d.x, d.y, d.x, d.y, d.x, d.y];
        }
        !(path[0] in {
          T: 1,
          Q: 1
        }) && (d.qx = d.qy = null);
        switch (path[0]) {
          case 'M':
            d.X = path[1];
            d.Y = path[2];
            break;
          case 'A':
            path = ['C', ...arcToCurve(d.x, d.y, ...path.slice(1))];
            break;
          case 'S':
            if (pathCommand == 'C' || pathCommand == 'S') {
              // In 'S' case we have to take into account, if the previous command is C/S.
              nx = d.x * 2 - d.bx;

              // And reflect the previous
              ny = d.y * 2 - d.by;

              // command's control point relative to the current point.
            } else {
              // or some else or nothing
              nx = d.x;
              ny = d.y;
            }
            path = ['C', nx, ny, ...path.slice(1)];
            break;
          case 'T':
            if (pathCommand == 'Q' || pathCommand == 'T') {
              // In 'T' case we have to take into account, if the previous command is Q/T.
              d.qx = d.x * 2 - d.qx;

              // And make a reflection similar
              d.qy = d.y * 2 - d.qy;

              // to case 'S'.
            } else {
              // or something else or nothing
              d.qx = d.x;
              d.qy = d.y;
            }
            path = ['C', ...qubicToCurve(d.x, d.y, d.qx, d.qy, path[1], path[2])];
            break;
          case 'Q':
            d.qx = path[1];
            d.qy = path[2];
            path = ['C', ...qubicToCurve(d.x, d.y, path[1], path[2], path[3], path[4])];
            break;
          case 'L':
            path = ['C', ...lineToCurve(d.x, d.y, path[1], path[2])];
            break;
          case 'H':
            path = ['C', ...lineToCurve(d.x, d.y, path[1], d.y)];
            break;
          case 'V':
            path = ['C', ...lineToCurve(d.x, d.y, d.x, path[1])];
            break;
          case 'Z':
            path = ['C', ...lineToCurve(d.x, d.y, d.X, d.Y)];
            break;
        }
        return path;
      },
      fixArc = function (pp, i) {
        if (pp[i].length > 7) {
          pp[i].shift();
          var pi = pp[i];
          while (pi.length) {
            pathCommands[i] = 'A'; // if created multiple C:s, their original seg is saved
            pp.splice(i++, 0, ['C', ...pi.splice(0, 6)]);
          }
          pp.splice(i, 1);
          ii = curvedPath.length;
        }
      },
      pathCommands = [],
      // path commands of original path p
      pfirst = '',
      // temporary holder for original path command
      pathCommand = ''; // holder for previous path command of original path

    for (var i = 0, ii = curvedPath.length; i < ii; i++) {
      curvedPath[i] && (pfirst = curvedPath[i][0]); // save current path command

      if (pfirst != 'C')
        // C is not saved yet, because it may be result of conversion
        {
          pathCommands[i] = pfirst; // Save current path command
          i && (pathCommand = pathCommands[i - 1]); // Get previous path command pathCommand
        }
      curvedPath[i] = processPath(curvedPath[i], attrs, pathCommand); // Previous path command is inputted to processPath

      if (pathCommands[i] != 'A' && pfirst == 'C') pathCommands[i] = 'C'; // A is the only command
      // which may produce multiple C:s
      // so we have to make sure that C is also C in original path

      fixArc(curvedPath, i); // fixArc adds also the right amount of A:s to pathCommands

      var seg = curvedPath[i],
        seglen = seg.length;
      attrs.x = seg[seglen - 2];
      attrs.y = seg[seglen - 1];
      attrs.bx = toFloat(seg[seglen - 4]) || attrs.x;
      attrs.by = toFloat(seg[seglen - 3]) || attrs.y;
    }

    // cache curve
    pth.curve = pathClone(curvedPath);
    return curvedPath;
  }

  /**
   * Checks whether a value is an instance of Connection.
   *
   * @param {any} value
   *
   * @return {boolean}
   */
  function isConnection(value) {
    return isObject$2(value) && has$3(value, 'waypoints');
  }

  /**
   * Checks whether a value is an instance of Label.
   *
   * @param {any} value
   *
   * @return {boolean}
   */
  function isLabel$1(value) {
    return isObject$2(value) && has$3(value, 'labelTarget');
  }

  /**
   * @typedef {import('../core/Types').ElementLike} Element
   * @typedef {import('../core/Types').ConnectionLike} Connection
   *
   * @typedef {import('../util/Types').DirectionTRBL} DirectionTRBL
   * @typedef {import('../util/Types').Intersection} Intersection
   * @typedef {import('../util/Types').Point} Point
   * @typedef {import('../util/Types').Rect} Rect
   * @typedef {import('../util/Types').RectTRBL} RectTRBL
   */

  /**
   * @param {Rect} bounds
   *
   * @returns {Rect}
   */
  function roundBounds(bounds) {
    return {
      x: Math.round(bounds.x),
      y: Math.round(bounds.y),
      width: Math.round(bounds.width),
      height: Math.round(bounds.height)
    };
  }

  /**
   * @param {Point} point
   *
   * @returns {Point}
   */
  function roundPoint(point) {
    return {
      x: Math.round(point.x),
      y: Math.round(point.y)
    };
  }

  /**
   * Convert the given bounds to a { top, left, bottom, right } descriptor.
   *
   * @param {Point|Rect} bounds
   *
   * @return {RectTRBL}
   */
  function asTRBL(bounds) {
    return {
      top: bounds.y,
      right: bounds.x + (bounds.width || 0),
      bottom: bounds.y + (bounds.height || 0),
      left: bounds.x
    };
  }

  /**
   * Convert a { top, left, bottom, right } to an objects bounds.
   *
   * @param {RectTRBL} trbl
   *
   * @return {Rect}
   */
  function asBounds(trbl) {
    return {
      x: trbl.left,
      y: trbl.top,
      width: trbl.right - trbl.left,
      height: trbl.bottom - trbl.top
    };
  }

  /**
   * Get the mid of the given bounds or point.
   *
   * @param {Point|Rect} bounds
   *
   * @return {Point}
   */
  function getBoundsMid(bounds) {
    return roundPoint({
      x: bounds.x + (bounds.width || 0) / 2,
      y: bounds.y + (bounds.height || 0) / 2
    });
  }

  /**
   * Get the mid of the given Connection.
   *
   * @param {Connection} connection
   *
   * @return {Point}
   */
  function getConnectionMid(connection) {
    var waypoints = connection.waypoints;

    // calculate total length and length of each segment
    var parts = waypoints.reduce(function (parts, point, index) {
      var lastPoint = waypoints[index - 1];
      if (lastPoint) {
        var lastPart = parts[parts.length - 1];
        var startLength = lastPart && lastPart.endLength || 0;
        var length = distance(lastPoint, point);
        parts.push({
          start: lastPoint,
          end: point,
          startLength: startLength,
          endLength: startLength + length,
          length: length
        });
      }
      return parts;
    }, []);
    var totalLength = parts.reduce(function (length, part) {
      return length + part.length;
    }, 0);

    // find which segement contains middle point
    var midLength = totalLength / 2;
    var i = 0;
    var midSegment = parts[i];
    while (midSegment.endLength < midLength) {
      midSegment = parts[++i];
    }

    // calculate relative position on mid segment
    var segmentProgress = (midLength - midSegment.startLength) / midSegment.length;
    var midPoint = {
      x: midSegment.start.x + (midSegment.end.x - midSegment.start.x) * segmentProgress,
      y: midSegment.start.y + (midSegment.end.y - midSegment.start.y) * segmentProgress
    };
    return midPoint;
  }

  /**
   * Get the mid of the given Element.
   *
   * @param {Element} element
   *
   * @return {Point}
   */
  function getMid(element) {
    if (isConnection(element)) {
      return getConnectionMid(element);
    }
    return getBoundsMid(element);
  }

  // orientation utils //////////////////////

  /**
   * Get orientation of the given rectangle with respect to
   * the reference rectangle.
   *
   * A padding (positive or negative) may be passed to influence
   * horizontal / vertical orientation and intersection.
   *
   * @param {Rect} rect
   * @param {Rect} reference
   * @param {Point|number} padding
   *
   * @return {DirectionTRBL|Intersection} the orientation; one of top, top-left, left, ..., bottom, right or intersect.
   */
  function getOrientation(rect, reference, padding) {
    padding = padding || 0;

    // make sure we can use an object, too
    // for individual { x, y } padding
    if (!isObject$2(padding)) {
      padding = {
        x: padding,
        y: padding
      };
    }
    var rectOrientation = asTRBL(rect),
      referenceOrientation = asTRBL(reference);
    var top = rectOrientation.bottom + padding.y <= referenceOrientation.top,
      right = rectOrientation.left - padding.x >= referenceOrientation.right,
      bottom = rectOrientation.top - padding.y >= referenceOrientation.bottom,
      left = rectOrientation.right + padding.x <= referenceOrientation.left;
    var vertical = top ? 'top' : bottom ? 'bottom' : null,
      horizontal = left ? 'left' : right ? 'right' : null;
    if (horizontal && vertical) {
      return vertical + '-' + horizontal;
    } else {
      return horizontal || vertical || 'intersect';
    }
  }

  // intersection utils //////////////////////

  /**
   * Get intersection between an element and a line path.
   *
   * @param {string} elementPath
   * @param {string} linePath
   * @param {boolean} cropStart Whether to crop start or end.
   *
   * @return {Point}
   */
  function getElementLineIntersection(elementPath, linePath, cropStart) {
    var intersections = getIntersections(elementPath, linePath);

    // recognize intersections
    // only one -> choose
    // two close together -> choose first
    // two or more distinct -> pull out appropriate one
    // none -> ok (fallback to point itself)
    if (intersections.length === 1) {
      return roundPoint(intersections[0]);
    } else if (intersections.length === 2 && pointDistance(intersections[0], intersections[1]) < 1) {
      return roundPoint(intersections[0]);
    } else if (intersections.length > 1) {
      // sort by intersections based on connection segment +
      // distance from start
      intersections = sortBy(intersections, function (i) {
        var distance = Math.floor(i.t2 * 100) || 1;
        distance = 100 - distance;
        distance = (distance < 10 ? '0' : '') + distance;

        // create a sort string that makes sure we sort
        // line segment ASC + line segment position DESC (for cropStart)
        // line segment ASC + line segment position ASC (for cropEnd)
        return i.segment2 + '#' + distance;
      });
      return roundPoint(intersections[cropStart ? 0 : intersections.length - 1]);
    }
    return null;
  }
  function getIntersections(a, b) {
    return findPathIntersections(a, b);
  }
  function filterRedundantWaypoints(waypoints) {
    // alter copy of waypoints, not original
    waypoints = waypoints.slice();
    var idx = 0,
      point,
      previousPoint,
      nextPoint;
    while (waypoints[idx]) {
      point = waypoints[idx];
      previousPoint = waypoints[idx - 1];
      nextPoint = waypoints[idx + 1];
      if (pointDistance(point, nextPoint) === 0 || pointsOnLine(previousPoint, nextPoint, point)) {
        // remove point, if overlapping with {nextPoint}
        // or on line with {previousPoint} -> {point} -> {nextPoint}
        waypoints.splice(idx, 1);
      } else {
        idx++;
      }
    }
    return waypoints;
  }

  // helpers //////////////////////

  function distance(a, b) {
    return Math.sqrt(Math.pow(a.x - b.x, 2) + Math.pow(a.y - b.y, 2));
  }

  /**
   * @typedef {import('./Types').ConnectionLike} ConnectionLike
   * @typedef {import('./Types').RootLike} RootLike
   * @typedef {import('./Types').ParentLike } ParentLike
   * @typedef {import('./Types').ShapeLike} ShapeLike
   *
   * @typedef { {
   *   container?: HTMLElement;
   *   deferUpdate?: boolean;
   *   width?: number;
   *   height?: number;
   * } } CanvasConfig
   * @typedef { {
   *   group: SVGElement;
   *   index: number;
   *   visible: boolean;
   * } } CanvasLayer
   * @typedef { {
   *   [key: string]: CanvasLayer;
   * } } CanvasLayers
   * @typedef { {
   *   rootElement: ShapeLike;
   *   layer: CanvasLayer;
   * } } CanvasPlane
   * @typedef { {
   *   scale: number;
   *   inner: Rect;
   *   outer: Dimensions;
   * } & Rect } CanvasViewbox
   *
   * @typedef {import('./ElementRegistry').default} ElementRegistry
   * @typedef {import('./EventBus').default} EventBus
   * @typedef {import('./GraphicsFactory').default} GraphicsFactory
   *
   * @typedef {import('../util/Types').Dimensions} Dimensions
   * @typedef {import('../util/Types').Point} Point
   * @typedef {import('../util/Types').Rect} Rect
   * @typedef {import('../util/Types').RectTRBL} RectTRBL
   * @typedef {import('../util/Types').ScrollDelta} ScrollDelta
   */

  function round$9(number, resolution) {
    return Math.round(number * resolution) / resolution;
  }
  function ensurePx(number) {
    return isNumber$3(number) ? number + 'px' : number;
  }
  function findRoot(element) {
    while (element.parent) {
      element = element.parent;
    }
    return element;
  }

  /**
   * Creates a HTML container element for a SVG element with
   * the given configuration
   *
   * @param {CanvasConfig} options
   *
   * @return {HTMLElement} the container element
   */
  function createContainer(options) {
    options = assign$4({}, {
      width: '100%',
      height: '100%'
    }, options);
    const container = options.container || document.body;

    // create a <div> around the svg element with the respective size
    // this way we can always get the correct container size
    // (this is impossible for <svg> elements at the moment)
    const parent = document.createElement('div');
    parent.setAttribute('class', 'djs-container djs-parent');
    assign(parent, {
      position: 'relative',
      overflow: 'hidden',
      width: ensurePx(options.width),
      height: ensurePx(options.height)
    });
    container.appendChild(parent);
    return parent;
  }
  function createGroup(parent, cls, childIndex) {
    const group = create$2('g');
    classes(group).add(cls);
    const index = childIndex !== undefined ? childIndex : parent.childNodes.length - 1;

    // must ensure second argument is node or _null_
    // cf. https://developer.mozilla.org/en-US/docs/Web/API/Node/insertBefore
    parent.insertBefore(group, parent.childNodes[index] || null);
    return group;
  }
  const BASE_LAYER = 'base';

  // render plane contents behind utility layers
  const PLANE_LAYER_INDEX = 0;
  const UTILITY_LAYER_INDEX = 1;
  const REQUIRED_MODEL_ATTRS = {
    shape: ['x', 'y', 'width', 'height'],
    connection: ['waypoints']
  };

  /**
   * The main drawing canvas.
   *
   * @class
   * @constructor
   *
   * @emits Canvas#canvas.init
   *
   * @param {CanvasConfig|null} config
   * @param {EventBus} eventBus
   * @param {GraphicsFactory} graphicsFactory
   * @param {ElementRegistry} elementRegistry
   */
  function Canvas(config, eventBus, graphicsFactory, elementRegistry) {
    this._eventBus = eventBus;
    this._elementRegistry = elementRegistry;
    this._graphicsFactory = graphicsFactory;

    /**
     * @type {number}
     */
    this._rootsIdx = 0;

    /**
     * @type {CanvasLayers}
     */
    this._layers = {};

    /**
     * @type {CanvasPlane[]}
     */
    this._planes = [];

    /**
     * @type {RootLike|null}
     */
    this._rootElement = null;

    /**
     * @type {boolean}
     */
    this._focused = false;
    this._init(config || {});
  }
  Canvas.$inject = ['config.canvas', 'eventBus', 'graphicsFactory', 'elementRegistry'];

  /**
   * Creates a <svg> element that is wrapped into a <div>.
   * This way we are always able to correctly figure out the size of the svg element
   * by querying the parent node.

   * (It is not possible to get the size of a svg element cross browser @ 2014-04-01)

   * <div class="djs-container" style="width: {desired-width}, height: {desired-height}">
   *   <svg width="100%" height="100%">
   *    ...
   *   </svg>
   * </div>
   *
   * @param {CanvasConfig} config
   */
  Canvas.prototype._init = function (config) {
    const eventBus = this._eventBus;

    // html container
    const container = this._container = createContainer(config);
    const svg = this._svg = create$2('svg');
    attr(svg, {
      width: '100%',
      height: '100%'
    });
    attr$1(svg, 'tabindex', 0);
    eventBus.on('element.hover', () => {
      this.restoreFocus();
    });
    svg.addEventListener('focusin', () => {
      this._focused = true;
      eventBus.fire('canvas.focus.changed', {
        focused: true
      });
    });
    svg.addEventListener('focusout', () => {
      this._focused = false;
      eventBus.fire('canvas.focus.changed', {
        focused: false
      });
    });
    append(container, svg);
    const viewport = this._viewport = createGroup(svg, 'viewport');

    // debounce canvas.viewbox.changed events when deferUpdate is set
    // to help with potential performance issues
    if (config.deferUpdate) {
      this._viewboxChanged = debounce(bind$3(this._viewboxChanged, this), 300);
    }
    eventBus.on('diagram.init', () => {
      /**
       * An event indicating that the canvas is ready to be drawn on.
       *
       * @memberOf Canvas
       *
       * @event canvas.init
       *
       * @type {Object}
       * @property {SVGElement} svg the created svg element
       * @property {SVGElement} viewport the direct parent of diagram elements and shapes
       */
      eventBus.fire('canvas.init', {
        svg: svg,
        viewport: viewport
      });
    });

    // reset viewbox on shape changes to
    // recompute the viewbox
    eventBus.on(['shape.added', 'connection.added', 'shape.removed', 'connection.removed', 'elements.changed', 'root.set'], () => {
      delete this._cachedViewbox;
    });
    eventBus.on('diagram.destroy', 500, this._destroy, this);
    eventBus.on('diagram.clear', 500, this._clear, this);
  };
  Canvas.prototype._destroy = function () {
    this._eventBus.fire('canvas.destroy', {
      svg: this._svg,
      viewport: this._viewport
    });
    const parent = this._container.parentNode;
    if (parent) {
      parent.removeChild(this._container);
    }
    delete this._svg;
    delete this._container;
    delete this._layers;
    delete this._planes;
    delete this._rootElement;
    delete this._viewport;
  };
  Canvas.prototype._clear = function () {
    const allElements = this._elementRegistry.getAll();

    // remove all elements
    allElements.forEach(element => {
      const type = getType$1(element);
      if (type === 'root') {
        this.removeRootElement(element);
      } else {
        this._removeElement(element, type);
      }
    });

    // remove all planes
    this._planes = [];
    this._rootElement = null;

    // force recomputation of view box
    delete this._cachedViewbox;
  };

  /**
  * Sets focus on the canvas SVG element.
  */
  Canvas.prototype.focus = function () {
    this._svg.focus({
      preventScroll: true
    });
  };

  /**
  * Sets focus on the canvas SVG element if `document.body` is currently focused.
  */
  Canvas.prototype.restoreFocus = function () {
    if (document.activeElement === document.body) {
      this.focus();
    }
  };

  /**
  * Returns true if the canvas is focused.
  *
  * @return {boolean}
  */
  Canvas.prototype.isFocused = function () {
    return this._focused;
  };

  /**
   * Returns the default layer on which
   * all elements are drawn.
   *
   * @return {SVGElement}  The SVG element of the layer.
   */
  Canvas.prototype.getDefaultLayer = function () {
    return this.getLayer(BASE_LAYER, PLANE_LAYER_INDEX);
  };

  /**
   * Returns a layer that is used to draw elements
   * or annotations on it.
   *
   * Non-existing layers retrieved through this method
   * will be created. During creation, the optional index
   * may be used to create layers below or above existing layers.
   * A layer with a certain index is always created above all
   * existing layers with the same index.
   *
   * @param {string} name The name of the layer.
   * @param {number} [index] The index of the layer.
   *
   * @return {SVGElement} The SVG element of the layer.
   */
  Canvas.prototype.getLayer = function (name, index) {
    if (!name) {
      throw new Error('must specify a name');
    }
    let layer = this._layers[name];
    if (!layer) {
      layer = this._layers[name] = this._createLayer(name, index);
    }

    // throw an error if layer creation / retrival is
    // requested on different index
    if (typeof index !== 'undefined' && layer.index !== index) {
      throw new Error('layer <' + name + '> already created at index <' + index + '>');
    }
    return layer.group;
  };

  /**
   * For a given index, return the number of layers that have a higher index and
   * are visible.
   *
   * This is used to determine the node a layer should be inserted at.
   *
   * @param {number} index
   *
   * @return {number}
   */
  Canvas.prototype._getChildIndex = function (index) {
    return reduce(this._layers, function (childIndex, layer) {
      if (layer.visible && index >= layer.index) {
        childIndex++;
      }
      return childIndex;
    }, 0);
  };

  /**
   * Creates a given layer and returns it.
   *
   * @param {string} name
   * @param {number} [index=0]
   *
   * @return {CanvasLayer}
   */
  Canvas.prototype._createLayer = function (name, index) {
    if (typeof index === 'undefined') {
      index = UTILITY_LAYER_INDEX;
    }
    const childIndex = this._getChildIndex(index);
    return {
      group: createGroup(this._viewport, 'layer-' + name, childIndex),
      index: index,
      visible: true
    };
  };

  /**
   * Shows a given layer.
   *
   * @param {string} name The name of the layer.
   *
   * @return {SVGElement} The SVG element of the layer.
   */
  Canvas.prototype.showLayer = function (name) {
    if (!name) {
      throw new Error('must specify a name');
    }
    const layer = this._layers[name];
    if (!layer) {
      throw new Error('layer <' + name + '> does not exist');
    }
    const viewport = this._viewport;
    const group = layer.group;
    const index = layer.index;
    if (layer.visible) {
      return group;
    }
    const childIndex = this._getChildIndex(index);
    viewport.insertBefore(group, viewport.childNodes[childIndex] || null);
    layer.visible = true;
    return group;
  };

  /**
   * Hides a given layer.
   *
   * @param {string} name The name of the layer.
   *
   * @return {SVGElement} The SVG element of the layer.
   */
  Canvas.prototype.hideLayer = function (name) {
    if (!name) {
      throw new Error('must specify a name');
    }
    const layer = this._layers[name];
    if (!layer) {
      throw new Error('layer <' + name + '> does not exist');
    }
    const group = layer.group;
    if (!layer.visible) {
      return group;
    }
    remove$3(group);
    layer.visible = false;
    return group;
  };
  Canvas.prototype._removeLayer = function (name) {
    const layer = this._layers[name];
    if (layer) {
      delete this._layers[name];
      remove$3(layer.group);
    }
  };

  /**
   * Returns the currently active layer. Can be null.
   *
   * @return {CanvasLayer|null} The active layer of `null`.
   */
  Canvas.prototype.getActiveLayer = function () {
    const plane = this._findPlaneForRoot(this.getRootElement());
    if (!plane) {
      return null;
    }
    return plane.layer;
  };

  /**
   * Returns the plane which contains the given element.
   *
   * @param {ShapeLike|ConnectionLike|string} element The element or its ID.
   *
   * @return {RootLike|undefined} The root of the element.
   */
  Canvas.prototype.findRoot = function (element) {
    if (typeof element === 'string') {
      element = this._elementRegistry.get(element);
    }
    if (!element) {
      return;
    }
    const plane = this._findPlaneForRoot(findRoot(element)) || {};
    return plane.rootElement;
  };

  /**
   * Return a list of all root elements on the diagram.
   *
   * @return {(RootLike)[]} The list of root elements.
   */
  Canvas.prototype.getRootElements = function () {
    return this._planes.map(function (plane) {
      return plane.rootElement;
    });
  };
  Canvas.prototype._findPlaneForRoot = function (rootElement) {
    return find$2(this._planes, function (plane) {
      return plane.rootElement === rootElement;
    });
  };

  /**
   * Returns the html element that encloses the
   * drawing canvas.
   *
   * @return {HTMLElement} The HTML element of the container.
   */
  Canvas.prototype.getContainer = function () {
    return this._container;
  };

  // markers //////////////////////

  Canvas.prototype._updateMarker = function (element, marker, add) {
    let container;
    if (!element.id) {
      element = this._elementRegistry.get(element);
    }
    element.markers = element.markers || new Set();

    // we need to access all
    container = this._elementRegistry._elements[element.id];
    if (!container) {
      return;
    }
    forEach$3([container.gfx, container.secondaryGfx], function (gfx) {
      if (gfx) {
        // invoke either addClass or removeClass based on mode
        if (add) {
          element.markers.add(marker);
          classes(gfx).add(marker);
        } else {
          element.markers.delete(marker);
          classes(gfx).remove(marker);
        }
      }
    });

    /**
     * An event indicating that a marker has been updated for an element
     *
     * @event element.marker.update
     * @type {Object}
     * @property {Element} element the shape
     * @property {SVGElement} gfx the graphical representation of the shape
     * @property {string} marker
     * @property {boolean} add true if the marker was added, false if it got removed
     */
    this._eventBus.fire('element.marker.update', {
      element: element,
      gfx: container.gfx,
      marker: marker,
      add: !!add
    });
  };

  /**
   * Adds a marker to an element (basically a css class).
   *
   * Fires the element.marker.update event, making it possible to
   * integrate extension into the marker life-cycle, too.
   *
   * @example
   *
   * ```javascript
   * canvas.addMarker('foo', 'some-marker');
   *
   * const fooGfx = canvas.getGraphics('foo');
   *
   * fooGfx; // <g class="... some-marker"> ... </g>
   * ```
   *
   * @param {ShapeLike|ConnectionLike|string} element The element or its ID.
   * @param {string} marker The marker.
   */
  Canvas.prototype.addMarker = function (element, marker) {
    this._updateMarker(element, marker, true);
  };

  /**
   * Remove a marker from an element.
   *
   * Fires the element.marker.update event, making it possible to
   * integrate extension into the marker life-cycle, too.
   *
   * @param {ShapeLike|ConnectionLike|string} element The element or its ID.
   * @param {string} marker The marker.
   */
  Canvas.prototype.removeMarker = function (element, marker) {
    this._updateMarker(element, marker, false);
  };

  /**
   * Check whether an element has a given marker.
   *
   * @param {ShapeLike|ConnectionLike|string} element The element or its ID.
   * @param {string} marker The marker.
   */
  Canvas.prototype.hasMarker = function (element, marker) {
    if (!element.id) {
      element = this._elementRegistry.get(element);
    }
    if (!element.markers) {
      return false;
    }
    return element.markers.has(marker);
  };

  /**
   * Toggles a marker on an element.
   *
   * Fires the element.marker.update event, making it possible to
   * integrate extension into the marker life-cycle, too.
   *
   * @param {ShapeLike|ConnectionLike|string} element The element or its ID.
   * @param {string} marker The marker.
   */
  Canvas.prototype.toggleMarker = function (element, marker) {
    if (this.hasMarker(element, marker)) {
      this.removeMarker(element, marker);
    } else {
      this.addMarker(element, marker);
    }
  };

  /**
   * Returns the current root element.
   *
   * Supports two different modes for handling root elements:
   *
   * 1. if no root element has been added before, an implicit root will be added
   * and returned. This is used in applications that don't require explicit
   * root elements.
   *
   * 2. when root elements have been added before calling `getRootElement`,
   * root elements can be null. This is used for applications that want to manage
   * root elements themselves.
   *
   * @return {RootLike} The current root element.
   */
  Canvas.prototype.getRootElement = function () {
    const rootElement = this._rootElement;

    // can return null if root elements are present but none was set yet
    if (rootElement || this._planes.length) {
      return rootElement;
    }
    return this.setRootElement(this.addRootElement(null));
  };

  /**
   * Adds a given root element and returns it.
   *
   * @param {RootLike} [rootElement] The root element to be added.
   *
   * @return {RootLike} The added root element or an implicit root element.
   */
  Canvas.prototype.addRootElement = function (rootElement) {
    const idx = this._rootsIdx++;
    if (!rootElement) {
      rootElement = {
        id: '__implicitroot_' + idx,
        children: [],
        isImplicit: true
      };
    }
    const layerName = rootElement.layer = 'root-' + idx;
    this._ensureValid('root', rootElement);
    const layer = this.getLayer(layerName, PLANE_LAYER_INDEX);
    this.hideLayer(layerName);
    this._addRoot(rootElement, layer);
    this._planes.push({
      rootElement: rootElement,
      layer: layer
    });
    return rootElement;
  };

  /**
   * Removes a given root element and returns it.
   *
   * @param {RootLike|string} rootElement element or element ID
   *
   * @return {RootLike|undefined} removed element
   */
  Canvas.prototype.removeRootElement = function (rootElement) {
    if (typeof rootElement === 'string') {
      rootElement = this._elementRegistry.get(rootElement);
    }
    const plane = this._findPlaneForRoot(rootElement);
    if (!plane) {
      return;
    }

    // hook up life-cycle events
    this._removeRoot(rootElement);

    // clean up layer
    this._removeLayer(rootElement.layer);

    // clean up plane
    this._planes = this._planes.filter(function (plane) {
      return plane.rootElement !== rootElement;
    });

    // clean up active root
    if (this._rootElement === rootElement) {
      this._rootElement = null;
    }
    return rootElement;
  };

  /**
   * Sets a given element as the new root element for the canvas
   * and returns the new root element.
   *
   * @param {RootLike} rootElement The root element to be set.
   *
   * @return {RootLike} The set root element.
   */
  Canvas.prototype.setRootElement = function (rootElement) {
    if (rootElement === this._rootElement) {
      return rootElement;
    }
    let plane;
    if (!rootElement) {
      throw new Error('rootElement required');
    }
    plane = this._findPlaneForRoot(rootElement);

    // give set add semantics for backwards compatibility
    if (!plane) {
      rootElement = this.addRootElement(rootElement);
    }
    this._setRoot(rootElement);
    return rootElement;
  };
  Canvas.prototype._removeRoot = function (element) {
    const elementRegistry = this._elementRegistry,
      eventBus = this._eventBus;

    // simulate element remove event sequence
    eventBus.fire('root.remove', {
      element: element
    });
    eventBus.fire('root.removed', {
      element: element
    });
    elementRegistry.remove(element);
  };
  Canvas.prototype._addRoot = function (element, gfx) {
    const elementRegistry = this._elementRegistry,
      eventBus = this._eventBus;

    // resemble element add event sequence
    eventBus.fire('root.add', {
      element: element
    });
    elementRegistry.add(element, gfx);
    eventBus.fire('root.added', {
      element: element,
      gfx: gfx
    });
  };
  Canvas.prototype._setRoot = function (rootElement, layer) {
    const currentRoot = this._rootElement;
    if (currentRoot) {
      // un-associate previous root element <svg>
      this._elementRegistry.updateGraphics(currentRoot, null, true);

      // hide previous layer
      this.hideLayer(currentRoot.layer);
    }
    if (rootElement) {
      if (!layer) {
        layer = this._findPlaneForRoot(rootElement).layer;
      }

      // associate element with <svg>
      this._elementRegistry.updateGraphics(rootElement, this._svg, true);

      // show root layer
      this.showLayer(rootElement.layer);
    }
    this._rootElement = rootElement;
    this._eventBus.fire('root.set', {
      element: rootElement
    });
  };
  Canvas.prototype._ensureValid = function (type, element) {
    if (!element.id) {
      throw new Error('element must have an id');
    }
    if (this._elementRegistry.get(element.id)) {
      throw new Error('element <' + element.id + '> already exists');
    }
    const requiredAttrs = REQUIRED_MODEL_ATTRS[type];
    const valid = every(requiredAttrs, function (attr) {
      return typeof element[attr] !== 'undefined';
    });
    if (!valid) {
      throw new Error('must supply { ' + requiredAttrs.join(', ') + ' } with ' + type);
    }
  };
  Canvas.prototype._setParent = function (element, parent, parentIndex) {
    add$2(parent.children, element, parentIndex);
    element.parent = parent;
  };

  /**
   * Adds an element to the canvas.
   *
   * This wires the parent <-> child relationship between the element and
   * a explicitly specified parent or an implicit root element.
   *
   * During add it emits the events
   *
   *  * <{type}.add> (element, parent)
   *  * <{type}.added> (element, gfx)
   *
   * Extensions may hook into these events to perform their magic.
   *
   * @param {string} type
   * @param {ConnectionLike|ShapeLike} element
   * @param {ShapeLike} [parent]
   * @param {number} [parentIndex]
   *
   * @return {ConnectionLike|ShapeLike} The added element.
   */
  Canvas.prototype._addElement = function (type, element, parent, parentIndex) {
    parent = parent || this.getRootElement();
    const eventBus = this._eventBus,
      graphicsFactory = this._graphicsFactory;
    this._ensureValid(type, element);
    eventBus.fire(type + '.add', {
      element: element,
      parent: parent
    });
    this._setParent(element, parent, parentIndex);

    // create graphics
    const gfx = graphicsFactory.create(type, element, parentIndex);
    this._elementRegistry.add(element, gfx);

    // update its visual
    graphicsFactory.update(type, element, gfx);
    eventBus.fire(type + '.added', {
      element: element,
      gfx: gfx
    });
    return element;
  };

  /**
   * Adds a shape to the canvas.
   *
   * @param {ShapeLike} shape The shape to be added
   * @param {ParentLike} [parent] The shape's parent.
   * @param {number} [parentIndex] The index at which to add the shape to the parent's children.
   *
   * @return {ShapeLike} The added shape.
   */
  Canvas.prototype.addShape = function (shape, parent, parentIndex) {
    return this._addElement('shape', shape, parent, parentIndex);
  };

  /**
   * Adds a connection to the canvas.
   *
   * @param {ConnectionLike} connection The connection to be added.
   * @param {ParentLike} [parent] The connection's parent.
   * @param {number} [parentIndex] The index at which to add the connection to the parent's children.
   *
   * @return {ConnectionLike} The added connection.
   */
  Canvas.prototype.addConnection = function (connection, parent, parentIndex) {
    return this._addElement('connection', connection, parent, parentIndex);
  };

  /**
   * Internal remove element
   */
  Canvas.prototype._removeElement = function (element, type) {
    const elementRegistry = this._elementRegistry,
      graphicsFactory = this._graphicsFactory,
      eventBus = this._eventBus;
    element = elementRegistry.get(element.id || element);
    if (!element) {
      // element was removed already
      return;
    }
    eventBus.fire(type + '.remove', {
      element: element
    });
    graphicsFactory.remove(element);

    // unset parent <-> child relationship
    remove$2(element.parent && element.parent.children, element);
    element.parent = null;
    eventBus.fire(type + '.removed', {
      element: element
    });
    elementRegistry.remove(element);
    return element;
  };

  /**
   * Removes a shape from the canvas.
   *
   * @fires ShapeRemoveEvent
   * @fires ShapeRemovedEvent
   *
   * @param {ShapeLike|string} shape The shape or its ID.
   *
   * @return {ShapeLike} The removed shape.
   */
  Canvas.prototype.removeShape = function (shape) {
    /**
     * An event indicating that a shape is about to be removed from the canvas.
     *
     * @memberOf Canvas
     *
     * @event ShapeRemoveEvent
     * @type {Object}
     * @property {ShapeLike} element The shape.
     * @property {SVGElement} gfx The graphical element.
     */

    /**
     * An event indicating that a shape has been removed from the canvas.
     *
     * @memberOf Canvas
     *
     * @event ShapeRemovedEvent
     * @type {Object}
     * @property {ShapeLike} element The shape.
     * @property {SVGElement} gfx The graphical element.
     */
    return this._removeElement(shape, 'shape');
  };

  /**
   * Removes a connection from the canvas.
   *
   * @fires ConnectionRemoveEvent
   * @fires ConnectionRemovedEvent
   *
   * @param {ConnectionLike|string} connection The connection or its ID.
   *
   * @return {ConnectionLike} The removed connection.
   */
  Canvas.prototype.removeConnection = function (connection) {
    /**
     * An event indicating that a connection is about to be removed from the canvas.
     *
     * @memberOf Canvas
     *
     * @event ConnectionRemoveEvent
     * @type {Object}
     * @property {ConnectionLike} element The connection.
     * @property {SVGElement} gfx The graphical element.
     */

    /**
     * An event indicating that a connection has been removed from the canvas.
     *
     * @memberOf Canvas
     *
     * @event ConnectionRemovedEvent
     * @type {Object}
     * @property {ConnectionLike} element The connection.
     * @property {SVGElement} gfx The graphical element.
     */
    return this._removeElement(connection, 'connection');
  };

  /**
   * Returns the graphical element of an element.
   *
   * @param {ShapeLike|ConnectionLike|string} element The element or its ID.
   * @param {boolean} [secondary=false] Whether to return the secondary graphical element.
   *
   * @return {SVGElement} The graphical element.
   */
  Canvas.prototype.getGraphics = function (element, secondary) {
    return this._elementRegistry.getGraphics(element, secondary);
  };

  /**
   * Perform a viewbox update via a given change function.
   *
   * @param {Function} changeFn
   */
  Canvas.prototype._changeViewbox = function (changeFn) {
    // notify others of the upcoming viewbox change
    this._eventBus.fire('canvas.viewbox.changing');

    // perform actual change
    changeFn.apply(this);

    // reset the cached viewbox so that
    // a new get operation on viewbox or zoom
    // triggers a viewbox re-computation
    this._cachedViewbox = null;

    // notify others of the change; this step
    // may or may not be debounced
    this._viewboxChanged();
  };
  Canvas.prototype._viewboxChanged = function () {
    this._eventBus.fire('canvas.viewbox.changed', {
      viewbox: this.viewbox()
    });
  };

  /**
   * Gets or sets the view box of the canvas, i.e. the
   * area that is currently displayed.
   *
   * The getter may return a cached viewbox (if it is currently
   * changing). To force a recomputation, pass `false` as the first argument.
   *
   * @example
   *
   * ```javascript
   * canvas.viewbox({ x: 100, y: 100, width: 500, height: 500 })
   *
   * // sets the visible area of the diagram to (100|100) -> (600|100)
   * // and and scales it according to the diagram width
   *
   * const viewbox = canvas.viewbox(); // pass `false` to force recomputing the box.
   *
   * console.log(viewbox);
   * // {
   * //   inner: Dimensions,
   * //   outer: Dimensions,
   * //   scale,
   * //   x, y,
   * //   width, height
   * // }
   *
   * // if the current diagram is zoomed and scrolled, you may reset it to the
   * // default zoom via this method, too:
   *
   * const zoomedAndScrolledViewbox = canvas.viewbox();
   *
   * canvas.viewbox({
   *   x: 0,
   *   y: 0,
   *   width: zoomedAndScrolledViewbox.outer.width,
   *   height: zoomedAndScrolledViewbox.outer.height
   * });
   * ```
   *
   * @param {Rect} [box] The viewbox to be set.
   *
   * @return {CanvasViewbox} The set viewbox.
   */
  Canvas.prototype.viewbox = function (box) {
    if (box === undefined && this._cachedViewbox) {
      return structuredClone(this._cachedViewbox);
    }
    const viewport = this._viewport,
      outerBox = this.getSize();
    let innerBox, matrix, activeLayer, transform, scale, x, y;
    if (!box) {
      // compute the inner box based on the
      // diagrams active layer. This allows us to exclude
      // external components, such as overlays

      activeLayer = this._rootElement ? this.getActiveLayer() : null;
      innerBox = activeLayer && activeLayer.getBBox() || {};
      transform = transform$1(viewport);
      matrix = transform ? transform.matrix : createMatrix();
      scale = round$9(matrix.a, 1000);
      x = round$9(-matrix.e || 0, 1000);
      y = round$9(-matrix.f || 0, 1000);
      box = this._cachedViewbox = {
        x: x ? x / scale : 0,
        y: y ? y / scale : 0,
        width: outerBox.width / scale,
        height: outerBox.height / scale,
        scale: scale,
        inner: {
          width: innerBox.width || 0,
          height: innerBox.height || 0,
          x: innerBox.x || 0,
          y: innerBox.y || 0
        },
        outer: outerBox
      };
      return box;
    } else {
      this._changeViewbox(function () {
        scale = Math.min(outerBox.width / box.width, outerBox.height / box.height);
        const matrix = this._svg.createSVGMatrix().scale(scale).translate(-box.x, -box.y);
        transform$1(viewport, matrix);
      });
    }
    return box;
  };

  /**
   * Gets or sets the scroll of the canvas.
   *
   * @param {ScrollDelta} [delta] The scroll to be set.
   *
   * @return {Point}
   */
  Canvas.prototype.scroll = function (delta) {
    const node = this._viewport;
    let matrix = node.getCTM();
    if (delta) {
      this._changeViewbox(function () {
        delta = assign$4({
          dx: 0,
          dy: 0
        }, delta || {});
        matrix = this._svg.createSVGMatrix().translate(delta.dx, delta.dy).multiply(matrix);
        setCTM(node, matrix);
      });
    }
    return {
      x: matrix.e,
      y: matrix.f
    };
  };

  /**
   * Scrolls the viewbox to contain the given element.
   * Optionally specify a padding to be applied to the edges.
   *
   * @param {ShapeLike|ConnectionLike|string} element The element to scroll to or its ID.
   * @param {RectTRBL|number} [padding=100] The padding to be applied. Can also specify top, bottom, left and right.
   */
  Canvas.prototype.scrollToElement = function (element, padding) {
    let defaultPadding = 100;
    if (typeof element === 'string') {
      element = this._elementRegistry.get(element);
    }

    // set to correct rootElement
    const rootElement = this.findRoot(element);
    if (rootElement !== this.getRootElement()) {
      this.setRootElement(rootElement);
    }

    // element is rootElement, do not change viewport
    if (rootElement === element) {
      return;
    }
    if (!padding) {
      padding = {};
    }
    if (typeof padding === 'number') {
      defaultPadding = padding;
    }
    padding = {
      top: padding.top || defaultPadding,
      right: padding.right || defaultPadding,
      bottom: padding.bottom || defaultPadding,
      left: padding.left || defaultPadding
    };
    const elementBounds = getBBox(element),
      elementTrbl = asTRBL(elementBounds),
      viewboxBounds = this.viewbox(),
      zoom = this.zoom();
    let dx, dy;

    // shrink viewboxBounds with padding
    viewboxBounds.y += padding.top / zoom;
    viewboxBounds.x += padding.left / zoom;
    viewboxBounds.width -= (padding.right + padding.left) / zoom;
    viewboxBounds.height -= (padding.bottom + padding.top) / zoom;
    const viewboxTrbl = asTRBL(viewboxBounds);
    const canFit = elementBounds.width < viewboxBounds.width && elementBounds.height < viewboxBounds.height;
    if (!canFit) {
      // top-left when element can't fit
      dx = elementBounds.x - viewboxBounds.x;
      dy = elementBounds.y - viewboxBounds.y;
    } else {
      const dRight = Math.max(0, elementTrbl.right - viewboxTrbl.right),
        dLeft = Math.min(0, elementTrbl.left - viewboxTrbl.left),
        dBottom = Math.max(0, elementTrbl.bottom - viewboxTrbl.bottom),
        dTop = Math.min(0, elementTrbl.top - viewboxTrbl.top);
      dx = dRight || dLeft;
      dy = dBottom || dTop;
    }
    this.scroll({
      dx: -dx * zoom,
      dy: -dy * zoom
    });
  };

  /**
   * Gets or sets the current zoom of the canvas, optionally zooming to the
   * specified position.
   *
   * The getter may return a cached zoom level. Call it with `false` as the first
   * argument to force recomputation of the current level.
   *
   * @param {number|'fit-viewport'} [newScale] The new zoom level, either a number,
   * i.e. 0.9, or `fit-viewport` to adjust the size to fit the current viewport.
   * @param {Point} [center] The reference point { x: ..., y: ...} to zoom to.
   *
   * @return {number} The set zoom level.
   */
  Canvas.prototype.zoom = function (newScale, center) {
    if (!newScale) {
      return this.viewbox(newScale).scale;
    }
    if (newScale === 'fit-viewport') {
      return this._fitViewport(center);
    }
    let outer, matrix;
    this._changeViewbox(function () {
      if (typeof center !== 'object') {
        outer = this.viewbox().outer;
        center = {
          x: outer.width / 2,
          y: outer.height / 2
        };
      }
      matrix = this._setZoom(newScale, center);
    });
    return round$9(matrix.a, 1000);
  };
  function setCTM(node, m) {
    const mstr = 'matrix(' + m.a + ',' + m.b + ',' + m.c + ',' + m.d + ',' + m.e + ',' + m.f + ')';
    node.setAttribute('transform', mstr);
  }
  Canvas.prototype._fitViewport = function (center) {
    const vbox = this.viewbox(),
      outer = vbox.outer,
      inner = vbox.inner;
    let newScale, newViewbox;

    // display the complete diagram without zooming in.
    // instead of relying on internal zoom, we perform a
    // hard reset on the canvas viewbox to realize this
    //
    // if diagram does not need to be zoomed in, we focus it around
    // the diagram origin instead

    if (inner.x >= 0 && inner.y >= 0 && inner.x + inner.width <= outer.width && inner.y + inner.height <= outer.height && !center) {
      newViewbox = {
        x: 0,
        y: 0,
        width: Math.max(inner.width + inner.x, outer.width),
        height: Math.max(inner.height + inner.y, outer.height)
      };
    } else {
      newScale = Math.min(1, outer.width / inner.width, outer.height / inner.height);
      newViewbox = {
        x: inner.x + (center ? inner.width / 2 - outer.width / newScale / 2 : 0),
        y: inner.y + (center ? inner.height / 2 - outer.height / newScale / 2 : 0),
        width: outer.width / newScale,
        height: outer.height / newScale
      };
    }
    this.viewbox(newViewbox);
    return this.viewbox(false).scale;
  };
  Canvas.prototype._setZoom = function (scale, center) {
    const svg = this._svg,
      viewport = this._viewport;
    const matrix = svg.createSVGMatrix();
    const point = svg.createSVGPoint();
    let centerPoint, originalPoint, currentMatrix, scaleMatrix, newMatrix;
    currentMatrix = viewport.getCTM();
    const currentScale = currentMatrix.a;
    if (center) {
      centerPoint = assign$4(point, center);

      // revert applied viewport transformations
      originalPoint = centerPoint.matrixTransform(currentMatrix.inverse());

      // create scale matrix
      scaleMatrix = matrix.translate(originalPoint.x, originalPoint.y).scale(1 / currentScale * scale).translate(-originalPoint.x, -originalPoint.y);
      newMatrix = currentMatrix.multiply(scaleMatrix);
    } else {
      newMatrix = matrix.scale(scale);
    }
    setCTM(this._viewport, newMatrix);
    return newMatrix;
  };

  /**
   * Returns the size of the canvas.
   *
   * @return {Dimensions} The size of the canvas.
   */
  Canvas.prototype.getSize = function () {
    return {
      width: this._container.clientWidth,
      height: this._container.clientHeight
    };
  };

  /**
   * Returns the absolute bounding box of an element.
   *
   * The absolute bounding box may be used to display overlays in the callers
   * (browser) coordinate system rather than the zoomed in/out canvas coordinates.
   *
   * @param {ShapeLike|ConnectionLike} element The element.
   *
   * @return {Rect} The element's absolute bounding box.
   */
  Canvas.prototype.getAbsoluteBBox = function (element) {
    const vbox = this.viewbox();
    let bbox;

    // connection
    // use svg bbox
    if (element.waypoints) {
      const gfx = this.getGraphics(element);
      bbox = gfx.getBBox();
    }

    // shapes
    // use data
    else {
      bbox = element;
    }
    const x = bbox.x * vbox.scale - vbox.x * vbox.scale;
    const y = bbox.y * vbox.scale - vbox.y * vbox.scale;
    const width = bbox.width * vbox.scale;
    const height = bbox.height * vbox.scale;
    return {
      x: x,
      y: y,
      width: width,
      height: height
    };
  };

  /**
   * Fires an event so other modules can react to the canvas resizing.
   */
  Canvas.prototype.resized = function () {
    // force recomputation of view box
    delete this._cachedViewbox;
    this._eventBus.fire('canvas.resized');
  };

  var ELEMENT_ID = 'data-element-id';

  /**
   * @typedef {import('./Types').ElementLike} ElementLike
   *
   * @typedef {import('./EventBus').default} EventBus
   *
   * @typedef { (element: ElementLike, gfx: SVGElement) => boolean|any } ElementRegistryFilterCallback
   * @typedef { (element: ElementLike, gfx: SVGElement) => any } ElementRegistryForEachCallback
   */

  /**
   * A registry that keeps track of all shapes in the diagram.
   *
   * @class
   * @constructor
   *
   * @param {EventBus} eventBus
   */
  function ElementRegistry$3(eventBus) {
    /**
     * @type { {
     *   [id: string]: {
     *     element: ElementLike;
     *     gfx?: SVGElement;
     *     secondaryGfx?: SVGElement;
     *   }
     * } }
     */
    this._elements = {};
    this._eventBus = eventBus;
  }
  ElementRegistry$3.$inject = ['eventBus'];

  /**
   * Add an element and its graphical representation(s) to the registry.
   *
   * @param {ElementLike} element The element to be added.
   * @param {SVGElement} gfx The primary graphical representation.
   * @param {SVGElement} [secondaryGfx] The secondary graphical representation.
   */
  ElementRegistry$3.prototype.add = function (element, gfx, secondaryGfx) {
    var id = element.id;
    this._validateId(id);

    // associate dom node with element
    attr(gfx, ELEMENT_ID, id);
    if (secondaryGfx) {
      attr(secondaryGfx, ELEMENT_ID, id);
    }
    this._elements[id] = {
      element: element,
      gfx: gfx,
      secondaryGfx: secondaryGfx
    };
  };

  /**
   * Remove an element from the registry.
   *
   * @param {ElementLike|string} element
   */
  ElementRegistry$3.prototype.remove = function (element) {
    var elements = this._elements,
      id = element.id || element,
      container = id && elements[id];
    if (container) {
      // unset element id on gfx
      attr(container.gfx, ELEMENT_ID, '');
      if (container.secondaryGfx) {
        attr(container.secondaryGfx, ELEMENT_ID, '');
      }
      delete elements[id];
    }
  };

  /**
   * Update an elements ID.
   *
   * @param {ElementLike|string} element The element or its ID.
   * @param {string} newId The new ID.
   */
  ElementRegistry$3.prototype.updateId = function (element, newId) {
    this._validateId(newId);
    if (typeof element === 'string') {
      element = this.get(element);
    }
    this._eventBus.fire('element.updateId', {
      element: element,
      newId: newId
    });
    var gfx = this.getGraphics(element),
      secondaryGfx = this.getGraphics(element, true);
    this.remove(element);
    element.id = newId;
    this.add(element, gfx, secondaryGfx);
  };

  /**
   * Update the graphical representation of an element.
   *
   * @param {ElementLike|string} filter The element or its ID.
   * @param {SVGElement} gfx The new graphical representation.
   * @param {boolean} [secondary=false] Whether to update the secondary graphical representation.
   */
  ElementRegistry$3.prototype.updateGraphics = function (filter, gfx, secondary) {
    var id = filter.id || filter;
    var container = this._elements[id];
    if (secondary) {
      container.secondaryGfx = gfx;
    } else {
      container.gfx = gfx;
    }
    if (gfx) {
      attr(gfx, ELEMENT_ID, id);
    }
    return gfx;
  };

  /**
   * Get the element with the given ID or graphical representation.
   *
   * @example
   *
   * ```javascript
   * elementRegistry.get('SomeElementId_1');
   *
   * elementRegistry.get(gfx);
   * ```
   *
   * @param {string|SVGElement} filter The elements ID or graphical representation.
   *
   * @return {ElementLike|undefined} The element.
   */
  ElementRegistry$3.prototype.get = function (filter) {
    var id;
    if (typeof filter === 'string') {
      id = filter;
    } else {
      id = filter && attr(filter, ELEMENT_ID);
    }
    var container = this._elements[id];
    return container && container.element;
  };

  /**
   * Return all elements that match a given filter function.
   *
   * @param {ElementRegistryFilterCallback} fn The filter function.
   *
   * @return {ElementLike[]} The matching elements.
   */
  ElementRegistry$3.prototype.filter = function (fn) {
    var filtered = [];
    this.forEach(function (element, gfx) {
      if (fn(element, gfx)) {
        filtered.push(element);
      }
    });
    return filtered;
  };

  /**
   * Return the first element that matches the given filter function.
   *
   * @param {ElementRegistryFilterCallback} fn The filter function.
   *
   * @return {ElementLike|undefined} The matching element.
   */
  ElementRegistry$3.prototype.find = function (fn) {
    var map = this._elements,
      keys = Object.keys(map);
    for (var i = 0; i < keys.length; i++) {
      var id = keys[i],
        container = map[id],
        element = container.element,
        gfx = container.gfx;
      if (fn(element, gfx)) {
        return element;
      }
    }
  };

  /**
   * Get all elements.
   *
   * @return {ElementLike[]} All elements.
   */
  ElementRegistry$3.prototype.getAll = function () {
    return this.filter(function (e) {
      return e;
    });
  };

  /**
   * Execute a given function for each element.
   *
   * @param {ElementRegistryForEachCallback} fn The function to execute.
   */
  ElementRegistry$3.prototype.forEach = function (fn) {
    var map = this._elements;
    Object.keys(map).forEach(function (id) {
      var container = map[id],
        element = container.element,
        gfx = container.gfx;
      return fn(element, gfx);
    });
  };

  /**
   * Return the graphical representation of an element.
   *
   * @example
   *
   * ```javascript
   * elementRegistry.getGraphics('SomeElementId_1');
   *
   * elementRegistry.getGraphics(rootElement); // <g ...>
   *
   * elementRegistry.getGraphics(rootElement, true); // <svg ...>
   * ```
   *
   * @param {ElementLike|string} filter The element or its ID.
   * @param {boolean} [secondary=false] Whether to return the secondary graphical representation.
   *
   * @return {SVGElement} The graphical representation.
   */
  ElementRegistry$3.prototype.getGraphics = function (filter, secondary) {
    var id = filter.id || filter;
    var container = this._elements[id];
    return container && (secondary ? container.secondaryGfx : container.gfx);
  };

  /**
   * Validate an ID and throw an error if invalid.
   *
   * @param {string} id
   *
   * @throws {Error} Error indicating that the ID is invalid or already assigned.
   */
  ElementRegistry$3.prototype._validateId = function (id) {
    if (!id) {
      throw new Error('element must have an id');
    }
    if (this._elements[id]) {
      throw new Error('element with id ' + id + ' already added');
    }
  };

  /**
   * Extends a collection with {@link Refs} aware methods
   *
   * @param {Array<Object>} collection
   * @param {Refs} refs instance
   * @param {Object} property represented by the collection
   * @param {Object} target object the collection is attached to
   *
   * @return {RefsCollection<Object>} the extended array
   */
  function extend(collection, refs, property, target) {
    var inverseProperty = property.inverse;

    /**
     * Removes the given element from the array and returns it.
     *
     * @method RefsCollection#remove
     *
     * @param {Object} element the element to remove
     */
    Object.defineProperty(collection, 'remove', {
      value: function (element) {
        var idx = this.indexOf(element);
        if (idx !== -1) {
          this.splice(idx, 1);

          // unset inverse
          refs.unset(element, inverseProperty, target);
        }
        return element;
      }
    });

    /**
     * Returns true if the collection contains the given element
     *
     * @method RefsCollection#contains
     *
     * @param {Object} element the element to check for
     */
    Object.defineProperty(collection, 'contains', {
      value: function (element) {
        return this.indexOf(element) !== -1;
      }
    });

    /**
     * Adds an element to the array, unless it exists already (set semantics).
     *
     * @method RefsCollection#add
     *
     * @param {Object} element the element to add
     * @param {Number} optional index to add element to
     *                 (possibly moving other elements around)
     */
    Object.defineProperty(collection, 'add', {
      value: function (element, idx) {
        var currentIdx = this.indexOf(element);
        if (typeof idx === 'undefined') {
          if (currentIdx !== -1) {
            // element already in collection (!)
            return;
          }

          // add to end of array, as no idx is specified
          idx = this.length;
        }

        // handle already in collection
        if (currentIdx !== -1) {
          // remove element from currentIdx
          this.splice(currentIdx, 1);
        }

        // add element at idx
        this.splice(idx, 0, element);
        if (currentIdx === -1) {
          // set inverse, unless element was
          // in collection already
          refs.set(element, inverseProperty, target);
        }
      }
    });

    // a simple marker, identifying this element
    // as being a refs collection
    Object.defineProperty(collection, '__refs_collection', {
      value: true
    });
    return collection;
  }

  /**
   * Checks if a given collection is extended
   *
   * @param {Array<Object>} collection
   *
   * @return {boolean}
   */
  function isExtended(collection) {
    return collection.__refs_collection === true;
  }
  function hasOwnProperty(e, property) {
    return Object.prototype.hasOwnProperty.call(e, property.name || property);
  }
  function defineCollectionProperty(ref, property, target) {
    var collection = extend(target[property.name] || [], ref, property, target);
    Object.defineProperty(target, property.name, {
      enumerable: property.enumerable,
      value: collection
    });
    if (collection.length) {
      collection.forEach(function (o) {
        ref.set(o, property.inverse, target);
      });
    }
  }
  function defineProperty$1(ref, property, target) {
    var inverseProperty = property.inverse;
    var _value = target[property.name];
    Object.defineProperty(target, property.name, {
      configurable: property.configurable,
      enumerable: property.enumerable,
      get: function () {
        return _value;
      },
      set: function (value) {
        // return if we already performed all changes
        if (value === _value) {
          return;
        }
        var old = _value;

        // temporary set null
        _value = null;
        if (old) {
          ref.unset(old, inverseProperty, target);
        }

        // set new value
        _value = value;

        // set inverse value
        ref.set(_value, inverseProperty, target);
      }
    });
  }

  /**
   * Creates a new references object defining two inversly related
   * attribute descriptors a and b.
   *
   * <p>
   *   When bound to an object using {@link Refs#bind} the references
   *   get activated and ensure that add and remove operations are applied
   *   reversely, too.
   * </p>
   *
   * <p>
   *   For attributes represented as collections {@link Refs} provides the
   *   {@link RefsCollection#add}, {@link RefsCollection#remove} and {@link RefsCollection#contains} extensions
   *   that must be used to properly hook into the inverse change mechanism.
   * </p>
   *
   * @class Refs
   *
   * @classdesc A bi-directional reference between two attributes.
   *
   * @param {Refs.AttributeDescriptor} a property descriptor
   * @param {Refs.AttributeDescriptor} b property descriptor
   *
   * @example
   *
   * var refs = Refs({ name: 'wheels', collection: true, enumerable: true }, { name: 'car' });
   *
   * var car = { name: 'toyota' };
   * var wheels = [{ pos: 'front-left' }, { pos: 'front-right' }];
   *
   * refs.bind(car, 'wheels');
   *
   * car.wheels // []
   * car.wheels.add(wheels[0]);
   * car.wheels.add(wheels[1]);
   *
   * car.wheels // [{ pos: 'front-left' }, { pos: 'front-right' }]
   *
   * wheels[0].car // { name: 'toyota' };
   * car.wheels.remove(wheels[0]);
   *
   * wheels[0].car // undefined
   */
  function Refs(a, b) {
    if (!(this instanceof Refs)) {
      return new Refs(a, b);
    }

    // link
    a.inverse = b;
    b.inverse = a;
    this.props = {};
    this.props[a.name] = a;
    this.props[b.name] = b;
  }

  /**
   * Binds one side of a bi-directional reference to a
   * target object.
   *
   * @memberOf Refs
   *
   * @param  {Object} target
   * @param  {String} property
   */
  Refs.prototype.bind = function (target, property) {
    if (typeof property === 'string') {
      if (!this.props[property]) {
        throw new Error('no property <' + property + '> in ref');
      }
      property = this.props[property];
    }
    if (property.collection) {
      defineCollectionProperty(this, property, target);
    } else {
      defineProperty$1(this, property, target);
    }
  };
  Refs.prototype.ensureRefsCollection = function (target, property) {
    var collection = target[property.name];
    if (!isExtended(collection)) {
      defineCollectionProperty(this, property, target);
    }
    return collection;
  };
  Refs.prototype.ensureBound = function (target, property) {
    if (!hasOwnProperty(target, property)) {
      this.bind(target, property);
    }
  };
  Refs.prototype.unset = function (target, property, value) {
    if (target) {
      this.ensureBound(target, property);
      if (property.collection) {
        this.ensureRefsCollection(target, property).remove(value);
      } else {
        target[property.name] = undefined;
      }
    }
  };
  Refs.prototype.set = function (target, property, value) {
    if (target) {
      this.ensureBound(target, property);
      if (property.collection) {
        this.ensureRefsCollection(target, property).add(value);
      } else {
        target[property.name] = value;
      }
    }
  };

  var parentRefs = new Refs({
      name: 'children',
      enumerable: true,
      collection: true
    }, {
      name: 'parent'
    }),
    labelRefs = new Refs({
      name: 'labels',
      enumerable: true,
      collection: true
    }, {
      name: 'labelTarget'
    }),
    attacherRefs = new Refs({
      name: 'attachers',
      collection: true
    }, {
      name: 'host'
    }),
    outgoingRefs = new Refs({
      name: 'outgoing',
      collection: true
    }, {
      name: 'source'
    }),
    incomingRefs = new Refs({
      name: 'incoming',
      collection: true
    }, {
      name: 'target'
    });

  /**
   * @typedef {import('./Types').Element} Element
   * @typedef {import('./Types').Shape} Shape
   * @typedef {import('./Types').Root} Root
   * @typedef {import('./Types').Label} Label
   * @typedef {import('./Types').Connection} Connection
   */

  /**
   * The basic graphical representation
   *
   * @class
   * @constructor
   */
  function ElementImpl() {
    /**
     * The object that backs up the shape
     *
     * @name Element#businessObject
     * @type Object
     */
    Object.defineProperty(this, 'businessObject', {
      writable: true
    });

    /**
     * Single label support, will mapped to multi label array
     *
     * @name Element#label
     * @type Object
     */
    Object.defineProperty(this, 'label', {
      get: function () {
        return this.labels[0];
      },
      set: function (newLabel) {
        var label = this.label,
          labels = this.labels;
        if (!newLabel && label) {
          labels.remove(label);
        } else {
          labels.add(newLabel, 0);
        }
      }
    });

    /**
     * The parent shape
     *
     * @name Element#parent
     * @type Shape
     */
    parentRefs.bind(this, 'parent');

    /**
     * The list of labels
     *
     * @name Element#labels
     * @type Label
     */
    labelRefs.bind(this, 'labels');

    /**
     * The list of outgoing connections
     *
     * @name Element#outgoing
     * @type Array<Connection>
     */
    outgoingRefs.bind(this, 'outgoing');

    /**
     * The list of incoming connections
     *
     * @name Element#incoming
     * @type Array<Connection>
     */
    incomingRefs.bind(this, 'incoming');
  }

  /**
   * A graphical object
   *
   * @class
   * @constructor
   *
   * @extends ElementImpl
   */
  function ShapeImpl() {
    ElementImpl.call(this);

    /**
     * Indicates frame shapes
     *
     * @name ShapeImpl#isFrame
     * @type boolean
     */

    /**
     * The list of children
     *
     * @name ShapeImpl#children
     * @type Element[]
     */
    parentRefs.bind(this, 'children');

    /**
     * @name ShapeImpl#host
     * @type Shape
     */
    attacherRefs.bind(this, 'host');

    /**
     * @name ShapeImpl#attachers
     * @type Shape
     */
    attacherRefs.bind(this, 'attachers');
  }
  e$3(ShapeImpl, ElementImpl);

  /**
   * A root graphical object
   *
   * @class
   * @constructor
   *
   * @extends ElementImpl
   */
  function RootImpl() {
    ElementImpl.call(this);

    /**
     * The list of children
     *
     * @name RootImpl#children
     * @type Element[]
     */
    parentRefs.bind(this, 'children');
  }
  e$3(RootImpl, ShapeImpl);

  /**
   * A label for an element
   *
   * @class
   * @constructor
   *
   * @extends ShapeImpl
   */
  function LabelImpl() {
    ShapeImpl.call(this);

    /**
     * The labeled element
     *
     * @name LabelImpl#labelTarget
     * @type Element
     */
    labelRefs.bind(this, 'labelTarget');
  }
  e$3(LabelImpl, ShapeImpl);

  /**
   * A connection between two elements
   *
   * @class
   * @constructor
   *
   * @extends ElementImpl
   */
  function ConnectionImpl() {
    ElementImpl.call(this);

    /**
     * The element this connection originates from
     *
     * @name ConnectionImpl#source
     * @type Element
     */
    outgoingRefs.bind(this, 'source');

    /**
     * The element this connection points to
     *
     * @name ConnectionImpl#target
     * @type Element
     */
    incomingRefs.bind(this, 'target');
  }
  e$3(ConnectionImpl, ElementImpl);
  var types$1 = {
    connection: ConnectionImpl,
    shape: ShapeImpl,
    label: LabelImpl,
    root: RootImpl
  };

  /**
   * Creates a root element.
   *
   * @overlord
   *
   * @example
   *
   * ```javascript
   * import * as Model from 'diagram-js/lib/model';
   *
   * const root = Model.create('root', {
   *   x: 100,
   *   y: 100,
   *   width: 100,
   *   height: 100
   * });
   * ```
   *
   * @param {'root'} type
   * @param {any} [attrs]
   *
   * @return {Root}
   */

  /**
   * Creates a connection.
   *
   * @overlord
   *
   * @example
   *
   * ```javascript
   * import * as Model from 'diagram-js/lib/model';
   *
   * const connection = Model.create('connection', {
   *   waypoints: [
   *     { x: 100, y: 100 },
   *     { x: 200, y: 100 }
   *   ]
   * });
   * ```
   *
   * @param {'connection'} type
   * @param {any} [attrs]
   *
   * @return {Connection}
   */

  /**
   * Creates a shape.
   *
   * @overlord
   *
   * @example
   *
   * ```javascript
   * import * as Model from 'diagram-js/lib/model';
   *
   * const shape = Model.create('shape', {
   *   x: 100,
   *   y: 100,
   *   width: 100,
   *   height: 100
   * });
   * ```
   *
   * @param {'shape'} type
   * @param {any} [attrs]
   *
   * @return {Shape}
   */

  /**
   * Creates a label.
   *
   * @example
   *
   * ```javascript
   * import * as Model from 'diagram-js/lib/model';
   *
   * const label = Model.create('label', {
   *   x: 100,
   *   y: 100,
   *   width: 100,
   *   height: 100,
   *   labelTarget: shape
   * });
   * ```
   *
   * @param {'label'} type
   * @param {Object} [attrs]
   *
   * @return {Label}
   */
  function create$1(type, attrs) {
    var Type = types$1[type];
    if (!Type) {
      throw new Error('unknown type: <' + type + '>');
    }
    return assign$4(new Type(), attrs);
  }

  /**
   * Checks whether an object is a model instance.
   *
   * @param {any} obj
   *
   * @return {boolean}
   */
  function isModelElement(obj) {
    return obj instanceof ElementImpl;
  }

  /**
   * @typedef {import('../model/Types').Element} Element
   * @typedef {import('../model/Types').Connection} Connection
   * @typedef {import('../model/Types').Label} Label
   * @typedef {import('../model/Types').Root} Root
   * @typedef {import('../model/Types').Shape} Shape
   */

  /**
   * A factory for model elements.
   *
   * @template {Connection} [T=Connection]
   * @template {Label} [U=Label]
   * @template {Root} [V=Root]
   * @template {Shape} [W=Shape]
   */
  function ElementFactory$3() {
    this._uid = 12;
  }

  /**
   * Create a root element.
   *
   * @param {Partial<Root>} [attrs]
   *
   * @return {V} The created root element.
   */
  ElementFactory$3.prototype.createRoot = function (attrs) {
    return this.create('root', attrs);
  };

  /**
   * Create a label.
   *
   * @param {Partial<Label>} [attrs]
   *
   * @return {U} The created label.
   */
  ElementFactory$3.prototype.createLabel = function (attrs) {
    return this.create('label', attrs);
  };

  /**
   * Create a shape.
   *
   * @param {Partial<Shape>} [attrs]
   *
   * @return {W} The created shape.
   */
  ElementFactory$3.prototype.createShape = function (attrs) {
    return this.create('shape', attrs);
  };

  /**
   * Create a connection.
   *
   * @param {Partial<Connection>} [attrs]
   *
   * @return {T} The created connection.
   */
  ElementFactory$3.prototype.createConnection = function (attrs) {
    return this.create('connection', attrs);
  };

  /**
   * Create a root element.
   *
   * @overlord
   * @param {'root'} type
   * @param {Partial<Root>} [attrs]
   * @return {V}
   */
  /**
   * Create a shape.
   *
   * @overlord
   * @param {'shape'} type
   * @param {Partial<Shape>} [attrs]
   * @return {W}
   */
  /**
   * Create a connection.
   *
   * @overlord
   * @param {'connection'} type
   * @param {Partial<Connection>} [attrs]
   * @return {T}
   */
  /**
   * Create a label.
   *
   * @param {'label'} type
   * @param {Partial<Label>} [attrs]
   * @return {U}
   */
  ElementFactory$3.prototype.create = function (type, attrs) {
    attrs = assign$4({}, attrs || {});
    if (!attrs.id) {
      attrs.id = type + '_' + this._uid++;
    }
    return create$1(type, attrs);
  };

  /**
   * SVGs for elements are generated by the {@link GraphicsFactory}.
   *
   * This utility gives quick access to the important semantic
   * parts of an element.
   */

  /**
   * Returns the visual part of a diagram element.
   *
   * @param {SVGElement} gfx
   *
   * @return {SVGElement}
   */
  function getVisual(gfx) {
    return gfx.childNodes[0];
  }

  /**
   * Returns the children for a given diagram element.
   *
   * @param {SVGElement} gfx
   * @return {SVGElement}
   */
  function getChildren$2(gfx) {
    return gfx.parentNode.childNodes[1];
  }

  /**
   * @param {SVGElement} gfx
   * @param {number} x
   * @param {number} y
   * @param {number} [angle]
   * @param {number} [amount]
   */
  function transform(gfx, x, y, angle, amount) {
    var translate = createTransform();
    translate.setTranslate(x, y);
    var rotate = createTransform();
    rotate.setRotate(0, 0, 0);
    var scale = createTransform();
    scale.setScale(1, 1);
    transform$1(gfx, [translate, rotate, scale]);
  }

  /**
   * @param {SVGElement} gfx
   * @param {number} x
   * @param {number} y
   */
  function translate$1(gfx, x, y) {
    var translate = createTransform();
    translate.setTranslate(x, y);
    transform$1(gfx, translate);
  }

  /**
   * @param {SVGElement} gfx
   * @param {number} angle
   */
  function rotate(gfx, angle) {
    var rotate = createTransform();
    rotate.setRotate(angle, 0, 0);
    transform$1(gfx, rotate);
  }

  /**
   * @typedef {import('./Types').ConnectionLike} ConnectionLike
   * @typedef {import('./Types').ElementLike} ElementLike
   * @typedef {import('./Types').ShapeLike} ShapeLike
   *
   * @typedef {import('./ElementRegistry').default} ElementRegistry
   * @typedef {import('./EventBus').default} EventBus
   */

  /**
   * A factory that creates graphical elements.
   *
   * @param {EventBus} eventBus
   * @param {ElementRegistry} elementRegistry
   */
  function GraphicsFactory(eventBus, elementRegistry) {
    this._eventBus = eventBus;
    this._elementRegistry = elementRegistry;
  }
  GraphicsFactory.$inject = ['eventBus', 'elementRegistry'];

  /**
   * @param { { parent?: any } } element
   * @return {SVGElement}
   */
  GraphicsFactory.prototype._getChildrenContainer = function (element) {
    var gfx = this._elementRegistry.getGraphics(element);
    var childrenGfx;

    // root element
    if (!element.parent) {
      childrenGfx = gfx;
    } else {
      childrenGfx = getChildren$2(gfx);
      if (!childrenGfx) {
        childrenGfx = create$2('g');
        classes(childrenGfx).add('djs-children');
        append(gfx.parentNode, childrenGfx);
      }
    }
    return childrenGfx;
  };

  /**
   * Clears the graphical representation of the element and returns the
   * cleared visual (the <g class="djs-visual" /> element).
   */
  GraphicsFactory.prototype._clear = function (gfx) {
    var visual = getVisual(gfx);
    clear$1(visual);
    return visual;
  };

  /**
   * Creates a gfx container for shapes and connections
   *
   * The layout is as follows:
   *
   * <g class="djs-group">
   *
   *   <!-- the gfx -->
   *   <g class="djs-element djs-(shape|connection|frame)">
   *     <g class="djs-visual">
   *       <!-- the renderer draws in here -->
   *     </g>
   *
   *     <!-- extensions (overlays, click box, ...) goes here
   *   </g>
   *
   *   <!-- the gfx child nodes -->
   *   <g class="djs-children"></g>
   * </g>
   *
   * @param {string} type the type of the element, i.e. shape | connection
   * @param {SVGElement} childrenGfx
   * @param {number} [parentIndex] position to create container in parent
   * @param {boolean} [isFrame] is frame element
   *
   * @return {SVGElement}
   */
  GraphicsFactory.prototype._createContainer = function (type, childrenGfx, parentIndex, isFrame) {
    var outerGfx = create$2('g');
    classes(outerGfx).add('djs-group');

    // insert node at position
    if (typeof parentIndex !== 'undefined') {
      prependTo(outerGfx, childrenGfx, childrenGfx.childNodes[parentIndex]);
    } else {
      append(childrenGfx, outerGfx);
    }
    var gfx = create$2('g');
    classes(gfx).add('djs-element');
    classes(gfx).add('djs-' + type);
    if (isFrame) {
      classes(gfx).add('djs-frame');
    }
    append(outerGfx, gfx);

    // create visual
    var visual = create$2('g');
    classes(visual).add('djs-visual');
    append(gfx, visual);
    return gfx;
  };

  /**
   * Create a graphical element.
   *
   * @param { 'shape' | 'connection' | 'label' | 'root' } type The type of the element.
   * @param {ElementLike} element The element.
   * @param {number} [parentIndex] The index at which to add the graphical element to its parent's children.
   *
   * @return {SVGElement} The graphical element.
   */
  GraphicsFactory.prototype.create = function (type, element, parentIndex) {
    var childrenGfx = this._getChildrenContainer(element.parent);
    return this._createContainer(type, childrenGfx, parentIndex, isFrameElement(element));
  };

  /**
   * Update the containments of the given elements.
   *
   * @param {ElementLike[]} elements The elements.
   */
  GraphicsFactory.prototype.updateContainments = function (elements) {
    var self = this,
      elementRegistry = this._elementRegistry,
      parents;
    parents = reduce(elements, function (map, e) {
      if (e.parent) {
        map[e.parent.id] = e.parent;
      }
      return map;
    }, {});

    // update all parents of changed and reorganized their children
    // in the correct order (as indicated in our model)
    forEach$3(parents, function (parent) {
      var children = parent.children;
      if (!children) {
        return;
      }
      var childrenGfx = self._getChildrenContainer(parent);
      forEach$3(children.slice().reverse(), function (child) {
        var childGfx = elementRegistry.getGraphics(child);
        prependTo(childGfx.parentNode, childrenGfx);
      });
    });
  };

  /**
   * Draw a shape.
   *
   * @param {SVGElement} visual The graphical element.
   * @param {ShapeLike} element The shape.
   * @param {Object} attrs Optional attributes.
   *
   * @return {SVGElement}
   */
  GraphicsFactory.prototype.drawShape = function (visual, element, attrs = {}) {
    var eventBus = this._eventBus;
    return eventBus.fire('render.shape', {
      gfx: visual,
      element,
      attrs
    });
  };

  /**
   * Get the path of a shape.
   *
   * @param {ShapeLike} element The shape.
   *
   * @return {string} The path of the shape.
   */
  GraphicsFactory.prototype.getShapePath = function (element) {
    var eventBus = this._eventBus;
    return eventBus.fire('render.getShapePath', element);
  };

  /**
   * Draw a connection.
   *
   * @param {SVGElement} visual The graphical element.
   * @param {ConnectionLike} element The connection.
   * @param {Object} attrs Optional attributes.
   *
   * @return {SVGElement}
   */
  GraphicsFactory.prototype.drawConnection = function (visual, element, attrs = {}) {
    var eventBus = this._eventBus;
    return eventBus.fire('render.connection', {
      gfx: visual,
      element,
      attrs
    });
  };

  /**
   * Get the path of a connection.
   *
   * @param {ConnectionLike} connection The connection.
   *
   * @return {string} The path of the connection.
   */
  GraphicsFactory.prototype.getConnectionPath = function (connection) {
    var eventBus = this._eventBus;
    return eventBus.fire('render.getConnectionPath', connection);
  };

  /**
   * Update an elements graphical representation.
   *
   * @param {'shape'|'connection'} type
   * @param {ElementLike} element
   * @param {SVGElement} gfx
   */
  GraphicsFactory.prototype.update = function (type, element, gfx) {
    // do NOT update root element
    if (!element.parent) {
      return;
    }
    var visual = this._clear(gfx);

    // redraw
    if (type === 'shape') {
      this.drawShape(visual, element);

      // update positioning
      translate$1(gfx, element.x, element.y);
    } else if (type === 'connection') {
      this.drawConnection(visual, element);
    } else {
      throw new Error('unknown type: ' + type);
    }
    if (element.hidden) {
      attr(gfx, 'display', 'none');
    } else {
      attr(gfx, 'display', 'block');
    }
  };

  /**
   * Remove a graphical element.
   *
   * @param {ElementLike} element The element.
   */
  GraphicsFactory.prototype.remove = function (element) {
    var gfx = this._elementRegistry.getGraphics(element);

    // remove
    remove$3(gfx.parentNode);
  };

  // helpers //////////

  function prependTo(newNode, parentNode, siblingNode) {
    var node = siblingNode || parentNode.firstChild;

    // do not prepend node to itself to prevent IE from crashing
    // https://github.com/bpmn-io/bpmn-js/issues/746
    if (newNode === node) {
      return;
    }
    parentNode.insertBefore(newNode, node);
  }

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var CoreModule$3 = {
    __depends__: [DrawModule$1],
    __init__: ['canvas'],
    canvas: ['type', Canvas],
    elementRegistry: ['type', ElementRegistry$3],
    elementFactory: ['type', ElementFactory$3],
    eventBus: ['type', EventBus],
    graphicsFactory: ['type', GraphicsFactory]
  };

  /**
   * @typedef {import('didi').InjectionContext} InjectionContext
   * @typedef {import('didi').LocalsMap} LocalsMap
   * @typedef {import('didi').ModuleDeclaration} ModuleDeclaration
   *
   * @typedef { {
   *   modules?: ModuleDeclaration[];
   * } & Record<string, any> } DiagramOptions
   */

  /**
   * @template T
   * @typedef {import('didi').FactoryFunction<T>} FactoryFunction
   */

  /**
   * @template T
   * @typedef {import('didi').ArrayFunc<T>} ArrayFunc
   */

  /**
   * Bootstrap an injector from a list of modules, instantiating a number of default components
   *
   * @param {ModuleDeclaration[]} modules
   *
   * @return {Injector} a injector to use to access the components
   */
  function bootstrap(modules) {
    var injector = new Injector(modules);
    injector.init();
    return injector;
  }

  /**
   * Creates an injector from passed options.
   *
   * @template ServiceMap
   * @param {DiagramOptions} [options]
   *
   * @return {Injector<ServiceMap>}
   */
  function createInjector$2(options) {
    options = options || {};

    /**
     * @type { ModuleDeclaration }
     */
    var configModule = {
      'config': ['value', options]
    };
    var modules = [configModule, CoreModule$3].concat(options.modules || []);
    return bootstrap(modules);
  }

  /**
   * The main diagram-js entry point that bootstraps the diagram with the given
   * configuration.
   *
   * To register extensions with the diagram, pass them as Array<Module> to the constructor.
   *
   * @class
   * @constructor
   * @template [ServiceMap=null]
   *
   * @example Creating a plug-in that logs whenever a shape is added to the canvas.
   *
   * ```javascript
   * // plug-in implementation
   * function MyLoggingPlugin(eventBus) {
   *   eventBus.on('shape.added', function(event) {
   *     console.log('shape ', event.shape, ' was added to the diagram');
   *   });
   * }
   *
   * // export as module
   * export default {
   *   __init__: [ 'myLoggingPlugin' ],
   *     myLoggingPlugin: [ 'type', MyLoggingPlugin ]
   * };
   * ```
   *
   * Use the plug-in in a Diagram instance:
   *
   * ```javascript
   * import MyLoggingModule from 'path-to-my-logging-plugin';
   *
   * var diagram = new Diagram({
   *   modules: [
   *     MyLoggingModule
   *   ]
   * });
   *
   * diagram.invoke([ 'canvas', function(canvas) {
   *   // add shape to drawing canvas
   *   canvas.addShape({ x: 10, y: 10 });
   * });
   *
   * // 'shape ... was added to the diagram' logged to console
   * ```
   *
   * @param {DiagramOptions} [options]
   * @param {Injector<ServiceMap>} [injector] An (optional) injector to bootstrap the diagram with.
   */
  function Diagram(options, injector) {
    /**
     * @type {Injector<ServiceMap>}
     */
    this._injector = injector || createInjector$2(options);

    // init

    /**
     * An event indicating that all plug-ins are loaded.
     *
     * Use this event to fire other events to interested plug-ins
     *
     * @memberOf Diagram
     *
     * @event diagram.init
     *
     * @example
     *
     * ```javascript
     * eventBus.on('diagram.init', function() {
     *   eventBus.fire('my-custom-event', { foo: 'BAR' });
     * });
     * ```
     *
     * @type {Object}
     */
    this.get('eventBus').fire('diagram.init');
  }

  /**
   * @overlord
   *
   * Resolves a diagram service.
   *
   * @template T
   *
   * @param {string} name The name of the service to get.
   *
   * @return {T}
   */
  /**
   * @overlord
   *
   * Resolves a diagram service.
   *
   * @template T
   *
   * @param {string} name The name of the service to get.
   * @param {true} strict If false, resolve missing services to null.
   *
   * @return {T}
   */
  /**
   * @overlord
   *
   * Resolves a diagram service.
   *
   * @template T
   *
   * @param {string} name The name of the service to get.
   * @param {boolean} strict If false, resolve missing services to null.
   *
   * @return {T|null}
   */
  /**
   * Resolves a diagram service.
   *
   * @template {keyof ServiceMap} Name
   *
   * @param {Name} name The name of the service to get.
   *
   * @return {ServiceMap[Name]}
   */
  Diagram.prototype.get = function (name, strict) {
    return this._injector.get(name, strict);
  };

  /**
   * @overlord
   *
   * Invoke the given function, injecting dependencies. Return the result.
   *
   * @template T
   *
   * @param {FactoryFunction<T>} func
   * @param {InjectionContext} [context]
   * @param {LocalsMap} [locals]
   *
   * @return {T}
   */
  /**
   * Invoke the given function, injecting dependencies provided in
   * array notation. Return the result.
   *
   * @template T
   *
   * @param {ArrayFunc<T>} func function to be invoked
   * @param {InjectionContext} [context] context of the invocation
   * @param {LocalsMap} [locals] locals provided
   *
   * @return {T}
   */
  Diagram.prototype.invoke = function (func, context, locals) {
    return this._injector.invoke(func, context, locals);
  };

  /**
   * Destroys the diagram
   */
  Diagram.prototype.destroy = function () {
    this.get('eventBus').fire('diagram.destroy');
  };

  /**
   * Clear the diagram, removing all contents.
   */
  Diagram.prototype.clear = function () {
    this.get('eventBus').fire('diagram.clear');
  };

  var diRefs = new Refs({
    name: 'dmnElementRef',
    enumerable: true
  }, {
    name: 'di',
    configurable: true
  });
  function DRDTreeWalker(handler, options) {
    // list of elements to handle deferred to ensure
    // prerequisites are drawn
    var deferred = [];
    function visit(element) {
      var gfx = element.gfx;

      // avoid multiple rendering of elements
      if (gfx) {
        throw new Error('already rendered ' + element.id);
      }

      // call handler
      return handler.element(element);
    }
    function visitRoot(element) {
      return handler.root(element);
    }
    function visitIfDi(element) {
      try {
        var gfx = element.di && visit(element);
        return gfx;
      } catch (e) {
        logError(e.message, {
          element: element,
          error: e
        });
      }
    }

    // Semantic handling //////////////////////

    /**
     * Handle definitions and return the rendered diagram (if any)
     *
     * @param {ModdleElement} definitions to walk and import
     * @param {ModdleElement} [diagram] specific diagram to import and display
     *
     * @throws {Error} if no diagram to display could be found
     */
    function handleDefinitions(definitions, diagram) {
      // make sure we walk the correct dmnElement
      var dmnDI = definitions.dmnDI;
      if (!dmnDI) {
        throw new Error('no dmndi:DMNDI');
      }
      var diagrams = dmnDI.diagrams || [];
      if (diagram && diagrams.indexOf(diagram) === -1) {
        throw new Error('diagram not part of dmndi:DMNDI');
      }
      if (!diagram && diagrams && diagrams.length) {
        diagram = diagrams[0];
      }

      // no diagram -> nothing to import
      if (!diagram) {
        throw new Error('no diagram to display');
      }

      // assign current diagram to definitions so that it can accessed later
      definitions.di = diagram;

      // load DI from selected diagram only
      handleDiagram(diagram);
      visitRoot(definitions);
      handleDrgElements(definitions.get('drgElement'));
      handleArtifacts(definitions.get('artifact'));
      handleDeferred();
    }
    function handleDrgElements(elements) {
      forEach$3(elements, function (element) {
        visitIfDi(element);
        handleRequirements(element);
      });
    }
    function handleArtifacts(elements) {
      forEach$3(elements, function (element) {
        if (is$1(element, 'dmn:Association')) {
          handleAssociation(element);
        } else {
          visitIfDi(element);
        }
      });
    }

    /**
     * Defer association visit until all shapes are visited.
     *
     * @param {ModdleElement} element
     */
    function handleAssociation(element) {
      defer(function () {
        visitIfDi(element);
      });
    }

    /**
     * Defer requirements visiting until all shapes are visited.
     *
     * @param {ModdleElement} element
     */
    function handleRequirements(element) {
      forEach$3(['informationRequirement', 'knowledgeRequirement', 'authorityRequirement'], function (requirements) {
        forEach$3(element[requirements], function (requirement) {
          defer(function () {
            visitIfDi(requirement);
          });
        });
      });
    }

    // DI handling //////////////////////
    function handleDiagram(diagram) {
      forEach$3(diagram.diagramElements, handleDiagramElement);
    }
    function handleDiagramElement(diagramElement) {
      registerDi(diagramElement);
    }
    function registerDi(di) {
      var dmnElement = di.dmnElementRef;
      if (dmnElement) {
        if (dmnElement.di) {
          logError('multiple DI elements defined for element', {
            element: dmnElement
          });
        } else {
          diRefs.bind(dmnElement, 'di');
          dmnElement.di = di;
        }
      } else {
        logError('no DMN element referenced in element', {
          element: di
        });
      }
    }
    function defer(fn) {
      deferred.push(fn);
    }
    function handleDeferred() {
      forEach$3(deferred, function (d) {
        d();
      });
    }
    function logError(message, context) {
      handler.error(message, context);
    }

    // API //////////////////////

    return {
      handleDefinitions: handleDefinitions
    };
  }

  /**
   * Import the definitions into a diagram.
   *
   * Errors and warnings are reported through the specified callback.
   *
   * @param  {Drd} drd
   * @param  {ModdleElement} definitions
   * @param  {Function} done
   *         the callback, invoked with (err, [ warning ]) once the import is done
   */
  function importDRD(drd, definitions, done) {
    var importer = drd.get('drdImporter'),
      eventBus = drd.get('eventBus');
    var error,
      warnings = [];
    function render(definitions) {
      var visitor = {
        root: function (element) {
          return importer.root(element);
        },
        element: function (element, di) {
          return importer.add(element, di);
        },
        error: function (message, context) {
          warnings.push({
            message: message,
            context: context
          });
        }
      };
      var walker = new DRDTreeWalker(visitor);

      // import
      walker.handleDefinitions(definitions);
    }
    eventBus.fire('import.start', {
      definitions: definitions
    });
    try {
      render(definitions);
    } catch (e) {
      error = e;
    }
    eventBus.fire('import.done', {
      error: error,
      warnings: warnings
    });
    done(error, warnings);
  }

  var NO_OP = '$NO_OP';
  var ERROR_MSG = 'a runtime error occured! Use Inferno in development environment to find the error.';
  var isBrowser = !!(typeof window !== 'undefined' && window.document);
  var isArray = Array.isArray;
  function isStringOrNumber(o) {
    var type = typeof o;
    return type === 'string' || type === 'number';
  }
  function isNullOrUndef(o) {
    return isUndefined(o) || isNull$2(o);
  }
  function isInvalid(o) {
    return isNull$2(o) || o === false || isTrue(o) || isUndefined(o);
  }
  function isFunction(o) {
    return typeof o === 'function';
  }
  function isString(o) {
    return typeof o === 'string';
  }
  function isNumber$2(o) {
    return typeof o === 'number';
  }
  function isNull$2(o) {
    return o === null;
  }
  function isTrue(o) {
    return o === true;
  }
  function isUndefined(o) {
    return o === void 0;
  }
  function isObject(o) {
    return typeof o === 'object';
  }
  function throwError(message) {
    if (!message) {
      message = ERROR_MSG;
    }
    throw new Error("Inferno Error: " + message);
  }
  function warning(message) {
    // tslint:disable-next-line:no-console
    console.error(message);
  }
  function combineFrom(first, second) {
    var out = {};
    if (first) {
      for (var key in first) {
        out[key] = first[key];
      }
    }
    if (second) {
      for (var _key in second) {
        out[_key] = second[_key];
      }
    }
    return out;
  }
  function getTagName(input) {
    var tagName;
    if (isArray(input)) {
      var arrayText = input.length > 3 ? input.slice(0, 3).toString() + ',...' : input.toString();
      tagName = 'Array(' + arrayText + ')';
    } else if (isStringOrNumber(input)) {
      tagName = 'Text(' + input + ')';
    } else if (isInvalid(input)) {
      tagName = 'InvalidVNode(' + input + ')';
    } else {
      var flags = input.flags;
      if (flags & 481 /* VNodeFlags.Element */) {
        tagName = "<" + input.type + (input.className ? ' class="' + input.className + '"' : '') + ">";
      } else if (flags & 16 /* VNodeFlags.Text */) {
        tagName = "Text(" + input.children + ")";
      } else if (flags & 1024 /* VNodeFlags.Portal */) {
        tagName = "Portal*";
      } else {
        var type = input.type;
        // Fallback for IE
        var componentName = type.name || type.displayName || type.constructor.name || (type.toString().match(/^function\s*([^\s(]+)/) || [])[1];
        tagName = "<" + componentName + " />";
      }
    }
    return '>> ' + tagName + '\n';
  }
  function DEV_ValidateKeys(vNodeTree, forceKeyed) {
    var foundKeys = {};
    for (var i = 0, len = vNodeTree.length; i < len; i++) {
      var childNode = vNodeTree[i];
      if (isArray(childNode)) {
        return 'Encountered ARRAY in mount, array must be flattened, or normalize used. Location: \n' + getTagName(childNode);
      }
      if (isInvalid(childNode)) {
        if (forceKeyed) {
          return 'Encountered invalid node when preparing to keyed algorithm. Location: \n' + getTagName(childNode);
        } else if (Object.keys(foundKeys).length !== 0) {
          return 'Encountered invalid node with mixed keys. Location: \n' + getTagName(childNode);
        }
        continue;
      }
      if (typeof childNode === 'object') {
        childNode.isValidated = true;
      }
      // Key can be undefined, null too. But typescript complains for no real reason
      var key = childNode.key;
      if (!isNullOrUndef(key) && !isStringOrNumber(key)) {
        return 'Encountered child vNode where key property is not string or number. Location: \n' + getTagName(childNode);
      }
      var children = childNode.children;
      var childFlags = childNode.childFlags;
      if (!isInvalid(children)) {
        var val = void 0;
        if (childFlags & 12 /* ChildFlags.MultipleChildren */) {
          val = DEV_ValidateKeys(children, childNode.childFlags & 8 /* ChildFlags.HasKeyedChildren */);
        } else if (childFlags === 2 /* ChildFlags.HasVNodeChildren */) {
          val = DEV_ValidateKeys([children], childNode.childFlags & 8 /* ChildFlags.HasKeyedChildren */);
        }
        if (val) {
          val += getTagName(childNode);
          return val;
        }
      }
      if (forceKeyed && isNullOrUndef(key)) {
        return 'Encountered child without key during keyed algorithm. If this error points to Array make sure children is flat list. Location: \n' + getTagName(childNode);
      } else if (!forceKeyed && isNullOrUndef(key)) {
        if (Object.keys(foundKeys).length !== 0) {
          return 'Encountered children with key missing. Location: \n' + getTagName(childNode);
        }
        continue;
      }
      if (foundKeys[key]) {
        return 'Encountered two children with same key: {' + key + '}. Location: \n' + getTagName(childNode);
      }
      foundKeys[key] = true;
    }
  }
  function validateVNodeElementChildren(vNode) {
    {
      if (vNode.childFlags & 1 /* ChildFlags.HasInvalidChildren */) {
        return;
      }
      if (vNode.flags & 64 /* VNodeFlags.InputElement */) {
        throwError("input elements can't have children.");
      }
      if (vNode.flags & 128 /* VNodeFlags.TextareaElement */) {
        throwError("textarea elements can't have children.");
      }
      if (vNode.flags & 481 /* VNodeFlags.Element */) {
        var voidTypes = ['area', 'base', 'br', 'col', 'command', 'embed', 'hr', 'img', 'input', 'keygen', 'link', 'meta', 'param', 'source', 'track', 'wbr'];
        var tag = vNode.type.toLowerCase();
        if (tag === 'media') {
          throwError("media elements can't have children.");
        }
        var idx = voidTypes.indexOf(tag);
        if (idx !== -1) {
          throwError(voidTypes[idx] + " elements can't have children.");
        }
      }
    }
  }
  function validateKeys(vNode) {
    {
      // Checks if there is any key missing or duplicate keys
      if (vNode.isValidated === false && vNode.children && vNode.flags & 481 /* VNodeFlags.Element */) {
        var error = DEV_ValidateKeys(Array.isArray(vNode.children) ? vNode.children : [vNode.children], (vNode.childFlags & 8 /* ChildFlags.HasKeyedChildren */) > 0);
        if (error) {
          throwError(error + getTagName(vNode));
        }
      }
      vNode.isValidated = true;
    }
  }
  var keyPrefix = '$';
  function getVNode(childFlags, children, className, flags, key, props, ref, type) {
    {
      return {
        childFlags: childFlags,
        children: children,
        className: className,
        dom: null,
        flags: flags,
        isValidated: false,
        key: key === void 0 ? null : key,
        parentVNode: null,
        props: props === void 0 ? null : props,
        ref: ref === void 0 ? null : ref,
        type: type
      };
    }
  }
  function createVNode(flags, type, className, children, childFlags, props, key, ref) {
    {
      if (flags & 14 /* VNodeFlags.Component */) {
        throwError('Creating Component vNodes using createVNode is not allowed. Use Inferno.createComponentVNode method.');
      }
    }
    var childFlag = childFlags === void 0 ? 1 /* ChildFlags.HasInvalidChildren */ : childFlags;
    var vNode = getVNode(childFlag, children, className, flags, key, props, ref, type);
    if (childFlag === 0 /* ChildFlags.UnknownChildren */) {
      normalizeChildren(vNode, vNode.children);
    }
    {
      validateVNodeElementChildren(vNode);
    }
    return vNode;
  }
  function createComponentVNode(flags, type, props, key, ref) {
    {
      if (flags & 1 /* VNodeFlags.HtmlElement */) {
        throwError('Creating element vNodes using createComponentVNode is not allowed. Use Inferno.createVNode method.');
      }
    }
    if ((flags & 2 /* VNodeFlags.ComponentUnknown */) > 0) {
      flags = type.prototype && isFunction(type.prototype.render) ? 4 /* VNodeFlags.ComponentClass */ : 8 /* VNodeFlags.ComponentFunction */;
    }
    // set default props
    var defaultProps = type.defaultProps;
    if (!isNullOrUndef(defaultProps)) {
      if (!props) {
        props = {}; // Props can be referenced and modified at application level so always create new object
      }
      for (var prop in defaultProps) {
        if (isUndefined(props[prop])) {
          props[prop] = defaultProps[prop];
        }
      }
    }
    if ((flags & 8 /* VNodeFlags.ComponentFunction */) > 0) {
      var defaultHooks = type.defaultHooks;
      if (!isNullOrUndef(defaultHooks)) {
        if (!ref) {
          // As ref cannot be referenced from application level, we can use the same refs object
          ref = defaultHooks;
        } else {
          for (var _prop in defaultHooks) {
            if (isUndefined(ref[_prop])) {
              ref[_prop] = defaultHooks[_prop];
            }
          }
        }
      }
    }
    var vNode = getVNode(1 /* ChildFlags.HasInvalidChildren */, null, null, flags, key, props, ref, type);
    var optsVNode = options.createVNode;
    if (isFunction(optsVNode)) {
      optsVNode(vNode);
    }
    return vNode;
  }
  function createTextVNode(text, key) {
    return getVNode(1 /* ChildFlags.HasInvalidChildren */, isNullOrUndef(text) ? '' : text, null, 16 /* VNodeFlags.Text */, key, null, null, null);
  }
  function normalizeProps(vNode) {
    var props = vNode.props;
    if (props) {
      var flags = vNode.flags;
      if (flags & 481 /* VNodeFlags.Element */) {
        if (props.children !== void 0 && isNullOrUndef(vNode.children)) {
          normalizeChildren(vNode, props.children);
        }
        if (props.className !== void 0) {
          vNode.className = props.className || null;
          props.className = undefined;
        }
      }
      if (props.key !== void 0) {
        vNode.key = props.key;
        props.key = undefined;
      }
      if (props.ref !== void 0) {
        if (flags & 8 /* VNodeFlags.ComponentFunction */) {
          vNode.ref = combineFrom(vNode.ref, props.ref);
        } else {
          vNode.ref = props.ref;
        }
        props.ref = undefined;
      }
    }
    return vNode;
  }
  function directClone(vNodeToClone) {
    var newVNode;
    var flags = vNodeToClone.flags;
    if (flags & 14 /* VNodeFlags.Component */) {
      var props;
      var propsToClone = vNodeToClone.props;
      if (!isNull$2(propsToClone)) {
        props = {};
        for (var key in propsToClone) {
          props[key] = propsToClone[key];
        }
      }
      newVNode = createComponentVNode(flags, vNodeToClone.type, props, vNodeToClone.key, vNodeToClone.ref);
    } else if (flags & 481 /* VNodeFlags.Element */) {
      newVNode = createVNode(flags, vNodeToClone.type, vNodeToClone.className, vNodeToClone.children, vNodeToClone.childFlags, vNodeToClone.props, vNodeToClone.key, vNodeToClone.ref);
    } else if (flags & 16 /* VNodeFlags.Text */) {
      newVNode = createTextVNode(vNodeToClone.children, vNodeToClone.key);
    } else if (flags & 1024 /* VNodeFlags.Portal */) {
      newVNode = vNodeToClone;
    }
    return newVNode;
  }
  function createVoidVNode() {
    return createTextVNode('', null);
  }
  function _normalizeVNodes(nodes, result, index, currentKey) {
    for (var len = nodes.length; index < len; index++) {
      var n = nodes[index];
      if (!isInvalid(n)) {
        var newKey = currentKey + keyPrefix + index;
        if (isArray(n)) {
          _normalizeVNodes(n, result, 0, newKey);
        } else {
          if (isStringOrNumber(n)) {
            n = createTextVNode(n, newKey);
          } else {
            var oldKey = n.key;
            var isPrefixedKey = isString(oldKey) && oldKey[0] === keyPrefix;
            if (!isNull$2(n.dom) || isPrefixedKey) {
              n = directClone(n);
            }
            if (isNull$2(oldKey) || isPrefixedKey) {
              n.key = newKey;
            } else {
              n.key = currentKey + oldKey;
            }
          }
          result.push(n);
        }
      }
    }
  }
  function normalizeChildren(vNode, children) {
    var newChildren;
    var newChildFlags = 1 /* ChildFlags.HasInvalidChildren */;
    // Don't change children to match strict equal (===) true in patching
    if (isInvalid(children)) {
      newChildren = children;
    } else if (isString(children)) {
      newChildFlags = 2 /* ChildFlags.HasVNodeChildren */;
      newChildren = createTextVNode(children);
    } else if (isNumber$2(children)) {
      newChildFlags = 2 /* ChildFlags.HasVNodeChildren */;
      newChildren = createTextVNode(children + '');
    } else if (isArray(children)) {
      var len = children.length;
      if (len === 0) {
        newChildren = null;
        newChildFlags = 1 /* ChildFlags.HasInvalidChildren */;
      } else {
        // we assign $ which basically means we've flagged this array for future note
        // if it comes back again, we need to clone it, as people are using it
        // in an immutable way
        // tslint:disable-next-line
        if (Object.isFrozen(children) || children['$'] === true) {
          children = children.slice();
        }
        newChildFlags = 8 /* ChildFlags.HasKeyedChildren */;
        for (var i = 0; i < len; i++) {
          var n = children[i];
          if (isInvalid(n) || isArray(n)) {
            newChildren = newChildren || children.slice(0, i);
            _normalizeVNodes(children, newChildren, i, '');
            break;
          } else if (isStringOrNumber(n)) {
            newChildren = newChildren || children.slice(0, i);
            newChildren.push(createTextVNode(n, keyPrefix + i));
          } else {
            var key = n.key;
            var isNullDom = isNull$2(n.dom);
            var isNullKey = isNull$2(key);
            var isPrefixed = !isNullKey && isString(key) && key[0] === keyPrefix;
            if (!isNullDom || isNullKey || isPrefixed) {
              newChildren = newChildren || children.slice(0, i);
              if (!isNullDom || isPrefixed) {
                n = directClone(n);
              }
              if (isNullKey || isPrefixed) {
                n.key = keyPrefix + i;
              }
              newChildren.push(n);
            } else if (newChildren) {
              newChildren.push(n);
            }
          }
        }
        newChildren = newChildren || children;
        newChildren.$ = true;
      }
    } else {
      newChildren = children;
      if (!isNull$2(children.dom)) {
        newChildren = directClone(children);
      }
      newChildFlags = 2 /* ChildFlags.HasVNodeChildren */;
    }
    vNode.children = newChildren;
    vNode.childFlags = newChildFlags;
    {
      validateVNodeElementChildren(vNode);
    }
    return vNode;
  }
  var options = {
    afterRender: null,
    beforeRender: null,
    createVNode: null,
    renderComplete: null
  };
  var xlinkNS = 'http://www.w3.org/1999/xlink';
  var xmlNS = 'http://www.w3.org/XML/1998/namespace';
  var svgNS = 'http://www.w3.org/2000/svg';
  var namespaces = {
    'xlink:actuate': xlinkNS,
    'xlink:arcrole': xlinkNS,
    'xlink:href': xlinkNS,
    'xlink:role': xlinkNS,
    'xlink:show': xlinkNS,
    'xlink:title': xlinkNS,
    'xlink:type': xlinkNS,
    'xml:base': xmlNS,
    'xml:lang': xmlNS,
    'xml:space': xmlNS
  };

  // We need EMPTY_OBJ defined in one place.
  // Its used for comparison so we cant inline it into shared
  var EMPTY_OBJ = {};
  var LIFECYCLE = [];
  {
    Object.freeze(EMPTY_OBJ);
  }
  function appendChild(parentDom, dom) {
    parentDom.appendChild(dom);
  }
  function insertOrAppend(parentDom, newNode, nextNode) {
    if (isNullOrUndef(nextNode)) {
      appendChild(parentDom, newNode);
    } else {
      parentDom.insertBefore(newNode, nextNode);
    }
  }
  function documentCreateElement(tag, isSVG) {
    if (isSVG) {
      return document.createElementNS(svgNS, tag);
    }
    return document.createElement(tag);
  }
  function replaceChild(parentDom, newDom, lastDom) {
    parentDom.replaceChild(newDom, lastDom);
  }
  function removeChild(parentDom, dom) {
    parentDom.removeChild(dom);
  }
  function callAll(arrayFn) {
    var listener;
    while ((listener = arrayFn.shift()) !== undefined) {
      listener();
    }
  }
  var attachedEventCounts = {};
  var attachedEvents = {};
  function handleEvent(name, nextEvent, dom) {
    var eventsLeft = attachedEventCounts[name];
    var eventsObject = dom.$EV;
    if (nextEvent) {
      if (!eventsLeft) {
        attachedEvents[name] = attachEventToDocument(name);
        attachedEventCounts[name] = 0;
      }
      if (!eventsObject) {
        eventsObject = dom.$EV = {};
      }
      if (!eventsObject[name]) {
        attachedEventCounts[name]++;
      }
      eventsObject[name] = nextEvent;
    } else if (eventsObject && eventsObject[name]) {
      attachedEventCounts[name]--;
      if (eventsLeft === 1) {
        document.removeEventListener(normalizeEventName(name), attachedEvents[name]);
        attachedEvents[name] = null;
      }
      eventsObject[name] = nextEvent;
    }
  }
  // When browsers fully support event.composedPath we could loop it through instead of using parentNode property
  function getTargetNode(event) {
    return isFunction(event.composedPath) ? event.composedPath()[0] : event.target;
  }
  function dispatchEvents(event, isClick, name, eventData) {
    var dom = getTargetNode(event);
    while (!isNull$2(dom)) {
      // Html Nodes can be nested fe: span inside button in that scenario browser does not handle disabled attribute on parent,
      // because the event listener is on document.body
      // Don't process clicks on disabled elements
      if (isClick && dom.disabled) {
        return;
      }
      var eventsObject = dom.$EV;
      if (eventsObject) {
        var currentEvent = eventsObject[name];
        if (currentEvent) {
          // linkEvent object
          eventData.dom = dom;
          if (currentEvent.event) {
            currentEvent.event(currentEvent.data, event);
          } else {
            currentEvent(event);
          }
          if (event.cancelBubble) {
            return;
          }
        }
      }
      dom = dom.parentNode;
    }
  }
  function normalizeEventName(name) {
    return name.substr(2).toLowerCase();
  }
  function stopPropagation$3() {
    this.cancelBubble = true;
    if (!this.immediatePropagationStopped) {
      this.stopImmediatePropagation();
    }
  }
  function attachEventToDocument(name) {
    var docEvent = function docEvent(event) {
      var type = event.type;
      var isClick = type === 'click' || type === 'dblclick';
      if (isClick && event.button !== 0) {
        // Firefox incorrectly triggers click event for mid/right mouse buttons.
        // This bug has been active for 12 years.
        // https://bugzilla.mozilla.org/show_bug.cgi?id=184051
        event.stopPropagation();
        return false;
      }
      event.stopPropagation = stopPropagation$3;
      // Event data needs to be object to save reference to currentTarget getter
      var eventData = {
        dom: document
      };
      Object.defineProperty(event, 'currentTarget', {
        configurable: true,
        get: function get() {
          return eventData.dom;
        }
      });
      dispatchEvents(event, isClick, name, eventData);
      return;
    };
    document.addEventListener(normalizeEventName(name), docEvent);
    return docEvent;
  }
  function isSameInnerHTML(dom, innerHTML) {
    var tempdom = document.createElement('i');
    tempdom.innerHTML = innerHTML;
    return tempdom.innerHTML === dom.innerHTML;
  }
  function isSamePropsInnerHTML(dom, props) {
    return Boolean(props && props.dangerouslySetInnerHTML && props.dangerouslySetInnerHTML.__html && isSameInnerHTML(dom, props.dangerouslySetInnerHTML.__html));
  }
  function triggerEventListener(props, methodName, e) {
    if (props[methodName]) {
      var listener = props[methodName];
      if (listener.event) {
        listener.event(listener.data, e);
      } else {
        listener(e);
      }
    } else {
      var nativeListenerName = methodName.toLowerCase();
      if (props[nativeListenerName]) {
        props[nativeListenerName](e);
      }
    }
  }
  function createWrappedFunction(methodName, applyValue) {
    var fnMethod = function fnMethod(e) {
      e.stopPropagation();
      var vNode = this.$V;
      // If vNode is gone by the time event fires, no-op
      if (!vNode) {
        return;
      }
      var props = vNode.props || EMPTY_OBJ;
      var dom = vNode.dom;
      if (isString(methodName)) {
        triggerEventListener(props, methodName, e);
      } else {
        for (var i = 0; i < methodName.length; i++) {
          triggerEventListener(props, methodName[i], e);
        }
      }
      if (isFunction(applyValue)) {
        var newVNode = this.$V;
        var newProps = newVNode.props || EMPTY_OBJ;
        applyValue(newProps, dom, false, newVNode);
      }
    };
    Object.defineProperty(fnMethod, 'wrapped', {
      configurable: false,
      enumerable: false,
      value: true,
      writable: false
    });
    return fnMethod;
  }
  function isCheckedType(type) {
    return type === 'checkbox' || type === 'radio';
  }
  var onTextInputChange = createWrappedFunction('onInput', applyValueInput);
  var wrappedOnChange$1 = createWrappedFunction(['onClick', 'onChange'], applyValueInput);
  /* tslint:disable-next-line:no-empty */
  function emptywrapper(event) {
    event.stopPropagation();
  }
  emptywrapper.wrapped = true;
  function inputEvents(dom, nextPropsOrEmpty) {
    if (isCheckedType(nextPropsOrEmpty.type)) {
      dom.onchange = wrappedOnChange$1;
      dom.onclick = emptywrapper;
    } else {
      dom.oninput = onTextInputChange;
    }
  }
  function applyValueInput(nextPropsOrEmpty, dom) {
    var type = nextPropsOrEmpty.type;
    var value = nextPropsOrEmpty.value;
    var checked = nextPropsOrEmpty.checked;
    var multiple = nextPropsOrEmpty.multiple;
    var defaultValue = nextPropsOrEmpty.defaultValue;
    var hasValue = !isNullOrUndef(value);
    if (type && type !== dom.type) {
      dom.setAttribute('type', type);
    }
    if (!isNullOrUndef(multiple) && multiple !== dom.multiple) {
      dom.multiple = multiple;
    }
    if (!isNullOrUndef(defaultValue) && !hasValue) {
      dom.defaultValue = defaultValue + '';
    }
    if (isCheckedType(type)) {
      if (hasValue) {
        dom.value = value;
      }
      if (!isNullOrUndef(checked)) {
        dom.checked = checked;
      }
    } else {
      if (hasValue && dom.value !== value) {
        dom.defaultValue = value;
        dom.value = value;
      } else if (!isNullOrUndef(checked)) {
        dom.checked = checked;
      }
    }
  }
  function updateChildOptionGroup(vNode, value) {
    var type = vNode.type;
    if (type === 'optgroup') {
      var children = vNode.children;
      var childFlags = vNode.childFlags;
      if (childFlags & 12 /* ChildFlags.MultipleChildren */) {
        for (var i = 0, len = children.length; i < len; i++) {
          updateChildOption(children[i], value);
        }
      } else if (childFlags === 2 /* ChildFlags.HasVNodeChildren */) {
        updateChildOption(children, value);
      }
    } else {
      updateChildOption(vNode, value);
    }
  }
  function updateChildOption(vNode, value) {
    var props = vNode.props || EMPTY_OBJ;
    var dom = vNode.dom;
    // we do this as multiple may have changed
    dom.value = props.value;
    if (isArray(value) && value.indexOf(props.value) !== -1 || props.value === value) {
      dom.selected = true;
    } else if (!isNullOrUndef(value) || !isNullOrUndef(props.selected)) {
      dom.selected = props.selected || false;
    }
  }
  var onSelectChange = createWrappedFunction('onChange', applyValueSelect);
  function selectEvents(dom) {
    dom.onchange = onSelectChange;
  }
  function applyValueSelect(nextPropsOrEmpty, dom, mounting, vNode) {
    var multiplePropInBoolean = Boolean(nextPropsOrEmpty.multiple);
    if (!isNullOrUndef(nextPropsOrEmpty.multiple) && multiplePropInBoolean !== dom.multiple) {
      dom.multiple = multiplePropInBoolean;
    }
    var childFlags = vNode.childFlags;
    if ((childFlags & 1 /* ChildFlags.HasInvalidChildren */) === 0) {
      var children = vNode.children;
      var value = nextPropsOrEmpty.value;
      if (mounting && isNullOrUndef(value)) {
        value = nextPropsOrEmpty.defaultValue;
      }
      if (childFlags & 12 /* ChildFlags.MultipleChildren */) {
        for (var i = 0, len = children.length; i < len; i++) {
          updateChildOptionGroup(children[i], value);
        }
      } else if (childFlags === 2 /* ChildFlags.HasVNodeChildren */) {
        updateChildOptionGroup(children, value);
      }
    }
  }
  var onTextareaInputChange = createWrappedFunction('onInput', applyValueTextArea);
  var wrappedOnChange = createWrappedFunction('onChange');
  function textAreaEvents(dom, nextPropsOrEmpty) {
    dom.oninput = onTextareaInputChange;
    if (nextPropsOrEmpty.onChange) {
      dom.onchange = wrappedOnChange;
    }
  }
  function applyValueTextArea(nextPropsOrEmpty, dom, mounting) {
    var value = nextPropsOrEmpty.value;
    var domValue = dom.value;
    if (isNullOrUndef(value)) {
      if (mounting) {
        var defaultValue = nextPropsOrEmpty.defaultValue;
        if (!isNullOrUndef(defaultValue) && defaultValue !== domValue) {
          dom.defaultValue = defaultValue;
          dom.value = defaultValue;
        }
      }
    } else if (domValue !== value) {
      /* There is value so keep it controlled */
      dom.defaultValue = value;
      dom.value = value;
    }
  }

  /**
   * There is currently no support for switching same input between controlled and nonControlled
   * If that ever becomes a real issue, then re design controlled elements
   * Currently user must choose either controlled or non-controlled and stick with that
   */
  function processElement(flags, vNode, dom, nextPropsOrEmpty, mounting, isControlled) {
    if (flags & 64 /* VNodeFlags.InputElement */) {
      applyValueInput(nextPropsOrEmpty, dom);
    } else if (flags & 256 /* VNodeFlags.SelectElement */) {
      applyValueSelect(nextPropsOrEmpty, dom, mounting, vNode);
    } else if (flags & 128 /* VNodeFlags.TextareaElement */) {
      applyValueTextArea(nextPropsOrEmpty, dom, mounting);
    }
    if (isControlled) {
      dom.$V = vNode;
    }
  }
  function addFormElementEventHandlers(flags, dom, nextPropsOrEmpty) {
    if (flags & 64 /* VNodeFlags.InputElement */) {
      inputEvents(dom, nextPropsOrEmpty);
    } else if (flags & 256 /* VNodeFlags.SelectElement */) {
      selectEvents(dom);
    } else if (flags & 128 /* VNodeFlags.TextareaElement */) {
      textAreaEvents(dom, nextPropsOrEmpty);
    }
  }
  function isControlledFormElement(nextPropsOrEmpty) {
    return nextPropsOrEmpty.type && isCheckedType(nextPropsOrEmpty.type) ? !isNullOrUndef(nextPropsOrEmpty.checked) : !isNullOrUndef(nextPropsOrEmpty.value);
  }
  function remove$1(vNode, parentDom) {
    unmount(vNode);
    if (parentDom && vNode.dom) {
      removeChild(parentDom, vNode.dom);
      // Let carbage collector free memory
      vNode.dom = null;
    }
  }
  function unmount(vNode) {
    var flags = vNode.flags;
    if (flags & 481 /* VNodeFlags.Element */) {
      var ref = vNode.ref;
      var props = vNode.props;
      if (isFunction(ref)) {
        ref(null);
      }
      var children = vNode.children;
      var childFlags = vNode.childFlags;
      if (childFlags & 12 /* ChildFlags.MultipleChildren */) {
        unmountAllChildren(children);
      } else if (childFlags === 2 /* ChildFlags.HasVNodeChildren */) {
        unmount(children);
      }
      if (!isNull$2(props)) {
        for (var name in props) {
          switch (name) {
            case 'onClick':
            case 'onDblClick':
            case 'onFocusIn':
            case 'onFocusOut':
            case 'onKeyDown':
            case 'onKeyPress':
            case 'onKeyUp':
            case 'onMouseDown':
            case 'onMouseMove':
            case 'onMouseUp':
            case 'onSubmit':
            case 'onTouchEnd':
            case 'onTouchMove':
            case 'onTouchStart':
              handleEvent(name, null, vNode.dom);
              break;
          }
        }
      }
    } else {
      var _children = vNode.children;
      // Safe guard for crashed VNode
      if (_children) {
        if (flags & 14 /* VNodeFlags.Component */) {
          var _ref = vNode.ref;
          if (flags & 4 /* VNodeFlags.ComponentClass */) {
            if (isFunction(_children.componentWillUnmount)) {
              _children.componentWillUnmount();
            }
            if (isFunction(_ref)) {
              _ref(null);
            }
            _children.$UN = true;
            if (_children.$LI) {
              unmount(_children.$LI);
            }
          } else {
            if (!isNullOrUndef(_ref) && isFunction(_ref.onComponentWillUnmount)) {
              _ref.onComponentWillUnmount(vNode.dom, vNode.props || EMPTY_OBJ);
            }
            unmount(_children);
          }
        } else if (flags & 1024 /* VNodeFlags.Portal */) {
          remove$1(_children, vNode.type);
        }
      }
    }
  }
  function unmountAllChildren(children) {
    for (var i = 0, len = children.length; i < len; i++) {
      unmount(children[i]);
    }
  }
  function removeAllChildren(dom, children) {
    unmountAllChildren(children);
    dom.textContent = '';
  }
  function createLinkEvent(linkEvent, nextValue) {
    return function (e) {
      linkEvent(nextValue.data, e);
    };
  }
  function patchEvent(name, nextValue, dom) {
    var nameLowerCase = name.toLowerCase();
    if (!isFunction(nextValue) && !isNullOrUndef(nextValue)) {
      var linkEvent = nextValue.event;
      if (linkEvent && isFunction(linkEvent)) {
        dom[nameLowerCase] = createLinkEvent(linkEvent, nextValue);
      } else {
        // Development warning
        {
          throwError("an event on a VNode \"" + name + "\". was not a function or a valid linkEvent.");
        }
      }
    } else {
      var domEvent = dom[nameLowerCase];
      // if the function is wrapped, that means it's been controlled by a wrapper
      if (!domEvent || !domEvent.wrapped) {
        dom[nameLowerCase] = nextValue;
      }
    }
  }
  function getNumberStyleValue(style, value) {
    switch (style) {
      case 'animationIterationCount':
      case 'borderImageOutset':
      case 'borderImageSlice':
      case 'borderImageWidth':
      case 'boxFlex':
      case 'boxFlexGroup':
      case 'boxOrdinalGroup':
      case 'columnCount':
      case 'fillOpacity':
      case 'flex':
      case 'flexGrow':
      case 'flexNegative':
      case 'flexOrder':
      case 'flexPositive':
      case 'flexShrink':
      case 'floodOpacity':
      case 'fontWeight':
      case 'gridColumn':
      case 'gridRow':
      case 'lineClamp':
      case 'lineHeight':
      case 'opacity':
      case 'order':
      case 'orphans':
      case 'stopOpacity':
      case 'strokeDasharray':
      case 'strokeDashoffset':
      case 'strokeMiterlimit':
      case 'strokeOpacity':
      case 'strokeWidth':
      case 'tabSize':
      case 'widows':
      case 'zIndex':
      case 'zoom':
        return value;
      default:
        return value + 'px';
    }
  }
  // We are assuming here that we come from patchProp routine
  // -nextAttrValue cannot be null or undefined
  function patchStyle(lastAttrValue, nextAttrValue, dom) {
    var domStyle = dom.style;
    var style;
    var value;
    if (isString(nextAttrValue)) {
      domStyle.cssText = nextAttrValue;
      return;
    }
    if (!isNullOrUndef(lastAttrValue) && !isString(lastAttrValue)) {
      for (style in nextAttrValue) {
        // do not add a hasOwnProperty check here, it affects performance
        value = nextAttrValue[style];
        if (value !== lastAttrValue[style]) {
          domStyle[style] = isNumber$2(value) ? getNumberStyleValue(style, value) : value;
        }
      }
      for (style in lastAttrValue) {
        if (isNullOrUndef(nextAttrValue[style])) {
          domStyle[style] = '';
        }
      }
    } else {
      for (style in nextAttrValue) {
        value = nextAttrValue[style];
        domStyle[style] = isNumber$2(value) ? getNumberStyleValue(style, value) : value;
      }
    }
  }
  function patchProp(prop, lastValue, nextValue, dom, isSVG, hasControlledValue, lastVNode) {
    switch (prop) {
      case 'onClick':
      case 'onDblClick':
      case 'onFocusIn':
      case 'onFocusOut':
      case 'onKeyDown':
      case 'onKeyPress':
      case 'onKeyUp':
      case 'onMouseDown':
      case 'onMouseMove':
      case 'onMouseUp':
      case 'onSubmit':
      case 'onTouchEnd':
      case 'onTouchMove':
      case 'onTouchStart':
        handleEvent(prop, nextValue, dom);
        break;
      case 'children':
      case 'childrenType':
      case 'className':
      case 'defaultValue':
      case 'key':
      case 'multiple':
      case 'ref':
        break;
      case 'autoFocus':
        dom.autofocus = !!nextValue;
        break;
      case 'allowfullscreen':
      case 'autoplay':
      case 'capture':
      case 'checked':
      case 'controls':
      case 'default':
      case 'disabled':
      case 'hidden':
      case 'indeterminate':
      case 'loop':
      case 'muted':
      case 'novalidate':
      case 'open':
      case 'readOnly':
      case 'required':
      case 'reversed':
      case 'scoped':
      case 'seamless':
      case 'selected':
        dom[prop] = !!nextValue;
        break;
      case 'defaultChecked':
      case 'value':
      case 'volume':
        if (hasControlledValue && prop === 'value') {
          return;
        }
        var value = isNullOrUndef(nextValue) ? '' : nextValue;
        if (dom[prop] !== value) {
          dom[prop] = value;
        }
        break;
      case 'dangerouslySetInnerHTML':
        var lastHtml = lastValue && lastValue.__html || '';
        var nextHtml = nextValue && nextValue.__html || '';
        if (lastHtml !== nextHtml) {
          if (!isNullOrUndef(nextHtml) && !isSameInnerHTML(dom, nextHtml)) {
            if (!isNull$2(lastVNode)) {
              if (lastVNode.childFlags & 12 /* ChildFlags.MultipleChildren */) {
                unmountAllChildren(lastVNode.children);
              } else if (lastVNode.childFlags === 2 /* ChildFlags.HasVNodeChildren */) {
                unmount(lastVNode.children);
              }
              lastVNode.children = null;
              lastVNode.childFlags = 1 /* ChildFlags.HasInvalidChildren */;
            }
            dom.innerHTML = nextHtml;
          }
        }
        break;
      default:
        if (prop[0] === 'o' && prop[1] === 'n') {
          patchEvent(prop, nextValue, dom);
        } else if (isNullOrUndef(nextValue)) {
          dom.removeAttribute(prop);
        } else if (prop === 'style') {
          patchStyle(lastValue, nextValue, dom);
        } else if (isSVG && namespaces[prop]) {
          // We optimize for isSVG being false
          // If we end up in this path we can read property again
          dom.setAttributeNS(namespaces[prop], prop, nextValue);
        } else {
          dom.setAttribute(prop, nextValue);
        }
        break;
    }
  }
  function mountProps(vNode, flags, props, dom, isSVG) {
    var hasControlledValue = false;
    var isFormElement = (flags & 448 /* VNodeFlags.FormElement */) > 0;
    if (isFormElement) {
      hasControlledValue = isControlledFormElement(props);
      if (hasControlledValue) {
        addFormElementEventHandlers(flags, dom, props);
      }
    }
    for (var prop in props) {
      // do not add a hasOwnProperty check here, it affects performance
      patchProp(prop, null, props[prop], dom, isSVG, hasControlledValue, null);
    }
    if (isFormElement) {
      processElement(flags, vNode, dom, props, true, hasControlledValue);
    }
  }
  function createClassComponentInstance(vNode, Component, props, context) {
    var instance = new Component(props, context);
    vNode.children = instance;
    instance.$V = vNode;
    instance.$BS = false;
    instance.context = context;
    if (instance.props === EMPTY_OBJ) {
      instance.props = props;
    }
    instance.$UN = false;
    if (isFunction(instance.componentWillMount)) {
      instance.$BR = true;
      instance.componentWillMount();
      if (instance.$PSS) {
        var state = instance.state;
        var pending = instance.$PS;
        if (isNull$2(state)) {
          instance.state = pending;
        } else {
          for (var key in pending) {
            state[key] = pending[key];
          }
        }
        instance.$PSS = false;
        instance.$PS = null;
      }
      instance.$BR = false;
    }
    if (isFunction(options.beforeRender)) {
      options.beforeRender(instance);
    }
    var input = handleComponentInput(instance.render(props, instance.state, context), vNode);
    var childContext;
    if (isFunction(instance.getChildContext)) {
      childContext = instance.getChildContext();
    }
    if (isNullOrUndef(childContext)) {
      instance.$CX = context;
    } else {
      instance.$CX = combineFrom(context, childContext);
    }
    if (isFunction(options.afterRender)) {
      options.afterRender(instance);
    }
    instance.$LI = input;
    return instance;
  }
  function handleComponentInput(input, componentVNode) {
    // Development validation
    {
      if (isArray(input)) {
        throwError('a valid Inferno VNode (or null) must be returned from a component render. You may have returned an array or an invalid object.');
      }
    }
    if (isInvalid(input)) {
      input = createVoidVNode();
    } else if (isStringOrNumber(input)) {
      input = createTextVNode(input, null);
    } else {
      if (input.dom) {
        input = directClone(input);
      }
      if (input.flags & 14 /* VNodeFlags.Component */) {
        // if we have an input that is also a component, we run into a tricky situation
        // where the root vNode needs to always have the correct DOM entry
        // we can optimise this in the future, but this gets us out of a lot of issues
        input.parentVNode = componentVNode;
      }
    }
    return input;
  }
  function mount(vNode, parentDom, context, isSVG) {
    var flags = vNode.flags;
    if (flags & 481 /* VNodeFlags.Element */) {
      return mountElement(vNode, parentDom, context, isSVG);
    }
    if (flags & 14 /* VNodeFlags.Component */) {
      return mountComponent(vNode, parentDom, context, isSVG, (flags & 4 /* VNodeFlags.ComponentClass */) > 0);
    }
    if (flags & 512 /* VNodeFlags.Void */ || flags & 16 /* VNodeFlags.Text */) {
      return mountText(vNode, parentDom);
    }
    if (flags & 1024 /* VNodeFlags.Portal */) {
      mount(vNode.children, vNode.type, context, false);
      return vNode.dom = mountText(createVoidVNode(), parentDom);
    }
    // Development validation, in production we don't need to throw because it crashes anyway
    {
      if (typeof vNode === 'object') {
        throwError("mount() received an object that's not a valid VNode, you should stringify it first, fix createVNode flags or call normalizeChildren. Object: \"" + JSON.stringify(vNode) + "\".");
      } else {
        throwError("mount() expects a valid VNode, instead it received an object with the type \"" + typeof vNode + "\".");
      }
    }
  }
  function mountText(vNode, parentDom) {
    var dom = vNode.dom = document.createTextNode(vNode.children);
    if (!isNull$2(parentDom)) {
      appendChild(parentDom, dom);
    }
    return dom;
  }
  function mountElement(vNode, parentDom, context, isSVG) {
    var flags = vNode.flags;
    var children = vNode.children;
    var props = vNode.props;
    var className = vNode.className;
    var ref = vNode.ref;
    var childFlags = vNode.childFlags;
    isSVG = isSVG || (flags & 32 /* VNodeFlags.SvgElement */) > 0;
    var dom = documentCreateElement(vNode.type, isSVG);
    vNode.dom = dom;
    if (!isNullOrUndef(className) && className !== '') {
      if (isSVG) {
        dom.setAttribute('class', className);
      } else {
        dom.className = className;
      }
    }
    {
      validateKeys(vNode);
    }
    if (!isNull$2(parentDom)) {
      appendChild(parentDom, dom);
    }
    if ((childFlags & 1 /* ChildFlags.HasInvalidChildren */) === 0) {
      var childrenIsSVG = isSVG === true && vNode.type !== 'foreignObject';
      if (childFlags === 2 /* ChildFlags.HasVNodeChildren */) {
        mount(children, dom, context, childrenIsSVG);
      } else if (childFlags & 12 /* ChildFlags.MultipleChildren */) {
        mountArrayChildren(children, dom, context, childrenIsSVG);
      }
    }
    if (!isNull$2(props)) {
      mountProps(vNode, flags, props, dom, isSVG);
    }
    {
      if (isString(ref)) {
        throwError('string "refs" are not supported in Inferno 1.0. Use callback "refs" instead.');
      }
    }
    if (isFunction(ref)) {
      mountRef(dom, ref);
    }
    return dom;
  }
  function mountArrayChildren(children, dom, context, isSVG) {
    for (var i = 0, len = children.length; i < len; i++) {
      var child = children[i];
      if (!isNull$2(child.dom)) {
        children[i] = child = directClone(child);
      }
      mount(child, dom, context, isSVG);
    }
  }
  function mountComponent(vNode, parentDom, context, isSVG, isClass) {
    var dom;
    var type = vNode.type;
    var props = vNode.props || EMPTY_OBJ;
    var ref = vNode.ref;
    if (isClass) {
      var instance = createClassComponentInstance(vNode, type, props, context);
      vNode.dom = dom = mount(instance.$LI, null, instance.$CX, isSVG);
      mountClassComponentCallbacks(vNode, ref, instance);
      instance.$UPD = false;
    } else {
      var input = handleComponentInput(type(props, context), vNode);
      vNode.children = input;
      vNode.dom = dom = mount(input, null, context, isSVG);
      mountFunctionalComponentCallbacks(props, ref, dom);
    }
    if (!isNull$2(parentDom)) {
      appendChild(parentDom, dom);
    }
    return dom;
  }
  function createClassMountCallback(instance) {
    return function () {
      instance.$UPD = true;
      instance.componentDidMount();
      instance.$UPD = false;
    };
  }
  function mountClassComponentCallbacks(vNode, ref, instance) {
    if (isFunction(ref)) {
      ref(instance);
    } else {
      {
        if (isStringOrNumber(ref)) {
          throwError('string "refs" are not supported in Inferno 1.0. Use callback "refs" instead.');
        } else if (!isNullOrUndef(ref) && isObject(ref) && vNode.flags & 4 /* VNodeFlags.ComponentClass */) {
          throwError('functional component lifecycle events are not supported on ES2015 class components.');
        }
      }
    }
    if (isFunction(instance.componentDidMount)) {
      LIFECYCLE.push(createClassMountCallback(instance));
    }
  }
  function createOnMountCallback(ref, dom, props) {
    return function () {
      return ref.onComponentDidMount(dom, props);
    };
  }
  function mountFunctionalComponentCallbacks(props, ref, dom) {
    if (!isNullOrUndef(ref)) {
      if (isFunction(ref.onComponentWillMount)) {
        ref.onComponentWillMount(props);
      }
      if (isFunction(ref.onComponentDidMount)) {
        LIFECYCLE.push(createOnMountCallback(ref, dom, props));
      }
    }
  }
  function mountRef(dom, value) {
    LIFECYCLE.push(function () {
      return value(dom);
    });
  }
  function hydrateComponent(vNode, dom, context, isSVG, isClass) {
    var type = vNode.type;
    var ref = vNode.ref;
    var props = vNode.props || EMPTY_OBJ;
    if (isClass) {
      var instance = createClassComponentInstance(vNode, type, props, context);
      var input = instance.$LI;
      hydrateVNode(input, dom, instance.$CX, isSVG);
      vNode.dom = input.dom;
      mountClassComponentCallbacks(vNode, ref, instance);
      instance.$UPD = false; // Mount finished allow going sync
    } else {
      var _input = handleComponentInput(type(props, context), vNode);
      hydrateVNode(_input, dom, context, isSVG);
      vNode.children = _input;
      vNode.dom = _input.dom;
      mountFunctionalComponentCallbacks(props, ref, dom);
    }
  }
  function hydrateElement(vNode, dom, context, isSVG) {
    var children = vNode.children;
    var props = vNode.props;
    var className = vNode.className;
    var flags = vNode.flags;
    var ref = vNode.ref;
    isSVG = isSVG || (flags & 32 /* VNodeFlags.SvgElement */) > 0;
    if (dom.nodeType !== 1 || dom.tagName.toLowerCase() !== vNode.type) {
      {
        warning("Inferno hydration: Server-side markup doesn't match client-side markup or Initial render target is not empty");
      }
      var newDom = mountElement(vNode, null, context, isSVG);
      vNode.dom = newDom;
      replaceChild(dom.parentNode, newDom, dom);
    } else {
      vNode.dom = dom;
      var childNode = dom.firstChild;
      var childFlags = vNode.childFlags;
      if ((childFlags & 1 /* ChildFlags.HasInvalidChildren */) === 0) {
        var nextSibling = null;
        while (childNode) {
          nextSibling = childNode.nextSibling;
          if (childNode.nodeType === 8) {
            if (childNode.data === '!') {
              dom.replaceChild(document.createTextNode(''), childNode);
            } else {
              dom.removeChild(childNode);
            }
          }
          childNode = nextSibling;
        }
        childNode = dom.firstChild;
        if (childFlags === 2 /* ChildFlags.HasVNodeChildren */) {
          if (isNull$2(childNode)) {
            mount(children, dom, context, isSVG);
          } else {
            nextSibling = childNode.nextSibling;
            hydrateVNode(children, childNode, context, isSVG);
            childNode = nextSibling;
          }
        } else if (childFlags & 12 /* ChildFlags.MultipleChildren */) {
          for (var i = 0, len = children.length; i < len; i++) {
            var child = children[i];
            if (isNull$2(childNode)) {
              mount(child, dom, context, isSVG);
            } else {
              nextSibling = childNode.nextSibling;
              hydrateVNode(child, childNode, context, isSVG);
              childNode = nextSibling;
            }
          }
        }
        // clear any other DOM nodes, there should be only a single entry for the root
        while (childNode) {
          nextSibling = childNode.nextSibling;
          dom.removeChild(childNode);
          childNode = nextSibling;
        }
      } else if (!isNull$2(dom.firstChild) && !isSamePropsInnerHTML(dom, props)) {
        dom.textContent = ''; // dom has content, but VNode has no children remove everything from DOM
        if (flags & 448 /* VNodeFlags.FormElement */) {
          // If element is form element, we need to clear defaultValue also
          dom.defaultValue = '';
        }
      }
      if (!isNull$2(props)) {
        mountProps(vNode, flags, props, dom, isSVG);
      }
      if (isNullOrUndef(className)) {
        if (dom.className !== '') {
          dom.removeAttribute('class');
        }
      } else if (isSVG) {
        dom.setAttribute('class', className);
      } else {
        dom.className = className;
      }
      if (isFunction(ref)) {
        mountRef(dom, ref);
      } else {
        {
          if (isString(ref)) {
            throwError('string "refs" are not supported in Inferno 1.0. Use callback "refs" instead.');
          }
        }
      }
    }
  }
  function hydrateText(vNode, dom) {
    if (dom.nodeType !== 3) {
      var newDom = mountText(vNode, null);
      vNode.dom = newDom;
      replaceChild(dom.parentNode, newDom, dom);
    } else {
      var text = vNode.children;
      if (dom.nodeValue !== text) {
        dom.nodeValue = text;
      }
      vNode.dom = dom;
    }
  }
  function hydrateVNode(vNode, dom, context, isSVG) {
    var flags = vNode.flags;
    if (flags & 14 /* VNodeFlags.Component */) {
      hydrateComponent(vNode, dom, context, isSVG, (flags & 4 /* VNodeFlags.ComponentClass */) > 0);
    } else if (flags & 481 /* VNodeFlags.Element */) {
      hydrateElement(vNode, dom, context, isSVG);
    } else if (flags & 16 /* VNodeFlags.Text */) {
      hydrateText(vNode, dom);
    } else if (flags & 512 /* VNodeFlags.Void */) {
      vNode.dom = dom;
    } else {
      {
        throwError("hydrate() expects a valid VNode, instead it received an object with the type \"" + typeof vNode + "\".");
      }
      throwError();
    }
  }
  function hydrate(input, parentDom, callback) {
    var dom = parentDom.firstChild;
    if (!isNull$2(dom)) {
      if (!isInvalid(input)) {
        hydrateVNode(input, dom, EMPTY_OBJ, false);
      }
      dom = parentDom.firstChild;
      // clear any other DOM nodes, there should be only a single entry for the root
      while (dom = dom.nextSibling) {
        parentDom.removeChild(dom);
      }
    }
    if (LIFECYCLE.length > 0) {
      callAll(LIFECYCLE);
    }
    parentDom.$V = input;
    if (isFunction(callback)) {
      callback();
    }
  }
  function replaceWithNewNode(lastNode, nextNode, parentDom, context, isSVG) {
    unmount(lastNode);
    replaceChild(parentDom, mount(nextNode, null, context, isSVG), lastNode.dom);
  }
  function patch(lastVNode, nextVNode, parentDom, context, isSVG) {
    var nextFlags = nextVNode.flags | 0;
    if (lastVNode.flags !== nextFlags || nextFlags & 2048 /* VNodeFlags.ReCreate */) {
      replaceWithNewNode(lastVNode, nextVNode, parentDom, context, isSVG);
    } else if (nextFlags & 481 /* VNodeFlags.Element */) {
      patchElement(lastVNode, nextVNode, parentDom, context, isSVG, nextFlags);
    } else if (nextFlags & 14 /* VNodeFlags.Component */) {
      patchComponent(lastVNode, nextVNode, parentDom, context, isSVG, (nextFlags & 4 /* VNodeFlags.ComponentClass */) > 0);
    } else if (nextFlags & 16 /* VNodeFlags.Text */) {
      patchText(lastVNode, nextVNode);
    } else if (nextFlags & 512 /* VNodeFlags.Void */) {
      nextVNode.dom = lastVNode.dom;
    } else {
      patchPortal(lastVNode, nextVNode, context);
    }
  }
  function patchContentEditableChildren(dom, nextVNode) {
    if (dom.textContent !== nextVNode.children) {
      dom.textContent = nextVNode.children;
    }
  }
  function patchPortal(lastVNode, nextVNode, context) {
    var lastContainer = lastVNode.type;
    var nextContainer = nextVNode.type;
    var nextChildren = nextVNode.children;
    patchChildren(lastVNode.childFlags, nextVNode.childFlags, lastVNode.children, nextChildren, lastContainer, context, false);
    nextVNode.dom = lastVNode.dom;
    if (lastContainer !== nextContainer && !isInvalid(nextChildren)) {
      var node = nextChildren.dom;
      lastContainer.removeChild(node);
      nextContainer.appendChild(node);
    }
  }
  function patchElement(lastVNode, nextVNode, parentDom, context, isSVG, nextFlags) {
    var nextTag = nextVNode.type;
    if (lastVNode.type !== nextTag) {
      replaceWithNewNode(lastVNode, nextVNode, parentDom, context, isSVG);
    } else {
      var dom = lastVNode.dom;
      var lastProps = lastVNode.props;
      var nextProps = nextVNode.props;
      var isFormElement = false;
      var hasControlledValue = false;
      var nextPropsOrEmpty;
      nextVNode.dom = dom;
      isSVG = isSVG || (nextFlags & 32 /* VNodeFlags.SvgElement */) > 0;
      // inlined patchProps  -- starts --
      if (lastProps !== nextProps) {
        var lastPropsOrEmpty = lastProps || EMPTY_OBJ;
        nextPropsOrEmpty = nextProps || EMPTY_OBJ;
        if (nextPropsOrEmpty !== EMPTY_OBJ) {
          isFormElement = (nextFlags & 448 /* VNodeFlags.FormElement */) > 0;
          if (isFormElement) {
            hasControlledValue = isControlledFormElement(nextPropsOrEmpty);
          }
          for (var prop in nextPropsOrEmpty) {
            var lastValue = lastPropsOrEmpty[prop];
            var nextValue = nextPropsOrEmpty[prop];
            if (lastValue !== nextValue) {
              patchProp(prop, lastValue, nextValue, dom, isSVG, hasControlledValue, lastVNode);
            }
          }
        }
        if (lastPropsOrEmpty !== EMPTY_OBJ) {
          for (var _prop in lastPropsOrEmpty) {
            if (!nextPropsOrEmpty.hasOwnProperty(_prop) && !isNullOrUndef(lastPropsOrEmpty[_prop])) {
              patchProp(_prop, lastPropsOrEmpty[_prop], null, dom, isSVG, hasControlledValue, lastVNode);
            }
          }
        }
      }
      var lastChildren = lastVNode.children;
      var nextChildren = nextVNode.children;
      var nextRef = nextVNode.ref;
      var lastClassName = lastVNode.className;
      var nextClassName = nextVNode.className;
      {
        validateKeys(nextVNode);
      }
      if (nextFlags & 4096 /* VNodeFlags.ContentEditable */) {
        patchContentEditableChildren(dom, nextChildren);
      } else {
        patchChildren(lastVNode.childFlags, nextVNode.childFlags, lastChildren, nextChildren, dom, context, isSVG && nextTag !== 'foreignObject');
      }
      if (isFormElement) {
        processElement(nextFlags, nextVNode, dom, nextPropsOrEmpty, false, hasControlledValue);
      }
      // inlined patchProps  -- ends --
      if (lastClassName !== nextClassName) {
        if (isNullOrUndef(nextClassName)) {
          dom.removeAttribute('class');
        } else if (isSVG) {
          dom.setAttribute('class', nextClassName);
        } else {
          dom.className = nextClassName;
        }
      }
      if (isFunction(nextRef) && lastVNode.ref !== nextRef) {
        mountRef(dom, nextRef);
      } else {
        {
          if (isString(nextRef)) {
            throwError('string "refs" are not supported in Inferno 1.0. Use callback "refs" instead.');
          }
        }
      }
    }
  }
  function patchChildren(lastChildFlags, nextChildFlags, lastChildren, nextChildren, parentDOM, context, isSVG) {
    switch (lastChildFlags) {
      case 2 /* ChildFlags.HasVNodeChildren */:
        switch (nextChildFlags) {
          case 2 /* ChildFlags.HasVNodeChildren */:
            patch(lastChildren, nextChildren, parentDOM, context, isSVG);
            break;
          case 1 /* ChildFlags.HasInvalidChildren */:
            remove$1(lastChildren, parentDOM);
            break;
          default:
            remove$1(lastChildren, parentDOM);
            mountArrayChildren(nextChildren, parentDOM, context, isSVG);
            break;
        }
        break;
      case 1 /* ChildFlags.HasInvalidChildren */:
        switch (nextChildFlags) {
          case 2 /* ChildFlags.HasVNodeChildren */:
            mount(nextChildren, parentDOM, context, isSVG);
            break;
          case 1 /* ChildFlags.HasInvalidChildren */:
            break;
          default:
            mountArrayChildren(nextChildren, parentDOM, context, isSVG);
            break;
        }
        break;
      default:
        if (nextChildFlags & 12 /* ChildFlags.MultipleChildren */) {
          var lastLength = lastChildren.length;
          var nextLength = nextChildren.length;
          // Fast path's for both algorithms
          if (lastLength === 0) {
            if (nextLength > 0) {
              mountArrayChildren(nextChildren, parentDOM, context, isSVG);
            }
          } else if (nextLength === 0) {
            removeAllChildren(parentDOM, lastChildren);
          } else if (nextChildFlags === 8 /* ChildFlags.HasKeyedChildren */ && lastChildFlags === 8 /* ChildFlags.HasKeyedChildren */) {
            patchKeyedChildren(lastChildren, nextChildren, parentDOM, context, isSVG, lastLength, nextLength);
          } else {
            patchNonKeyedChildren(lastChildren, nextChildren, parentDOM, context, isSVG, lastLength, nextLength);
          }
        } else if (nextChildFlags === 1 /* ChildFlags.HasInvalidChildren */) {
          removeAllChildren(parentDOM, lastChildren);
        } else if (nextChildFlags === 2 /* ChildFlags.HasVNodeChildren */) {
          removeAllChildren(parentDOM, lastChildren);
          mount(nextChildren, parentDOM, context, isSVG);
        }
        break;
    }
  }
  function updateClassComponent(instance, nextState, nextVNode, nextProps, parentDom, context, isSVG, force, fromSetState) {
    var lastState = instance.state;
    var lastProps = instance.props;
    nextVNode.children = instance;
    var renderOutput;
    if (instance.$UN) {
      {
        warning('Inferno Error: Can only update a mounted or mounting component. This usually means you called setState() or forceUpdate() on an unmounted component. This is a no-op.');
      }
      return;
    }
    if (lastProps !== nextProps || nextProps === EMPTY_OBJ) {
      if (!fromSetState && isFunction(instance.componentWillReceiveProps)) {
        instance.$BR = true;
        instance.componentWillReceiveProps(nextProps, context);
        // If instance component was removed during its own update do nothing.
        if (instance.$UN) {
          return;
        }
        instance.$BR = false;
      }
      if (instance.$PSS) {
        nextState = combineFrom(nextState, instance.$PS);
        instance.$PSS = false;
        instance.$PS = null;
      }
    }
    /* Update if scu is not defined, or it returns truthy value or force */
    var hasSCU = Boolean(instance.shouldComponentUpdate);
    if (force || !hasSCU || hasSCU && instance.shouldComponentUpdate(nextProps, nextState, context)) {
      if (isFunction(instance.componentWillUpdate)) {
        instance.$BS = true;
        instance.componentWillUpdate(nextProps, nextState, context);
        instance.$BS = false;
      }
      instance.props = nextProps;
      instance.state = nextState;
      instance.context = context;
      if (isFunction(options.beforeRender)) {
        options.beforeRender(instance);
      }
      renderOutput = instance.render(nextProps, nextState, context);
      if (isFunction(options.afterRender)) {
        options.afterRender(instance);
      }
      var didUpdate = renderOutput !== NO_OP;
      var childContext;
      if (isFunction(instance.getChildContext)) {
        childContext = instance.getChildContext();
      }
      if (isNullOrUndef(childContext)) {
        childContext = context;
      } else {
        childContext = combineFrom(context, childContext);
      }
      instance.$CX = childContext;
      if (didUpdate) {
        var lastInput = instance.$LI;
        var nextInput = handleComponentInput(renderOutput, nextVNode);
        patch(lastInput, nextInput, parentDom, childContext, isSVG);
        instance.$LI = nextInput;
        if (isFunction(instance.componentDidUpdate)) {
          instance.componentDidUpdate(lastProps, lastState);
        }
      }
    } else {
      instance.props = nextProps;
      instance.state = nextState;
      instance.context = context;
    }
    nextVNode.dom = instance.$LI.dom;
  }
  function patchComponent(lastVNode, nextVNode, parentDom, context, isSVG, isClass) {
    var nextType = nextVNode.type;
    var lastKey = lastVNode.key;
    var nextKey = nextVNode.key;
    if (lastVNode.type !== nextType || lastKey !== nextKey) {
      replaceWithNewNode(lastVNode, nextVNode, parentDom, context, isSVG);
    } else {
      var nextProps = nextVNode.props || EMPTY_OBJ;
      if (isClass) {
        var instance = lastVNode.children;
        instance.$UPD = true;
        instance.$V = nextVNode;
        updateClassComponent(instance, instance.state, nextVNode, nextProps, parentDom, context, isSVG, false, false);
        instance.$UPD = false;
      } else {
        var shouldUpdate = true;
        var lastProps = lastVNode.props;
        var nextHooks = nextVNode.ref;
        var nextHooksDefined = !isNullOrUndef(nextHooks);
        var lastInput = lastVNode.children;
        nextVNode.dom = lastVNode.dom;
        nextVNode.children = lastInput;
        if (nextHooksDefined && isFunction(nextHooks.onComponentShouldUpdate)) {
          shouldUpdate = nextHooks.onComponentShouldUpdate(lastProps, nextProps);
        }
        if (shouldUpdate !== false) {
          if (nextHooksDefined && isFunction(nextHooks.onComponentWillUpdate)) {
            nextHooks.onComponentWillUpdate(lastProps, nextProps);
          }
          var nextInput = nextType(nextProps, context);
          if (nextInput !== NO_OP) {
            nextInput = handleComponentInput(nextInput, nextVNode);
            patch(lastInput, nextInput, parentDom, context, isSVG);
            nextVNode.children = nextInput;
            nextVNode.dom = nextInput.dom;
            if (nextHooksDefined && isFunction(nextHooks.onComponentDidUpdate)) {
              nextHooks.onComponentDidUpdate(lastProps, nextProps);
            }
          }
        } else if (lastInput.flags & 14 /* VNodeFlags.Component */) {
          lastInput.parentVNode = nextVNode;
        }
      }
    }
  }
  function patchText(lastVNode, nextVNode) {
    var nextText = nextVNode.children;
    var dom = lastVNode.dom;
    if (nextText !== lastVNode.children) {
      dom.nodeValue = nextText;
    }
    nextVNode.dom = dom;
  }
  function patchNonKeyedChildren(lastChildren, nextChildren, dom, context, isSVG, lastChildrenLength, nextChildrenLength) {
    var commonLength = lastChildrenLength > nextChildrenLength ? nextChildrenLength : lastChildrenLength;
    var i = 0;
    var nextChild;
    var lastChild;
    for (; i < commonLength; i++) {
      nextChild = nextChildren[i];
      lastChild = lastChildren[i];
      if (nextChild.dom) {
        nextChild = nextChildren[i] = directClone(nextChild);
      }
      patch(lastChild, nextChild, dom, context, isSVG);
      lastChildren[i] = nextChild;
    }
    if (lastChildrenLength < nextChildrenLength) {
      for (i = commonLength; i < nextChildrenLength; i++) {
        nextChild = nextChildren[i];
        if (nextChild.dom) {
          nextChild = nextChildren[i] = directClone(nextChild);
        }
        mount(nextChild, dom, context, isSVG);
      }
    } else if (lastChildrenLength > nextChildrenLength) {
      for (i = commonLength; i < lastChildrenLength; i++) {
        remove$1(lastChildren[i], dom);
      }
    }
  }
  function patchKeyedChildren(a, b, dom, context, isSVG, aLength, bLength) {
    var aEnd = aLength - 1;
    var bEnd = bLength - 1;
    var i;
    var j = 0;
    var aNode = a[j];
    var bNode = b[j];
    var nextPos;
    // Step 1
    // tslint:disable-next-line
    outer: {
      // Sync nodes with the same key at the beginning.
      while (aNode.key === bNode.key) {
        if (bNode.dom) {
          b[j] = bNode = directClone(bNode);
        }
        patch(aNode, bNode, dom, context, isSVG);
        a[j] = bNode;
        j++;
        if (j > aEnd || j > bEnd) {
          break outer;
        }
        aNode = a[j];
        bNode = b[j];
      }
      aNode = a[aEnd];
      bNode = b[bEnd];
      // Sync nodes with the same key at the end.
      while (aNode.key === bNode.key) {
        if (bNode.dom) {
          b[bEnd] = bNode = directClone(bNode);
        }
        patch(aNode, bNode, dom, context, isSVG);
        a[aEnd] = bNode;
        aEnd--;
        bEnd--;
        if (j > aEnd || j > bEnd) {
          break outer;
        }
        aNode = a[aEnd];
        bNode = b[bEnd];
      }
    }
    if (j > aEnd) {
      if (j <= bEnd) {
        nextPos = bEnd + 1;
        var nextNode = nextPos < bLength ? b[nextPos].dom : null;
        while (j <= bEnd) {
          bNode = b[j];
          if (bNode.dom) {
            b[j] = bNode = directClone(bNode);
          }
          j++;
          insertOrAppend(dom, mount(bNode, null, context, isSVG), nextNode);
        }
      }
    } else if (j > bEnd) {
      while (j <= aEnd) {
        remove$1(a[j++], dom);
      }
    } else {
      var aStart = j;
      var bStart = j;
      var aLeft = aEnd - j + 1;
      var bLeft = bEnd - j + 1;
      var sources = [];
      for (i = 0; i < bLeft; i++) {
        sources.push(0);
      }
      // Keep track if its possible to remove whole DOM using textContent = '';
      var canRemoveWholeContent = aLeft === aLength;
      var moved = false;
      var pos = 0;
      var patched = 0;
      // When sizes are small, just loop them through
      if (bLength < 4 || (aLeft | bLeft) < 32) {
        for (i = aStart; i <= aEnd; i++) {
          aNode = a[i];
          if (patched < bLeft) {
            for (j = bStart; j <= bEnd; j++) {
              bNode = b[j];
              if (aNode.key === bNode.key) {
                sources[j - bStart] = i + 1;
                if (canRemoveWholeContent) {
                  canRemoveWholeContent = false;
                  while (i > aStart) {
                    remove$1(a[aStart++], dom);
                  }
                }
                if (pos > j) {
                  moved = true;
                } else {
                  pos = j;
                }
                if (bNode.dom) {
                  b[j] = bNode = directClone(bNode);
                }
                patch(aNode, bNode, dom, context, isSVG);
                patched++;
                break;
              }
            }
            if (!canRemoveWholeContent && j > bEnd) {
              remove$1(aNode, dom);
            }
          } else if (!canRemoveWholeContent) {
            remove$1(aNode, dom);
          }
        }
      } else {
        var keyIndex = {};
        // Map keys by their index
        for (i = bStart; i <= bEnd; i++) {
          keyIndex[b[i].key] = i;
        }
        // Try to patch same keys
        for (i = aStart; i <= aEnd; i++) {
          aNode = a[i];
          if (patched < bLeft) {
            j = keyIndex[aNode.key];
            if (j !== void 0) {
              if (canRemoveWholeContent) {
                canRemoveWholeContent = false;
                while (i > aStart) {
                  remove$1(a[aStart++], dom);
                }
              }
              bNode = b[j];
              sources[j - bStart] = i + 1;
              if (pos > j) {
                moved = true;
              } else {
                pos = j;
              }
              if (bNode.dom) {
                b[j] = bNode = directClone(bNode);
              }
              patch(aNode, bNode, dom, context, isSVG);
              patched++;
            } else if (!canRemoveWholeContent) {
              remove$1(aNode, dom);
            }
          } else if (!canRemoveWholeContent) {
            remove$1(aNode, dom);
          }
        }
      }
      // fast-path: if nothing patched remove all old and add all new
      if (canRemoveWholeContent) {
        removeAllChildren(dom, a);
        mountArrayChildren(b, dom, context, isSVG);
      } else {
        if (moved) {
          var seq = lis_algorithm(sources);
          j = seq.length - 1;
          for (i = bLeft - 1; i >= 0; i--) {
            if (sources[i] === 0) {
              pos = i + bStart;
              bNode = b[pos];
              if (bNode.dom) {
                b[pos] = bNode = directClone(bNode);
              }
              nextPos = pos + 1;
              insertOrAppend(dom, mount(bNode, null, context, isSVG), nextPos < bLength ? b[nextPos].dom : null);
            } else if (j < 0 || i !== seq[j]) {
              pos = i + bStart;
              bNode = b[pos];
              nextPos = pos + 1;
              insertOrAppend(dom, bNode.dom, nextPos < bLength ? b[nextPos].dom : null);
            } else {
              j--;
            }
          }
        } else if (patched !== bLeft) {
          // when patched count doesn't match b length we need to insert those new ones
          // loop backwards so we can use insertBefore
          for (i = bLeft - 1; i >= 0; i--) {
            if (sources[i] === 0) {
              pos = i + bStart;
              bNode = b[pos];
              if (bNode.dom) {
                b[pos] = bNode = directClone(bNode);
              }
              nextPos = pos + 1;
              insertOrAppend(dom, mount(bNode, null, context, isSVG), nextPos < bLength ? b[nextPos].dom : null);
            }
          }
        }
      }
    }
  }
  // https://en.wikipedia.org/wiki/Longest_increasing_subsequence
  function lis_algorithm(arr) {
    var p = arr.slice();
    var result = [0];
    var i;
    var j;
    var u;
    var v;
    var c;
    var len = arr.length;
    for (i = 0; i < len; i++) {
      var arrI = arr[i];
      if (arrI !== 0) {
        j = result[result.length - 1];
        if (arr[j] < arrI) {
          p[i] = j;
          result.push(i);
          continue;
        }
        u = 0;
        v = result.length - 1;
        while (u < v) {
          c = (u + v) / 2 | 0;
          if (arr[result[c]] < arrI) {
            u = c + 1;
          } else {
            v = c;
          }
        }
        if (arrI < arr[result[u]]) {
          if (u > 0) {
            p[i] = result[u - 1];
          }
          result[u] = i;
        }
      }
    }
    u = result.length;
    v = result[u - 1];
    while (u-- > 0) {
      result[u] = v;
      v = p[v];
    }
    return result;
  }
  {
    if (isBrowser && document.body === null) {
      warning('Inferno warning: you cannot initialize inferno without "document.body". Wait on "DOMContentLoaded" event, add script to bottom of body, or use async/defer attributes on script tag.');
    }
  }
  var documentBody = isBrowser ? document.body : null;
  function render(input, parentDom, callback) {
    // Development warning
    {
      if (documentBody === parentDom) {
        throwError('you cannot render() to the "document.body". Use an empty element as a container instead.');
      }
    }
    if (input === NO_OP) {
      return;
    }
    var rootInput = parentDom.$V;
    if (isNullOrUndef(rootInput)) {
      if (!isInvalid(input)) {
        if (input.dom) {
          input = directClone(input);
        }
        if (isNull$2(parentDom.firstChild)) {
          mount(input, parentDom, EMPTY_OBJ, false);
          parentDom.$V = input;
        } else {
          hydrate(input, parentDom);
        }
        rootInput = input;
      }
    } else {
      if (isNullOrUndef(input)) {
        remove$1(rootInput, parentDom);
        parentDom.$V = null;
      } else {
        if (input.dom) {
          input = directClone(input);
        }
        patch(rootInput, input, parentDom, EMPTY_OBJ, false);
        rootInput = parentDom.$V = input;
      }
    }
    if (LIFECYCLE.length > 0) {
      callAll(LIFECYCLE);
    }
    if (isFunction(callback)) {
      callback();
    }
    if (isFunction(options.renderComplete)) {
      options.renderComplete(rootInput);
    }
    if (rootInput && rootInput.flags & 14 /* VNodeFlags.Component */) {
      return rootInput.children;
    }
  }
  function createPortal(children, container) {
    return createVNode(1024 /* VNodeFlags.Portal */, container, null, children, 0 /* ChildFlags.UnknownChildren */, null, isInvalid(children) ? null : children.key, null);
  }
  var resolvedPromise = typeof Promise === 'undefined' ? null : Promise.resolve();
  // raf.bind(window) is needed to work around bug in IE10-IE11 strict mode (TypeError: Invalid calling object)
  var fallbackMethod = typeof requestAnimationFrame === 'undefined' ? setTimeout : requestAnimationFrame.bind(window);
  function nextTick(fn) {
    if (resolvedPromise) {
      return resolvedPromise.then(fn);
    }
    return fallbackMethod(fn);
  }
  function queueStateChanges(component, newState, callback, force) {
    if (isFunction(newState)) {
      newState = newState(component.state, component.props, component.context);
    }
    var pending = component.$PS;
    if (isNullOrUndef(pending)) {
      component.$PS = newState;
    } else {
      for (var stateKey in newState) {
        pending[stateKey] = newState[stateKey];
      }
    }
    if (!component.$PSS && !component.$BR) {
      if (!component.$UPD) {
        component.$PSS = true;
        component.$UPD = true;
        applyState(component, force, callback);
        component.$UPD = false;
      } else {
        // Async
        var queue = component.$QU;
        if (isNull$2(queue)) {
          queue = component.$QU = [];
          nextTick(promiseCallback(component, queue));
        }
        if (isFunction(callback)) {
          queue.push(callback);
        }
      }
    } else {
      component.$PSS = true;
      if (component.$BR && isFunction(callback)) {
        LIFECYCLE.push(callback.bind(component));
      }
    }
  }
  function promiseCallback(component, queue) {
    return function () {
      component.$QU = null;
      component.$UPD = true;
      applyState(component, false, function () {
        for (var i = 0, len = queue.length; i < len; i++) {
          queue[i].call(component);
        }
      });
      component.$UPD = false;
    };
  }
  function applyState(component, force, callback) {
    if (component.$UN) {
      return;
    }
    if (force || !component.$BR) {
      component.$PSS = false;
      var pendingState = component.$PS;
      var prevState = component.state;
      var nextState = combineFrom(prevState, pendingState);
      var props = component.props;
      var context = component.context;
      component.$PS = null;
      var vNode = component.$V;
      var lastInput = component.$LI;
      var parentDom = lastInput.dom && lastInput.dom.parentNode;
      updateClassComponent(component, nextState, vNode, props, parentDom, context, (vNode.flags & 32 /* VNodeFlags.SvgElement */) > 0, force, true);
      if (component.$UN) {
        return;
      }
      if ((component.$LI.flags & 1024 /* VNodeFlags.Portal */) === 0) {
        var dom = component.$LI.dom;
        while (!isNull$2(vNode = vNode.parentVNode)) {
          if ((vNode.flags & 14 /* VNodeFlags.Component */) > 0) {
            vNode.dom = dom;
          }
        }
      }
      if (LIFECYCLE.length > 0) {
        callAll(LIFECYCLE);
      }
    } else {
      component.state = component.$PS;
      component.$PS = null;
    }
    if (isFunction(callback)) {
      callback.call(component);
    }
  }
  var Component = /*#__PURE__*/function () {
    // QUEUE
    function Component(props, context) {
      this.state = null;
      this.props = void 0;
      this.context = void 0;
      this.refs = void 0;
      // Internal properties
      this.$BR = false;
      // BLOCK RENDER
      this.$BS = true;
      // BLOCK STATE
      this.$PSS = false;
      // PENDING SET STATE
      this.$PS = null;
      // PENDING STATE (PARTIAL or FULL)
      this.$LI = null;
      // LAST INPUT
      this.$V = null;
      // VNODE
      this.$UN = false;
      // UNMOUNTED
      this.$CX = null;
      // CHILDCONTEXT
      this.$UPD = true;
      // UPDATING
      this.$QU = null;
      /** @type {object} */
      this.props = props || EMPTY_OBJ;
      /** @type {object} */
      this.context = context || EMPTY_OBJ; // context should not be mutable
    }
    var _proto = Component.prototype;
    _proto.forceUpdate = function forceUpdate(callback) {
      if (this.$UN) {
        return;
      }
      // Do not allow double render during force update
      queueStateChanges(this, {}, callback, true);
    };
    _proto.setState = function setState(newState, callback) {
      if (this.$UN) {
        return;
      }
      if (!this.$BS) {
        queueStateChanges(this, newState, callback, false);
      } else {
        // Development warning
        {
          throwError('cannot update state via setState() in componentWillUpdate() or constructor.');
        }
        return;
      }
    }
    // tslint:disable-next-line:no-empty
    ;
    _proto.render = function render(_nextProps, _nextState, _nextContext) {};
    return Component;
  }();
  // Public
  Component.defaultProps = void 0;
  {
    /* tslint:disable-next-line:no-empty */
    var testFunc = function testFn() {};
    /* tslint:disable-next-line*/
    console.info('Inferno is in development mode.');
    if ((testFunc.name || testFunc.toString()).indexOf('testFn') === -1) {
      warning("It looks like you're using a minified copy of the development build " + 'of Inferno. When deploying Inferno apps to production, make sure to use ' + 'the production build which skips development warnings and is faster. ' + 'See http://infernojs.org for more details.');
    }
  }

  // inlined ../../../../resources/logo.svg
  var BPMNIO_IMG = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 14.02 5.57" width="53" height="21" style="vertical-align:middle"><path fill="#000000" d="M1.88.92v.14c0 .41-.13.68-.4.8.33.14.46.44.46.86v.33c0 .61-.33.95-.95.95H0V0h.95c.65 0 .93.3.93.92zM.63.57v1.06h.24c.24 0 .38-.1.38-.43V.98c0-.28-.1-.4-.32-.4zm0 1.63v1.22h.36c.2 0 .32-.1.32-.39v-.35c0-.37-.12-.48-.4-.48H.63zM4.18.99v.52c0 .64-.31.98-.94.98h-.3V4h-.62V0h.92c.63 0 .94.35.94.99zM2.94.57v1.35h.3c.2 0 .3-.09.3-.37v-.6c0-.29-.1-.38-.3-.38h-.3zm2.89 2.27L6.25 0h.88v4h-.6V1.12L6.1 3.99h-.6l-.46-2.82v2.82h-.55V0h.87zM8.14 1.1V4h-.56V0h.79L9 2.4V0h.56v4h-.64zm2.49 2.29v.6h-.6v-.6zM12.12 1c0-.63.33-1 .95-1 .61 0 .95.37.95 1v2.04c0 .64-.34 1-.95 1-.62 0-.95-.37-.95-1zm.62 2.08c0 .28.13.39.33.39s.32-.1.32-.4V.98c0-.29-.12-.4-.32-.4s-.33.11-.33.4z"/><path fill="#000000" d="M0 4.53h14.02v1.04H0zM11.08 0h.63v.62h-.63zm.63 4V1h-.63v2.98z"/></svg>';

  /**
   * Adds the project logo to the diagram container as
   * required by the bpmn.io license.
   *
   * @see http://bpmn.io/license
   *
   * @param {Element} container
   */
  function addProjectLogo(container) {
    var linkMarkup = '<a href="http://bpmn.io" ' + 'target="_blank" ' + 'class="bjs-powered-by" ' + 'title="Powered by bpmn.io" ' + 'style="position: absolute; bottom: 15px; right: 15px; z-index: 100;">' + BPMNIO_IMG + '</a>';
    var linkElement = domify$1(linkMarkup);
    container.appendChild(linkElement);
    event.bind(linkElement, 'click', function (event) {
      open();
      event.preventDefault();
    });
  }
  class PoweredByComponent extends Component {
    constructor(props, context) {
      super(props, context);
      this.node = null;
    }
    componentDidMount() {
      addProjectLogo(this.node);
    }
    render() {
      return createVNode(1, "div", null, null, 1, null, null, node => this.node = node);
    }
  }
  function css(attrs) {
    return attrs.join(';');
  }
  var LIGHTBOX_STYLES = css(['z-index: 1001', 'position: fixed', 'top: 0', 'left: 0', 'right: 0', 'bottom: 0']);
  var BACKDROP_STYLES = css(['width: 100%', 'height: 100%', 'background: rgba(40,40,40,0.2)']);
  var NOTICE_STYLES = css(['position: absolute', 'left: 50%', 'top: 40%', 'transform: translate(-50%)', 'width: 260px', 'padding: 10px', 'background: white', 'box-shadow: 0 1px 4px rgba(0,0,0,0.3)', 'font-family: Helvetica, Arial, sans-serif', 'font-size: 14px', 'display: flex', 'line-height: 1.3']);
  var LIGHTBOX_MARKUP = '<div class="bjs-powered-by-lightbox" style="' + LIGHTBOX_STYLES + '">' + '<div class="backdrop" style="' + BACKDROP_STYLES + '"></div>' + '<div class="notice" style="' + NOTICE_STYLES + '">' + '<a href="https://bpmn.io" target="_blank" rel="noopener" style="margin: 15px 20px 15px 10px; align-self: center;' + '">' + BPMNIO_IMG + '</a>' + '<span>' + 'Web-based tooling for BPMN, DMN and CMMN diagrams ' + 'powered by <a href="https://bpmn.io" target="_blank" rel="noopener">bpmn.io</a>.' + '</span>' + '</div>' + '</div>';
  var lightbox;
  function open() {
    if (!lightbox) {
      lightbox = domify$1(LIGHTBOX_MARKUP);
      delegate.bind(lightbox, '.backdrop', 'click', function (event) {
        document.body.removeChild(lightbox);
      });
    }
    document.body.appendChild(lightbox);
  }

  var RENDERER_IDS = new Ids$1();
  var black = 'hsl(225, 10%, 15%)';

  /**
   * Renderer for the DRD view. The default colors are configurable.
   * When default label color is not provided, it will take the default
   * stroke color.
   *
   * @example
   * ```javascript
   * // for simple DRD viewer
   * const viewer = new DrdViewer({
   *   drdRenderer: {
   *     defaultFillColor: '#ffd700',
   *     defaultStrokeColor: '#0057b8',
   *     defaultLabelColor: '#0057b8'
   *   }
   * });
   *
   * // in dmn-js
   * const modeler = new DmnModeler({
   *   drd: {
   *     drdRenderer: {
   *       defaultFillColor: '#ffd700',
   *       defaultStrokeColor: '#0057b8',
   *       defaultLabelColor: '#0057b8'
   *     }
   *   }
   * });
   * ```
   */
  function DrdRenderer(config, eventBus, pathMap, styles, textRenderer, canvas) {
    BaseRenderer.call(this, eventBus);
    var rendererId = RENDERER_IDS.next();
    var computeStyle = styles.computeStyle;
    var markers = {};
    var defaultFillColor = config && config.defaultFillColor || 'white',
      defaultStrokeColor = config && config.defaultStrokeColor || black,
      defaultLabelColor = config && config.defaultLabelColor;
    function marker(type, fill, stroke) {
      var id = type + '-' + colorEscape(fill) + '-' + colorEscape(stroke) + '-' + rendererId;
      if (!markers[id]) {
        createMarker(id, type, fill, stroke);
      }
      return 'url(#' + id + ')';
    }
    function addMarker(id, options) {
      var attrs = assign$4({
        strokeWidth: 1,
        strokeLinecap: 'round',
        strokeDasharray: 'none'
      }, options.attrs);
      var ref = options.ref || {
        x: 0,
        y: 0
      };
      var scale = options.scale || 1;

      // fix for safari / chrome / firefox bug not correctly
      // resetting stroke dash array
      if (attrs.strokeDasharray === 'none') {
        attrs.strokeDasharray = [10000, 1];
      }
      var marker = create$2('marker');
      attr(options.element, attrs);
      append(marker, options.element);
      attr(marker, {
        id: id,
        viewBox: '0 0 20 20',
        refX: ref.x,
        refY: ref.y,
        markerWidth: 20 * scale,
        markerHeight: 20 * scale,
        orient: 'auto'
      });
      var defs = query('defs', canvas._svg);
      if (!defs) {
        defs = create$2('defs');
        append(canvas._svg, defs);
      }
      append(defs, marker);
      markers[id] = marker;
    }
    function createMarker(id, type, fill, stroke) {
      if (type === 'association-start') {
        var associationStart = create$2('path');
        attr(associationStart, {
          d: 'M 11 5 L 1 10 L 11 15'
        });
        addMarker(id, {
          element: associationStart,
          attrs: {
            fill: 'none',
            stroke: stroke,
            strokeWidth: 1.5
          },
          ref: {
            x: 1,
            y: 10
          },
          scale: 0.5
        });
      } else if (type === 'association-end') {
        var associationEnd = create$2('path');
        attr(associationEnd, {
          d: 'M 1 5 L 11 10 L 1 15'
        });
        addMarker(id, {
          element: associationEnd,
          attrs: {
            fill: 'none',
            stroke: stroke,
            strokeWidth: 1.5
          },
          ref: {
            x: 12,
            y: 10
          },
          scale: 0.5
        });
      } else if (type === 'information-requirement-end') {
        var informationRequirementEnd = create$2('path');
        attr(informationRequirementEnd, {
          d: 'M 1 5 L 11 10 L 1 15 Z'
        });
        addMarker(id, {
          element: informationRequirementEnd,
          attrs: {
            fill: stroke,
            stroke: 'none'
          },
          ref: {
            x: 11,
            y: 10
          },
          scale: 1
        });
      } else if (type === 'knowledge-requirement-end') {
        var knowledgeRequirementEnd = create$2('path');
        attr(knowledgeRequirementEnd, {
          d: 'M 1 3 L 11 10 L 1 17'
        });
        addMarker(id, {
          element: knowledgeRequirementEnd,
          attrs: {
            fill: 'none',
            stroke: stroke,
            strokeWidth: 2
          },
          ref: {
            x: 11,
            y: 10
          },
          scale: 0.8
        });
      } else if (type === 'authority-requirement-end') {
        var authorityRequirementEnd = create$2('circle');
        attr(authorityRequirementEnd, {
          cx: 3,
          cy: 3,
          r: 3
        });
        addMarker(id, {
          element: authorityRequirementEnd,
          attrs: {
            fill: stroke,
            stroke: 'none'
          },
          ref: {
            x: 3,
            y: 3
          },
          scale: 0.9
        });
      }
    }
    function drawRect(p, width, height, r, offset, attrs) {
      if (isObject$2(offset)) {
        attrs = offset;
        offset = 0;
      }
      offset = offset || 0;
      attrs = computeStyle(attrs, {
        stroke: black,
        strokeWidth: 2,
        fill: 'white'
      });
      var rect = create$2('rect');
      attr(rect, {
        x: offset,
        y: offset,
        width: width - offset * 2,
        height: height - offset * 2,
        rx: r,
        ry: r
      });
      attr(rect, attrs);
      append(p, rect);
      return rect;
    }
    function renderLabel(p, label, options) {
      var text = textRenderer.createText(label || '', options);
      attr$1(text, 'class', 'djs-label');
      append(p, text);
      return text;
    }
    function renderEmbeddedLabel(p, element, align, options) {
      var name = getName(element);
      options = assign$4({
        box: element,
        align: align,
        padding: 5,
        style: {
          fill: getLabelColor(element, defaultLabelColor, defaultStrokeColor)
        }
      }, options);
      return renderLabel(p, name, options);
    }
    function drawPath(p, d, attrs) {
      attrs = computeStyle(attrs, ['no-fill'], {
        strokeWidth: 2,
        stroke: black
      });
      var path = create$2('path');
      attr(path, {
        d: d
      });
      attr(path, attrs);
      append(p, path);
      return path;
    }
    var handlers = {
      'dmn:Decision': function (p, element) {
        var rect = drawRect(p, element.width, element.height, 0, {
          stroke: getStrokeColor(element, defaultStrokeColor),
          fill: getFillColor(element, defaultFillColor)
        });
        renderEmbeddedLabel(p, element, 'center-middle');
        return rect;
      },
      'dmn:KnowledgeSource': function (p, element) {
        var pathData = pathMap.getScaledPath('KNOWLEDGE_SOURCE', {
          xScaleFactor: 1.021,
          yScaleFactor: 1,
          containerWidth: element.width,
          containerHeight: element.height,
          position: {
            mx: 0.0,
            my: 0.075
          }
        });
        var knowledgeSource = drawPath(p, pathData, {
          strokeWidth: 2,
          fill: getFillColor(element, defaultFillColor),
          stroke: getStrokeColor(element, defaultStrokeColor)
        });
        renderEmbeddedLabel(p, element, 'center-middle');
        return knowledgeSource;
      },
      'dmn:BusinessKnowledgeModel': function (p, element) {
        var pathData = pathMap.getScaledPath('BUSINESS_KNOWLEDGE_MODEL', {
          xScaleFactor: 1,
          yScaleFactor: 1,
          containerWidth: element.width,
          containerHeight: element.height,
          position: {
            mx: 0.0,
            my: 0.3
          }
        });
        var businessKnowledge = drawPath(p, pathData, {
          strokeWidth: 2,
          fill: getFillColor(element, defaultFillColor),
          stroke: getStrokeColor(element, defaultStrokeColor)
        });
        renderEmbeddedLabel(p, element, 'center-middle');
        return businessKnowledge;
      },
      'dmn:InputData': function (p, element) {
        var rect = drawRect(p, element.width, element.height, 22, {
          stroke: getStrokeColor(element, defaultStrokeColor),
          fill: getFillColor(element, defaultFillColor)
        });
        renderEmbeddedLabel(p, element, 'center-middle');
        return rect;
      },
      'dmn:TextAnnotation': function (p, element) {
        var style = {
          'fill': 'none',
          'stroke': 'none'
        };
        var textElement = drawRect(p, element.width, element.height, 0, 0, style);
        var textPathData = pathMap.getScaledPath('TEXT_ANNOTATION', {
          xScaleFactor: 1,
          yScaleFactor: 1,
          containerWidth: element.width,
          containerHeight: element.height,
          position: {
            mx: 0.0,
            my: 0.0
          }
        });
        drawPath(p, textPathData, {
          stroke: getStrokeColor(element, defaultStrokeColor)
        });
        var text = getSemantic(element).text || '';
        renderLabel(p, text, {
          style: {
            fill: getLabelColor(element, defaultLabelColor, defaultStrokeColor)
          },
          box: element,
          align: 'left-top',
          padding: 5
        });
        return textElement;
      },
      'dmn:Association': function (p, element) {
        var semantic = getSemantic(element);
        var fill = getFillColor(element, defaultFillColor),
          stroke = getStrokeColor(element, defaultStrokeColor),
          attrs = {
            stroke: stroke,
            strokeDasharray: '0.5, 5',
            strokeLinecap: 'round',
            strokeLinejoin: 'round',
            fill: 'none'
          };
        if (semantic.associationDirection === 'One' || semantic.associationDirection === 'Both') {
          attrs.markerEnd = marker('association-end', fill, stroke);
        }
        if (semantic.associationDirection === 'Both') {
          attrs.markerStart = marker('association-start', fill, stroke);
        }
        return drawLine(p, element.waypoints, attrs);
      },
      'dmn:InformationRequirement': function (p, element) {
        var fill = getFillColor(element, defaultFillColor),
          stroke = getStrokeColor(element, defaultStrokeColor),
          attrs = {
            stroke: stroke,
            strokeWidth: 1,
            strokeLinecap: 'round',
            strokeLinejoin: 'round',
            markerEnd: marker('information-requirement-end', fill, stroke)
          };
        return drawLine(p, element.waypoints, attrs);
      },
      'dmn:KnowledgeRequirement': function (p, element) {
        var fill = getFillColor(element, defaultFillColor),
          stroke = getStrokeColor(element, defaultStrokeColor);
        var attrs = {
          stroke: stroke,
          strokeWidth: 1,
          strokeDasharray: 5,
          strokeLinecap: 'round',
          strokeLinejoin: 'round',
          markerEnd: marker('knowledge-requirement-end', fill, stroke)
        };
        return drawLine(p, element.waypoints, attrs);
      },
      'dmn:AuthorityRequirement': function (p, element) {
        var fill = getFillColor(element, defaultFillColor),
          stroke = getStrokeColor(element, defaultStrokeColor),
          attrs = {
            stroke: stroke,
            strokeWidth: 1.5,
            strokeDasharray: 5,
            strokeLinecap: 'round',
            strokeLinejoin: 'round',
            markerEnd: marker('authority-requirement-end', fill, stroke)
          };
        return drawLine(p, element.waypoints, attrs);
      }
    };

    // draw shape and connection //////////////////

    function drawShape(parent, element) {
      var h = handlers[element.type];
      if (!h) {
        return BaseRenderer.prototype.drawShape.apply(this, [parent, element]);
      } else {
        return h(parent, element);
      }
    }
    function drawConnection(parent, element) {
      var type = element.type;
      var h = handlers[type];
      if (!h) {
        return BaseRenderer.prototype.drawConnection.apply(this, [parent, element]);
      } else {
        return h(parent, element);
      }
    }
    function drawLine(p, waypoints, attrs) {
      attrs = computeStyle(attrs, ['no-fill'], {
        stroke: black,
        strokeWidth: 2,
        fill: 'none'
      });
      var line = createLine(waypoints, attrs);
      append(p, line);
      return line;
    }
    this.canRender = function (element) {
      return is$1(element, 'dmn:DMNElement') || is$1(element, 'dmn:InformationRequirement') || is$1(element, 'dmn:KnowledgeRequirement') || is$1(element, 'dmn:AuthorityRequirement');
    };
    this.drawShape = drawShape;
    this.drawConnection = drawConnection;
  }
  e$3(DrdRenderer, BaseRenderer);
  DrdRenderer.$inject = ['config.drdRenderer', 'eventBus', 'pathMap', 'styles', 'textRenderer', 'canvas'];

  // helper functions //////////////////////

  function getSemantic(element) {
    return element.businessObject;
  }
  function colorEscape(str) {
    // only allow characters and numbers
    return str.replace(/[^0-9a-zA-z]+/g, '_');
  }
  function getStrokeColor(element, defaultColor) {
    return defaultColor;
  }
  function getFillColor(element, defaultColor) {
    return defaultColor;
  }
  function getLabelColor(element, defaultColor, defaultStrokeColor) {
    return defaultColor || getStrokeColor(element, defaultStrokeColor);
  }

  /**
   * @typedef {import('../util/Types').Dimensions} Dimensions
   *
   * @typedef { {
   *   top: number;
   *   left: number;
   *   right: number;
   *   bottom: number;
   * } } Padding
   *
   * @typedef { number | Partial<Padding> } PaddingConfig
   *
   * @typedef { {
   *   horizontal: 'center' | 'left' | 'right';
   *   vertical: 'top' | 'middle';
   * } } Alignment
   *
   *  @typedef { 'center-middle' | 'center-top' } AlignmentConfig
   *
   * @typedef { Partial<{
   *   align: AlignmentConfig;
   *   style: Record<string, number | string>;
   *   padding: PaddingConfig;
   * }> } BaseTextConfig
   *
   * @typedef { BaseTextConfig & Partial<{
   *   size: Dimensions;
   * }> } TextConfig
   *
   * @typedef { BaseTextConfig & Partial<{
   *   box: Dimensions;
   *   fitBox: boolean;
   * }> } TextLayoutConfig
   *
   *  @typedef { Dimensions & {
   *  text: string;
   * } } LineDescriptor
   */

  var DEFAULT_BOX_PADDING = 0;
  var DEFAULT_LABEL_SIZE = {
    width: 150,
    height: 50
  };

  /**
   * @param {AlignmentConfig} align
   * @return {Alignment}
   */
  function parseAlign(align) {
    var parts = align.split('-');
    return {
      horizontal: parts[0] || 'center',
      vertical: parts[1] || 'top'
    };
  }

  /**
   * @param {PaddingConfig} padding
   *
   * @return {Padding}
   */
  function parsePadding(padding) {
    if (isObject$2(padding)) {
      return assign$4({
        top: 0,
        left: 0,
        right: 0,
        bottom: 0
      }, padding);
    } else {
      return {
        top: padding,
        left: padding,
        right: padding,
        bottom: padding
      };
    }
  }

  /**
   * @param {string} text
   * @param {SVGTextElement} fakeText
   *
   * @return {import('../util/Types').Dimensions}
   */
  function getTextBBox(text, fakeText) {
    fakeText.textContent = text;
    var textBBox;
    try {
      var bbox,
        emptyLine = text === '';

      // add dummy text, when line is empty to
      // determine correct height
      fakeText.textContent = emptyLine ? 'dummy' : text;
      textBBox = fakeText.getBBox();

      // take text rendering related horizontal
      // padding into account
      bbox = {
        width: textBBox.width + textBBox.x * 2,
        height: textBBox.height
      };
      if (emptyLine) {
        // correct width
        bbox.width = 0;
      }
      return bbox;
    } catch (e) {
      console.log(e);
      return {
        width: 0,
        height: 0
      };
    }
  }

  /**
   * Layout the next line and return the layouted element.
   *
   * Alters the lines passed.
   *
   * @param {string[]} lines
   * @param {number} maxWidth
   * @param {SVGTextElement} fakeText
   *
   * @return {LineDescriptor} the line descriptor
   */
  function layoutNext(lines, maxWidth, fakeText) {
    var originalLine = lines.shift(),
      fitLine = originalLine;
    var textBBox;
    for (;;) {
      textBBox = getTextBBox(fitLine, fakeText);
      textBBox.width = fitLine ? textBBox.width : 0;

      // try to fit
      if (fitLine === ' ' || fitLine === '' || textBBox.width < Math.round(maxWidth) || fitLine.length < 2) {
        return fit(lines, fitLine, originalLine, textBBox);
      }
      fitLine = shortenLine(fitLine, textBBox.width, maxWidth);
    }
  }

  /**
   * @param {string[]} lines
   * @param {string} fitLine
   * @param {string} originalLine
   * @param {Dimensions} textBBox
   *
   * @return {LineDescriptor}
   */
  function fit(lines, fitLine, originalLine, textBBox) {
    if (fitLine.length < originalLine.length) {
      var remainder = originalLine.slice(fitLine.length).trim();
      lines.unshift(remainder);
    }
    return {
      width: textBBox.width,
      height: textBBox.height,
      text: fitLine
    };
  }
  var SOFT_BREAK = '\u00AD';

  /**
   * Shortens a line based on spacing and hyphens.
   * Returns the shortened result on success.
   *
   * @param {string} line
   * @param {number} maxLength the maximum characters of the string
   *
   * @return {string} the shortened string
   */
  function semanticShorten(line, maxLength) {
    var parts = line.split(/(\s|-|\u00AD)/g),
      part,
      shortenedParts = [],
      length = 0;

    // try to shorten via break chars
    if (parts.length > 1) {
      while (part = parts.shift()) {
        if (part.length + length < maxLength) {
          shortenedParts.push(part);
          length += part.length;
        } else {
          // remove previous part, too if hyphen does not fit anymore
          if (part === '-' || part === SOFT_BREAK) {
            shortenedParts.pop();
          }
          break;
        }
      }
    }
    var last = shortenedParts[shortenedParts.length - 1];

    // translate trailing soft break to actual hyphen
    if (last && last === SOFT_BREAK) {
      shortenedParts[shortenedParts.length - 1] = '-';
    }
    return shortenedParts.join('');
  }

  /**
   * @param {string} line
   * @param {number} width
   * @param {number} maxWidth
   *
   * @return {string}
   */
  function shortenLine(line, width, maxWidth) {
    var length = Math.max(line.length * (maxWidth / width), 1);

    // try to shorten semantically (i.e. based on spaces and hyphens)
    var shortenedLine = semanticShorten(line, length);
    if (!shortenedLine) {
      // force shorten by cutting the long word
      shortenedLine = line.slice(0, Math.max(Math.round(length - 1), 1));
    }
    return shortenedLine;
  }

  /**
   * @return {SVGSVGElement}
   */
  function getHelperSvg() {
    var helperSvg = document.getElementById('helper-svg');
    if (!helperSvg) {
      helperSvg = create$2('svg');
      attr(helperSvg, {
        id: 'helper-svg'
      });
      assign(helperSvg, {
        visibility: 'hidden',
        position: 'fixed',
        width: 0,
        height: 0
      });
      document.body.appendChild(helperSvg);
    }
    return helperSvg;
  }

  /**
   * Creates a new label utility
   *
   * @param {TextConfig} [config]
   */
  function Text$1(config) {
    this._config = assign$4({}, {
      size: DEFAULT_LABEL_SIZE,
      padding: DEFAULT_BOX_PADDING,
      style: {},
      align: 'center-top'
    }, config || {});
  }

  /**
   * Returns the layouted text as an SVG element.
   *
   * @param {string} text
   * @param {TextLayoutConfig} options
   *
   * @return {SVGElement}
   */
  Text$1.prototype.createText = function (text, options) {
    return this.layoutText(text, options).element;
  };

  /**
   * Returns a labels layouted dimensions.
   *
   * @param {string} text to layout
   * @param {TextLayoutConfig} options
   *
   * @return {Dimensions}
   */
  Text$1.prototype.getDimensions = function (text, options) {
    return this.layoutText(text, options).dimensions;
  };

  /**
   * Creates and returns a label and its bounding box.
   *
   * @param {string} text the text to render on the label
   * @param {TextLayoutConfig} options
   *
   * @return { {
   *   element: SVGElement,
   *   dimensions: Dimensions
   * } }
   */
  Text$1.prototype.layoutText = function (text, options) {
    var box = assign$4({}, this._config.size, options.box),
      style = assign$4({}, this._config.style, options.style),
      align = parseAlign(options.align || this._config.align),
      padding = parsePadding(options.padding !== undefined ? options.padding : this._config.padding),
      fitBox = options.fitBox || false;
    var lineHeight = getLineHeight(style);

    // we split text by lines and normalize
    // {soft break} + {line break} => { line break }
    var lines = text.split(/\u00AD?\r?\n/),
      layouted = [];
    var maxWidth = box.width - padding.left - padding.right;

    // ensure correct rendering by attaching helper text node to invisible SVG
    var helperText = create$2('text');
    attr(helperText, {
      x: 0,
      y: 0
    });
    attr(helperText, style);
    var helperSvg = getHelperSvg();
    append(helperSvg, helperText);
    while (lines.length) {
      layouted.push(layoutNext(lines, maxWidth, helperText));
    }
    if (align.vertical === 'middle') {
      padding.top = padding.bottom = 0;
    }
    var totalHeight = reduce(layouted, function (sum, line, idx) {
      return sum + (lineHeight || line.height);
    }, 0) + padding.top + padding.bottom;
    var maxLineWidth = reduce(layouted, function (sum, line, idx) {
      return line.width > sum ? line.width : sum;
    }, 0);

    // the y position of the next line
    var y = padding.top;
    if (align.vertical === 'middle') {
      y += (box.height - totalHeight) / 2;
    }

    // magic number initial offset
    y -= (lineHeight || layouted[0].height) / 4;
    var textElement = create$2('text');
    attr(textElement, style);

    // layout each line taking into account that parent
    // shape might resize to fit text size
    forEach$3(layouted, function (line) {
      var x;
      y += lineHeight || line.height;
      switch (align.horizontal) {
        case 'left':
          x = padding.left;
          break;
        case 'right':
          x = (fitBox ? maxLineWidth : maxWidth) - padding.right - line.width;
          break;
        default:
          // aka center
          x = Math.max(((fitBox ? maxLineWidth : maxWidth) - line.width) / 2 + padding.left, 0);
      }
      var tspan = create$2('tspan');
      attr(tspan, {
        x: x,
        y: y
      });
      tspan.textContent = line.text;
      append(textElement, tspan);
    });
    remove$3(helperText);
    var dimensions = {
      width: maxLineWidth,
      height: totalHeight
    };
    return {
      dimensions: dimensions,
      element: textElement
    };
  };
  function getLineHeight(style) {
    if ('fontSize' in style && 'lineHeight' in style) {
      return style.lineHeight * parseInt(style.fontSize, 10);
    }
  }

  var DEFAULT_FONT_SIZE = 12;
  var LINE_HEIGHT_RATIO = 1.2;
  var MIN_TEXT_ANNOTATION_HEIGHT = 30;
  function TextRenderer(config) {
    var defaultStyle = assign$4({
      fontFamily: 'Arial, sans-serif',
      fontSize: DEFAULT_FONT_SIZE,
      fontWeight: 'normal',
      lineHeight: LINE_HEIGHT_RATIO
    }, config && config.defaultStyle || {});
    var fontSize = parseInt(defaultStyle.fontSize, 10) - 1;
    var externalStyle = assign$4({}, defaultStyle, {
      fontSize: fontSize
    }, config && config.externalStyle || {});
    var textUtil = new Text$1({
      style: defaultStyle
    });

    /**
     * Get the new bounds of an externally rendered,
     * layouted label.
     *
     * @param  {Bounds} bounds
     * @param  {string} text
     *
     * @return {Bounds}
     */
    this.getExternalLabelBounds = function (bounds, text) {
      var layoutedDimensions = textUtil.getDimensions(text, {
        box: {
          width: 90,
          height: 30,
          x: bounds.width / 2 + bounds.x,
          y: bounds.height / 2 + bounds.y
        },
        style: externalStyle
      });

      // resize label shape to fit label text
      return {
        x: Math.round(bounds.x + bounds.width / 2 - layoutedDimensions.width / 2),
        y: Math.round(bounds.y),
        width: Math.ceil(layoutedDimensions.width),
        height: Math.ceil(layoutedDimensions.height)
      };
    };

    /**
     * Get the new bounds of text annotation.
     *
     * @param  {Bounds} bounds
     * @param  {string} text
     *
     * @return {Bounds}
     */
    this.getTextAnnotationBounds = function (bounds, text) {
      var layoutedDimensions = textUtil.getDimensions(text, {
        box: bounds,
        style: defaultStyle,
        align: 'left-top',
        padding: 5
      });
      return {
        x: bounds.x,
        y: bounds.y,
        width: bounds.width,
        height: Math.max(MIN_TEXT_ANNOTATION_HEIGHT, Math.round(layoutedDimensions.height))
      };
    };

    /**
     * Create a layouted text element.
     *
     * @param {string} text
     * @param {Object} [options]
     *
     * @return {SVGElement} rendered text
     */
    this.createText = function (text, options) {
      return textUtil.createText(text, options || {});
    };

    /**
     * Get default text style.
     */
    this.getDefaultStyle = function () {
      return defaultStyle;
    };

    /**
     * Get the external text style.
     */
    this.getExternalStyle = function () {
      return externalStyle;
    };
  }
  TextRenderer.$inject = ['config.textRenderer'];

  /**
   * Map containing SVG paths needed by BpmnRenderer.
   */

  function PathMap() {
    /**
     * Contains a map of path elements
     *
     * <h1>Path definition</h1>
     * A parameterized path is defined like this:
     * <pre>
     * 'GATEWAY_PARALLEL': {
     *   d: 'm {mx},{my} {e.x0},0 0,{e.x1} {e.x1},0 0,{e.y0} -{e.x1},0 0,{e.y1} ' +
            '-{e.x0},0 0,-{e.y1} -{e.x1},0 0,-{e.y0} {e.x1},0 z',
     *   height: 17.5,
     *   width:  17.5,
     *   heightElements: [2.5, 7.5],
     *   widthElements: [2.5, 7.5]
     * }
     * </pre>
     * <p>It's important to specify a correct <b>height and width</b> for the path as the scaling
     * is based on the ratio between the specified height and width in this object and the
     * height and width that is set as scale target (Note x,y coordinates will be scaled with
     * individual ratios).</p>
     * <p>The '<b>heightElements</b>' and '<b>widthElements</b>' array must contain the values that will be scaled.
     * The scaling is based on the computed ratios.
     * Coordinates on the y axis should be in the <b>heightElement</b>'s array, they will be scaled using
     * the computed ratio coefficient.
     * In the parameterized path the scaled values can be accessed through the 'e' object in {} brackets.
     *   <ul>
     *    <li>The values for the y axis can be accessed in the path string using {e.y0}, {e.y1}, ....</li>
     *    <li>The values for the x axis can be accessed in the path string using {e.x0}, {e.x1}, ....</li>
     *   </ul>
     *   The numbers x0, x1 respectively y0, y1, ... map to the corresponding array index.
     * </p>
      m1,1
      l 0,55.3
      c 29.8,19.7 48.4,-4.2 67.2,-6.7
      c 12.2,-2.3 19.8,1.6 30.8,6.2
      l 0,-54.6
      z
      */
    this.pathMap = {
      'KNOWLEDGE_SOURCE': {
        d: 'm {mx},{my} ' + 'l 0,{e.y0} ' + 'c {e.x0},{e.y1} {e.x1},-{e.y2} {e.x2},-{e.y3} ' + 'c {e.x3},-{e.y4} {e.x4},{e.y5} {e.x5},{e.y6} ' + 'l 0,-{e.y7}z',
        width: 100,
        height: 65,
        widthElements: [29.8, 48.4, 67.2, 12.2, 19.8, 30.8],
        heightElements: [55.3, 19.7, 4.2, 6.7, 2.3, 1.6, 6.2, 54.6]
      },
      'BUSINESS_KNOWLEDGE_MODEL': {
        d: 'm {mx},{my} l {e.x0},-{e.y0} l {e.x1},0 l 0,{e.y1} l -{e.x2},{e.y2} l -{e.x3},0z',
        width: 125,
        height: 45,
        widthElements: [13.8, 109.2, 13.8, 109.1],
        heightElements: [13.2, 29.8, 13.2]
      },
      'TEXT_ANNOTATION': {
        d: 'm {mx}, {my} m 10,0 l -10,0 l 0,{e.y0} l 10,0',
        width: 10,
        height: 30,
        widthElements: [10],
        heightElements: [30]
      }
    };
    this.getRawPath = function getRawPath(pathId) {
      return this.pathMap[pathId].d;
    };

    /**
     * Scales the path to the given height and width.
     * <h1>Use case</h1>
     * <p>Use case is to scale the content of elements (event, gateways) based
     * on the element bounding box's size.
     * </p>
     * <h1>Why not transform</h1>
     * <p>Scaling a path with transform() will also scale the stroke and IE does not support
     * the option 'non-scaling-stroke' to prevent this.
     * Also there are use cases where only some parts of a path should be
     * scaled.</p>
     *
     * @param {string} pathId The ID of the path.
     * @param {Object} param <p>
     *   Example param object scales the path to 60% size of the container (data.width, data.height).
     *   <pre>
     *   {
     *     xScaleFactor: 0.6,
     *     yScaleFactor:0.6,
     *     containerWidth: data.width,
     *     containerHeight: data.height,
     *     position: {
     *       mx: 0.46,
     *       my: 0.2,
     *     }
     *   }
     *   </pre>
     *   <ul>
     *    <li>targetpathwidth = xScaleFactor * containerWidth</li>
     *    <li>targetpathheight = yScaleFactor * containerHeight</li>
     *    <li>Position is used to set the starting coordinate of the path. M is computed:
      *    <ul>
      *      <li>position.x * containerWidth</li>
      *      <li>position.y * containerHeight</li>
      *    </ul>
      *    Center of the container <pre> position: {
     *       mx: 0.5,
     *       my: 0.5,
     *     }</pre>
     *     Upper left corner of the container
     *     <pre> position: {
     *       mx: 0.0,
     *       my: 0.0,
     *     }</pre>
     *    </li>
     *   </ul>
     * </p>
     *
     */
    this.getScaledPath = function getScaledPath(pathId, param) {
      var rawPath = this.pathMap[pathId];

      // positioning
      // compute the start point of the path
      var mx, my;
      if (param.abspos) {
        mx = param.abspos.x;
        my = param.abspos.y;
      } else {
        mx = param.containerWidth * param.position.mx;
        my = param.containerHeight * param.position.my;
      }
      var coordinates = {}; // map for the scaled coordinates
      if (param.position) {
        // path
        var heightRatio = param.containerHeight / rawPath.height * param.yScaleFactor;
        var widthRatio = param.containerWidth / rawPath.width * param.xScaleFactor;

        // Apply height ratio
        for (var heightIndex = 0; heightIndex < rawPath.heightElements.length; heightIndex++) {
          coordinates['y' + heightIndex] = rawPath.heightElements[heightIndex] * heightRatio;
        }

        // Apply width ratio
        for (var widthIndex = 0; widthIndex < rawPath.widthElements.length; widthIndex++) {
          coordinates['x' + widthIndex] = rawPath.widthElements[widthIndex] * widthRatio;
        }
      }

      // Apply value to raw path
      var path = format(rawPath.d, {
        mx: mx,
        my: my,
        e: coordinates
      });
      return path;
    };
  }

  // helpers //////////////////////

  // copied and adjusted from https://github.com/adobe-webplatform/Snap.svg/blob/master/src/svg.js
  var tokenRegex = /\{([^{}]+)\}/g,
    objNotationRegex = /(?:(?:^|\.)(.+?)(?=\[|\.|$|\()|\[('|")(.+?)\2\])(\(\))?/g; // matches .xxxxx or ["xxxxx"] to run over object properties

  function replacer(all, key, obj) {
    var res = obj;
    key.replace(objNotationRegex, function (all, name, quote, quotedName, isFunc) {
      name = name || quotedName;
      if (res) {
        if (name in res) {
          res = res[name];
        }
        typeof res == 'function' && isFunc && (res = res());
      }
    });
    res = (res == null || res == obj ? all : res) + '';
    return res;
  }
  function format(str, obj) {
    return String(str).replace(tokenRegex, function (all, key) {
      return replacer(all, key, obj);
    });
  }

  var DrawModule = {
    __init__: ['drdRenderer'],
    drdRenderer: ['type', DrdRenderer],
    textRenderer: ['type', TextRenderer],
    pathMap: ['type', PathMap]
  };

  function DrdImporter(eventBus, canvas, elementFactory, elementRegistry) {
    this._eventBus = eventBus;
    this._canvas = canvas;
    this._elementRegistry = elementRegistry;
    this._elementFactory = elementFactory;
  }
  DrdImporter.$inject = ['eventBus', 'canvas', 'elementFactory', 'elementRegistry'];
  DrdImporter.prototype.root = function (semantic) {
    var element = this._elementFactory.createRoot(elementData$1(semantic));
    this._canvas.setRootElement(element);
    return element;
  };

  /**
   * Add drd element (semantic) to the canvas.
   */
  DrdImporter.prototype.add = function (semantic) {
    var elementFactory = this._elementFactory,
      canvas = this._canvas,
      eventBus = this._eventBus,
      di = semantic.di;
    var element, waypoints, source, target, elementDefinition, bounds;
    if (di.$instanceOf('dmndi:DMNShape')) {
      bounds = di.bounds;
      elementDefinition = elementData$1(semantic, {
        x: Math.round(bounds.x),
        y: Math.round(bounds.y),
        width: Math.round(bounds.width),
        height: Math.round(bounds.height)
      });
      element = elementFactory.createShape(elementDefinition);
      canvas.addShape(element);
      eventBus.fire('drdElement.added', {
        element: element,
        di: di
      });
    } else if (di.$instanceOf('dmndi:DMNEdge')) {
      waypoints = collectWaypoints(di);
      source = this._getSource(semantic);
      target = this._getTarget(semantic);
      if (source && target) {
        elementDefinition = elementData$1(semantic, {
          hidden: false,
          source: source,
          target: target,
          waypoints: waypoints
        });
        element = elementFactory.createConnection(elementDefinition);
        canvas.addConnection(element);
        eventBus.fire('drdElement.added', {
          element: element,
          di: di
        });
      }
    } else {
      throw new Error('unknown di for element ' + semantic.id);
    }
    return element;
  };
  DrdImporter.prototype._getSource = function (semantic) {
    var href, elementReference;
    if (is$1(semantic, 'dmn:Association')) {
      elementReference = semantic.sourceRef;
    } else if (is$1(semantic, 'dmn:InformationRequirement')) {
      elementReference = semantic.requiredDecision || semantic.requiredInput;
    } else if (is$1(semantic, 'dmn:KnowledgeRequirement')) {
      elementReference = semantic.requiredKnowledge;
    } else if (is$1(semantic, 'dmn:AuthorityRequirement')) {
      elementReference = semantic.requiredDecision || semantic.requiredInput || semantic.requiredAuthority;
    }
    if (elementReference) {
      href = elementReference.href;
    }
    if (href) {
      return this._getShape(getIdFromHref(href));
    }
  };
  DrdImporter.prototype._getTarget = function (semantic) {
    if (is$1(semantic, 'dmn:Association')) {
      return semantic.targetRef && this._getShape(getIdFromHref(semantic.targetRef.href));
    }
    return this._getShape(semantic.$parent.id);
  };
  DrdImporter.prototype._getShape = function (id) {
    return this._elementRegistry.get(id);
  };

  // helper /////
  function elementData$1(semantic, attrs) {
    return assign$4({
      id: semantic.id,
      type: semantic.$type,
      businessObject: semantic
    }, attrs);
  }
  function collectWaypoints(edge) {
    var waypoints = edge.waypoint;
    if (waypoints) {
      return map$3(waypoints, function (waypoint) {
        var position = {
          x: waypoint.x,
          y: waypoint.y
        };
        return assign$4({
          original: position
        }, position);
      });
    }
  }
  function getIdFromHref(href) {
    return href.split('#').pop();
  }

  var ImportModule = {
    drdImporter: ['type', DrdImporter]
  };

  var CoreModule$2 = {
    __depends__: [DrawModule, ImportModule]
  };

  /**
   * @typedef { {
   *   [key: string]: string;
   * } } TranslateReplacements
   */

  /**
   * A simple translation stub to be used for multi-language support
   * in diagrams. Can be easily replaced with a more sophisticated
   * solution.
   *
   * @example
   *
   * ```javascript
   * // use it inside any diagram component by injecting `translate`.
   *
   * function MyService(translate) {
   *   alert(translate('HELLO {you}', { you: 'You!' }));
   * }
   * ```
   *
   * @param {string} template to interpolate
   * @param {TranslateReplacements} [replacements] a map with substitutes
   *
   * @return {string} the translated string
   */
  function translate(template, replacements) {
    replacements = replacements || {};
    return template.replace(/{([^}]+)}/g, function (_, key) {
      return replacements[key] || '{' + key + '}';
    });
  }

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var TranslateModule = {
    translate: ['value', translate]
  };

  /**
   * @typedef {import('../util/Types').Point} Point
   */

  function __stopPropagation(event) {
    if (!event || typeof event.stopPropagation !== 'function') {
      return;
    }
    event.stopPropagation();
  }

  /**
   * @param {import('../core/EventBus').Event} event
   *
   * @return {Event}
   */
  function getOriginal$1(event) {
    return event.originalEvent || event.srcEvent;
  }

  /**
   * @param {Event|import('../core/EventBus').Event} event
   */
  function stopPropagation$2(event) {
    __stopPropagation(event);
    __stopPropagation(getOriginal$1(event));
  }

  /**
   * @param {Event} event
   *
   * @return {Point|null}
   */
  function toPoint(event) {
    if (event.pointers && event.pointers.length) {
      event = event.pointers[0];
    }
    if (event.touches && event.touches.length) {
      event = event.touches[0];
    }
    return event ? {
      x: event.clientX,
      y: event.clientY
    } : null;
  }

  function isMac() {
    return /mac/i.test(navigator.platform);
  }

  /**
   * @param {MouseEvent} event
   * @param {string} button
   *
   * @return {boolean}
   */
  function isButton(event, button) {
    return (getOriginal$1(event) || event).button === button;
  }

  /**
   * @param {MouseEvent} event
   *
   * @return {boolean}
   */
  function isPrimaryButton(event) {
    // button === 0 -> left áka primary mouse button
    return isButton(event, 0);
  }

  /**
   * @param {MouseEvent} event
   *
   * @return {boolean}
   */
  function isAuxiliaryButton(event) {
    // button === 1 -> auxiliary áka wheel button
    return isButton(event, 1);
  }

  /**
   * @param {MouseEvent} event
   *
   * @return {boolean}
   */
  function hasPrimaryModifier(event) {
    var originalEvent = getOriginal$1(event) || event;
    if (!isPrimaryButton(event)) {
      return false;
    }

    // Use cmd as primary modifier key for mac OS
    if (isMac()) {
      return originalEvent.metaKey;
    } else {
      return originalEvent.ctrlKey;
    }
  }

  /**
   * @param {MouseEvent} event
   *
   * @return {boolean}
   */
  function hasSecondaryModifier(event) {
    var originalEvent = getOriginal$1(event) || event;
    return isPrimaryButton(event) && originalEvent.shiftKey;
  }

  /**
   * @typedef {import('../../model/Types').Element} Element
   *
   * @typedef {import('../../core/ElementRegistry').default} ElementRegistry
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('../../draw/Styles').default} Styles
   *
   * @typedef {import('../../util/Types').Point} Point
   */

  function allowAll(event) {
    return true;
  }
  function allowPrimaryAndAuxiliary(event) {
    return isPrimaryButton(event) || isAuxiliaryButton(event);
  }
  var LOW_PRIORITY$m = 500;

  /**
   * A plugin that provides interaction events for diagram elements.
   *
   * It emits the following events:
   *
   *   * element.click
   *   * element.contextmenu
   *   * element.dblclick
   *   * element.hover
   *   * element.mousedown
   *   * element.mousemove
   *   * element.mouseup
   *   * element.out
   *
   * Each event is a tuple { element, gfx, originalEvent }.
   *
   * Canceling the event via Event#preventDefault()
   * prevents the original DOM operation.
   *
   * @param {EventBus} eventBus
   * @param {ElementRegistry} elementRegistry
   * @param {Styles} styles
   */
  function InteractionEvents$1(eventBus, elementRegistry, styles) {
    var self = this;

    /**
     * Fire an interaction event.
     *
     * @param {string} type local event name, e.g. element.click.
     * @param {MouseEvent|TouchEvent} event native event
     * @param {Element} [element] the diagram element to emit the event on;
     *                                   defaults to the event target
     */
    function fire(type, event, element) {
      if (isIgnored(type, event)) {
        return;
      }
      var target, gfx, returnValue;
      if (!element) {
        target = event.delegateTarget || event.target;
        if (target) {
          gfx = target;
          element = elementRegistry.get(gfx);
        }
      } else {
        gfx = elementRegistry.getGraphics(element);
      }
      if (!gfx || !element) {
        return;
      }
      returnValue = eventBus.fire(type, {
        element: element,
        gfx: gfx,
        originalEvent: event
      });
      if (returnValue === false) {
        event.stopPropagation();
        event.preventDefault();
      }
    }

    // TODO(nikku): document this
    var handlers = {};
    function mouseHandler(localEventName) {
      return handlers[localEventName];
    }
    function isIgnored(localEventName, event) {
      var filter = ignoredFilters[localEventName] || isPrimaryButton;

      // only react on left mouse button interactions
      // except for interaction events that are enabled
      // for secundary mouse button
      return !filter(event);
    }
    var bindings = {
      click: 'element.click',
      contextmenu: 'element.contextmenu',
      dblclick: 'element.dblclick',
      mousedown: 'element.mousedown',
      mousemove: 'element.mousemove',
      mouseover: 'element.hover',
      mouseout: 'element.out',
      mouseup: 'element.mouseup'
    };
    var ignoredFilters = {
      'element.contextmenu': allowAll,
      'element.mousedown': allowPrimaryAndAuxiliary,
      'element.mouseup': allowPrimaryAndAuxiliary,
      'element.click': allowPrimaryAndAuxiliary,
      'element.dblclick': allowPrimaryAndAuxiliary
    };

    // manual event trigger //////////

    /**
     * Trigger an interaction event (based on a native dom event)
     * on the target shape or connection.
     *
     * @param {string} eventName the name of the triggered DOM event
     * @param {MouseEvent|TouchEvent} event
     * @param {Element} targetElement
     */
    function triggerMouseEvent(eventName, event, targetElement) {
      // i.e. element.mousedown...
      var localEventName = bindings[eventName];
      if (!localEventName) {
        throw new Error('unmapped DOM event name <' + eventName + '>');
      }
      return fire(localEventName, event, targetElement);
    }
    var ELEMENT_SELECTOR = 'svg, .djs-element';

    // event handling ///////

    function registerEvent(node, event, localEvent, ignoredFilter) {
      var handler = handlers[localEvent] = function (event) {
        fire(localEvent, event);
      };
      if (ignoredFilter) {
        ignoredFilters[localEvent] = ignoredFilter;
      }
      handler.$delegate = delegate.bind(node, ELEMENT_SELECTOR, event, handler);
    }
    function unregisterEvent(node, event, localEvent) {
      var handler = mouseHandler(localEvent);
      if (!handler) {
        return;
      }
      delegate.unbind(node, event, handler.$delegate);
    }
    function registerEvents(svg) {
      forEach$3(bindings, function (val, key) {
        registerEvent(svg, key, val);
      });
    }
    function unregisterEvents(svg) {
      forEach$3(bindings, function (val, key) {
        unregisterEvent(svg, key, val);
      });
    }
    eventBus.on('canvas.destroy', function (event) {
      unregisterEvents(event.svg);
    });
    eventBus.on('canvas.init', function (event) {
      registerEvents(event.svg);
    });

    // hit box updating ////////////////

    eventBus.on(['shape.added', 'connection.added'], function (event) {
      var element = event.element,
        gfx = event.gfx;
      eventBus.fire('interactionEvents.createHit', {
        element: element,
        gfx: gfx
      });
    });

    // Update djs-hit on change.
    // A low priortity is necessary, because djs-hit of labels has to be updated
    // after the label bounds have been updated in the renderer.
    eventBus.on(['shape.changed', 'connection.changed'], LOW_PRIORITY$m, function (event) {
      var element = event.element,
        gfx = event.gfx;
      eventBus.fire('interactionEvents.updateHit', {
        element: element,
        gfx: gfx
      });
    });
    eventBus.on('interactionEvents.createHit', LOW_PRIORITY$m, function (event) {
      var element = event.element,
        gfx = event.gfx;
      self.createDefaultHit(element, gfx);
    });
    eventBus.on('interactionEvents.updateHit', function (event) {
      var element = event.element,
        gfx = event.gfx;
      self.updateDefaultHit(element, gfx);
    });

    // hit styles ////////////

    var STROKE_HIT_STYLE = createHitStyle('djs-hit djs-hit-stroke');
    var CLICK_STROKE_HIT_STYLE = createHitStyle('djs-hit djs-hit-click-stroke');
    var ALL_HIT_STYLE = createHitStyle('djs-hit djs-hit-all');
    var NO_MOVE_HIT_STYLE = createHitStyle('djs-hit djs-hit-no-move');
    var HIT_TYPES = {
      'all': ALL_HIT_STYLE,
      'click-stroke': CLICK_STROKE_HIT_STYLE,
      'stroke': STROKE_HIT_STYLE,
      'no-move': NO_MOVE_HIT_STYLE
    };
    function createHitStyle(classNames, attrs) {
      attrs = assign$4({
        stroke: 'white',
        strokeWidth: 15
      }, attrs || {});
      return styles.cls(classNames, ['no-fill', 'no-border'], attrs);
    }

    // style helpers ///////////////

    function applyStyle(hit, type) {
      var attrs = HIT_TYPES[type];
      if (!attrs) {
        throw new Error('invalid hit type <' + type + '>');
      }
      attr(hit, attrs);
      return hit;
    }
    function appendHit(gfx, hit) {
      append(gfx, hit);
    }

    // API

    /**
     * Remove hints on the given graphics.
     *
     * @param {SVGElement} gfx
     */
    this.removeHits = function (gfx) {
      var hits = all('.djs-hit', gfx);
      forEach$3(hits, remove$3);
    };

    /**
     * Create default hit for the given element.
     *
     * @param {Element} element
     * @param {SVGElement} gfx
     *
     * @return {SVGElement} created hit
     */
    this.createDefaultHit = function (element, gfx) {
      var waypoints = element.waypoints,
        isFrame = element.isFrame,
        boxType;
      if (waypoints) {
        return this.createWaypointsHit(gfx, waypoints);
      } else {
        boxType = isFrame ? 'stroke' : 'all';
        return this.createBoxHit(gfx, boxType, {
          width: element.width,
          height: element.height
        });
      }
    };

    /**
     * Create hits for the given waypoints.
     *
     * @param {SVGElement} gfx
     * @param {Point[]} waypoints
     *
     * @return {SVGElement}
     */
    this.createWaypointsHit = function (gfx, waypoints) {
      var hit = createLine(waypoints);
      applyStyle(hit, 'stroke');
      appendHit(gfx, hit);
      return hit;
    };

    /**
     * Create hits for a box.
     *
     * @param {SVGElement} gfx
     * @param {string} type
     * @param {Object} attrs
     *
     * @return {SVGElement}
     */
    this.createBoxHit = function (gfx, type, attrs) {
      attrs = assign$4({
        x: 0,
        y: 0
      }, attrs);
      var hit = create$2('rect');
      applyStyle(hit, type);
      attr(hit, attrs);
      appendHit(gfx, hit);
      return hit;
    };

    /**
     * Update default hit of the element.
     *
     * @param {Element} element
     * @param {SVGElement} gfx
     *
     * @return {SVGElement} updated hit
     */
    this.updateDefaultHit = function (element, gfx) {
      var hit = query('.djs-hit', gfx);
      if (!hit) {
        return;
      }
      if (element.waypoints) {
        updateLine(hit, element.waypoints);
      } else {
        attr(hit, {
          width: element.width,
          height: element.height
        });
      }
      return hit;
    };
    this.fire = fire;
    this.triggerMouseEvent = triggerMouseEvent;
    this.mouseHandler = mouseHandler;
    this.registerEvent = registerEvent;
    this.unregisterEvent = unregisterEvent;
  }
  InteractionEvents$1.$inject = ['eventBus', 'elementRegistry', 'styles'];

  /**
   * An event indicating that the mouse hovered over an element
   *
   * @event element.hover
   *
   * @type {Object}
   * @property {Element} element
   * @property {SVGElement} gfx
   * @property {Event} originalEvent
   */

  /**
   * An event indicating that the mouse has left an element
   *
   * @event element.out
   *
   * @type {Object}
   * @property {Element} element
   * @property {SVGElement} gfx
   * @property {Event} originalEvent
   */

  /**
   * An event indicating that the mouse has clicked an element
   *
   * @event element.click
   *
   * @type {Object}
   * @property {Element} element
   * @property {SVGElement} gfx
   * @property {Event} originalEvent
   */

  /**
   * An event indicating that the mouse has double clicked an element
   *
   * @event element.dblclick
   *
   * @type {Object}
   * @property {Element} element
   * @property {SVGElement} gfx
   * @property {Event} originalEvent
   */

  /**
   * An event indicating that the mouse has gone down on an element.
   *
   * @event element.mousedown
   *
   * @type {Object}
   * @property {Element} element
   * @property {SVGElement} gfx
   * @property {Event} originalEvent
   */

  /**
   * An event indicating that the mouse has gone up on an element.
   *
   * @event element.mouseup
   *
   * @type {Object}
   * @property {Element} element
   * @property {SVGElement} gfx
   * @property {Event} originalEvent
   */

  /**
   * An event indicating that the context menu action is triggered
   * via mouse or touch controls.
   *
   * @event element.contextmenu
   *
   * @type {Object}
   * @property {Element} element
   * @property {SVGElement} gfx
   * @property {Event} originalEvent
   */

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var InteractionEventsModule = {
    __init__: ['interactionEvents'],
    interactionEvents: ['type', InteractionEvents$1]
  };

  /**
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/EventBus').default} EventBus
   */

  /**
   * A service that offers the current selection in a diagram.
   * Offers the api to control the selection, too.
   *
   * @param {EventBus} eventBus
   * @param {Canvas} canvas
   */
  function Selection$1(eventBus, canvas) {
    this._eventBus = eventBus;
    this._canvas = canvas;

    /**
     * @type {Object[]}
     */
    this._selectedElements = [];
    var self = this;
    eventBus.on(['shape.remove', 'connection.remove'], function (e) {
      var element = e.element;
      self.deselect(element);
    });
    eventBus.on(['diagram.clear', 'root.set'], function (e) {
      self.select(null);
    });
  }
  Selection$1.$inject = ['eventBus', 'canvas'];

  /**
   * Deselect an element.
   *
   * @param {Object} element The element to deselect.
   */
  Selection$1.prototype.deselect = function (element) {
    var selectedElements = this._selectedElements;
    var idx = selectedElements.indexOf(element);
    if (idx !== -1) {
      var oldSelection = selectedElements.slice();
      selectedElements.splice(idx, 1);
      this._eventBus.fire('selection.changed', {
        oldSelection: oldSelection,
        newSelection: selectedElements
      });
    }
  };

  /**
   * Get the selected elements.
   *
   * @return {Object[]} The selected elements.
   */
  Selection$1.prototype.get = function () {
    return this._selectedElements;
  };

  /**
   * Check whether an element is selected.
   *
   * @param {Object} element The element.
   *
   * @return {boolean} Whether the element is selected.
   */
  Selection$1.prototype.isSelected = function (element) {
    return this._selectedElements.indexOf(element) !== -1;
  };

  /**
   * Select one or many elements.
   *
   * @param {Object|Object[]} elements The element(s) to select.
   * @param {boolean} [add] Whether to add the element(s) to the selected elements.
   * Defaults to `false`.
   */
  Selection$1.prototype.select = function (elements, add) {
    var selectedElements = this._selectedElements,
      oldSelection = selectedElements.slice();
    if (!isArray$6(elements)) {
      elements = elements ? [elements] : [];
    }
    var canvas = this._canvas;
    var rootElement = canvas.getRootElement();
    elements = elements.filter(function (element) {
      var elementRoot = canvas.findRoot(element);
      return rootElement === elementRoot;
    });

    // selection may be cleared by passing an empty array or null
    // to the method
    if (add) {
      forEach$3(elements, function (element) {
        if (selectedElements.indexOf(element) !== -1) {
          // already selected
          return;
        } else {
          selectedElements.push(element);
        }
      });
    } else {
      this._selectedElements = selectedElements = elements.slice();
    }
    this._eventBus.fire('selection.changed', {
      oldSelection: oldSelection,
      newSelection: selectedElements
    });
  };

  /**
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/EventBus').default} EventBus
   */

  var MARKER_HOVER = 'hover',
    MARKER_SELECTED = 'selected';

  /**
   * A plugin that adds a visible selection UI to shapes and connections
   * by appending the <code>hover</code> and <code>selected</code> classes to them.
   *
   * @class
   *
   * Makes elements selectable, too.
   *
   * @param {Canvas} canvas
   * @param {EventBus} eventBus
   */
  function SelectionVisuals(canvas, eventBus) {
    this._canvas = canvas;
    function addMarker(e, cls) {
      canvas.addMarker(e, cls);
    }
    function removeMarker(e, cls) {
      canvas.removeMarker(e, cls);
    }
    eventBus.on('element.hover', function (event) {
      addMarker(event.element, MARKER_HOVER);
    });
    eventBus.on('element.out', function (event) {
      removeMarker(event.element, MARKER_HOVER);
    });
    eventBus.on('selection.changed', function (event) {
      function deselect(s) {
        removeMarker(s, MARKER_SELECTED);
      }
      function select(s) {
        addMarker(s, MARKER_SELECTED);
      }
      var oldSelection = event.oldSelection,
        newSelection = event.newSelection;
      forEach$3(oldSelection, function (e) {
        if (newSelection.indexOf(e) === -1) {
          deselect(e);
        }
      });
      forEach$3(newSelection, function (e) {
        if (oldSelection.indexOf(e) === -1) {
          select(e);
        }
      });
    });
  }
  SelectionVisuals.$inject = ['canvas', 'eventBus'];

  /**
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/ElementRegistry').default} ElementRegistry
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('./Selection').default} Selection
   */

  /**
   * @param {EventBus} eventBus
   * @param {Selection} selection
   * @param {Canvas} canvas
   * @param {ElementRegistry} elementRegistry
   */
  function SelectionBehavior$1(eventBus, selection, canvas, elementRegistry) {
    // Select elements on create
    eventBus.on('create.end', 500, function (event) {
      var context = event.context,
        canExecute = context.canExecute,
        elements = context.elements,
        hints = context.hints || {},
        autoSelect = hints.autoSelect;
      if (canExecute) {
        if (autoSelect === false) {
          // Select no elements
          return;
        }
        if (isArray$6(autoSelect)) {
          selection.select(autoSelect);
        } else {
          // Select all elements by default
          selection.select(elements.filter(isShown));
        }
      }
    });

    // Select connection targets on connect
    eventBus.on('connect.end', 500, function (event) {
      var context = event.context,
        connection = context.connection;
      if (connection) {
        selection.select(connection);
      }
    });

    // Select shapes on move
    eventBus.on('shape.move.end', 500, function (event) {
      var previousSelection = event.previousSelection || [];
      var shape = elementRegistry.get(event.context.shape.id);

      // Always select main shape on move
      var isSelected = find$2(previousSelection, function (selectedShape) {
        return shape.id === selectedShape.id;
      });
      if (!isSelected) {
        selection.select(shape);
      }
    });

    // Select elements on click
    eventBus.on('element.click', function (event) {
      if (!isPrimaryButton(event)) {
        return;
      }
      var element = event.element;
      if (element === canvas.getRootElement()) {
        element = null;
      }
      var isSelected = selection.isSelected(element),
        isMultiSelect = selection.get().length > 1;

      // Add to selection if SHIFT pressed
      var add = hasSecondaryModifier(event);
      if (isSelected && isMultiSelect) {
        if (add) {
          // Deselect element
          return selection.deselect(element);
        } else {
          // Select element only
          return selection.select(element);
        }
      } else if (!isSelected) {
        // Select element
        selection.select(element, add);
      } else {
        // Deselect element
        selection.deselect(element);
      }
    });
  }
  SelectionBehavior$1.$inject = ['eventBus', 'selection', 'canvas', 'elementRegistry'];
  function isShown(element) {
    return !element.hidden;
  }

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var SelectionModule = {
    __init__: ['selectionVisuals', 'selectionBehavior'],
    __depends__: [InteractionEventsModule],
    selection: ['type', Selection$1],
    selectionVisuals: ['type', SelectionVisuals],
    selectionBehavior: ['type', SelectionBehavior$1]
  };

  /**
   * Util that provides unique IDs.
   *
   * @class
   * @constructor
   *
   * The ids can be customized via a given prefix and contain a random value to avoid collisions.
   *
   * @param {string} [prefix] a prefix to prepend to generated ids (for better readability)
   */
  function IdGenerator(prefix) {
    this._counter = 0;
    this._prefix = (prefix ? prefix + '-' : '') + Math.floor(Math.random() * 1000000000) + '-';
  }

  /**
   * Returns a next unique ID.
   *
   * @return {string} the id
   */
  IdGenerator.prototype.next = function () {
    return this._prefix + ++this._counter;
  };

  // document wide unique overlay ids
  var ids = new IdGenerator('ov');
  var LOW_PRIORITY$l = 500;

  /**
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/ElementRegistry').default} ElementRegistry
   * @typedef {import('../../core/EventBus').default} EventBus
   *
   * @typedef {import('../../model/Types').Element} Element
   *
   * @typedef { {
   *   minZoom?: number,
   *   maxZoom?: number
   * } } OverlaysConfigShow
   *
   * @typedef { {
   *   min?: number,
   *   max?: number
   * } } OverlaysConfigScale
   *
   * @typedef { {
  *   id: string,
  *   type: string | null,
  *   element: Element | string
  * } & OverlayAttrs } Overlay
  *
   * @typedef { {
   *   html: HTMLElement | string,
   *   position: {
   *     top?: number,
   *     right?: number,
   *     bottom?: number,
   *     left?: number
   *   }
   * } & OverlaysConfigDefault } OverlayAttrs
   *
   * @typedef { {
   *   html: HTMLElement,
   *   element: Element,
   *   overlays: Overlay[]
   * } } OverlayContainer
   *
   * @typedef {{
   *   defaults?: OverlaysConfigDefault
   * }} OverlaysConfig
   *
   * @typedef { {
   *  show?: OverlaysConfigShow,
   *  scale?: OverlaysConfigScale | boolean
   * } } OverlaysConfigDefault
   *
   * @typedef { {
   *   id?: string;
   *   element?: Element | string;
   *   type?: string;
   * } | string } OverlaysFilter
   */

  /**
   * A service that allows users to attach overlays to diagram elements.
   *
   * The overlay service will take care of overlay positioning during updates.
   *
   * @example
   *
   * ```javascript
   * // add a pink badge on the top left of the shape
   *
   * overlays.add(someShape, {
   *   position: {
   *     top: -5,
   *     left: -5
   *   },
   *   html: '<div style="width: 10px; background: fuchsia; color: white;">0</div>'
   * });
   *
   * // or add via shape id
   *
   * overlays.add('some-element-id', {
   *   position: {
   *     top: -5,
   *     left: -5
   *   }
   *   html: '<div style="width: 10px; background: fuchsia; color: white;">0</div>'
   * });
   *
   * // or add with optional type
   *
   * overlays.add(someShape, 'badge', {
   *   position: {
   *     top: -5,
   *     left: -5
   *   }
   *   html: '<div style="width: 10px; background: fuchsia; color: white;">0</div>'
   * });
   * ```
   *
   * ```javascript
   * // remove an overlay
   *
   * var id = overlays.add(...);
   * overlays.remove(id);
   *
   *
   * You may configure overlay defaults during tool by providing a `config` module
   * with `overlays.defaults` as an entry:
   *
   * {
   *   overlays: {
   *     defaults: {
   *       show: {
   *         minZoom: 0.7,
   *         maxZoom: 5.0
   *       },
   *       scale: {
   *         min: 1
   *       }
   *     }
   * }
   * ```
   *
   * @param {OverlaysConfig} config
   * @param {EventBus} eventBus
   * @param {Canvas} canvas
   * @param {ElementRegistry} elementRegistry
   */
  function Overlays(config, eventBus, canvas, elementRegistry) {
    this._eventBus = eventBus;
    this._canvas = canvas;
    this._elementRegistry = elementRegistry;
    this._ids = ids;

    /**
     * @type {OverlaysConfigDefault}
     */
    this._overlayDefaults = assign$4({
      // no show constraints
      show: null,
      // always scale
      scale: true
    }, config && config.defaults);

    /**
     * @type {Record<string, Overlay>}
     */
    this._overlays = {};

    /**
     * @type {OverlayContainer[]}
     */
    this._overlayContainers = [];

    /**
     * @type {HTMLElement}
     */
    this._overlayRoot = createRoot(canvas.getContainer());
    this._init();
  }
  Overlays.$inject = ['config.overlays', 'eventBus', 'canvas', 'elementRegistry'];

  /**
   * Returns the overlay with the specified ID or a list of overlays
   * for an element with a given type.
   *
   * @example
   *
   * ```javascript
   * // return the single overlay with the given ID
   * overlays.get('some-id');
   *
   * // return all overlays for the shape
   * overlays.get({ element: someShape });
   *
   * // return all overlays on shape with type 'badge'
   * overlays.get({ element: someShape, type: 'badge' });
   *
   * // shape can also be specified as ID
   * overlays.get({ element: 'element-id', type: 'badge' });
   * ```
   *
   * @param {OverlaysFilter} search The filter to be used to find the overlay(s).
   *
   * @return {Overlay|Overlay[]} The overlay(s).
   */
  Overlays.prototype.get = function (search) {
    if (isString$4(search)) {
      search = {
        id: search
      };
    }
    if (isString$4(search.element)) {
      search.element = this._elementRegistry.get(search.element);
    }
    if (search.element) {
      var container = this._getOverlayContainer(search.element, true);

      // return a list of overlays when searching by element (+type)
      if (container) {
        return search.type ? filter$2(container.overlays, matchPattern({
          type: search.type
        })) : container.overlays.slice();
      } else {
        return [];
      }
    } else if (search.type) {
      return filter$2(this._overlays, matchPattern({
        type: search.type
      }));
    } else {
      // return single element when searching by id
      return search.id ? this._overlays[search.id] : null;
    }
  };

  /**
   * Adds an HTML overlay to an element.
   *
   * @param {Element|string} element The element to add the overlay to.
   * @param {string} [type] An optional type that can be used to filter.
   * @param {OverlayAttrs} overlay The overlay.
   *
   * @return {string} The overlay's ID that can be used to get or remove it.
   */
  Overlays.prototype.add = function (element, type, overlay) {
    if (isObject$2(type)) {
      overlay = type;
      type = null;
    }
    if (!element.id) {
      element = this._elementRegistry.get(element);
    }
    if (!overlay.position) {
      throw new Error('must specifiy overlay position');
    }
    if (!overlay.html) {
      throw new Error('must specifiy overlay html');
    }
    if (!element) {
      throw new Error('invalid element specified');
    }
    var id = this._ids.next();
    overlay = assign$4({}, this._overlayDefaults, overlay, {
      id: id,
      type: type,
      element: element,
      html: overlay.html
    });
    this._addOverlay(overlay);
    return id;
  };

  /**
   * Remove an overlay with the given ID or all overlays matching the given filter.
   *
   * @see Overlays#get for filter options.
   *
   * @param {OverlaysFilter} filter The filter to be used to find the overlay.
   */
  Overlays.prototype.remove = function (filter) {
    var overlays = this.get(filter) || [];
    if (!isArray$6(overlays)) {
      overlays = [overlays];
    }
    var self = this;
    forEach$3(overlays, function (overlay) {
      var container = self._getOverlayContainer(overlay.element, true);
      if (overlay) {
        remove$4(overlay.html);
        remove$4(overlay.htmlContainer);
        delete overlay.htmlContainer;
        delete overlay.element;
        delete self._overlays[overlay.id];
      }
      if (container) {
        var idx = container.overlays.indexOf(overlay);
        if (idx !== -1) {
          container.overlays.splice(idx, 1);
        }
      }
    });
  };

  /**
   * Checks whether overlays are shown.
   *
   * @return {boolean} Whether overlays are shown.
   */
  Overlays.prototype.isShown = function () {
    return this._overlayRoot.style.display !== 'none';
  };

  /**
   * Show all overlays.
   */
  Overlays.prototype.show = function () {
    setVisible(this._overlayRoot);
  };

  /**
   * Hide all overlays.
   */
  Overlays.prototype.hide = function () {
    setVisible(this._overlayRoot, false);
  };

  /**
   * Remove all overlays and their container.
   */
  Overlays.prototype.clear = function () {
    this._overlays = {};
    this._overlayContainers = [];
    clear$1(this._overlayRoot);
  };
  Overlays.prototype._updateOverlayContainer = function (container) {
    var element = container.element,
      html = container.html;

    // update container left,top according to the elements x,y coordinates
    // this ensures we can attach child elements relative to this container

    var x = element.x,
      y = element.y;
    if (element.waypoints) {
      var bbox = getBBox(element);
      x = bbox.x;
      y = bbox.y;
    }
    setPosition(html, x, y);
    attr$1(container.html, 'data-container-id', element.id);
  };
  Overlays.prototype._updateOverlay = function (overlay) {
    var position = overlay.position,
      htmlContainer = overlay.htmlContainer,
      element = overlay.element;

    // update overlay html relative to shape because
    // it is already positioned on the element

    // update relative
    var left = position.left,
      top = position.top;
    if (position.right !== undefined) {
      var width;
      if (element.waypoints) {
        width = getBBox(element).width;
      } else {
        width = element.width;
      }
      left = position.right * -1 + width;
    }
    if (position.bottom !== undefined) {
      var height;
      if (element.waypoints) {
        height = getBBox(element).height;
      } else {
        height = element.height;
      }
      top = position.bottom * -1 + height;
    }
    setPosition(htmlContainer, left || 0, top || 0);
    this._updateOverlayVisibilty(overlay, this._canvas.viewbox());
  };
  Overlays.prototype._createOverlayContainer = function (element) {
    var html = domify$1('<div class="djs-overlays" />');
    assign(html, {
      position: 'absolute'
    });
    this._overlayRoot.appendChild(html);
    var container = {
      html: html,
      element: element,
      overlays: []
    };
    this._updateOverlayContainer(container);
    this._overlayContainers.push(container);
    return container;
  };
  Overlays.prototype._updateRoot = function (viewbox) {
    var scale = viewbox.scale || 1;
    var matrix = 'matrix(' + [scale, 0, 0, scale, -1 * viewbox.x * scale, -1 * viewbox.y * scale].join(',') + ')';
    setTransform(this._overlayRoot, matrix);
  };
  Overlays.prototype._getOverlayContainer = function (element, raw) {
    var container = find$2(this._overlayContainers, function (c) {
      return c.element === element;
    });
    if (!container && !raw) {
      return this._createOverlayContainer(element);
    }
    return container;
  };
  Overlays.prototype._addOverlay = function (overlay) {
    var id = overlay.id,
      element = overlay.element,
      html = overlay.html,
      htmlContainer,
      overlayContainer;

    // unwrap jquery (for those who need it)
    if (html.get && html.constructor.prototype.jquery) {
      html = html.get(0);
    }

    // create proper html elements from
    // overlay HTML strings
    if (isString$4(html)) {
      html = domify$1(html);
    }
    overlayContainer = this._getOverlayContainer(element);
    htmlContainer = domify$1('<div class="djs-overlay" data-overlay-id="' + id + '">');
    assign(htmlContainer, {
      position: 'absolute'
    });
    htmlContainer.appendChild(html);
    if (overlay.type) {
      classes$1(htmlContainer).add('djs-overlay-' + overlay.type);
    }
    var elementRoot = this._canvas.findRoot(element);
    var activeRoot = this._canvas.getRootElement();
    setVisible(htmlContainer, elementRoot === activeRoot);
    overlay.htmlContainer = htmlContainer;
    overlayContainer.overlays.push(overlay);
    overlayContainer.html.appendChild(htmlContainer);
    this._overlays[id] = overlay;
    this._updateOverlay(overlay);
    this._updateOverlayVisibilty(overlay, this._canvas.viewbox());
  };
  Overlays.prototype._updateOverlayVisibilty = function (overlay, viewbox) {
    var show = overlay.show,
      rootElement = this._canvas.findRoot(overlay.element),
      minZoom = show && show.minZoom,
      maxZoom = show && show.maxZoom,
      htmlContainer = overlay.htmlContainer,
      activeRootElement = this._canvas.getRootElement(),
      visible = true;
    if (rootElement !== activeRootElement) {
      visible = false;
    } else if (show) {
      if (isDefined(minZoom) && minZoom > viewbox.scale || isDefined(maxZoom) && maxZoom < viewbox.scale) {
        visible = false;
      }
    }
    setVisible(htmlContainer, visible);
    this._updateOverlayScale(overlay, viewbox);
  };
  Overlays.prototype._updateOverlayScale = function (overlay, viewbox) {
    var shouldScale = overlay.scale,
      minScale,
      maxScale,
      htmlContainer = overlay.htmlContainer;
    var scale,
      transform = '';
    if (shouldScale !== true) {
      if (shouldScale === false) {
        minScale = 1;
        maxScale = 1;
      } else {
        minScale = shouldScale.min;
        maxScale = shouldScale.max;
      }
      if (isDefined(minScale) && viewbox.scale < minScale) {
        scale = (1 / viewbox.scale || 1) * minScale;
      }
      if (isDefined(maxScale) && viewbox.scale > maxScale) {
        scale = (1 / viewbox.scale || 1) * maxScale;
      }
    }
    if (isDefined(scale)) {
      transform = 'scale(' + scale + ',' + scale + ')';
    }
    setTransform(htmlContainer, transform);
  };
  Overlays.prototype._updateOverlaysVisibilty = function (viewbox) {
    var self = this;
    forEach$3(this._overlays, function (overlay) {
      self._updateOverlayVisibilty(overlay, viewbox);
    });
  };
  Overlays.prototype._init = function () {
    var eventBus = this._eventBus;
    var self = this;

    // scroll/zoom integration

    function updateViewbox(viewbox) {
      self._updateRoot(viewbox);
      self._updateOverlaysVisibilty(viewbox);
      self.show();
    }
    eventBus.on('canvas.viewbox.changing', function (event) {
      self.hide();
    });
    eventBus.on('canvas.viewbox.changed', function (event) {
      updateViewbox(event.viewbox);
    });

    // remove integration

    eventBus.on(['shape.remove', 'connection.remove'], function (e) {
      var element = e.element;
      var overlays = self.get({
        element: element
      });
      forEach$3(overlays, function (o) {
        self.remove(o.id);
      });
      var container = self._getOverlayContainer(element);
      if (container) {
        remove$4(container.html);
        var i = self._overlayContainers.indexOf(container);
        if (i !== -1) {
          self._overlayContainers.splice(i, 1);
        }
      }
    });

    // move integration

    eventBus.on('element.changed', LOW_PRIORITY$l, function (e) {
      var element = e.element;
      var container = self._getOverlayContainer(element, true);
      if (container) {
        forEach$3(container.overlays, function (overlay) {
          self._updateOverlay(overlay);
        });
        self._updateOverlayContainer(container);
      }
    });

    // marker integration, simply add them on the overlays as classes, too.

    eventBus.on('element.marker.update', function (e) {
      var container = self._getOverlayContainer(e.element, true);
      if (container) {
        classes$1(container.html)[e.add ? 'add' : 'remove'](e.marker);
      }
    });
    eventBus.on('root.set', function () {
      self._updateOverlaysVisibilty(self._canvas.viewbox());
    });

    // clear overlays with diagram

    eventBus.on('diagram.clear', this.clear, this);
  };

  // helpers /////////////////////////////

  function createRoot(parentNode) {
    var root = domify$1('<div class="djs-overlay-container" />');
    assign(root, {
      position: 'absolute',
      width: 0,
      height: 0
    });
    parentNode.insertBefore(root, parentNode.firstChild);
    return root;
  }
  function setPosition(el, x, y) {
    assign(el, {
      left: x + 'px',
      top: y + 'px'
    });
  }

  /**
   * Set element visible
   *
   * @param {DOMElement} el
   * @param {boolean} [visible=true]
   */
  function setVisible(el, visible) {
    el.style.display = visible === false ? 'none' : '';
  }
  function setTransform(el, transform) {
    el.style['transform-origin'] = 'top left';
    ['', '-ms-', '-webkit-'].forEach(function (prefix) {
      el.style[prefix + 'transform'] = transform;
    });
  }

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var OverlaysModule = {
    __init__: ['overlays'],
    overlays: ['type', Overlays]
  };

  function DefinitionPropertiesView(eventBus, canvas, translate) {
    this._eventBus = eventBus;
    this._canvas = canvas;
    this._translate = translate;
    eventBus.on('diagram.init', function () {
      this._init();
    }, this);
    eventBus.on('import.done', function (event) {
      if (!event.error) {
        this.update();
      }
    }, this);

    /* markup definition */

    this.HTML_MARKUP = '<div class="dmn-definitions">' + '<div class="dmn-definitions-name" title="' + this._translate('Definition name') + '" spellcheck="false">' + '</div>' + '<div class="dmn-definitions-id" title="' + this._translate('Definition ID') + '" spellcheck="false">' + '</div>' + '</div>';
  }
  DefinitionPropertiesView.$inject = ['eventBus', 'canvas', 'translate'];

  /**
   * Initialize
   */
  DefinitionPropertiesView.prototype._init = function () {
    var canvas = this._canvas,
      eventBus = this._eventBus;
    var parent = canvas.getContainer(),
      container = this._container = domify$1(this.HTML_MARKUP);
    parent.appendChild(container);
    this.nameElement = query('.dmn-definitions-name', this._container);
    this.idElement = query('.dmn-definitions-id', this._container);
    delegate.bind(container, '.dmn-definitions-name, .dmn-definitions-id', 'mousedown', function (event) {
      event.stopPropagation();
    });
    eventBus.fire('definitionIdView.create', {
      html: container
    });
  };
  DefinitionPropertiesView.prototype.update = function () {
    var businessObject = this._canvas.getRootElement().businessObject;
    this.nameElement.textContent = businessObject.name;
    this.idElement.textContent = businessObject.id;
  };

  function PaletteAdapter(eventBus, canvas) {
    function toggleMarker(cls, on) {
      var container = canvas.getContainer();
      classes$1(container).toggle(cls, on);
    }
    eventBus.on('palette.create', function () {
      toggleMarker('with-palette', true);
    });
    eventBus.on('palette.changed', function (event) {
      toggleMarker('with-palette-two-column', event.twoColumn);
    });
  }
  PaletteAdapter.$inject = ['eventBus', 'canvas'];

  var DefinitionPropertiesViewer = {
    __depends__: [TranslateModule],
    __init__: ['definitionPropertiesView', 'definitionPropertiesPaletteAdapter'],
    definitionPropertiesView: ['type', DefinitionPropertiesView],
    definitionPropertiesPaletteAdapter: ['type', PaletteAdapter]
  };

  var PROVIDERS = [{
    className: 'dmn-icon-decision-table',
    matches: function (el) {
      var businessObject = getBusinessObject(el);
      return is$1(businessObject, 'dmn:Decision') && is$1(businessObject.decisionLogic, 'dmn:DecisionTable');
    },
    title: 'Open decision table'
  }, {
    className: 'dmn-icon-literal-expression',
    matches: function (el) {
      var boxedExpression = getBoxedExpression(el);
      return is$1(boxedExpression, 'dmn:LiteralExpression');
    },
    title: 'Open literal expression'
  }];

  /**
   * Displays overlays that can be clicked in order to drill
   * down into a DMN element.
   */
  class DrillDown {
    constructor(injector, eventBus, overlays, config, translate) {
      this._injector = injector;
      this._eventBus = eventBus;
      this._overlays = overlays;
      this._translate = translate;
      this._config = config || {
        enabled: true
      };
      eventBus.on(['shape.added'], ({
        element
      }) => {
        for (let i = 0; i < PROVIDERS.length; i++) {
          const {
            matches,
            className,
            title
          } = PROVIDERS[i];
          var editable = matches && matches(element);
          if (editable) {
            this.addOverlay(element, className, title);
          }
        }
      });
    }

    /**
     * Add overlay to an element that enables drill down.
     *
     * @param {Object} element Element to add overlay to.
     * @param {string} className
     *        CSS class that will be added to overlay in order to display icon.
     * @param {string} title added to the button
     */
    addOverlay(element, className, title) {
      const enabled = this._config.enabled !== false;
      const node = this._getOverlayNode(className, title, enabled);
      const overlayId = this._overlays.add(element, {
        position: {
          top: 2,
          left: 2
        },
        html: node
      });

      // TODO(nikku): can we remove renamed to drillDown.enabled
      if (enabled) {
        classes$1(node).add('interactive');
        this.bindEventListener(element, node, overlayId);
      }
    }
    _getOverlayNode(className, title, enabled) {
      const container = document.createElement('div');
      container.className = 'drill-down-overlay';
      if (!enabled) {
        const icon = document.createElement('span');
        icon.className = className;
        container.appendChild(icon);
        return container;
      }
      const button = document.createElement('button');
      button.type = 'button';
      button.className = className;
      button.title = this._translate(title);
      container.appendChild(button);
      return container;
    }

    /**
     * @param {Object} element
     * @param {Object} overlay
     * @param {string} id
     */
    bindEventListener(element, overlay, id) {
      const overlays = this._overlays,
        eventBus = this._eventBus;
      const overlaysRoot = overlays._overlayRoot;
      delegate.bind(overlaysRoot, '[data-overlay-id="' + id + '"]', 'click', () => {
        const triggerDefault = eventBus.fire('drillDown.click', {
          element
        });
        if (triggerDefault === false) {
          return;
        }
        this.drillDown(element);
      });
    }

    /**
     * Drill down into the specific element.
     *
     * @param  {djs.model.Base} element
     *
     * @return {boolean} whether drill down was executed
     */
    drillDown(element) {
      const parent = this._injector.get('_parent', false);

      // no parent; skip drill down
      if (!parent) {
        return false;
      }
      const view = parent.getView(element.businessObject);

      // no view to drill down to
      if (!view) {
        return false;
      }
      parent.open(view);
      return true;
    }
  }
  DrillDown.$inject = ['injector', 'eventBus', 'overlays', 'config.drillDown', 'translate'];

  var DrillDownModule = {
    __depends__: [OverlaysModule, TranslateModule],
    __init__: ['drillDown'],
    drillDown: ['type', DrillDown]
  };

  /**
   * The code in the <project-logo></project-logo> area
   * must not be changed.
   *
   * @see http://bpmn.io/license for more information.
   */


  /**
   * @typedef {import('dmn-js-shared/lib/base/View).OpenResult} OpenResult
   */

  /**
   * @typedef {import('dmn-js-shared/lib/base/View).OpenError} OpenError
   */

  /**
   * A viewer for DMN diagrams.
   *
   * Have a look at {@link NavigatedViewer} or {@link Modeler} for bundles that include
   * additional features.
   *
   *
   * ## Extending the Viewer
   *
   * In order to extend the viewer pass extension modules to bootstrap via the
   * `additionalModules` option. An extension module is an object that exposes
   * named services.
   *
   * The following example depicts the integration of a simple
   * logging component that integrates with interaction events:
   *
   *
   * ```javascript
   *
   * // logging component
   * function InteractionLogger(eventBus) {
   *   eventBus.on('element.hover', function(event) {
   *     console.log()
   *   })
   * }
   *
   * InteractionLogger.$inject = [ 'eventBus' ]; // minification save
   *
   * // extension module
   * var extensionModule = {
   *   __init__: [ 'interactionLogger' ],
   *   interactionLogger: [ 'type', InteractionLogger ]
   * };
   *
   * // extend the viewer
   * var drdViewer = new Viewer({ additionalModules: [ extensionModule ] });
   * drdViewer.importXML(...);
   * ```
   *
   * @param {Object} options configuration options to pass to the viewer
   * @param {DOMElement} [options.container]
   *        the container to render the viewer in, defaults to body
   * @param {Array<didi.Module>} [options.modules]
   *        a list of modules to override the default modules
   * @param {Array<didi.Module>} [options.additionalModules]
   *        a list of modules to use with the default modules
   */
  function Viewer$4(options) {
    this._container = this._createContainer();

    /* <project-logo> */

    addProjectLogo(this._container);

    /* </project-logo> */

    this._init(this._container, options);
  }
  e$3(Viewer$4, Diagram);

  /**
   * The saveSVG result.
   *
   * @typedef {Object} SaveSVGResult
   *
   * @property {string} svg
   */

  /**
   * Export the currently displayed DMN diagram as
   * an SVG image.
   *
   * @param {Object} [options]
   *
   * @return {Promise<SaveSVGResult>}
   */
  Viewer$4.prototype.saveSVG = wrapForCompatibility(function (options) {
    var self = this;
    return new Promise(function (resolve) {
      var canvas = self.get('canvas');
      var contentNode = canvas.getActiveLayer(),
        defsNode = query('defs', canvas._svg);
      var contents = innerSVG(contentNode),
        defs = defsNode && defsNode.outerHTML || '';
      var bbox = contentNode.getBBox();
      var svg = '<?xml version="1.0" encoding="utf-8"?>\n' + '<!-- created with dmn-js / http://bpmn.io -->\n' + '<!DOCTYPE svg PUBLIC "-//W3C//DTD SVG 1.1//EN" "http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd">\n' + '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" ' + 'width="' + bbox.width + '" height="' + bbox.height + '" ' + 'viewBox="' + bbox.x + ' ' + bbox.y + ' ' + bbox.width + ' ' + bbox.height + '" version="1.1">' + defs + contents + '</svg>';
      resolve({
        svg
      });
    });
  });
  Viewer$4.prototype.getModules = function () {
    return this._modules;
  };

  /**
   * Destroy the viewer instance and remove all its
   * remainders from the document tree.
   */
  Viewer$4.prototype.destroy = function () {
    // diagram destroy
    Diagram.prototype.destroy.call(this);

    // dom detach
    remove$4(this._container);
  };

  /**
   * Register an event listener
   *
   * Remove a previously added listener via {@link #off(event, callback)}.
   *
   * @param {string} event
   * @param {number} [priority]
   * @param {Function} callback
   * @param {Object} [that]
   */
  Viewer$4.prototype.on = function (event, priority, callback, target) {
    return this.get('eventBus').on(event, priority, callback, target);
  };

  /**
   * De-register an event listener
   *
   * @param {string} event
   * @param {Function} callback
   */
  Viewer$4.prototype.off = function (event, callback) {
    this.get('eventBus').off(event, callback);
  };
  Viewer$4.prototype._init = function (container, options) {
    var {
      additionalModules,
      canvas,
      ...additionalOptions
    } = options;
    var baseModules = options.modules || this.getModules(),
      staticModules = [{
        drd: ['value', this]
      }];
    var modules = [...staticModules, ...baseModules, ...(additionalModules || [])];
    var diagramOptions = {
      ...additionalOptions,
      canvas: {
        ...canvas,
        container
      },
      modules
    };

    // invoke diagram constructor
    Diagram.call(this, diagramOptions);
    if (options && options.container) {
      this.attachTo(options.container);
    }
  };

  /**
   * Emit an event on the underlying {@link EventBus}
   *
   * @param  {string} type
   * @param  {Object} event
   *
   * @return {Object} event processing result (if any)
   */
  Viewer$4.prototype._emit = function (type, event) {
    return this.get('eventBus').fire(type, event);
  };
  Viewer$4.prototype._createContainer = function () {
    return domify$1('<div class="dmn-drd-container"></div>');
  };

  /**
   * Open diagram element.
   *
   * @param  {ModdleElement} definitions
   * @returns {Promise} Resolves with {OpenResult} when successful
   * or rejects with {OpenError}
   */
  Viewer$4.prototype.open = function (definitions) {
    var self = this;
    return new Promise((resolve, reject) => {
      var err;

      // use try/catch to not swallow synchronous exceptions
      // that may be raised during model parsing
      try {
        if (self._definitions) {
          // clear existing rendered diagram
          self.clear();
        }

        // update definitions
        self._definitions = definitions;

        // perform graphical import
        return importDRD(self, definitions, function (err, warnings) {
          if (err) {
            err.warnings = warnings || [];
            reject(err);
          } else {
            resolve({
              warnings: warnings || []
            });
          }
        });
      } catch (e) {
        err = e;
      }
      if (err) {
        err.warnings = err.warnings || [];
        reject(err);
      } else {
        resolve({
          warnings: []
        });
      }
    });
  };

  /**
   * Attach viewer to given parent node.
   *
   * @param  {Element} parentNode
   */
  Viewer$4.prototype.attachTo = function (parentNode) {
    if (!parentNode) {
      throw new Error('parentNode required');
    }

    // ensure we detach from the
    // previous, old parent
    this.detach();
    var container = this._container;
    parentNode.appendChild(container);
    this._emit('attach', {});
    this.get('canvas').resized();
  };

  /**
   * Detach viewer from parent node, if attached.
   */
  Viewer$4.prototype.detach = function () {
    var container = this._container,
      parentNode = container.parentNode;
    if (!parentNode) {
      return;
    }
    this._emit('detach', {});
    parentNode.removeChild(container);
  };
  Viewer$4.prototype._modules = [CoreModule$2, TranslateModule, SelectionModule, OverlaysModule, DefinitionPropertiesViewer, DrillDownModule];

  /**
   * @typedef {import('../util/Types').Point} Point
   * @typedef {import('../util/Types').Rect} Rect
   */

  /**
   * @param {Rect} bounds
   * @return {Point}
   */
  function center(bounds) {
    return {
      x: bounds.x + bounds.width / 2,
      y: bounds.y + bounds.height / 2
    };
  }

  /**
   * @param {Point} a
   * @param {Point} b
   * @return {Point}
   */
  function delta(a, b) {
    return {
      x: a.x - b.x,
      y: a.y - b.y
    };
  }

  /**
   * Get the logarithm of x with base 10.
   *
   * @param {number} x
   */
  function log10(x) {
    return Math.log(x) / Math.log(10);
  }

  /**
   * Get step size for given range and number of steps.
   *
   * @param {Object} range
   * @param {number} range.min
   * @param {number} range.max
   * @param {number} steps
   */
  function getStepSize(range, steps) {
    var minLinearRange = log10(range.min),
      maxLinearRange = log10(range.max);
    var absoluteLinearRange = Math.abs(minLinearRange) + Math.abs(maxLinearRange);
    return absoluteLinearRange / steps;
  }

  /**
   * @param {Object} range
   * @param {number} range.min
   * @param {number} range.max
   * @param {number} scale
   */
  function cap(range, scale) {
    return Math.max(range.min, Math.min(range.max, scale));
  }

  /**
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/EventBus').default} EventBus
   *
   * @typedef {import('../../util/Types').Point} Point
   * @typedef {import('../../util/Types').ScrollDelta} ScrollDelta
   */

  var sign = Math.sign || function (n) {
    return n >= 0 ? 1 : -1;
  };
  var RANGE$2 = {
      min: 0.2,
      max: 4
    },
    NUM_STEPS = 10;
  var DELTA_THRESHOLD = 0.1;
  var DEFAULT_SCALE = 0.75;

  /**
   * An implementation of zooming and scrolling within the
   * {@link Canvas} via the mouse wheel.
   *
   * Mouse wheel zooming / scrolling may be disabled using
   * the {@link toggle(enabled)} method.
   *
   * @param {Object} [config]
   * @param {boolean} [config.enabled=true] default enabled state
   * @param {number} [config.scale=.75] scroll sensivity
   * @param {EventBus} eventBus
   * @param {Canvas} canvas
   */
  function ZoomScroll$1(config, eventBus, canvas) {
    config = config || {};
    this._enabled = false;
    this._canvas = canvas;
    this._container = canvas._container;
    this._handleWheel = bind$3(this._handleWheel, this);
    this._totalDelta = 0;
    this._scale = config.scale || DEFAULT_SCALE;
    var self = this;
    eventBus.on('canvas.init', function (e) {
      self._init(config.enabled !== false);
    });
  }
  ZoomScroll$1.$inject = ['config.zoomScroll', 'eventBus', 'canvas'];

  /**
   * @param {ScrollDelta} delta
   */
  ZoomScroll$1.prototype.scroll = function scroll(delta) {
    this._canvas.scroll(delta);
  };
  ZoomScroll$1.prototype.reset = function reset() {
    this._canvas.zoom('fit-viewport');
  };

  /**
   * Zoom depending on delta.
   *
   * @param {number} delta
   * @param {Point} position
   */
  ZoomScroll$1.prototype.zoom = function zoom(delta, position) {
    // zoom with half the step size of stepZoom
    var stepSize = getStepSize(RANGE$2, NUM_STEPS * 2);

    // add until threshold reached
    this._totalDelta += delta;
    if (Math.abs(this._totalDelta) > DELTA_THRESHOLD) {
      this._zoom(delta, position, stepSize);

      // reset
      this._totalDelta = 0;
    }
  };
  ZoomScroll$1.prototype._handleWheel = function handleWheel(event) {
    // event is already handled by '.djs-scrollable'
    if (closest(event.target, '.djs-scrollable', true)) {
      return;
    }
    var element = this._container;
    event.preventDefault();

    // pinch to zoom is mapped to wheel + ctrlKey = true
    // in modern browsers (!)

    var isZoom = event.ctrlKey || isMac() && event.metaKey;
    var isHorizontalScroll = event.shiftKey;
    var factor = -1 * this._scale,
      delta;
    if (isZoom) {
      factor *= event.deltaMode === 0 ? 0.020 : 0.32;
    } else {
      factor *= event.deltaMode === 0 ? 1.0 : 16.0;
    }
    if (isZoom) {
      var elementRect = element.getBoundingClientRect();
      var offset = {
        x: event.clientX - elementRect.left,
        y: event.clientY - elementRect.top
      };
      delta = Math.sqrt(Math.pow(event.deltaY, 2) + Math.pow(event.deltaX, 2)) * sign(event.deltaY) * factor;

      // zoom in relative to diagram {x,y} coordinates
      this.zoom(delta, offset);
    } else {
      if (isHorizontalScroll) {
        delta = {
          dx: factor * event.deltaY,
          dy: 0
        };
      } else {
        delta = {
          dx: factor * event.deltaX,
          dy: factor * event.deltaY
        };
      }
      this.scroll(delta);
    }
  };

  /**
   * Zoom with fixed step size.
   *
   * @param {number} delta Zoom delta (1 for zooming in, -1 for zooming out).
   * @param {Point} [position]
   */
  ZoomScroll$1.prototype.stepZoom = function stepZoom(delta, position) {
    var stepSize = getStepSize(RANGE$2, NUM_STEPS);
    this._zoom(delta, position, stepSize);
  };

  /**
   * Zoom in/out given a step size.
   *
   * @param {number} delta
   * @param {Point} [position]
   * @param {number} stepSize
   */
  ZoomScroll$1.prototype._zoom = function (delta, position, stepSize) {
    var canvas = this._canvas;
    var direction = delta > 0 ? 1 : -1;
    var currentLinearZoomLevel = log10(canvas.zoom());

    // snap to a proximate zoom step
    var newLinearZoomLevel = Math.round(currentLinearZoomLevel / stepSize) * stepSize;

    // increase or decrease one zoom step in the given direction
    newLinearZoomLevel += stepSize * direction;

    // calculate the absolute logarithmic zoom level based on the linear zoom level
    // (e.g. 2 for an absolute x2 zoom)
    var newLogZoomLevel = Math.pow(10, newLinearZoomLevel);
    canvas.zoom(cap(RANGE$2, newLogZoomLevel), position);
  };

  /**
   * Toggle the zoom scroll ability via mouse wheel.
   *
   * @param {boolean} [newEnabled] new enabled state
   */
  ZoomScroll$1.prototype.toggle = function toggle(newEnabled) {
    var element = this._container;
    var handleWheel = this._handleWheel;
    var oldEnabled = this._enabled;
    if (typeof newEnabled === 'undefined') {
      newEnabled = !oldEnabled;
    }

    // only react on actual changes
    if (oldEnabled !== newEnabled) {
      // add or remove wheel listener based on
      // changed enabled state
      event[newEnabled ? 'bind' : 'unbind'](element, 'wheel', handleWheel, false);
    }
    this._enabled = newEnabled;
    return newEnabled;
  };
  ZoomScroll$1.prototype._init = function (newEnabled) {
    this.toggle(newEnabled);
  };

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var ZoomScroll = {
    __init__: ['zoomScroll'],
    zoomScroll: ['type', ZoomScroll$1]
  };

  var CURSOR_CLS_PATTERN = /^djs-cursor-.*$/;

  /**
   * @param {string} mode
   */
  function set(mode) {
    var classes = classes$1(document.body);
    classes.removeMatching(CURSOR_CLS_PATTERN);
    if (mode) {
      classes.add('djs-cursor-' + mode);
    }
  }
  function unset() {
    set(null);
  }

  /**
   * @typedef {import('../core/EventBus').default} EventBus
   */

  var TRAP_PRIORITY = 5000;

  /**
   * Installs a click trap that prevents a ghost click following a dragging operation.
   *
   * @param {EventBus} eventBus
   * @param {string} [eventName='element.click']
   *
   * @return {() => void} a function to immediately remove the installed trap.
   */
  function install(eventBus, eventName) {
    eventName = eventName || 'element.click';
    function trap() {
      return false;
    }
    eventBus.once(eventName, TRAP_PRIORITY, trap);
    return function () {
      eventBus.off(eventName, trap);
    };
  }

  /**
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/EventBus').default} EventBus
   */

  var THRESHOLD$1 = 15;

  /**
   * Move the canvas via mouse.
   *
   * @param {EventBus} eventBus
   * @param {Canvas} canvas
   */
  function MoveCanvas$1(eventBus, canvas) {
    var context;

    // listen for move on element mouse down;
    // allow others to hook into the event before us though
    // (dragging / element moving will do this)
    eventBus.on('element.mousedown', 500, function (e) {
      return handleStart(e.originalEvent);
    });
    function handleMove(event) {
      var start = context.start,
        button = context.button,
        position = toPoint(event),
        delta$1 = delta(position, start);
      if (!context.dragging && length(delta$1) > THRESHOLD$1) {
        context.dragging = true;
        if (button === 0) {
          install(eventBus);
        }
        set('grab');
      }
      if (context.dragging) {
        var lastPosition = context.last || context.start;
        delta$1 = delta(position, lastPosition);
        canvas.scroll({
          dx: delta$1.x,
          dy: delta$1.y
        });
        context.last = position;
      }

      // prevent select
      event.preventDefault();
    }
    function handleEnd(event$1) {
      event.unbind(document, 'mousemove', handleMove);
      event.unbind(document, 'mouseup', handleEnd);
      context = null;
      unset();
    }
    function handleStart(event$1) {
      // event is already handled by '.djs-draggable'
      if (closest(event$1.target, '.djs-draggable')) {
        return;
      }
      var button = event$1.button;

      // reject right mouse button or modifier key
      if (button >= 2 || event$1.ctrlKey || event$1.shiftKey || event$1.altKey) {
        return;
      }
      context = {
        button: button,
        start: toPoint(event$1)
      };
      event.bind(document, 'mousemove', handleMove);
      event.bind(document, 'mouseup', handleEnd);

      // we've handled the event
      return true;
    }
    this.isActive = function () {
      return !!context;
    };
  }
  MoveCanvas$1.$inject = ['eventBus', 'canvas'];

  // helpers ///////

  function length(point) {
    return Math.sqrt(Math.pow(point.x, 2) + Math.pow(point.y, 2));
  }

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var MoveCanvas = {
    __init__: ['moveCanvas'],
    moveCanvas: ['type', MoveCanvas$1]
  };

  /**
   * @param {string} str
   *
   * @return {string}
   */
  function escapeCSS(str) {
    return CSS.escape(str);
  }
  var HTML_ESCAPE_MAP = {
    '&': '&amp;',
    '<': '&lt;',
    '>': '&gt;',
    '"': '&quot;',
    '\'': '&#39;'
  };

  /**
   * @param {string} str
   *
   * @return {string}
   */
  function escapeHTML(str) {
    str = '' + str;
    return str && str.replace(/[&<>"']/g, function (match) {
      return HTML_ESCAPE_MAP[match];
    });
  }

  var KEYS_COPY = ['c', 'C'];
  var KEYS_PASTE = ['v', 'V'];
  var KEYS_REDO = ['y', 'Y'];
  var KEYS_UNDO = ['z', 'Z'];

  /**
   * Returns true if event was triggered with any modifier
   * @param {KeyboardEvent} event
   */
  function hasModifier(event) {
    return event.ctrlKey || event.metaKey || event.shiftKey || event.altKey;
  }

  /**
   * @param {KeyboardEvent} event
   * @return {boolean}
   */
  function isCmd$5(event) {
    // ensure we don't react to AltGr
    // (mapped to CTRL + ALT)
    if (event.altKey) {
      return false;
    }
    return event.ctrlKey || event.metaKey;
  }

  /**
   * Checks if key pressed is one of provided keys.
   *
   * @param {string|string[]} keys
   * @param {KeyboardEvent} event
   * @return {boolean}
   */
  function isKey(keys, event) {
    keys = isArray$6(keys) ? keys : [keys];
    return keys.indexOf(event.key) !== -1 || keys.indexOf(event.code) !== -1;
  }

  /**
   * @param {KeyboardEvent} event
   */
  function isShift$2(event) {
    return event.shiftKey;
  }

  /**
   * @param {KeyboardEvent} event
   */
  function isCopy(event) {
    return isCmd$5(event) && isKey(KEYS_COPY, event);
  }

  /**
   * @param {KeyboardEvent} event
   */
  function isPaste(event) {
    return isCmd$5(event) && isKey(KEYS_PASTE, event);
  }

  /**
   * @param {KeyboardEvent} event
   */
  function isUndo(event) {
    return isCmd$5(event) && !isShift$2(event) && isKey(KEYS_UNDO, event);
  }

  /**
   * @param {KeyboardEvent} event
   */
  function isRedo(event) {
    return isCmd$5(event) && (isKey(KEYS_REDO, event) || isKey(KEYS_UNDO, event) && isShift$2(event));
  }

  /**
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('../selection/Selection').default} Selection
   * @typedef {import('../../i18n/translate/translate.js').default} Translate
   *
   * @typedef {import('../../util/Types').Dimensions} Dimensions
   *
   * @typedef {import('./SearchPadProvider').default} SearchPadProvider
   * @typedef {import('./SearchPadProvider').SearchResult} SearchResult
   * @typedef {import('./SearchPadProvider').Token} Token
   */

  var SCROLL_TO_ELEMENT_PADDING = 300;

  /**
   * Provides searching infrastructure.
   *
   * @param {Canvas} canvas
   * @param {EventBus} eventBus
   * @param {Selection} selection
   * @param {Translate} translate
   */
  function SearchPad(canvas, eventBus, selection, translate) {
    this._open = false;
    this._results = {};
    this._eventMaps = [];
    this._cachedRootElement = null;
    this._cachedSelection = null;
    this._cachedViewbox = null;
    this._canvas = canvas;
    this._eventBus = eventBus;
    this._selection = selection;
    this._translate = translate;

    // setup elements
    this._container = this._getBoxHtml();
    this._searchInput = query(SearchPad.INPUT_SELECTOR, this._container);
    this._resultsContainer = query(SearchPad.RESULTS_CONTAINER_SELECTOR, this._container);

    // attach search pad
    this._canvas.getContainer().appendChild(this._container);

    // cleanup whenever appropriate
    eventBus.on(['canvas.destroy', 'diagram.destroy', 'drag.init', 'elements.changed'], this.close, this);
  }
  SearchPad.$inject = ['canvas', 'eventBus', 'selection', 'translate'];

  /**
   * Binds and keeps track of all event listereners
   */
  SearchPad.prototype._bindEvents = function () {
    var self = this;
    function listen(el, selector, type, fn) {
      self._eventMaps.push({
        el: el,
        type: type,
        listener: delegate.bind(el, selector, type, fn)
      });
    }

    // close search on clicking anywhere outside
    listen(document, 'html', 'click', function (e) {
      self.close(false);
    });

    // stop event from propagating and closing search
    // focus on input
    listen(this._container, SearchPad.INPUT_SELECTOR, 'click', function (e) {
      e.stopPropagation();
      e.delegateTarget.focus();
    });

    // preselect result on hover
    listen(this._container, SearchPad.RESULT_SELECTOR, 'mouseover', function (e) {
      e.stopPropagation();
      self._scrollToNode(e.delegateTarget);
      self._preselect(e.delegateTarget);
    });

    // selects desired result on mouse click
    listen(this._container, SearchPad.RESULT_SELECTOR, 'click', function (e) {
      e.stopPropagation();
      self._select(e.delegateTarget);
    });

    // prevent cursor in input from going left and right when using up/down to
    // navigate results
    listen(this._container, SearchPad.INPUT_SELECTOR, 'keydown', function (e) {
      if (isKey('ArrowUp', e)) {
        e.preventDefault();
      }
      if (isKey('ArrowDown', e)) {
        e.preventDefault();
      }
    });

    // handle keyboard input
    listen(this._container, SearchPad.INPUT_SELECTOR, 'keyup', function (e) {
      if (isKey('Escape', e)) {
        return self.close();
      }
      if (isKey('Enter', e)) {
        var selected = self._getCurrentResult();
        return selected ? self._select(selected) : self.close(false);
      }
      if (isKey('ArrowUp', e)) {
        return self._scrollToDirection(true);
      }
      if (isKey('ArrowDown', e)) {
        return self._scrollToDirection();
      }

      // do not search while navigating text input
      if (isKey(['ArrowLeft', 'ArrowRight'], e)) {
        return;
      }

      // anything else
      self._search(e.delegateTarget.value);
    });
  };

  /**
   * Unbinds all previously established listeners
   */
  SearchPad.prototype._unbindEvents = function () {
    this._eventMaps.forEach(function (m) {
      delegate.unbind(m.el, m.type, m.listener);
    });
  };

  /**
   * Performs a search for the given pattern.
   *
   * @param {string} pattern
   */
  SearchPad.prototype._search = function (pattern) {
    var self = this;
    this._clearResults();

    // do not search on empty query
    if (!pattern || pattern === '') {
      return;
    }
    var searchResults = this._searchProvider.find(pattern);
    searchResults = searchResults.filter(function (searchResult) {
      return !self._canvas.getRootElements().includes(searchResult.element);
    });
    if (!searchResults.length) {
      this._selection.select(null);
      return;
    }

    // append new results
    searchResults.forEach(function (result) {
      var id = result.element.id;
      var node = self._createResultNode(result, id);
      self._results[id] = {
        element: result.element,
        node: node
      };
    });

    // preselect first result
    var node = query(SearchPad.RESULT_SELECTOR, this._resultsContainer);
    this._scrollToNode(node);
    this._preselect(node);
  };

  /**
   * Navigate to the previous/next result. Defaults to next result.
   *
   * @param {boolean} previous
   */
  SearchPad.prototype._scrollToDirection = function (previous) {
    var selected = this._getCurrentResult();
    if (!selected) {
      return;
    }
    var node = previous ? selected.previousElementSibling : selected.nextElementSibling;
    if (node) {
      this._scrollToNode(node);
      this._preselect(node);
    }
  };

  /**
   * Scroll to the node if it is not visible.
   *
   * @param {HTMLElement} node
   */
  SearchPad.prototype._scrollToNode = function (node) {
    if (!node || node === this._getCurrentResult()) {
      return;
    }
    var nodeOffset = node.offsetTop;
    var containerScroll = this._resultsContainer.scrollTop;
    var bottomScroll = nodeOffset - this._resultsContainer.clientHeight + node.clientHeight;
    if (nodeOffset < containerScroll) {
      this._resultsContainer.scrollTop = nodeOffset;
    } else if (containerScroll < bottomScroll) {
      this._resultsContainer.scrollTop = bottomScroll;
    }
  };

  /**
   * Clears all results data.
   */
  SearchPad.prototype._clearResults = function () {
    clear$1(this._resultsContainer);
    this._results = {};
    this._eventBus.fire('searchPad.cleared');
  };

  /**
   * Get currently selected result.
   *
   * @return {HTMLElement}
   */
  SearchPad.prototype._getCurrentResult = function () {
    return query(SearchPad.RESULT_SELECTED_SELECTOR, this._resultsContainer);
  };

  /**
   * Create result DOM element within results container
   * that corresponds to a search result.
   *
   * 'result' : one of the elements returned by Pad
   * 'id' : id attribute value to assign to the new DOM node
   * return : created DOM element
   *
   * @param {SearchResult} result
   * @param {string} id
   *
   * @return {HTMLElement}
   */
  SearchPad.prototype._createResultNode = function (result, id) {
    var node = domify$1(SearchPad.RESULT_HTML);

    // create only if available
    if (result.primaryTokens.length > 0) {
      createInnerTextNode(node, result.primaryTokens, SearchPad.RESULT_PRIMARY_HTML);
    }

    // secondary tokens (represent element ID) are allways available
    createInnerTextNode(node, result.secondaryTokens, SearchPad.RESULT_SECONDARY_HTML);
    attr$1(node, SearchPad.RESULT_ID_ATTRIBUTE, id);
    this._resultsContainer.appendChild(node);
    return node;
  };

  /**
   * Register search element provider.
   *
   * @param {SearchPadProvider} provider
   */
  SearchPad.prototype.registerProvider = function (provider) {
    this._searchProvider = provider;
  };

  /**
   * Open search pad.
   */
  SearchPad.prototype.open = function () {
    if (!this._searchProvider) {
      throw new Error('no search provider registered');
    }
    if (this.isOpen()) {
      return;
    }
    this._cachedRootElement = this._canvas.getRootElement();
    this._cachedSelection = this._selection.get();
    this._cachedViewbox = this._canvas.viewbox();
    this._selection.select(null);
    this._bindEvents();
    this._open = true;
    classes$1(this._canvas.getContainer()).add('djs-search-open');
    classes$1(this._container).add('open');
    this._searchInput.focus();
    this._eventBus.fire('searchPad.opened');
  };

  /**
   * Close search pad.
   */
  SearchPad.prototype.close = function (restoreCached = true) {
    if (!this.isOpen()) {
      return;
    }
    if (restoreCached) {
      if (this._cachedRootElement) {
        this._canvas.setRootElement(this._cachedRootElement);
      }
      if (this._cachedSelection) {
        this._selection.select(this._cachedSelection);
      }
      if (this._cachedViewbox) {
        this._canvas.viewbox(this._cachedViewbox);
      }
      this._eventBus.fire('searchPad.restored');
    }
    this._cachedRootElement = null;
    this._cachedSelection = null;
    this._cachedViewbox = null;
    this._unbindEvents();
    this._open = false;
    classes$1(this._canvas.getContainer()).remove('djs-search-open');
    classes$1(this._container).remove('open');
    this._clearResults();
    this._searchInput.value = '';
    this._searchInput.blur();
    this._eventBus.fire('searchPad.closed');
    this._canvas.restoreFocus();
  };

  /**
   * Toggles search pad on/off.
   */
  SearchPad.prototype.toggle = function () {
    this.isOpen() ? this.close() : this.open();
  };

  /**
   * Report state of search pad.
   */
  SearchPad.prototype.isOpen = function () {
    return this._open;
  };

  /**
   * Preselect result entry.
   *
   * @param {HTMLElement} element
   */
  SearchPad.prototype._preselect = function (node) {
    var selectedNode = this._getCurrentResult();

    // already selected
    if (node === selectedNode) {
      return;
    }

    // removing preselection from current node
    if (selectedNode) {
      classes$1(selectedNode).remove(SearchPad.RESULT_SELECTED_CLASS);
    }
    var id = attr$1(node, SearchPad.RESULT_ID_ATTRIBUTE);
    var element = this._results[id].element;
    classes$1(node).add(SearchPad.RESULT_SELECTED_CLASS);
    this._canvas.scrollToElement(element, {
      top: SCROLL_TO_ELEMENT_PADDING
    });
    this._selection.select(element);
    this._eventBus.fire('searchPad.preselected', element);
  };

  /**
   * Select result node.
   *
   * @param {HTMLElement} element
   */
  SearchPad.prototype._select = function (node) {
    var id = attr$1(node, SearchPad.RESULT_ID_ATTRIBUTE);
    var element = this._results[id].element;
    this._cachedSelection = null;
    this._cachedViewbox = null;
    this.close(false);
    this._canvas.scrollToElement(element, {
      top: SCROLL_TO_ELEMENT_PADDING
    });
    this._selection.select(element);
    this._eventBus.fire('searchPad.selected', element);
  };
  SearchPad.prototype._getBoxHtml = function () {
    const box = domify$1(SearchPad.BOX_HTML);
    const input = query(SearchPad.INPUT_SELECTOR, box);
    if (input) {
      input.setAttribute('aria-label', this._translate('Search in diagram'));
    }
    return box;
  };

  /**
   * Creates and appends child node from result tokens and HTML template.
   *
   * @param {HTMLElement} node
   * @param {Token[]} tokens
   * @param {string} template
   */
  function createInnerTextNode(parentNode, tokens, template) {
    var text = createHtmlText(tokens);
    var childNode = domify$1(template);
    childNode.innerHTML = text;
    parentNode.appendChild(childNode);
  }

  /**
   * Create internal HTML markup from result tokens.
   * Caters for highlighting pattern matched tokens.
   *
   * @param {Token[]} tokens
   *
   * @return {string|null}
   */
  function createHtmlText(tokens) {
    var htmlText = '';
    tokens.forEach(function (t) {
      var text = escapeHTML(t.value || t.matched || t.normal);
      var match = t.match || t.matched;
      if (match) {
        htmlText += '<b class="' + SearchPad.RESULT_HIGHLIGHT_CLASS + '">' + text + '</b>';
      } else {
        htmlText += text;
      }
    });
    return htmlText !== '' ? htmlText : null;
  }

  /**
   * CONSTANTS
   */
  SearchPad.CONTAINER_SELECTOR = '.djs-search-container';
  SearchPad.INPUT_SELECTOR = '.djs-search-input input';
  SearchPad.RESULTS_CONTAINER_SELECTOR = '.djs-search-results';
  SearchPad.RESULT_SELECTOR = '.djs-search-result';
  SearchPad.RESULT_SELECTED_CLASS = 'djs-search-result-selected';
  SearchPad.RESULT_SELECTED_SELECTOR = '.' + SearchPad.RESULT_SELECTED_CLASS;
  SearchPad.RESULT_ID_ATTRIBUTE = 'data-result-id';
  SearchPad.RESULT_HIGHLIGHT_CLASS = 'djs-search-highlight';
  SearchPad.BOX_HTML = `<div class="djs-search-container djs-scrollable">
  <div class="djs-search-input">
    <svg class="djs-search-icon" width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg">
      <path fill-rule="evenodd" clip-rule="evenodd" d="M9.0325 8.5H9.625L13.3675 12.25L12.25 13.3675L8.5 9.625V9.0325L8.2975 8.8225C7.4425 9.5575 6.3325 10 5.125 10C2.4325 10 0.25 7.8175 0.25 5.125C0.25 2.4325 2.4325 0.25 5.125 0.25C7.8175 0.25 10 2.4325 10 5.125C10 6.3325 9.5575 7.4425 8.8225 8.2975L9.0325 8.5ZM1.75 5.125C1.75 6.9925 3.2575 8.5 5.125 8.5C6.9925 8.5 8.5 6.9925 8.5 5.125C8.5 3.2575 6.9925 1.75 5.125 1.75C3.2575 1.75 1.75 3.2575 1.75 5.125Z" fill="#22242A"/>
    </svg>
    <input type="text" spellcheck="false" />
  </div>
  <div class="djs-search-results" />
</div>`;
  SearchPad.RESULT_HTML = '<div class="djs-search-result"></div>';
  SearchPad.RESULT_PRIMARY_HTML = '<div class="djs-search-result-primary"></div>';
  SearchPad.RESULT_SECONDARY_HTML = '<p class="djs-search-result-secondary"></p>';

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var SearchPadModule = {
    __depends__: [TranslateModule, OverlaysModule, SelectionModule],
    searchPad: ['type', SearchPad]
  };

  function getLabelAttr(semantic) {
    if (is$1(semantic, 'dmn:Decision') || is$1(semantic, 'dmn:BusinessKnowledgeModel') || is$1(semantic, 'dmn:InputData') || is$1(semantic, 'dmn:KnowledgeSource')) {
      return 'name';
    }
    if (is$1(semantic, 'dmn:TextAnnotation')) {
      return 'text';
    }
  }
  function getLabel(element) {
    var semantic = element.businessObject,
      attr = getLabelAttr(semantic);
    if (attr) {
      return semantic[attr] || '';
    }
  }
  function setLabel(element, text, isExternal) {
    var semantic = element.businessObject,
      attr = getLabelAttr(semantic);
    if (attr) {
      semantic[attr] = text;
    }

    // show external label if not empty
    if (isExternal) {
      element.hidden = !text;
    }
    return element;
  }

  /**
   * @typedef {import('diagram-js/lib/core/Canvas').default} Canvas
   * @typedef {import('diagram-js/lib/core/ElementRegistry').default} ElementRegistry
   * @typedef {import('diagram-js/lib/features/search-pad/SearchPad').default} SearchPad
   *
   * @typedef {import('diagram-js/lib/features/search-pad/SearchPadProvider').default
   * } SearchPadProvider
   * @typedef {import('diagram-js/lib/features/search-pad/SearchPadProvider').SearchResult
   * } SearchResult
   */

  /**
   * Provides ability to search for DMN elements.
   *
   * @implements {SearchPadProvider}
   *
   * @param {ElementRegistry} elementRegistry
   * @param {SearchPad} searchPad
   * @param {Canvas} canvas
   */
  function DmnSearchProvider(elementRegistry, searchPad, canvas) {
    this._elementRegistry = elementRegistry;
    this._canvas = canvas;
    searchPad.registerProvider(this);
  }
  DmnSearchProvider.$inject = ['elementRegistry', 'searchPad', 'canvas'];

  /**
   * @param {string} pattern
   *
   * @return {SearchResult[]}
   */
  DmnSearchProvider.prototype.find = function (pattern) {
    const rootElement = this._canvas.getRootElement();
    let elements = this._elementRegistry.filter(function (element) {
      if (element.labelTarget) {
        return false;
      }
      return true;
    });

    // do not include root element
    elements = filter$2(elements, function (element) {
      return element !== rootElement;
    });
    elements = map$3(elements, function (element) {
      return {
        primaryTokens: matchAndSplit(getLabel(element), pattern),
        secondaryTokens: matchAndSplit(element.id, pattern),
        element: element
      };
    });

    // exclude non-matched elements
    elements = filter$2(elements, function (element) {
      return hasMatched(element.primaryTokens) || hasMatched(element.secondaryTokens);
    });
    elements = sortBy(elements, function (element) {
      return getLabel(element.element) + element.element.id;
    });
    return elements;
  };

  /**
   * @param {Token[]} tokens
   *
   * @return {boolean}
   */
  function hasMatched(tokens) {
    const matched = filter$2(tokens, function (token) {
      return !!token.matched;
    });
    return matched.length > 0;
  }

  /**
   * @param {string} text
   * @param {string} pattern
   *
   * @return {Token[]}
   */
  function matchAndSplit(text, pattern) {
    const tokens = [],
      originalText = text;
    if (!text) {
      return tokens;
    }
    text = text.toLowerCase();
    pattern = pattern.toLowerCase();
    const i = text.indexOf(pattern);
    if (i > -1) {
      if (i !== 0) {
        tokens.push({
          normal: originalText.substr(0, i)
        });
      }
      tokens.push({
        matched: originalText.substr(i, pattern.length)
      });
      if (pattern.length + i < text.length) {
        tokens.push({
          normal: originalText.substr(pattern.length + i, text.length)
        });
      }
    } else {
      tokens.push({
        normal: originalText
      });
    }
    return tokens;
  }

  var DmnSearchModule = {
    __depends__: [SearchPadModule],
    __init__: ['dmnSearch'],
    dmnSearch: ['type', DmnSearchProvider]
  };

  /**
   * A viewer that includes mouse navigation facilities
   *
   * @param {Object} options
   */
  function NavigatedViewer(options) {
    Viewer$4.call(this, options);
  }
  e$3(NavigatedViewer, Viewer$4);
  NavigatedViewer.prototype._navigationModules = [ZoomScroll, MoveCanvas, DmnSearchModule];
  NavigatedViewer.prototype._modules = [].concat(NavigatedViewer.prototype._modules, NavigatedViewer.prototype._navigationModules);

  /**
   * @typedef {import('../../model/Types').Element} Element
   *
   * @typedef {import('../modeling/Modeling').default} Modeling
   * @typedef {import('../rules/Rules').default} Rules
   *
   * @typedef {import('../../util/Types').Axis} Axis
   * @typedef {import('../../util/Types').Dimension} Dimension
   *
   * @typedef { 'top' | 'right' | 'bottom' | 'left' | 'center' | 'middle' } Alignment
   */

  function last$1(arr) {
    return arr && arr[arr.length - 1];
  }
  function sortTopOrMiddle(element) {
    return element.y;
  }
  function sortLeftOrCenter(element) {
    return element.x;
  }

  /**
   * Sorting functions for different alignments.
   *
   * @type {Record<string, Function>}
   */
  var ALIGNMENT_SORTING = {
    left: sortLeftOrCenter,
    center: sortLeftOrCenter,
    right: function (element) {
      return element.x + element.width;
    },
    top: sortTopOrMiddle,
    middle: sortTopOrMiddle,
    bottom: function (element) {
      return element.y + element.height;
    }
  };

  /**
   * @param {Modeling} modeling
   * @param {Rules} rules
   */
  function AlignElements$1(modeling, rules) {
    this._modeling = modeling;
    this._rules = rules;
  }
  AlignElements$1.$inject = ['modeling', 'rules'];

  /**
   * Get relevant axis and dimension for given alignment.
   *
   * @param {Alignment} type
   *
   * @return { {
   *   axis: Axis;
   *   dimension: Dimension;
   * } }
   */
  AlignElements$1.prototype._getOrientationDetails = function (type) {
    var vertical = ['top', 'bottom', 'middle'],
      axis = 'x',
      dimension = 'width';
    if (vertical.indexOf(type) !== -1) {
      axis = 'y';
      dimension = 'height';
    }
    return {
      axis: axis,
      dimension: dimension
    };
  };
  AlignElements$1.prototype._isType = function (type, types) {
    return types.indexOf(type) !== -1;
  };

  /**
   * Get point on relevant axis for given alignment.
   *
   * @param {Alignment} type
   * @param {Element[]} sortedElements
   *
   * @return {Partial<Record<Alignment, number>>}
   */
  AlignElements$1.prototype._alignmentPosition = function (type, sortedElements) {
    var orientation = this._getOrientationDetails(type),
      axis = orientation.axis,
      dimension = orientation.dimension,
      alignment = {},
      centers = {},
      hasSharedCenters = false,
      centeredElements,
      firstElement,
      lastElement;
    function getMiddleOrTop(first, last) {
      return Math.round((first[axis] + last[axis] + last[dimension]) / 2);
    }
    if (this._isType(type, ['left', 'top'])) {
      alignment[type] = sortedElements[0][axis];
    } else if (this._isType(type, ['right', 'bottom'])) {
      lastElement = last$1(sortedElements);
      alignment[type] = lastElement[axis] + lastElement[dimension];
    } else if (this._isType(type, ['center', 'middle'])) {
      // check if there is a center shared by more than one shape
      // if not, just take the middle of the range
      forEach$3(sortedElements, function (element) {
        var center = element[axis] + Math.round(element[dimension] / 2);
        if (centers[center]) {
          centers[center].elements.push(element);
        } else {
          centers[center] = {
            elements: [element],
            center: center
          };
        }
      });
      centeredElements = sortBy(centers, function (center) {
        if (center.elements.length > 1) {
          hasSharedCenters = true;
        }
        return center.elements.length;
      });
      if (hasSharedCenters) {
        alignment[type] = last$1(centeredElements).center;
        return alignment;
      }
      firstElement = sortedElements[0];
      sortedElements = sortBy(sortedElements, function (element) {
        return element[axis] + element[dimension];
      });
      lastElement = last$1(sortedElements);
      alignment[type] = getMiddleOrTop(firstElement, lastElement);
    }
    return alignment;
  };

  /**
   * Align elements on relevant axis for given alignment.
   *
   * @param {Element[]} elements
   * @param {Alignment} type
   */
  AlignElements$1.prototype.trigger = function (elements, type) {
    var modeling = this._modeling,
      allowed;

    // filter out elements which cannot be aligned
    var filteredElements = filter$2(elements, function (element) {
      return !(element.waypoints || element.host || element.labelTarget);
    });

    // filter out elements via rules
    allowed = this._rules.allowed('elements.align', {
      elements: filteredElements
    });
    if (isArray$6(allowed)) {
      filteredElements = allowed;
    }
    if (filteredElements.length < 2 || !allowed) {
      return;
    }
    var sortFn = ALIGNMENT_SORTING[type];
    var sortedElements = sortBy(filteredElements, sortFn);
    var alignment = this._alignmentPosition(type, sortedElements);
    modeling.alignElements(sortedElements, alignment);
  };

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var AlignElementsModule = {
    __init__: ['alignElements'],
    alignElements: ['type', AlignElements$1]
  };

  /**
   * @typedef {import('../../model/Types').Connection} Connection
   * @typedef {import('../../model/Types').Element} Element
   * @typedef {import('../../model/Types').Shape} Shape
   *
   * @typedef {import('../../util/Types').Point} Point
   */

  // padding to detect element placement
  var PLACEMENT_DETECTION_PAD = 10;
  var DEFAULT_DISTANCE = 50;
  var DEFAULT_MAX_DISTANCE = 250;

  /**
   * Get free position starting from given position.
   *
   * @param {Shape} source
   * @param {Shape} element
   * @param {Point} position
   * @param {(element: Element, position: Point, connectedAtPosition: Element) => Point} getNextPosition
   *
   * @return {Point}
   */
  function findFreePosition(source, element, position, getNextPosition) {
    var connectedAtPosition;
    while (connectedAtPosition = getConnectedAtPosition(source, position, element)) {
      position = getNextPosition(element, position, connectedAtPosition);
    }
    return position;
  }

  /**
   * Returns function that returns next position.
   *
   * @param {Object} nextPositionDirection
   * @param {Object} [nextPositionDirection.x]
   * @param {Object} [nextPositionDirection.y]
   *
   * @return {(element: Element, previousPosition: Point, connectedAtPosition: Element) => Point}
   */
  function generateGetNextPosition(nextPositionDirection) {
    return function (element, previousPosition, connectedAtPosition) {
      var nextPosition = {
        x: previousPosition.x,
        y: previousPosition.y
      };
      ['x', 'y'].forEach(function (axis) {
        var nextPositionDirectionForAxis = nextPositionDirection[axis];
        if (!nextPositionDirectionForAxis) {
          return;
        }
        var dimension = axis === 'x' ? 'width' : 'height';
        var margin = nextPositionDirectionForAxis.margin,
          minDistance = nextPositionDirectionForAxis.minDistance;
        if (margin < 0) {
          nextPosition[axis] = Math.min(connectedAtPosition[axis] + margin - element[dimension] / 2, previousPosition[axis] - minDistance + margin);
        } else {
          nextPosition[axis] = Math.max(connectedAtPosition[axis] + connectedAtPosition[dimension] + margin + element[dimension] / 2, previousPosition[axis] + minDistance + margin);
        }
      });
      return nextPosition;
    };
  }

  /**
   * Return connected element at given position and within given bounds. Takes
   * connected elements from host and attachers into account, too.
   *
   * @param {Shape} source
   * @param {Point} position
   * @param {Shape} element
   *
   * @return {Shape|undefined}
   */
  function getConnectedAtPosition(source, position, element) {
    var bounds = {
      x: position.x - element.width / 2,
      y: position.y - element.height / 2,
      width: element.width,
      height: element.height
    };
    var closure = getAutoPlaceClosure(source);
    return find$2(closure, function (target) {
      if (target === element) {
        return false;
      }
      var orientation = getOrientation(target, bounds, PLACEMENT_DETECTION_PAD);
      return orientation === 'intersect';
    });
  }

  /**
  * Compute optimal distance between source and target based on existing connections to and from source.
  * Assumes left-to-right and top-to-down modeling.
  *
  * @param {Shape} source
  * @param {Object} [hints]
  * @param {number} [hints.defaultDistance]
  * @param {string} [hints.direction]
  * @param {(connection: Connection) => boolean} [hints.filter]
  * @param {(connection: Connection) => number} [hints.getWeight]
  * @param {number} [hints.maxDistance]
  * @param {'start'|'center'|'end'} [hints.reference]
  *
  * @return {number}
  */
  function getConnectedDistance(source, hints) {
    if (!hints) {
      hints = {};
    }

    // targets > sources by default
    function getDefaultWeight(connection) {
      return connection.source === source ? 1 : -1;
    }
    var defaultDistance = hints.defaultDistance || DEFAULT_DISTANCE,
      direction = hints.direction || 'e',
      filter = hints.filter,
      getWeight = hints.getWeight || getDefaultWeight,
      maxDistance = hints.maxDistance || DEFAULT_MAX_DISTANCE,
      reference = hints.reference || 'start';
    if (!filter) {
      filter = noneFilter;
    }
    function getDistance(a, b) {
      if (direction === 'n') {
        if (reference === 'start') {
          return asTRBL(a).top - asTRBL(b).bottom;
        } else if (reference === 'center') {
          return asTRBL(a).top - getMid(b).y;
        } else {
          return asTRBL(a).top - asTRBL(b).top;
        }
      } else if (direction === 'w') {
        if (reference === 'start') {
          return asTRBL(a).left - asTRBL(b).right;
        } else if (reference === 'center') {
          return asTRBL(a).left - getMid(b).x;
        } else {
          return asTRBL(a).left - asTRBL(b).left;
        }
      } else if (direction === 's') {
        if (reference === 'start') {
          return asTRBL(b).top - asTRBL(a).bottom;
        } else if (reference === 'center') {
          return getMid(b).y - asTRBL(a).bottom;
        } else {
          return asTRBL(b).bottom - asTRBL(a).bottom;
        }
      } else {
        if (reference === 'start') {
          return asTRBL(b).left - asTRBL(a).right;
        } else if (reference === 'center') {
          return getMid(b).x - asTRBL(a).right;
        } else {
          return asTRBL(b).right - asTRBL(a).right;
        }
      }
    }
    var sourcesDistances = source.incoming.filter(filter).map(function (connection) {
      var weight = getWeight(connection);
      var distance = weight < 0 ? getDistance(connection.source, source) : getDistance(source, connection.source);
      return {
        id: connection.source.id,
        distance: distance,
        weight: weight
      };
    });
    var targetsDistances = source.outgoing.filter(filter).map(function (connection) {
      var weight = getWeight(connection);
      var distance = weight > 0 ? getDistance(source, connection.target) : getDistance(connection.target, source);
      return {
        id: connection.target.id,
        distance: distance,
        weight: weight
      };
    });
    var distances = sourcesDistances.concat(targetsDistances).reduce(function (accumulator, currentValue) {
      accumulator[currentValue.id + '__weight_' + currentValue.weight] = currentValue;
      return accumulator;
    }, {});
    var distancesGrouped = reduce(distances, function (accumulator, currentValue) {
      var distance = currentValue.distance,
        weight = currentValue.weight;
      if (distance < 0 || distance > maxDistance) {
        return accumulator;
      }
      if (!accumulator[String(distance)]) {
        accumulator[String(distance)] = 0;
      }
      accumulator[String(distance)] += 1 * weight;
      if (!accumulator.distance || accumulator[accumulator.distance] < accumulator[String(distance)]) {
        accumulator.distance = distance;
      }
      return accumulator;
    }, {});
    return distancesGrouped.distance || defaultDistance;
  }

  /**
   * Returns all elements connected to given source.
   *
   * This includes:
   *
   *   - elements connected to source
   *   - elements connected to host if source is an attacher
   *   - elements connected to attachers if source is a host
   *
   * @param {Shape} source
   *
   * @return {Shape[]}
   */
  function getAutoPlaceClosure(source) {
    var allConnected = getConnected(source);
    if (source.host) {
      allConnected = allConnected.concat(getConnected(source.host));
    }
    if (source.attachers) {
      allConnected = allConnected.concat(source.attachers.reduce(function (shapes, attacher) {
        return shapes.concat(getConnected(attacher));
      }, []));
    }
    return allConnected;
  }

  /**
   * Get all connected elements.
   *
   * @param {Shape} element
   *
   * @returns {Shape[]}
   */
  function getConnected(element) {
    return getTargets(element).concat(getSources(element));
  }
  function getSources(shape) {
    return shape.incoming.map(function (connection) {
      return connection.source;
    });
  }
  function getTargets(shape) {
    return shape.outgoing.map(function (connection) {
      return connection.target;
    });
  }
  function noneFilter() {
    return true;
  }

  /**
   * @typedef {import('../../core/Types').ShapeLike} Shape
   *
   * @typedef {import('../../util/Types').Point} Point
   *
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('../modeling/Modeling').default} Modeling
   */

  var LOW_PRIORITY$k = 100;

  /**
   * A service that places elements connected to existing ones
   * to an appropriate position in an _automated_ fashion.
   *
   * @param {EventBus} eventBus
   * @param {Modeling} modeling
   * @param {Canvas} canvas
   */
  function AutoPlace$1(eventBus, modeling, canvas) {
    eventBus.on('autoPlace', LOW_PRIORITY$k, function (context) {
      var shape = context.shape,
        source = context.source;
      return getNewShapePosition$1(source, shape);
    });
    eventBus.on('autoPlace.end', function (event) {
      canvas.scrollToElement(event.shape);
    });

    /**
     * Append shape to source at appropriate position.
     *
     * @param {Shape} source
     * @param {Shape} shape
     * @param {any} [hints={}]
     *
     * @return {Shape} appended shape
     */
    this.append = function (source, shape, hints) {
      eventBus.fire('autoPlace.start', {
        source: source,
        shape: shape
      });

      // allow others to provide the position
      var position = eventBus.fire('autoPlace', {
        source: source,
        shape: shape
      });
      var newShape = modeling.appendShape(source, shape, position, source.parent, hints);
      eventBus.fire('autoPlace.end', {
        source: source,
        shape: newShape
      });
      return newShape;
    };
  }
  AutoPlace$1.$inject = ['eventBus', 'modeling', 'canvas'];

  // helpers //////////

  /**
   * Find the new position for the target element to
   * connect to source.
   *
   * @param {Shape} source
   * @param {Shape} element
   * @param {Object} [hints]
   * @param {Object} [hints.defaultDistance]
   *
   * @return {Point}
   */
  function getNewShapePosition$1(source, element, hints) {
    if (!hints) {
      hints = {};
    }
    var distance = hints.defaultDistance || DEFAULT_DISTANCE;
    var sourceMid = getMid(source),
      sourceTrbl = asTRBL(source);

    // simply put element right next to source
    return {
      x: sourceTrbl.right + distance + element.width / 2,
      y: sourceMid.y
    };
  }

  /**
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('../selection/Selection').default} Selection
   */

  /**
   * Select element after auto placement.
   *
   * @param {EventBus} eventBus
   * @param {Selection} selection
   */
  function AutoPlaceSelectionBehavior(eventBus, selection) {
    eventBus.on('autoPlace.end', 500, function (e) {
      selection.select(e.shape);
    });
  }
  AutoPlaceSelectionBehavior.$inject = ['eventBus', 'selection'];

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var AutoPlaceModule$1 = {
    __init__: ['autoPlaceSelectionBehavior'],
    autoPlace: ['type', AutoPlace$1],
    autoPlaceSelectionBehavior: ['type', AutoPlaceSelectionBehavior]
  };

  var BUSINESS_KNOWLEDGE_MODEL_SIZE = {
    width: 135,
    height: 46
  };
  var DECISION_SIZE = {
    width: 180,
    height: 80
  };
  var INPUT_DATA_SIZE = {
    width: 125,
    height: 45
  };
  var KNOWLEDGE_SOURCE_SIZE = {
    width: 100,
    height: 63
  };

  /**
   * A drd-aware factory for diagram-js shapes
   */
  function ElementFactory$2(drdFactory) {
    ElementFactory$3.call(this);
    this._drdFactory = drdFactory;
  }
  e$3(ElementFactory$2, ElementFactory$3);
  ElementFactory$2.$inject = ['drdFactory'];
  ElementFactory$2.prototype.baseCreate = ElementFactory$3.prototype.create;
  ElementFactory$2.prototype.create = function (elementType, attrs) {
    return this.createDrdElement(elementType, attrs);
  };
  ElementFactory$2.prototype.createDrdElement = function (elementType, attrs) {
    var drdFactory = this._drdFactory;
    var size;
    attrs = attrs || {};
    var businessObject = attrs.businessObject;
    if (!businessObject) {
      if (!attrs.type) {
        throw new Error('no shape type specified');
      }
      businessObject = drdFactory.create(attrs.type);
    }
    if (!businessObject.di) {
      if (elementType === 'connection') {
        businessObject.di = drdFactory.createDiEdge(businessObject, []);
      } else if (elementType === 'shape') {
        businessObject.di = drdFactory.createDiShape(businessObject, {});
      }
    }
    size = this._getDefaultSize(businessObject);
    attrs = assign$4({
      businessObject: businessObject,
      id: businessObject.id
    }, size, attrs);
    return this.baseCreate(elementType, attrs);
  };
  ElementFactory$2.prototype._getDefaultSize = function (semantic) {
    if (is$1(semantic, 'dmn:BusinessKnowledgeModel')) {
      return BUSINESS_KNOWLEDGE_MODEL_SIZE;
    }
    if (is$1(semantic, 'dmn:Decision')) {
      return DECISION_SIZE;
    }
    if (is$1(semantic, 'dmn:InputData')) {
      return INPUT_DATA_SIZE;
    }
    if (is$1(semantic, 'dmn:KnowledgeSource')) {
      return KNOWLEDGE_SOURCE_SIZE;
    }
    return {
      width: 100,
      height: 80
    };
  };

  var DIRECTION_LEFT = 'left',
    DIRECTION_RIGHT = 'right';
  var DRG_ELEMENT_MARGIN = 60,
    DRG_ELEMENT_ROW_SIZE = DECISION_SIZE.width;
  function getNewShapePosition(source, element) {
    if (is$1(element, 'dmn:TextAnnotation')) {
      return getTextAnnotationPosition(source, element);
    }
    if (is$1(element, 'dmn:DRGElement')) {
      return getDRGElementPosition(source, element);
    }
  }

  /**
   * Always try to place text annotations top right of source.
   */
  function getTextAnnotationPosition(source, element) {
    var sourceTrbl = asTRBL(source);
    var position = {
      x: sourceTrbl.right + element.width / 2,
      y: sourceTrbl.top - 50 - element.height / 2
    };
    var nextPositionDirection = {
      y: {
        margin: -30,
        minDistance: 20
      }
    };
    return findFreePosition(source, element, position, generateGetNextPosition(nextPositionDirection));
  }

  /**
   * Get position for DRG elements.
   *
   * @param {djs.model.Shape} source
   * @param {djs.model.Shape} element
   *
   * @returns {Point}
   */
  function getDRGElementPosition(source, element) {
    var sourceTrbl = asTRBL(source),
      sourceMid = getMid(source);
    function getWeight(connection) {
      return connection.target === source ? 1 : -1;
    }
    var verticalDistance = getConnectedDistance(source, {
      defaultDistance: 180,
      direction: 's',
      getWeight: getWeight,
      filter: filter,
      reference: 'center'
    });
    var position = {
      x: sourceMid.x,
      y: sourceTrbl.bottom + verticalDistance
    };
    return findFreePosition(source, element, position, generateGetNextDRGElementPosition(source));
  }

  // helpers //////////

  function filter(connection) {
    return !is$1(connection, 'dmn:Association');
  }
  function getHorizontalDistance(a, b) {
    return Math.abs(b.x - a.x);
  }
  function generateGetNextDRGElementPosition(source) {
    var sourceMid = getMid(source);
    var connectedAtPositionLeft, connectedAtPositionRight;
    return function (element, previousPosition, connectedAtPreviousPosition) {
      var direction;

      // (1) get direction
      if (!connectedAtPositionLeft) {
        connectedAtPositionLeft = connectedAtPreviousPosition;
        connectedAtPositionRight = connectedAtPreviousPosition;
        if (getMid(connectedAtPreviousPosition).x - sourceMid.x > 0) {
          direction = DIRECTION_LEFT;
        } else {
          direction = DIRECTION_RIGHT;
        }
      } else {
        if (previousPosition.x < sourceMid.x) {
          connectedAtPositionLeft = connectedAtPreviousPosition;
        } else {
          connectedAtPositionRight = connectedAtPreviousPosition;
        }
        if (getHorizontalDistance(sourceMid, getMid(connectedAtPositionLeft)) < getHorizontalDistance(sourceMid, getMid(connectedAtPositionRight))) {
          direction = DIRECTION_LEFT;
        } else {
          direction = DIRECTION_RIGHT;
        }
      }

      // (2) get next position
      if (direction === DIRECTION_LEFT) {
        return {
          x: Math.min(getMid(connectedAtPositionLeft).x - DRG_ELEMENT_ROW_SIZE - DRG_ELEMENT_MARGIN, asTRBL(connectedAtPositionLeft).left - DRG_ELEMENT_MARGIN - element.width / 2),
          y: previousPosition.y
        };
      } else {
        return {
          x: Math.max(getMid(connectedAtPositionRight).x + DRG_ELEMENT_ROW_SIZE + DRG_ELEMENT_MARGIN, asTRBL(connectedAtPositionRight).right + DRG_ELEMENT_MARGIN + element.width / 2),
          y: previousPosition.y
        };
      }
    };
  }

  /**
   * DMN auto-place behavior.
   *
   * @param {EventBus} eventBus
   */
  function AutoPlace(eventBus) {
    eventBus.on('autoPlace', function (context) {
      var shape = context.shape,
        source = context.source;
      return getNewShapePosition(source, shape);
    });
  }
  AutoPlace.$inject = ['eventBus'];

  var AutoPlaceModule = {
    __depends__: [AutoPlaceModule$1],
    __init__: ['dmnAutoPlace'],
    dmnAutoPlace: ['type', AutoPlace]
  };

  /**
   * @typedef {import('didi').Injector} Injector
   *
   * @typedef {import('../../core/ElementRegistry').default} ElementRegistry
   * @typedef {import('../../core/EventBus').default} EventBus
   */

  var HIGH_PRIORITY$b = 1500;

  /**
   * Browsers may swallow certain events (hover, out ...) if users are to
   * fast with the mouse.
   *
   * @see http://stackoverflow.com/questions/7448468/why-cant-i-reliably-capture-a-mouseout-event
   *
   * The fix implemented in this component ensure that we
   *
   * 1) have a hover state after a successful drag.move event
   * 2) have an out event when dragging leaves an element
   *
   * @param {ElementRegistry} elementRegistry
   * @param {EventBus} eventBus
   * @param {Injector} injector
   */
  function HoverFix(elementRegistry, eventBus, injector) {
    var self = this;
    var dragging = injector.get('dragging', false);

    /**
     * Make sure we are god damn hovering!
     *
     * @param {Event} dragging event
     */
    function ensureHover(event) {
      if (event.hover) {
        return;
      }
      var originalEvent = event.originalEvent;
      var gfx = self._findTargetGfx(originalEvent);
      var element = gfx && elementRegistry.get(gfx);
      if (gfx && element) {
        // 1) cancel current mousemove
        event.stopPropagation();

        // 2) emit fake hover for new target
        dragging.hover({
          element: element,
          gfx: gfx
        });

        // 3) re-trigger move event
        dragging.move(originalEvent);
      }
    }
    if (dragging) {
      /**
       * We wait for a specific sequence of events before
       * emitting a fake drag.hover event.
       *
       * Event Sequence:
       *
       * drag.start
       * drag.move >> ensure we are hovering
       */
      eventBus.on('drag.start', function (event) {
        eventBus.once('drag.move', HIGH_PRIORITY$b, function (event) {
          ensureHover(event);
        });
      });
    }

    /**
     * We make sure that element.out is always fired, even if the
     * browser swallows an element.out event.
     *
     * Event sequence:
     *
     * element.hover
     * (element.out >> sometimes swallowed)
     * element.hover >> ensure we fired element.out
     */
    (function () {
      var hoverGfx;
      var hover;
      eventBus.on('element.hover', function (event) {
        // (1) remember current hover element
        hoverGfx = event.gfx;
        hover = event.element;
      });
      eventBus.on('element.hover', HIGH_PRIORITY$b, function (event) {
        // (3) am I on an element still?
        if (hover) {
          // (4) that is a problem, gotta "simulate the out"
          eventBus.fire('element.out', {
            element: hover,
            gfx: hoverGfx
          });
        }
      });
      eventBus.on('element.out', function () {
        // (2) unset hover state if we correctly outed us *GG*
        hoverGfx = null;
        hover = null;
      });
    })();
    this._findTargetGfx = function (event) {
      var position, target;
      if (!(event instanceof MouseEvent)) {
        return;
      }
      position = toPoint(event);

      // damn expensive operation, ouch!
      target = document.elementFromPoint(position.x, position.y);
      return getGfx(target);
    };
  }
  HoverFix.$inject = ['elementRegistry', 'eventBus', 'injector'];

  // helpers /////////////////////

  function getGfx(target) {
    return closest(target, 'svg, .djs-element', true);
  }

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var HoverFixModule = {
    __init__: ['hoverFix'],
    hoverFix: ['type', HoverFix]
  };

  var round$8 = Math.round;

  /**
   * @typedef {import('../../util/Types').Point} Point
   *
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/ElementRegistry').default} ElementRegistry
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('../selection/Selection').default} Selection
   */

  var DRAG_ACTIVE_CLS = 'djs-drag-active';
  function preventDefault$1(event) {
    event.preventDefault();
  }
  function isTouchEvent(event) {
    // check for TouchEvent being available first
    // (i.e. not available on desktop Firefox)
    return typeof TouchEvent !== 'undefined' && event instanceof TouchEvent;
  }
  function getLength(point) {
    return Math.sqrt(Math.pow(point.x, 2) + Math.pow(point.y, 2));
  }

  /**
   * A helper that fires canvas localized drag events and realizes
   * the general "drag-and-drop" look and feel.
   *
   * Calling {@link Dragging#activate} activates dragging on a canvas.
   *
   * It provides the following:
   *
   *   * emits life cycle events, namespaced with a prefix assigned
   *     during dragging activation
   *   * sets and restores the cursor
   *   * sets and restores the selection if elements still exist
   *   * ensures there can be only one drag operation active at a time
   *
   * Dragging may be canceled manually by calling {@link Dragging#cancel}
   * or by pressing ESC.
   *
   *
   * ## Life-cycle events
   *
   * Dragging can be in three different states, off, initialized
   * and active.
   *
   * (1) off: no dragging operation is in progress
   * (2) initialized: a new drag operation got initialized but not yet
   *                  started (i.e. because of no initial move)
   * (3) started: dragging is in progress
   *
   * Eventually dragging will be off again after a drag operation has
   * been ended or canceled via user click or ESC key press.
   *
   * To indicate transitions between these states dragging emits generic
   * life-cycle events with the `drag.` prefix _and_ events namespaced
   * to a prefix choosen by a user during drag initialization.
   *
   * The following events are emitted (appropriately prefixed) via
   * the {@link EventBus}.
   *
   * * `init`
   * * `start`
   * * `move`
   * * `end`
   * * `ended` (dragging already in off state)
   * * `cancel` (only if previously started)
   * * `canceled` (dragging already in off state, only if previously started)
   * * `cleanup`
   *
   *
   * @example
   *
   * ```javascript
   * function MyDragComponent(eventBus, dragging) {
   *
   *   eventBus.on('mydrag.start', function(event) {
   *     console.log('yes, we start dragging');
   *   });
   *
   *   eventBus.on('mydrag.move', function(event) {
   *     console.log('canvas local coordinates', event.x, event.y, event.dx, event.dy);
   *
   *     // local drag data is passed with the event
   *     event.context.foo; // "BAR"
   *
   *     // the original mouse event, too
   *     event.originalEvent; // MouseEvent(...)
   *   });
   *
   *   eventBus.on('element.click', function(event) {
   *     dragging.init(event, 'mydrag', {
   *       cursor: 'grabbing',
   *       data: {
   *         context: {
   *           foo: "BAR"
   *         }
   *       }
   *     });
   *   });
   * }
   * ```
   *
   * @param {EventBus} eventBus
   * @param {Canvas} canvas
   * @param {Selection} selection
   * @param {ElementRegistry} elementRegistry
   */
  function Dragging(eventBus, canvas, selection, elementRegistry) {
    var defaultOptions = {
      threshold: 5,
      trapClick: true
    };

    // the currently active drag operation
    // dragging is active as soon as this context exists.
    //
    // it is visually _active_ only when a context.active flag is set to true.
    var context;

    /* convert a global event into local coordinates */
    function toLocalPoint(globalPosition) {
      var viewbox = canvas.viewbox();
      var clientRect = canvas._container.getBoundingClientRect();
      return {
        x: viewbox.x + (globalPosition.x - clientRect.left) / viewbox.scale,
        y: viewbox.y + (globalPosition.y - clientRect.top) / viewbox.scale
      };
    }

    // helpers

    function fire(type, dragContext) {
      dragContext = dragContext || context;
      var event = eventBus.createEvent(assign$4({}, dragContext.payload, dragContext.data, {
        isTouch: dragContext.isTouch
      }));

      // default integration
      if (eventBus.fire('drag.' + type, event) === false) {
        return false;
      }
      return eventBus.fire(dragContext.prefix + '.' + type, event);
    }
    function restoreSelection(previousSelection) {
      var existingSelection = previousSelection.filter(function (element) {
        return elementRegistry.get(element.id);
      });
      existingSelection.length && selection.select(existingSelection);
    }

    // event listeners

    function move(event, activate) {
      var payload = context.payload,
        displacement = context.displacement;
      var globalStart = context.globalStart,
        globalCurrent = toPoint(event),
        globalDelta = delta(globalCurrent, globalStart);
      var localStart = context.localStart,
        localCurrent = toLocalPoint(globalCurrent),
        localDelta = delta(localCurrent, localStart);

      // activate context explicitly or once threshold is reached
      if (!context.active && (activate || getLength(globalDelta) > context.threshold)) {
        // fire start event with original
        // starting coordinates

        assign$4(payload, {
          x: round$8(localStart.x + displacement.x),
          y: round$8(localStart.y + displacement.y),
          dx: 0,
          dy: 0
        }, {
          originalEvent: event
        });
        if (false === fire('start')) {
          return cancel();
        }
        context.active = true;

        // unset selection and remember old selection
        // the previous (old) selection will always passed
        // with the event via the event.previousSelection property
        if (!context.keepSelection) {
          payload.previousSelection = selection.get();
          selection.select(null);
        }

        // allow custom cursor
        if (context.cursor) {
          set(context.cursor);
        }

        // indicate dragging via marker on root element
        canvas.addMarker(canvas.getRootElement(), DRAG_ACTIVE_CLS);
      }
      stopPropagation$2(event);
      if (context.active) {
        // update payload with actual coordinates
        assign$4(payload, {
          x: round$8(localCurrent.x + displacement.x),
          y: round$8(localCurrent.y + displacement.y),
          dx: round$8(localDelta.x),
          dy: round$8(localDelta.y)
        }, {
          originalEvent: event
        });

        // emit move event
        fire('move');
      }
    }
    function end(event) {
      var previousContext,
        returnValue = true;
      if (context.active) {
        if (event) {
          context.payload.originalEvent = event;

          // suppress original event (click, ...)
          // because we just ended a drag operation
          stopPropagation$2(event);
        }

        // implementations may stop restoring the
        // original state (selections, ...) by preventing the
        // end events default action
        returnValue = fire('end');
      }
      if (returnValue === false) {
        fire('rejected');
      }
      previousContext = cleanup(returnValue !== true);

      // last event to be fired when all drag operations are done
      // at this point in time no drag operation is in progress anymore
      fire('ended', previousContext);
    }

    // cancel active drag operation if the user presses
    // the ESC key on the keyboard

    function checkCancel(event) {
      if (isKey('Escape', event)) {
        preventDefault$1(event);
        cancel();
      }
    }

    // prevent ghost click that might occur after a finished
    // drag and drop session

    function trapClickAndEnd(event) {
      var untrap;

      // trap the click in case we are part of an active
      // drag operation. This will effectively prevent
      // the ghost click that cannot be canceled otherwise.
      if (context.active) {
        untrap = install(eventBus);

        // remove trap after minimal delay
        setTimeout(untrap, 400);

        // prevent default action (click)
        preventDefault$1(event);
      }
      end(event);
    }
    function trapTouch(event) {
      move(event);
    }

    // update the drag events model element (`hover`) and graphical element (`hoverGfx`)
    // properties during hover and out and fire {prefix}.hover and {prefix}.out properties
    // respectively

    function hover(event) {
      var payload = context.payload;
      payload.hoverGfx = event.gfx;
      payload.hover = event.element;
      fire('hover');
    }
    function out(event) {
      fire('out');
      var payload = context.payload;
      payload.hoverGfx = null;
      payload.hover = null;
    }

    // life-cycle methods

    function cancel(restore) {
      var previousContext;
      if (!context) {
        return;
      }
      var wasActive = context.active;
      if (wasActive) {
        fire('cancel');
      }
      previousContext = cleanup(restore);
      if (wasActive) {
        // last event to be fired when all drag operations are done
        // at this point in time no drag operation is in progress anymore
        fire('canceled', previousContext);
      }
    }
    function cleanup(restore) {
      var previousContext, endDrag;
      fire('cleanup');

      // reset cursor
      unset();
      if (context.trapClick) {
        endDrag = trapClickAndEnd;
      } else {
        endDrag = end;
      }

      // reset dom listeners
      event.unbind(document, 'mousemove', move);
      event.unbind(document, 'dragstart', preventDefault$1);
      event.unbind(document, 'selectstart', preventDefault$1);
      event.unbind(document, 'mousedown', endDrag, true);
      event.unbind(document, 'mouseup', endDrag, true);
      event.unbind(document, 'keyup', checkCancel);
      event.unbind(document, 'touchstart', trapTouch, true);
      event.unbind(document, 'touchcancel', cancel, true);
      event.unbind(document, 'touchmove', move, true);
      event.unbind(document, 'touchend', end, true);
      eventBus.off('element.hover', hover);
      eventBus.off('element.out', out);

      // remove drag marker on root element
      canvas.removeMarker(canvas.getRootElement(), DRAG_ACTIVE_CLS);

      // restore selection, unless it has changed
      var previousSelection = context.payload.previousSelection;
      if (restore !== false && previousSelection && !selection.get().length) {
        restoreSelection(previousSelection);
      }
      previousContext = context;
      context = null;
      return previousContext;
    }

    /**
     * Initialize a drag operation.
     *
     * If `localPosition` is given, drag events will be emitted
     * relative to it.
     *
     * @param {MouseEvent|TouchEvent} [event]
     * @param {Point} [relativeTo] actual diagram local position this drag operation should start at
     * @param {string} prefix
     * @param {Object} [options]
     */
    function init(event$1, relativeTo, prefix, options) {
      // only one drag operation may be active, at a time
      if (context) {
        cancel(false);
      }
      if (typeof relativeTo === 'string') {
        options = prefix;
        prefix = relativeTo;
        relativeTo = null;
      }
      options = assign$4({}, defaultOptions, options || {});
      var data = options.data || {},
        originalEvent,
        globalStart,
        localStart,
        endDrag,
        isTouch;
      if (options.trapClick) {
        endDrag = trapClickAndEnd;
      } else {
        endDrag = end;
      }
      if (event$1) {
        originalEvent = getOriginal$1(event$1) || event$1;
        globalStart = toPoint(event$1);
        stopPropagation$2(event$1);

        // prevent default browser dragging behavior
        if (originalEvent.type === 'dragstart') {
          preventDefault$1(originalEvent);
        }
      } else {
        originalEvent = null;
        globalStart = {
          x: 0,
          y: 0
        };
      }
      localStart = toLocalPoint(globalStart);
      if (!relativeTo) {
        relativeTo = localStart;
      }
      isTouch = isTouchEvent(originalEvent);
      context = assign$4({
        prefix: prefix,
        data: data,
        payload: {},
        globalStart: globalStart,
        displacement: delta(relativeTo, localStart),
        localStart: localStart,
        isTouch: isTouch
      }, options);

      // skip dom registration if trigger
      // is set to manual (during testing)
      if (!options.manual) {
        // add dom listeners

        if (isTouch) {
          event.bind(document, 'touchstart', trapTouch, true);
          event.bind(document, 'touchcancel', cancel, true);
          event.bind(document, 'touchmove', move, true);
          event.bind(document, 'touchend', end, true);
        } else {
          // assume we use the mouse to interact per default
          event.bind(document, 'mousemove', move);

          // prevent default browser drag and text selection behavior
          event.bind(document, 'dragstart', preventDefault$1);
          event.bind(document, 'selectstart', preventDefault$1);
          event.bind(document, 'mousedown', endDrag, true);
          event.bind(document, 'mouseup', endDrag, true);
        }
        event.bind(document, 'keyup', checkCancel);
        eventBus.on('element.hover', hover);
        eventBus.on('element.out', out);
      }
      fire('init');
      if (options.autoActivate) {
        move(event$1, true);
      }
    }

    // cancel on diagram destruction
    eventBus.on('diagram.destroy', cancel);

    // API

    this.init = init;
    this.move = move;
    this.hover = hover;
    this.out = out;
    this.end = end;
    this.cancel = cancel;

    // for introspection

    this.context = function () {
      return context;
    };
    this.setOptions = function (options) {
      assign$4(defaultOptions, options);
    };
  }
  Dragging.$inject = ['eventBus', 'canvas', 'selection', 'elementRegistry'];

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var DraggingModule = {
    __depends__: [HoverFixModule, SelectionModule],
    dragging: ['type', Dragging]
  };

  /**
   * @typedef {import('../../util/Types').Point} Point
   *
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('../../core/Canvas').default} Canvas
   */

  /**
   * Initiates canvas scrolling if current cursor point is close to a border.
   * Cancelled when current point moves back inside the scrolling borders
   * or cancelled manually.
   *
   * Default options :
   *   scrollThresholdIn: [ 20, 20, 20, 20 ],
   *   scrollThresholdOut: [ 0, 0, 0, 0 ],
   *   scrollRepeatTimeout: 15,
   *   scrollStep: 10
   *
   * Threshold order:
   *   [ left, top, right, bottom ]
   *
   * @param {Object} config
   * @param {EventBus} eventBus
   * @param {Canvas} canvas
   */
  function AutoScroll(config, eventBus, canvas) {
    this._canvas = canvas;
    this._opts = assign$4({
      scrollThresholdIn: [20, 20, 20, 20],
      scrollThresholdOut: [0, 0, 0, 0],
      scrollRepeatTimeout: 15,
      scrollStep: 10
    }, config);
    var self = this;
    eventBus.on('drag.move', function (e) {
      var point = self._toBorderPoint(e);
      self.startScroll(point);
    });
    eventBus.on(['drag.cleanup'], function () {
      self.stopScroll();
    });
  }
  AutoScroll.$inject = ['config.autoScroll', 'eventBus', 'canvas'];

  /**
   * Starts scrolling loop.
   * Point is given in global scale in canvas container box plane.
   *
   * @param {Point} point
   */
  AutoScroll.prototype.startScroll = function (point) {
    var canvas = this._canvas;
    var opts = this._opts;
    var self = this;
    var clientRect = canvas.getContainer().getBoundingClientRect();
    var diff = [point.x, point.y, clientRect.width - point.x, clientRect.height - point.y];
    this.stopScroll();
    var dx = 0,
      dy = 0;
    for (var i = 0; i < 4; i++) {
      if (between(diff[i], opts.scrollThresholdOut[i], opts.scrollThresholdIn[i])) {
        if (i === 0) {
          dx = opts.scrollStep;
        } else if (i == 1) {
          dy = opts.scrollStep;
        } else if (i == 2) {
          dx = -opts.scrollStep;
        } else if (i == 3) {
          dy = -opts.scrollStep;
        }
      }
    }
    if (dx !== 0 || dy !== 0) {
      canvas.scroll({
        dx: dx,
        dy: dy
      });
      this._scrolling = setTimeout(function () {
        self.startScroll(point);
      }, opts.scrollRepeatTimeout);
    }
  };
  function between(val, start, end) {
    if (start < val && val < end) {
      return true;
    }
    return false;
  }

  /**
   * Stops scrolling loop.
   */
  AutoScroll.prototype.stopScroll = function () {
    clearTimeout(this._scrolling);
  };

  /**
   * Overrides defaults options.
   *
   * @param {Object} options
   */
  AutoScroll.prototype.setOptions = function (options) {
    this._opts = assign$4({}, this._opts, options);
  };

  /**
   * Converts event to a point in canvas container plane in global scale.
   *
   * @param {Event} event
   * @return {Point}
   */
  AutoScroll.prototype._toBorderPoint = function (event) {
    var clientRect = this._canvas._container.getBoundingClientRect();
    var globalPosition = toPoint(event.originalEvent);
    return {
      x: globalPosition.x - clientRect.left,
      y: globalPosition.y - clientRect.top
    };
  };

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var AutoScrollModule = {
    __depends__: [DraggingModule],
    __init__: ['autoScroll'],
    autoScroll: ['type', AutoScroll]
  };

  /**
   * @typedef {import('didi').Injector} Injector
   */

  /**
   * A service that provides rules for certain diagram actions.
   *
   * The default implementation will hook into the {@link CommandStack}
   * to perform the actual rule evaluation. Make sure to provide the
   * `commandStack` service with this module if you plan to use it.
   *
   * Together with this implementation you may use the {@link import('./RuleProvider').default}
   * to implement your own rule checkers.
   *
   * This module is ment to be easily replaced, thus the tiny foot print.
   *
   * @param {Injector} injector
   */
  function Rules$6(injector) {
    this._commandStack = injector.get('commandStack', false);
  }
  Rules$6.$inject = ['injector'];

  /**
   * Returns whether or not a given modeling action can be executed
   * in the specified context.
   *
   * This implementation will respond with allow unless anyone
   * objects.
   *
   * @param {string} action The action to be allowed or disallowed.
   * @param {Object} [context] The context for allowing or disallowing the action.
   *
   * @return {boolean|null} Wether the action is allowed. Returns `null` if the action
   * is to be ignored.
   */
  Rules$6.prototype.allowed = function (action, context) {
    var allowed = true;
    var commandStack = this._commandStack;
    if (commandStack) {
      allowed = commandStack.canExecute(action, context);
    }

    // map undefined to true, i.e. no rules
    return allowed === undefined ? true : allowed;
  };

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var Rules$5 = {
    __init__: ['rules'],
    rules: ['type', Rules$6]
  };

  /**
   * @typedef {import('../util/Types').Point} Point
   *
   * @typedef { {
   *   bendpoint?: boolean;
   *   index: number;
   *   point: Point;
   * } } Intersection
   */

  var round$7 = Math.round,
    max$2 = Math.max;
  function circlePath(center, r) {
    var x = center.x,
      y = center.y;
    return [['M', x, y], ['m', 0, -r], ['a', r, r, 0, 1, 1, 0, 2 * r], ['a', r, r, 0, 1, 1, 0, -2 * r], ['z']];
  }
  function linePath(points) {
    var segments = [];
    points.forEach(function (p, idx) {
      segments.push([idx === 0 ? 'M' : 'L', p.x, p.y]);
    });
    return segments;
  }
  var INTERSECTION_THRESHOLD = 10;

  /**
   * @param {Point[]} waypoints
   * @param {Point} reference
   *
   * @return {Intersection|null}
   */
  function getBendpointIntersection(waypoints, reference) {
    var i, w;
    for (i = 0; w = waypoints[i]; i++) {
      if (pointDistance(w, reference) <= INTERSECTION_THRESHOLD) {
        return {
          point: waypoints[i],
          bendpoint: true,
          index: i
        };
      }
    }
    return null;
  }

  /**
   * @param {Point[]} waypoints
   * @param {Point} reference
   *
   * @return {Intersection|null}
   */
  function getPathIntersection(waypoints, reference) {
    var intersections = findPathIntersections(circlePath(reference, INTERSECTION_THRESHOLD), linePath(waypoints));
    var a = intersections[0],
      b = intersections[intersections.length - 1],
      idx;
    if (!a) {
      // no intersection
      return null;
    }
    if (a !== b) {
      if (a.segment2 !== b.segment2) {
        // we use the bendpoint in between both segments
        // as the intersection point

        idx = max$2(a.segment2, b.segment2) - 1;
        return {
          point: waypoints[idx],
          bendpoint: true,
          index: idx
        };
      }
      return {
        point: {
          x: round$7(a.x + b.x) / 2,
          y: round$7(a.y + b.y) / 2
        },
        index: a.segment2
      };
    }
    return {
      point: {
        x: round$7(a.x),
        y: round$7(a.y)
      },
      index: a.segment2
    };
  }

  /**
   * Returns the closest point on the connection towards a given reference point.
   *
   * @param {Point[]} waypoints
   * @param {Point} reference
   *
   * @return {Intersection|null}
   */
  function getApproxIntersection(waypoints, reference) {
    return getBendpointIntersection(waypoints, reference) || getPathIntersection(waypoints, reference);
  }

  /**
   * @typedef {import('../../util/Types').Point} Point
   * @typedef {import('../../util/Types').Vector} Vector
   */

  /**
   * Returns the length of a vector.
   *
   * @param {Vector} vector
   *
   * @return {number}
   */
  function vectorLength(vector) {
    return Math.sqrt(Math.pow(vector.x, 2) + Math.pow(vector.y, 2));
  }

  /**
   * Solves a 2D equation system
   * a + r*b = c, where a,b,c are 2D vectors
   *
   * @param {Vector} a
   * @param {Vector} b
   * @param {Vector} c
   *
   * @return {number}
   */
  function solveLambaSystem(a, b, c) {
    // the 2d system
    var system = [{
      n: a[0] - c[0],
      lambda: b[0]
    }, {
      n: a[1] - c[1],
      lambda: b[1]
    }];

    // solve
    var n = system[0].n * b[0] + system[1].n * b[1],
      l = system[0].lambda * b[0] + system[1].lambda * b[1];
    return -n / l;
  }

  /**
   * Calculates the position of the perpendicular foot.
   *
   * @param {Point} point
   * @param {Point[]} line
   *
   * @return {Point}
   */
  function perpendicularFoot(point, line) {
    var a = line[0],
      b = line[1];

    // relative position of b from a
    var bd = {
      x: b.x - a.x,
      y: b.y - a.y
    };

    // solve equation system to the parametrized vectors param real value
    var r = solveLambaSystem([a.x, a.y], [bd.x, bd.y], [point.x, point.y]);
    return {
      x: a.x + r * bd.x,
      y: a.y + r * bd.y
    };
  }

  /**
   * Calculates the distance between a point and a line.
   *
   * @param {Point} point
   * @param {Point[]} line
   *
   * @return {number}
   */
  function getDistancePointLine(point, line) {
    var pfPoint = perpendicularFoot(point, line);

    // distance vector
    var connectionVector = {
      x: pfPoint.x - point.x,
      y: pfPoint.y - point.y
    };
    return vectorLength(connectionVector);
  }

  /**
   * @typedef {import('../../core/Types').ConnectionLike} Connection
   *
   * @typedef {import('../../util/Types').Point} Point
   */

  var BENDPOINT_CLS = 'djs-bendpoint';
  var SEGMENT_DRAGGER_CLS = 'djs-segment-dragger';
  function toCanvasCoordinates(canvas, event) {
    var position = toPoint(event),
      clientRect = canvas._container.getBoundingClientRect(),
      offset;

    // canvas relative position

    offset = {
      x: clientRect.left,
      y: clientRect.top
    };

    // update actual event payload with canvas relative measures

    var viewbox = canvas.viewbox();
    return {
      x: viewbox.x + (position.x - offset.x) / viewbox.scale,
      y: viewbox.y + (position.y - offset.y) / viewbox.scale
    };
  }
  function getConnectionIntersection(canvas, waypoints, event) {
    var localPosition = toCanvasCoordinates(canvas, event),
      intersection = getApproxIntersection(waypoints, localPosition);
    return intersection;
  }
  function addBendpoint(parentGfx, cls) {
    var groupGfx = create$2('g');
    classes(groupGfx).add(BENDPOINT_CLS);
    append(parentGfx, groupGfx);
    var visual = create$2('circle');
    attr(visual, {
      cx: 0,
      cy: 0,
      r: 4
    });
    classes(visual).add('djs-visual');
    append(groupGfx, visual);
    var hit = create$2('circle');
    attr(hit, {
      cx: 0,
      cy: 0,
      r: 10
    });
    classes(hit).add('djs-hit');
    append(groupGfx, hit);
    if (cls) {
      classes(groupGfx).add(cls);
    }
    return groupGfx;
  }
  function createParallelDragger(parentGfx, segmentStart, segmentEnd, alignment) {
    var draggerGfx = create$2('g');
    append(parentGfx, draggerGfx);
    var width = 18,
      height = 6,
      padding = 11,
      hitWidth = calculateHitWidth(segmentStart, segmentEnd, alignment),
      hitHeight = height + padding;
    var visual = create$2('rect');
    attr(visual, {
      x: -width / 2,
      y: -height / 2,
      width: width,
      height: height
    });
    classes(visual).add('djs-visual');
    append(draggerGfx, visual);
    var hit = create$2('rect');
    attr(hit, {
      x: -hitWidth / 2,
      y: -hitHeight / 2,
      width: hitWidth,
      height: hitHeight
    });
    classes(hit).add('djs-hit');
    append(draggerGfx, hit);
    rotate(draggerGfx, alignment === 'v' ? 90 : 0);
    return draggerGfx;
  }
  function addSegmentDragger(parentGfx, segmentStart, segmentEnd) {
    var groupGfx = create$2('g'),
      mid = getMidPoint(segmentStart, segmentEnd),
      alignment = pointsAligned(segmentStart, segmentEnd);
    append(parentGfx, groupGfx);
    createParallelDragger(groupGfx, segmentStart, segmentEnd, alignment);
    classes(groupGfx).add(SEGMENT_DRAGGER_CLS);
    classes(groupGfx).add(alignment === 'h' ? 'horizontal' : 'vertical');
    translate$1(groupGfx, mid.x, mid.y);
    return groupGfx;
  }

  /**
   * Calculates region for segment move which is 2/3 of the full segment length
   * @param {number} segmentLength
   *
   * @return {number}
   */
  function calculateSegmentMoveRegion(segmentLength) {
    return Math.abs(Math.round(segmentLength * 2 / 3));
  }

  /**
   * Returns the point with the closest distance that is on the connection path.
   *
   * @param {Point} position
   * @param {Connection} connection
   * @return {Point}
   */
  function getClosestPointOnConnection(position, connection) {
    var segment = getClosestSegment(position, connection);
    return perpendicularFoot(position, segment);
  }

  // helper //////////

  function calculateHitWidth(segmentStart, segmentEnd, alignment) {
    var segmentLengthXAxis = segmentEnd.x - segmentStart.x,
      segmentLengthYAxis = segmentEnd.y - segmentStart.y;
    return alignment === 'h' ? calculateSegmentMoveRegion(segmentLengthXAxis) : calculateSegmentMoveRegion(segmentLengthYAxis);
  }
  function getClosestSegment(position, connection) {
    var waypoints = connection.waypoints;
    var minDistance = Infinity,
      segmentIndex;
    for (var i = 0; i < waypoints.length - 1; i++) {
      var start = waypoints[i],
        end = waypoints[i + 1],
        distance = getDistancePointLine(position, [start, end]);
      if (distance < minDistance) {
        minDistance = distance;
        segmentIndex = i;
      }
    }
    return [waypoints[segmentIndex], waypoints[segmentIndex + 1]];
  }

  /**
   * @typedef {import('../bendpoints/BendpointMove').default} BendpointMove
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../bendpoints/ConnectionSegmentMove').default} ConnectionSegmentMove
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('../interaction-events/InteractionEvents').default} InteractionEvents
   */

  /**
   * A service that adds editable bendpoints to connections.
   *
   * @param {EventBus} eventBus
   * @param {Canvas} canvas
   * @param {InteractionEvents} interactionEvents
   * @param {BendpointMove} bendpointMove
   * @param {ConnectionSegmentMove} connectionSegmentMove
   */
  function Bendpoints(eventBus, canvas, interactionEvents, bendpointMove, connectionSegmentMove) {
    /**
     * Returns true if intersection point is inside middle region of segment, adjusted by
     * optional threshold
     */
    function isIntersectionMiddle(intersection, waypoints, treshold) {
      var idx = intersection.index,
        p = intersection.point,
        p0,
        p1,
        mid,
        aligned,
        xDelta,
        yDelta;
      if (idx <= 0 || intersection.bendpoint) {
        return false;
      }
      p0 = waypoints[idx - 1];
      p1 = waypoints[idx];
      mid = getMidPoint(p0, p1), aligned = pointsAligned(p0, p1);
      xDelta = Math.abs(p.x - mid.x);
      yDelta = Math.abs(p.y - mid.y);
      return aligned && xDelta <= treshold && yDelta <= treshold;
    }

    /**
     * Calculates the threshold from a connection's middle which fits the two-third-region
     */
    function calculateIntersectionThreshold(connection, intersection) {
      var waypoints = connection.waypoints,
        relevantSegment,
        alignment,
        segmentLength,
        threshold;
      if (intersection.index <= 0 || intersection.bendpoint) {
        return null;
      }

      // segment relative to connection intersection
      relevantSegment = {
        start: waypoints[intersection.index - 1],
        end: waypoints[intersection.index]
      };
      alignment = pointsAligned(relevantSegment.start, relevantSegment.end);
      if (!alignment) {
        return null;
      }
      if (alignment === 'h') {
        segmentLength = relevantSegment.end.x - relevantSegment.start.x;
      } else {
        segmentLength = relevantSegment.end.y - relevantSegment.start.y;
      }

      // calculate threshold relative to 2/3 of segment length
      threshold = calculateSegmentMoveRegion(segmentLength) / 2;
      return threshold;
    }
    function activateBendpointMove(event, connection) {
      var waypoints = connection.waypoints,
        intersection = getConnectionIntersection(canvas, waypoints, event),
        threshold;
      if (!intersection) {
        return;
      }
      threshold = calculateIntersectionThreshold(connection, intersection);
      if (isIntersectionMiddle(intersection, waypoints, threshold)) {
        connectionSegmentMove.start(event, connection, intersection.index);
      } else {
        bendpointMove.start(event, connection, intersection.index, !intersection.bendpoint);
      }

      // we've handled the event
      return true;
    }
    function bindInteractionEvents(node, eventName, element) {
      event.bind(node, eventName, function (event) {
        interactionEvents.triggerMouseEvent(eventName, event, element);
        event.stopPropagation();
      });
    }
    function getBendpointsContainer(element, create) {
      var layer = canvas.getLayer('overlays'),
        gfx = query('.djs-bendpoints[data-element-id="' + escapeCSS(element.id) + '"]', layer);
      if (!gfx && create) {
        gfx = create$2('g');
        attr(gfx, {
          'data-element-id': element.id
        });
        classes(gfx).add('djs-bendpoints');
        append(layer, gfx);
        bindInteractionEvents(gfx, 'mousedown', element);
        bindInteractionEvents(gfx, 'click', element);
        bindInteractionEvents(gfx, 'dblclick', element);
      }
      return gfx;
    }
    function getSegmentDragger(idx, parentGfx) {
      return query('.djs-segment-dragger[data-segment-idx="' + idx + '"]', parentGfx);
    }
    function createBendpoints(gfx, connection) {
      connection.waypoints.forEach(function (p, idx) {
        var bendpoint = addBendpoint(gfx);
        append(gfx, bendpoint);
        translate$1(bendpoint, p.x, p.y);
      });

      // add floating bendpoint
      addBendpoint(gfx, 'floating');
    }
    function createSegmentDraggers(gfx, connection) {
      var waypoints = connection.waypoints;
      var segmentStart, segmentEnd, segmentDraggerGfx;
      for (var i = 1; i < waypoints.length; i++) {
        segmentStart = waypoints[i - 1];
        segmentEnd = waypoints[i];
        if (pointsAligned(segmentStart, segmentEnd)) {
          segmentDraggerGfx = addSegmentDragger(gfx, segmentStart, segmentEnd);
          attr(segmentDraggerGfx, {
            'data-segment-idx': i
          });
          bindInteractionEvents(segmentDraggerGfx, 'mousemove', connection);
        }
      }
    }
    function clearBendpoints(gfx) {
      forEach$3(all('.' + BENDPOINT_CLS, gfx), function (node) {
        remove$3(node);
      });
    }
    function clearSegmentDraggers(gfx) {
      forEach$3(all('.' + SEGMENT_DRAGGER_CLS, gfx), function (node) {
        remove$3(node);
      });
    }
    function addHandles(connection) {
      var gfx = getBendpointsContainer(connection);
      if (!gfx) {
        gfx = getBendpointsContainer(connection, true);
        createBendpoints(gfx, connection);
        createSegmentDraggers(gfx, connection);
      }
      return gfx;
    }
    function updateHandles(connection) {
      var gfx = getBendpointsContainer(connection);
      if (gfx) {
        clearSegmentDraggers(gfx);
        clearBendpoints(gfx);
        createSegmentDraggers(gfx, connection);
        createBendpoints(gfx, connection);
      }
    }
    function updateFloatingBendpointPosition(parentGfx, intersection) {
      var floating = query('.floating', parentGfx),
        point = intersection.point;
      if (!floating) {
        return;
      }
      translate$1(floating, point.x, point.y);
    }
    function updateSegmentDraggerPosition(parentGfx, intersection, waypoints) {
      var draggerGfx = getSegmentDragger(intersection.index, parentGfx),
        segmentStart = waypoints[intersection.index - 1],
        segmentEnd = waypoints[intersection.index],
        point = intersection.point,
        mid = getMidPoint(segmentStart, segmentEnd),
        alignment = pointsAligned(segmentStart, segmentEnd),
        draggerVisual,
        relativePosition;
      if (!draggerGfx) {
        return;
      }
      draggerVisual = getDraggerVisual(draggerGfx);
      relativePosition = {
        x: point.x - mid.x,
        y: point.y - mid.y
      };
      if (alignment === 'v') {
        // rotate position
        relativePosition = {
          x: relativePosition.y,
          y: relativePosition.x
        };
      }
      translate$1(draggerVisual, relativePosition.x, relativePosition.y);
    }
    eventBus.on('connection.changed', function (event) {
      updateHandles(event.element);
    });
    eventBus.on('connection.remove', function (event) {
      var gfx = getBendpointsContainer(event.element);
      if (gfx) {
        remove$3(gfx);
      }
    });
    eventBus.on('element.marker.update', function (event) {
      var element = event.element,
        bendpointsGfx;
      if (!element.waypoints) {
        return;
      }
      bendpointsGfx = addHandles(element);
      if (event.add) {
        classes(bendpointsGfx).add(event.marker);
      } else {
        classes(bendpointsGfx).remove(event.marker);
      }
    });
    eventBus.on('element.mousemove', function (event) {
      var element = event.element,
        waypoints = element.waypoints,
        bendpointsGfx,
        intersection;
      if (waypoints) {
        bendpointsGfx = getBendpointsContainer(element, true);
        intersection = getConnectionIntersection(canvas, waypoints, event.originalEvent);
        if (!intersection) {
          return;
        }
        updateFloatingBendpointPosition(bendpointsGfx, intersection);
        if (!intersection.bendpoint) {
          updateSegmentDraggerPosition(bendpointsGfx, intersection, waypoints);
        }
      }
    });
    eventBus.on('element.mousedown', function (event) {
      if (!isPrimaryButton(event)) {
        return;
      }
      var originalEvent = event.originalEvent,
        element = event.element;
      if (!element.waypoints) {
        return;
      }
      return activateBendpointMove(originalEvent, element);
    });
    eventBus.on('selection.changed', function (event) {
      var newSelection = event.newSelection,
        primary = newSelection[0];
      if (primary && primary.waypoints) {
        addHandles(primary);
      }
    });
    eventBus.on('element.hover', function (event) {
      var element = event.element;
      if (element.waypoints) {
        addHandles(element);
        interactionEvents.registerEvent(event.gfx, 'mousemove', 'element.mousemove');
      }
    });
    eventBus.on('element.out', function (event) {
      interactionEvents.unregisterEvent(event.gfx, 'mousemove', 'element.mousemove');
    });

    // update bendpoint container data attribute on element ID change
    eventBus.on('element.updateId', function (context) {
      var element = context.element,
        newId = context.newId;
      if (element.waypoints) {
        var bendpointContainer = getBendpointsContainer(element);
        if (bendpointContainer) {
          attr(bendpointContainer, {
            'data-element-id': newId
          });
        }
      }
    });

    // API

    this.addHandles = addHandles;
    this.updateHandles = updateHandles;
    this.getBendpointsContainer = getBendpointsContainer;
    this.getSegmentDragger = getSegmentDragger;
  }
  Bendpoints.$inject = ['eventBus', 'canvas', 'interactionEvents', 'bendpointMove', 'connectionSegmentMove'];

  // helper /////////////

  function getDraggerVisual(draggerGfx) {
    return query('.djs-visual', draggerGfx);
  }

  /**
   * @typedef {import('didi').Injector} Injector
   *
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../dragging/Dragging').default} Dragging
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('../modeling/Modeling').default} Modeling
   * @typedef {import('../rules/Rules').default} Rules
   */

  var round$6 = Math.round;
  var RECONNECT_START$2 = 'reconnectStart',
    RECONNECT_END$2 = 'reconnectEnd',
    UPDATE_WAYPOINTS$1 = 'updateWaypoints';

  /**
   * Move bendpoints through drag and drop to add/remove bendpoints or reconnect connection.
   *
   * @param {Injector} injector
   * @param {EventBus} eventBus
   * @param {Canvas} canvas
   * @param {Dragging} dragging
   * @param {Rules} rules
   * @param {Modeling} modeling
   */
  function BendpointMove(injector, eventBus, canvas, dragging, rules, modeling) {
    this._injector = injector;
    this.start = function (event, connection, bendpointIndex, insert) {
      var gfx = canvas.getGraphics(connection),
        source = connection.source,
        target = connection.target,
        waypoints = connection.waypoints,
        type;
      if (!insert && bendpointIndex === 0) {
        type = RECONNECT_START$2;
      } else if (!insert && bendpointIndex === waypoints.length - 1) {
        type = RECONNECT_END$2;
      } else {
        type = UPDATE_WAYPOINTS$1;
      }
      var command = type === UPDATE_WAYPOINTS$1 ? 'connection.updateWaypoints' : 'connection.reconnect';
      var allowed = rules.allowed(command, {
        connection: connection,
        source: source,
        target: target
      });
      if (allowed === false) {
        allowed = rules.allowed(command, {
          connection: connection,
          source: target,
          target: source
        });
      }
      if (allowed === false) {
        return;
      }
      dragging.init(event, 'bendpoint.move', {
        data: {
          connection: connection,
          connectionGfx: gfx,
          context: {
            allowed: allowed,
            bendpointIndex: bendpointIndex,
            connection: connection,
            source: source,
            target: target,
            insert: insert,
            type: type
          }
        }
      });
    };
    eventBus.on('bendpoint.move.hover', function (event) {
      var context = event.context,
        connection = context.connection,
        source = connection.source,
        target = connection.target,
        hover = event.hover,
        type = context.type;

      // cache hover state
      context.hover = hover;
      var allowed;
      if (!hover) {
        return;
      }
      var command = type === UPDATE_WAYPOINTS$1 ? 'connection.updateWaypoints' : 'connection.reconnect';
      allowed = context.allowed = rules.allowed(command, {
        connection: connection,
        source: type === RECONNECT_START$2 ? hover : source,
        target: type === RECONNECT_END$2 ? hover : target
      });
      if (allowed) {
        context.source = type === RECONNECT_START$2 ? hover : source;
        context.target = type === RECONNECT_END$2 ? hover : target;
        return;
      }
      if (allowed === false) {
        allowed = context.allowed = rules.allowed(command, {
          connection: connection,
          source: type === RECONNECT_END$2 ? hover : target,
          target: type === RECONNECT_START$2 ? hover : source
        });
      }
      if (allowed) {
        context.source = type === RECONNECT_END$2 ? hover : target;
        context.target = type === RECONNECT_START$2 ? hover : source;
      }
    });
    eventBus.on(['bendpoint.move.out', 'bendpoint.move.cleanup'], function (event) {
      var context = event.context,
        type = context.type;
      context.hover = null;
      context.source = null;
      context.target = null;
      if (type !== UPDATE_WAYPOINTS$1) {
        context.allowed = false;
      }
    });
    eventBus.on('bendpoint.move.end', function (event) {
      var context = event.context,
        allowed = context.allowed,
        bendpointIndex = context.bendpointIndex,
        connection = context.connection,
        insert = context.insert,
        newWaypoints = connection.waypoints.slice(),
        source = context.source,
        target = context.target,
        type = context.type,
        hints = context.hints || {};

      // ensure integer values (important if zoom level was > 1 during move)
      var docking = {
        x: round$6(event.x),
        y: round$6(event.y)
      };
      if (!allowed) {
        return false;
      }
      if (type === UPDATE_WAYPOINTS$1) {
        if (insert) {
          // insert new bendpoint
          newWaypoints.splice(bendpointIndex, 0, docking);
        } else {
          // swap previous waypoint with moved one
          newWaypoints[bendpointIndex] = docking;
        }

        // pass hints about actual moved bendpoint
        // useful for connection/label layout
        hints.bendpointMove = {
          insert: insert,
          bendpointIndex: bendpointIndex
        };
        newWaypoints = this.cropWaypoints(connection, newWaypoints);
        modeling.updateWaypoints(connection, filterRedundantWaypoints(newWaypoints), hints);
      } else {
        if (type === RECONNECT_START$2) {
          hints.docking = 'source';
          if (isReverse$1(context)) {
            hints.docking = 'target';
            hints.newWaypoints = newWaypoints.reverse();
          }
        } else if (type === RECONNECT_END$2) {
          hints.docking = 'target';
          if (isReverse$1(context)) {
            hints.docking = 'source';
            hints.newWaypoints = newWaypoints.reverse();
          }
        }
        modeling.reconnect(connection, source, target, docking, hints);
      }
    }, this);
  }
  BendpointMove.$inject = ['injector', 'eventBus', 'canvas', 'dragging', 'rules', 'modeling'];
  BendpointMove.prototype.cropWaypoints = function (connection, newWaypoints) {
    var connectionDocking = this._injector.get('connectionDocking', false);
    if (!connectionDocking) {
      return newWaypoints;
    }
    var waypoints = connection.waypoints;
    connection.waypoints = newWaypoints;
    connection.waypoints = connectionDocking.getCroppedWaypoints(connection);
    newWaypoints = connection.waypoints;
    connection.waypoints = waypoints;
    return newWaypoints;
  };

  // helpers //////////

  function isReverse$1(context) {
    var hover = context.hover,
      source = context.source,
      target = context.target,
      type = context.type;
    if (type === RECONNECT_START$2) {
      return hover && target && hover === target && source !== target;
    }
    if (type === RECONNECT_END$2) {
      return hover && source && hover === source && source !== target;
    }
  }

  /**
   * @typedef {import('didi').Injector} Injector
   *
   * @typedef {import('../bendpoints/BendpointMove').default} BendpointMove
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/EventBus').default} EventBus
   */

  var RECONNECT_START$1 = 'reconnectStart',
    RECONNECT_END$1 = 'reconnectEnd',
    UPDATE_WAYPOINTS = 'updateWaypoints';
  var MARKER_OK$3 = 'connect-ok',
    MARKER_NOT_OK$3 = 'connect-not-ok',
    MARKER_CONNECT_HOVER$1 = 'connect-hover',
    MARKER_CONNECT_UPDATING$1 = 'djs-updating',
    MARKER_DRAGGER = 'djs-dragging';
  var HIGH_PRIORITY$a = 1100;

  /**
   * Preview connection while moving bendpoints.
   *
   * @param {BendpointMove} bendpointMove
   * @param {Injector} injector
   * @param {EventBus} eventBus
   * @param {Canvas} canvas
   */
  function BendpointMovePreview(bendpointMove, injector, eventBus, canvas) {
    this._injector = injector;
    var connectionPreview = injector.get('connectionPreview', false);
    eventBus.on('bendpoint.move.start', function (event) {
      var context = event.context,
        bendpointIndex = context.bendpointIndex,
        connection = context.connection,
        insert = context.insert,
        waypoints = connection.waypoints,
        newWaypoints = waypoints.slice();
      context.waypoints = waypoints;
      if (insert) {
        // insert placeholder for new bendpoint
        newWaypoints.splice(bendpointIndex, 0, {
          x: event.x,
          y: event.y
        });
      }
      connection.waypoints = newWaypoints;

      // add dragger gfx
      var draggerGfx = context.draggerGfx = addBendpoint(canvas.getLayer('overlays'));
      classes(draggerGfx).add('djs-dragging');
      canvas.addMarker(connection, MARKER_DRAGGER);
      canvas.addMarker(connection, MARKER_CONNECT_UPDATING$1);
    });
    eventBus.on('bendpoint.move.hover', function (event) {
      var context = event.context,
        allowed = context.allowed,
        hover = context.hover,
        type = context.type;
      if (hover) {
        canvas.addMarker(hover, MARKER_CONNECT_HOVER$1);
        if (type === UPDATE_WAYPOINTS) {
          return;
        }
        if (allowed) {
          canvas.removeMarker(hover, MARKER_NOT_OK$3);
          canvas.addMarker(hover, MARKER_OK$3);
        } else if (allowed === false) {
          canvas.removeMarker(hover, MARKER_OK$3);
          canvas.addMarker(hover, MARKER_NOT_OK$3);
        }
      }
    });
    eventBus.on(['bendpoint.move.out', 'bendpoint.move.cleanup'], HIGH_PRIORITY$a, function (event) {
      var context = event.context,
        hover = context.hover,
        target = context.target;
      if (hover) {
        canvas.removeMarker(hover, MARKER_CONNECT_HOVER$1);
        canvas.removeMarker(hover, target ? MARKER_OK$3 : MARKER_NOT_OK$3);
      }
    });
    eventBus.on('bendpoint.move.move', function (event) {
      var context = event.context,
        allowed = context.allowed,
        bendpointIndex = context.bendpointIndex,
        draggerGfx = context.draggerGfx,
        hover = context.hover,
        type = context.type,
        connection = context.connection,
        source = connection.source,
        target = connection.target,
        newWaypoints = connection.waypoints.slice(),
        bendpoint = {
          x: event.x,
          y: event.y
        },
        hints = context.hints || {},
        drawPreviewHints = {};
      if (connectionPreview) {
        if (hints.connectionStart) {
          drawPreviewHints.connectionStart = hints.connectionStart;
        }
        if (hints.connectionEnd) {
          drawPreviewHints.connectionEnd = hints.connectionEnd;
        }
        if (type === RECONNECT_START$1) {
          if (isReverse$1(context)) {
            drawPreviewHints.connectionEnd = drawPreviewHints.connectionEnd || bendpoint;
            drawPreviewHints.source = target;
            drawPreviewHints.target = hover || source;
            newWaypoints = newWaypoints.reverse();
          } else {
            drawPreviewHints.connectionStart = drawPreviewHints.connectionStart || bendpoint;
            drawPreviewHints.source = hover || source;
            drawPreviewHints.target = target;
          }
        } else if (type === RECONNECT_END$1) {
          if (isReverse$1(context)) {
            drawPreviewHints.connectionStart = drawPreviewHints.connectionStart || bendpoint;
            drawPreviewHints.source = hover || target;
            drawPreviewHints.target = source;
            newWaypoints = newWaypoints.reverse();
          } else {
            drawPreviewHints.connectionEnd = drawPreviewHints.connectionEnd || bendpoint;
            drawPreviewHints.source = source;
            drawPreviewHints.target = hover || target;
          }
        } else {
          drawPreviewHints.noCropping = true;
          drawPreviewHints.noLayout = true;
          newWaypoints[bendpointIndex] = bendpoint;
        }
        if (type === UPDATE_WAYPOINTS) {
          newWaypoints = bendpointMove.cropWaypoints(connection, newWaypoints);
        }
        drawPreviewHints.waypoints = newWaypoints;
        connectionPreview.drawPreview(context, allowed, drawPreviewHints);
      }
      translate$1(draggerGfx, event.x, event.y);
    }, this);
    eventBus.on(['bendpoint.move.end', 'bendpoint.move.cancel'], HIGH_PRIORITY$a, function (event) {
      var context = event.context,
        connection = context.connection,
        draggerGfx = context.draggerGfx,
        hover = context.hover,
        target = context.target,
        waypoints = context.waypoints;
      connection.waypoints = waypoints;

      // remove dragger gfx
      remove$3(draggerGfx);
      canvas.removeMarker(connection, MARKER_CONNECT_UPDATING$1);
      canvas.removeMarker(connection, MARKER_DRAGGER);
      if (hover) {
        canvas.removeMarker(hover, MARKER_OK$3);
        canvas.removeMarker(hover, target ? MARKER_OK$3 : MARKER_NOT_OK$3);
      }
      if (connectionPreview) {
        connectionPreview.cleanUp(context);
      }
    });
  }
  BendpointMovePreview.$inject = ['bendpointMove', 'injector', 'eventBus', 'canvas'];

  var MARKER_CONNECT_HOVER = 'connect-hover',
    MARKER_CONNECT_UPDATING = 'djs-updating';

  /**
   * @typedef {import('../../model/Types').Shape} Shape
   *
   * @typedef {import('../../util/Types').Axis} Axis
   * @typedef {import('../../util/Types').Point} Point
   *
   * @typedef {import('didi').Injector} Injector
   *
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../dragging/Dragging').default} Dragging
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('../../core/GraphicsFactory').default} GraphicsFactory
   * @typedef {import('../modeling/Modeling').default} Modeling
   */

  function axisAdd(point, axis, delta) {
    return axisSet(point, axis, point[axis] + delta);
  }
  function axisSet(point, axis, value) {
    return {
      x: axis === 'x' ? value : point.x,
      y: axis === 'y' ? value : point.y
    };
  }
  function axisFenced(position, segmentStart, segmentEnd, axis) {
    var maxValue = Math.max(segmentStart[axis], segmentEnd[axis]),
      minValue = Math.min(segmentStart[axis], segmentEnd[axis]);
    var padding = 20;
    var fencedValue = Math.min(Math.max(minValue + padding, position[axis]), maxValue - padding);
    return axisSet(segmentStart, axis, fencedValue);
  }
  function flipAxis(axis) {
    return axis === 'x' ? 'y' : 'x';
  }

  /**
   * Get the docking point on the given element.
   *
   * Compute a reasonable docking, if non exists.
   *
   * @param {Point} point
   * @param {Shape} referenceElement
   * @param {Axis} moveAxis
   *
   * @return {Point}
   */
  function getDocking$1(point, referenceElement, moveAxis) {
    var referenceMid, inverseAxis;
    if (point.original) {
      return point.original;
    } else {
      referenceMid = getMid(referenceElement);
      inverseAxis = flipAxis(moveAxis);
      return axisSet(point, inverseAxis, referenceMid[inverseAxis]);
    }
  }

  /**
   * A component that implements moving of bendpoints.
   *
   * @param {Injector} injector
   * @param {EventBus} eventBus
   * @param {Canvas} canvas
   * @param {Canvas} dragging
   * @param {GraphicsFactory} graphicsFactory
   * @param {Modeling} modeling
   */
  function ConnectionSegmentMove(injector, eventBus, canvas, dragging, graphicsFactory, modeling) {
    // optional connection docking integration
    var connectionDocking = injector.get('connectionDocking', false);

    // API

    this.start = function (event, connection, idx) {
      var context,
        gfx = canvas.getGraphics(connection),
        segmentStartIndex = idx - 1,
        segmentEndIndex = idx,
        waypoints = connection.waypoints,
        segmentStart = waypoints[segmentStartIndex],
        segmentEnd = waypoints[segmentEndIndex],
        intersection = getConnectionIntersection(canvas, waypoints, event),
        direction,
        axis,
        dragPosition;
      direction = pointsAligned(segmentStart, segmentEnd);

      // do not move diagonal connection
      if (!direction) {
        return;
      }

      // the axis where we are going to move things
      axis = direction === 'v' ? 'x' : 'y';
      if (segmentStartIndex === 0) {
        segmentStart = getDocking$1(segmentStart, connection.source, axis);
      }
      if (segmentEndIndex === waypoints.length - 1) {
        segmentEnd = getDocking$1(segmentEnd, connection.target, axis);
      }
      if (intersection) {
        dragPosition = intersection.point;
      } else {
        // set to segment center as default
        dragPosition = {
          x: (segmentStart.x + segmentEnd.x) / 2,
          y: (segmentStart.y + segmentEnd.y) / 2
        };
      }
      context = {
        connection: connection,
        segmentStartIndex: segmentStartIndex,
        segmentEndIndex: segmentEndIndex,
        segmentStart: segmentStart,
        segmentEnd: segmentEnd,
        axis: axis,
        dragPosition: dragPosition
      };
      dragging.init(event, dragPosition, 'connectionSegment.move', {
        cursor: axis === 'x' ? 'resize-ew' : 'resize-ns',
        data: {
          connection: connection,
          connectionGfx: gfx,
          context: context
        }
      });
    };

    /**
     * Crop connection if connection cropping is provided.
     *
     * @param {Connection} connection
     * @param {Point[]} newWaypoints
     *
     * @return {Point[]} cropped connection waypoints
     */
    function cropConnection(connection, newWaypoints) {
      // crop connection, if docking service is provided only
      if (!connectionDocking) {
        return newWaypoints;
      }
      var oldWaypoints = connection.waypoints,
        croppedWaypoints;

      // temporary set new waypoints
      connection.waypoints = newWaypoints;
      croppedWaypoints = connectionDocking.getCroppedWaypoints(connection);

      // restore old waypoints
      connection.waypoints = oldWaypoints;
      return croppedWaypoints;
    }

    // DRAGGING IMPLEMENTATION

    function redrawConnection(data) {
      graphicsFactory.update('connection', data.connection, data.connectionGfx);
    }
    function updateDragger(context, segmentOffset, event) {
      var newWaypoints = context.newWaypoints,
        segmentStartIndex = context.segmentStartIndex + segmentOffset,
        segmentStart = newWaypoints[segmentStartIndex],
        segmentEndIndex = context.segmentEndIndex + segmentOffset,
        segmentEnd = newWaypoints[segmentEndIndex],
        axis = flipAxis(context.axis);

      // make sure the dragger does not move
      // outside the connection
      var draggerPosition = axisFenced(event, segmentStart, segmentEnd, axis);

      // update dragger
      translate$1(context.draggerGfx, draggerPosition.x, draggerPosition.y);
    }

    /**
     * Filter waypoints for redundant ones (i.e. on the same axis).
     * Returns the filtered waypoints and the offset related to the segment move.
     *
     * @param {Point[]} waypoints
     * @param {Integer} segmentStartIndex of moved segment start
     *
     * @return {Object} { filteredWaypoints, segmentOffset }
     */
    function filterRedundantWaypoints(waypoints, segmentStartIndex) {
      var segmentOffset = 0;
      var filteredWaypoints = waypoints.filter(function (r, idx) {
        if (pointsOnLine(waypoints[idx - 1], waypoints[idx + 1], r)) {
          // remove point and increment offset
          segmentOffset = idx <= segmentStartIndex ? segmentOffset - 1 : segmentOffset;
          return false;
        }

        // dont remove point
        return true;
      });
      return {
        waypoints: filteredWaypoints,
        segmentOffset: segmentOffset
      };
    }
    eventBus.on('connectionSegment.move.start', function (event) {
      var context = event.context,
        connection = event.connection,
        layer = canvas.getLayer('overlays');
      context.originalWaypoints = connection.waypoints.slice();

      // add dragger gfx
      context.draggerGfx = addSegmentDragger(layer, context.segmentStart, context.segmentEnd);
      classes(context.draggerGfx).add('djs-dragging');
      canvas.addMarker(connection, MARKER_CONNECT_UPDATING);
    });
    eventBus.on('connectionSegment.move.move', function (event) {
      var context = event.context,
        connection = context.connection,
        segmentStartIndex = context.segmentStartIndex,
        segmentEndIndex = context.segmentEndIndex,
        segmentStart = context.segmentStart,
        segmentEnd = context.segmentEnd,
        axis = context.axis;
      var newWaypoints = context.originalWaypoints.slice(),
        newSegmentStart = axisAdd(segmentStart, axis, event['d' + axis]),
        newSegmentEnd = axisAdd(segmentEnd, axis, event['d' + axis]);

      // original waypoint count and added / removed
      // from start waypoint delta. We use the later
      // to retrieve the updated segmentStartIndex / segmentEndIndex
      var waypointCount = newWaypoints.length,
        segmentOffset = 0;

      // move segment start / end by axis delta
      newWaypoints[segmentStartIndex] = newSegmentStart;
      newWaypoints[segmentEndIndex] = newSegmentEnd;
      var sourceToSegmentOrientation, targetToSegmentOrientation;

      // handle first segment
      if (segmentStartIndex < 2) {
        sourceToSegmentOrientation = getOrientation(connection.source, newSegmentStart);

        // first bendpoint, remove first segment if intersecting
        if (segmentStartIndex === 1) {
          if (sourceToSegmentOrientation === 'intersect') {
            newWaypoints.shift();
            newWaypoints[0] = newSegmentStart;
            segmentOffset--;
          }
        }

        // docking point, add segment if not intersecting anymore
        else {
          if (sourceToSegmentOrientation !== 'intersect') {
            newWaypoints.unshift(segmentStart);
            segmentOffset++;
          }
        }
      }

      // handle last segment
      if (segmentEndIndex > waypointCount - 3) {
        targetToSegmentOrientation = getOrientation(connection.target, newSegmentEnd);

        // last bendpoint, remove last segment if intersecting
        if (segmentEndIndex === waypointCount - 2) {
          if (targetToSegmentOrientation === 'intersect') {
            newWaypoints.pop();
            newWaypoints[newWaypoints.length - 1] = newSegmentEnd;
          }
        }

        // last bendpoint, remove last segment if intersecting
        else {
          if (targetToSegmentOrientation !== 'intersect') {
            newWaypoints.push(segmentEnd);
          }
        }
      }

      // update connection waypoints
      context.newWaypoints = connection.waypoints = cropConnection(connection, newWaypoints);

      // update dragger position
      updateDragger(context, segmentOffset, event);

      // save segmentOffset in context
      context.newSegmentStartIndex = segmentStartIndex + segmentOffset;

      // redraw connection
      redrawConnection(event);
    });
    eventBus.on('connectionSegment.move.hover', function (event) {
      event.context.hover = event.hover;
      canvas.addMarker(event.hover, MARKER_CONNECT_HOVER);
    });
    eventBus.on(['connectionSegment.move.out', 'connectionSegment.move.cleanup'], function (event) {
      // remove connect marker
      // if it was added
      var hover = event.context.hover;
      if (hover) {
        canvas.removeMarker(hover, MARKER_CONNECT_HOVER);
      }
    });
    eventBus.on('connectionSegment.move.cleanup', function (event) {
      var context = event.context,
        connection = context.connection;

      // remove dragger gfx
      if (context.draggerGfx) {
        remove$3(context.draggerGfx);
      }
      canvas.removeMarker(connection, MARKER_CONNECT_UPDATING);
    });
    eventBus.on(['connectionSegment.move.cancel', 'connectionSegment.move.end'], function (event) {
      var context = event.context,
        connection = context.connection;
      connection.waypoints = context.originalWaypoints;
      redrawConnection(event);
    });
    eventBus.on('connectionSegment.move.end', function (event) {
      var context = event.context,
        connection = context.connection,
        newWaypoints = context.newWaypoints,
        newSegmentStartIndex = context.newSegmentStartIndex;

      // ensure we have actual pixel values bendpoint
      // coordinates (important when zoom level was > 1 during move)
      newWaypoints = newWaypoints.map(function (p) {
        return {
          original: p.original,
          x: Math.round(p.x),
          y: Math.round(p.y)
        };
      });

      // apply filter redunant waypoints
      var filtered = filterRedundantWaypoints(newWaypoints, newSegmentStartIndex);

      // get filtered waypoints
      var filteredWaypoints = filtered.waypoints,
        croppedWaypoints = cropConnection(connection, filteredWaypoints),
        segmentOffset = filtered.segmentOffset;
      var hints = {
        segmentMove: {
          segmentStartIndex: context.segmentStartIndex,
          newSegmentStartIndex: newSegmentStartIndex + segmentOffset
        }
      };
      modeling.updateWaypoints(connection, croppedWaypoints, hints);
    });
  }
  ConnectionSegmentMove.$inject = ['injector', 'eventBus', 'canvas', 'dragging', 'graphicsFactory', 'modeling'];

  /**
   * @typedef {import('../../core/Types').ConnectionLike} Connection
   * @typedef {import('../../core/Types').ShapeLike} Shape
   *
   * @typedef {import('../../core/EventBus').Event} Event
   *
   * @typedef {import('../../util/Types').Axis} Axis
   */

  var abs$1 = Math.abs,
    round$5 = Math.round;

  /**
   * Snap value to a collection of reference values.
   *
   * @param {number} value
   * @param {Array<number>} values
   * @param {number} [tolerance=10]
   *
   * @return {number} the value we snapped to or null, if none snapped
   */
  function snapTo(value, values, tolerance) {
    tolerance = tolerance === undefined ? 10 : tolerance;
    var idx, snapValue;
    for (idx = 0; idx < values.length; idx++) {
      snapValue = values[idx];
      if (abs$1(snapValue - value) <= tolerance) {
        return snapValue;
      }
    }
  }
  function topLeft(bounds) {
    return {
      x: bounds.x,
      y: bounds.y
    };
  }
  function bottomRight(bounds) {
    return {
      x: bounds.x + bounds.width,
      y: bounds.y + bounds.height
    };
  }
  function mid$1(bounds, defaultValue) {
    if (!bounds || isNaN(bounds.x) || isNaN(bounds.y)) {
      return defaultValue;
    }
    return {
      x: round$5(bounds.x + bounds.width / 2),
      y: round$5(bounds.y + bounds.height / 2)
    };
  }

  /**
   * Retrieve the snap state of the given event.
   *
   * @param {Event} event
   * @param {Axis} axis
   *
   * @return {boolean} the snapped state
   *
   */
  function isSnapped(event, axis) {
    var snapped = event.snapped;
    if (!snapped) {
      return false;
    }
    if (typeof axis === 'string') {
      return snapped[axis];
    }
    return snapped.x && snapped.y;
  }

  /**
   * Set the given event as snapped.
   *
   * This method may change the x and/or y position of the shape
   * from the given event!
   *
   * @param {Event} event
   * @param {Axis} axis
   * @param {number|boolean} value
   *
   * @return {number} old value
   */
  function setSnapped(event, axis, value) {
    if (typeof axis !== 'string') {
      throw new Error('axis must be in [x, y]');
    }
    if (typeof value !== 'number' && value !== false) {
      throw new Error('value must be Number or false');
    }
    var delta,
      previousValue = event[axis];
    var snapped = event.snapped = event.snapped || {};
    if (value === false) {
      snapped[axis] = false;
    } else {
      snapped[axis] = true;
      delta = value - previousValue;
      event[axis] += delta;
      event['d' + axis] += delta;
    }
    return previousValue;
  }

  /**
   * Get children of a shape.
   *
   * @param {Shape} parent
   *
   * @return {Array<Shape|Connection>}
   */
  function getChildren$1(parent) {
    return parent.children || [];
  }

  /**
   * @typedef {import('../../core/EventBus').default} EventBus
   */
  var abs = Math.abs,
    round$4 = Math.round;
  var TOLERANCE = 10;

  /**
   * @param {EventBus} eventBus
   */
  function BendpointSnapping(eventBus) {
    function snapTo(values, value) {
      if (isArray$6(values)) {
        var i = values.length;
        while (i--) if (abs(values[i] - value) <= TOLERANCE) {
          return values[i];
        }
      } else {
        values = +values;
        var rem = value % values;
        if (rem < TOLERANCE) {
          return value - rem;
        }
        if (rem > values - TOLERANCE) {
          return value - rem + values;
        }
      }
      return value;
    }
    function getSnapPoint(element, event) {
      if (element.waypoints) {
        return getClosestPointOnConnection(event, element);
      }
      if (element.width) {
        return {
          x: round$4(element.width / 2 + element.x),
          y: round$4(element.height / 2 + element.y)
        };
      }
    }

    // connection segment snapping //////////////////////

    function getConnectionSegmentSnaps(event) {
      var context = event.context,
        snapPoints = context.snapPoints,
        connection = context.connection,
        waypoints = connection.waypoints,
        segmentStart = context.segmentStart,
        segmentStartIndex = context.segmentStartIndex,
        segmentEnd = context.segmentEnd,
        segmentEndIndex = context.segmentEndIndex,
        axis = context.axis;
      if (snapPoints) {
        return snapPoints;
      }
      var referenceWaypoints = [waypoints[segmentStartIndex - 1], segmentStart, segmentEnd, waypoints[segmentEndIndex + 1]];
      if (segmentStartIndex < 2) {
        referenceWaypoints.unshift(getSnapPoint(connection.source, event));
      }
      if (segmentEndIndex > waypoints.length - 3) {
        referenceWaypoints.unshift(getSnapPoint(connection.target, event));
      }
      context.snapPoints = snapPoints = {
        horizontal: [],
        vertical: []
      };
      forEach$3(referenceWaypoints, function (p) {
        // we snap on existing bendpoints only,
        // not placeholders that are inserted during add
        if (p) {
          p = p.original || p;
          if (axis === 'y') {
            snapPoints.horizontal.push(p.y);
          }
          if (axis === 'x') {
            snapPoints.vertical.push(p.x);
          }
        }
      });
      return snapPoints;
    }
    eventBus.on('connectionSegment.move.move', 1500, function (event) {
      var snapPoints = getConnectionSegmentSnaps(event),
        x = event.x,
        y = event.y,
        sx,
        sy;
      if (!snapPoints) {
        return;
      }

      // snap
      sx = snapTo(snapPoints.vertical, x);
      sy = snapTo(snapPoints.horizontal, y);

      // correction x/y
      var cx = x - sx,
        cy = y - sy;

      // update delta
      assign$4(event, {
        dx: event.dx - cx,
        dy: event.dy - cy,
        x: sx,
        y: sy
      });

      // only set snapped if actually snapped
      if (cx || snapPoints.vertical.indexOf(x) !== -1) {
        setSnapped(event, 'x', sx);
      }
      if (cy || snapPoints.horizontal.indexOf(y) !== -1) {
        setSnapped(event, 'y', sy);
      }
    });

    // bendpoint snapping //////////////////////

    function getBendpointSnaps(context) {
      var snapPoints = context.snapPoints,
        waypoints = context.connection.waypoints,
        bendpointIndex = context.bendpointIndex;
      if (snapPoints) {
        return snapPoints;
      }
      var referenceWaypoints = [waypoints[bendpointIndex - 1], waypoints[bendpointIndex + 1]];
      context.snapPoints = snapPoints = {
        horizontal: [],
        vertical: []
      };
      forEach$3(referenceWaypoints, function (p) {
        // we snap on existing bendpoints only,
        // not placeholders that are inserted during add
        if (p) {
          p = p.original || p;
          snapPoints.horizontal.push(p.y);
          snapPoints.vertical.push(p.x);
        }
      });
      return snapPoints;
    }

    // Snap Endpoint of new connection
    eventBus.on(['connect.hover', 'connect.move', 'connect.end'], 1500, function (event) {
      var context = event.context,
        hover = context.hover,
        hoverMid = hover && getSnapPoint(hover, event);

      // only snap on connections, elements can have multiple connect endpoints
      if (!isConnection(hover) || !hoverMid || !hoverMid.x || !hoverMid.y) {
        return;
      }
      setSnapped(event, 'x', hoverMid.x);
      setSnapped(event, 'y', hoverMid.y);
    });
    eventBus.on(['bendpoint.move.move', 'bendpoint.move.end'], 1500, function (event) {
      var context = event.context,
        snapPoints = getBendpointSnaps(context),
        hover = context.hover,
        hoverMid = hover && getSnapPoint(hover, event),
        x = event.x,
        y = event.y,
        sx,
        sy;
      if (!snapPoints) {
        return;
      }

      // snap to hover mid
      sx = snapTo(hoverMid ? snapPoints.vertical.concat([hoverMid.x]) : snapPoints.vertical, x);
      sy = snapTo(hoverMid ? snapPoints.horizontal.concat([hoverMid.y]) : snapPoints.horizontal, y);

      // correction x/y
      var cx = x - sx,
        cy = y - sy;

      // update delta
      assign$4(event, {
        dx: event.dx - cx,
        dy: event.dy - cy,
        x: event.x - cx,
        y: event.y - cy
      });

      // only set snapped if actually snapped
      if (cx || snapPoints.vertical.indexOf(x) !== -1) {
        setSnapped(event, 'x', sx);
      }
      if (cy || snapPoints.horizontal.indexOf(y) !== -1) {
        setSnapped(event, 'y', sy);
      }
    });
  }
  BendpointSnapping.$inject = ['eventBus'];

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var BendpointsModule = {
    __depends__: [DraggingModule, Rules$5],
    __init__: ['bendpoints', 'bendpointSnapping', 'bendpointMovePreview'],
    bendpoints: ['type', Bendpoints],
    bendpointMove: ['type', BendpointMove],
    bendpointMovePreview: ['type', BendpointMovePreview],
    connectionSegmentMove: ['type', ConnectionSegmentMove],
    bendpointSnapping: ['type', BendpointSnapping]
  };

  const Ids = new IdGenerator();

  /**
   * @typedef { {
   *   promise: Promise<unknown>,
   *   executionId: number
   * } } ScheduledTask
   */

  /**
   * A utility that allows you to schedule async tasks.
   *
   * @class
   * @constructor
   *
   * @param { import('../core/EventBus').default } eventBus
   */
  function Scheduler(eventBus) {
    /**
     * @type { Record<string, ScheduledTask> }
     */
    this._scheduled = {};
    eventBus.on('diagram.destroy', () => {
      Object.keys(this._scheduled).forEach(id => {
        this.cancel(id);
      });
    });
  }
  Scheduler.$inject = ['eventBus'];

  /**
   * Schedule execution of a task in the next tick.
   *
   * Call with an id to ensure only the latest call will be executed.
   *
   * @template T

   * @param {(...args: any[]) => T} taskFn function to be executed
   * @param {string} [id] identifying the task to ensure uniqueness
   *
   * @return Promise<T> result of the executed task
   */
  Scheduler.prototype.schedule = function (taskFn, id = Ids.next()) {
    this.cancel(id);
    const newScheduled = this._schedule(taskFn, id);
    this._scheduled[id] = newScheduled;
    return newScheduled.promise;
  };
  Scheduler.prototype._schedule = function (taskFn, id) {
    const deferred = defer();
    const executionId = setTimeout(() => {
      try {
        this._scheduled[id] = null;
        try {
          deferred.resolve(taskFn());
        } catch (error) {
          deferred.reject(error);
        }
      } catch (error) {
        console.error('Scheduler#_schedule execution failed', error);
      }
    });
    return {
      executionId,
      promise: deferred.promise
    };
  };

  /**
   * Cancel a previously scheduled task.
   *
   * @param {string} id
   */
  Scheduler.prototype.cancel = function (id) {
    const scheduled = this._scheduled[id];
    if (scheduled) {
      this._cancel(scheduled);
      this._scheduled[id] = null;
    }
  };
  Scheduler.prototype._cancel = function (scheduled) {
    clearTimeout(scheduled.executionId);
  };

  /**
   * @return { {
   *   promise: Promise,
   *   resolve: Function,
   *   reject: Function
   * } }
   */
  function defer() {
    const deferred = {};
    deferred.promise = new Promise((resolve, reject) => {
      deferred.resolve = resolve;
      deferred.reject = reject;
    });
    return deferred;
  }

  var SchedulerModule = {
    scheduler: ['type', Scheduler]
  };

  var MARKER_HIDDEN = 'djs-element-hidden';

  /**
   * @typedef {import('../../model/Types').Element} Element
   *
   * @typedef {import('../../util/Types').Rect} Rect
   * @typedef {import('../../util/Types').RectTRBL} RectTRBL
   *
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/ElementRegistry').default} ElementRegistry
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('../../scheduler/Scheduler').default} Scheduler
   *
   * @typedef {import('./ContextPadProvider').default} ContextPadProvider
   * @typedef {import('./ContextPadProvider').ContextPadEntries} ContextPadEntries
   *
   */

  /**
   * @template {Element} [ElementType=Element]
   *
   * @typedef {ElementType|ElementType[]} ContextPadTarget
   */

  var entrySelector = '.entry';
  var DEFAULT_PRIORITY$7 = 1000;
  var CONTEXT_PAD_MARGIN = 8;
  var HOVER_DELAY = 300;

  /**
   * A context pad that displays element specific, contextual actions next
   * to a diagram element.
   *
   * @param {Canvas} canvas
   * @param {ElementRegistry} elementRegistry
   * @param {EventBus} eventBus
   * @param {Scheduler} scheduler
   */
  function ContextPad(canvas, elementRegistry, eventBus, scheduler) {
    this._canvas = canvas;
    this._elementRegistry = elementRegistry;
    this._eventBus = eventBus;
    this._scheduler = scheduler;
    this._current = null;
    this._init();
  }
  ContextPad.$inject = ['canvas', 'elementRegistry', 'eventBus', 'scheduler'];

  /**
   * Registers events needed for interaction with other components.
   */
  ContextPad.prototype._init = function () {
    var self = this;
    this._eventBus.on('selection.changed', function (event) {
      var selection = event.newSelection;
      var target = selection.length ? selection.length === 1 ? selection[0] : selection : null;
      if (target) {
        self.open(target, true);
      } else {
        self.close();
      }
    });
    this._eventBus.on('elements.changed', function (event) {
      var elements = event.elements,
        current = self._current;
      if (!current) {
        return;
      }
      var target = current.target;
      var targets = isArray$6(target) ? target : [target];
      var targetsChanged = targets.filter(function (element) {
        return elements.includes(element);
      });
      if (targetsChanged.length) {
        // (1) close
        self.close();
        var targetsNew = targets.filter(function (element) {
          return self._elementRegistry.get(element.id);
        });
        if (targetsNew.length) {
          // (2) re-open with new targets being all previous targets that still
          // exist
          self._updateAndOpen(targetsNew.length > 1 ? targetsNew : targetsNew[0]);
        }
      }
    });
    this._eventBus.on('canvas.viewbox.changed', function () {
      self._updatePosition();
    });
    this._eventBus.on('element.marker.update', function (event) {
      if (!self.isOpen()) {
        return;
      }
      var element = event.element;
      var current = self._current;
      var targets = isArray$6(current.target) ? current.target : [current.target];
      if (!targets.includes(element)) {
        return;
      }
      self._updateVisibility();
    });
    this._container = this._createContainer();
  };
  ContextPad.prototype._createContainer = function () {
    var container = domify$1('<div class="djs-context-pad-parent"></div>');
    this._canvas.getContainer().appendChild(container);
    return container;
  };

  /**
   * @overlord
   *
   * Register a context pad provider with the default priority. See
   * {@link ContextPadProvider} for examples.
   *
   * @param {ContextPadProvider} provider
   */

  /**
   * Register a context pad provider with the given priority. See
   * {@link ContextPadProvider} for examples.
   *
   * @param {number} priority
   * @param {ContextPadProvider} provider
   */
  ContextPad.prototype.registerProvider = function (priority, provider) {
    if (!provider) {
      provider = priority;
      priority = DEFAULT_PRIORITY$7;
    }
    this._eventBus.on('contextPad.getProviders', priority, function (event) {
      event.providers.push(provider);
    });
  };

  /**
   * Get context pad entries for given elements.
   *
   * @param {ContextPadTarget} target
   *
   * @return {ContextPadEntries} list of entries
   */
  ContextPad.prototype.getEntries = function (target) {
    var providers = this._getProviders();
    var provideFn = isArray$6(target) ? 'getMultiElementContextPadEntries' : 'getContextPadEntries';
    var entries = {};

    // loop through all providers and their entries.
    // group entries by id so that overriding an entry is possible
    forEach$3(providers, function (provider) {
      if (!isFunction$2(provider[provideFn])) {
        return;
      }
      var entriesOrUpdater = provider[provideFn](target);
      if (isFunction$2(entriesOrUpdater)) {
        entries = entriesOrUpdater(entries);
      } else {
        forEach$3(entriesOrUpdater, function (entry, id) {
          entries[id] = entry;
        });
      }
    });
    return entries;
  };

  /**
   * Trigger context pad via DOM event.
   *
   * The entry to trigger is determined by the target element.
   *
   * @param {string} action
   * @param {Event} event
   * @param {boolean} [autoActivate=false]
   */
  ContextPad.prototype.trigger = function (action, event, autoActivate) {
    var self = this;
    var entry,
      originalEvent,
      button = event.delegateTarget || event.target;
    if (!button) {
      return event.preventDefault();
    }
    entry = attr$1(button, 'data-action');
    originalEvent = event.originalEvent || event;
    if (action === 'mouseover') {
      this._timeout = setTimeout(function () {
        self._mouseout = self.triggerEntry(entry, 'hover', originalEvent, autoActivate);
      }, HOVER_DELAY);
      return;
    } else if (action === 'mouseout') {
      clearTimeout(this._timeout);
      if (this._mouseout) {
        this._mouseout();
        this._mouseout = null;
      }
      return;
    }
    return this.triggerEntry(entry, action, originalEvent, autoActivate);
  };

  /**
   * Trigger action on context pad entry entry, e.g. click, mouseover or mouseout.
   *
   * @param {string} entryId
   * @param {string} action
   * @param {Event} event
   * @param {boolean} [autoActivate=false]
   */
  ContextPad.prototype.triggerEntry = function (entryId, action, event, autoActivate) {
    if (!this.isShown()) {
      return;
    }
    var target = this._current.target,
      entries = this._current.entries;
    var entry = entries[entryId];
    if (!entry) {
      return;
    }
    var handler = entry.action;
    if (this._eventBus.fire('contextPad.trigger', {
      entry,
      event
    }) === false) {
      return;
    }

    // simple action (via callback function)
    if (isFunction$2(handler)) {
      if (action === 'click') {
        return handler(event, target, autoActivate);
      }
    } else {
      if (handler[action]) {
        return handler[action](event, target, autoActivate);
      }
    }

    // silence other actions
    event.preventDefault();
  };

  /**
   * Open the context pad for given elements.
   *
   * @param {ContextPadTarget} target
   * @param {boolean} [force=false] - Force re-opening context pad.
   */
  ContextPad.prototype.open = function (target, force) {
    if (!force && this.isOpen(target)) {
      return;
    }
    this.close();
    this._updateAndOpen(target);
  };
  ContextPad.prototype._getProviders = function () {
    var event = this._eventBus.createEvent({
      type: 'contextPad.getProviders',
      providers: []
    });
    this._eventBus.fire(event);
    return event.providers;
  };

  /**
   * @param {ContextPadTarget} target
   */
  ContextPad.prototype._updateAndOpen = function (target) {
    var entries = this.getEntries(target),
      html = this._createHtml(target),
      image;
    forEach$3(entries, function (entry, id) {
      var grouping = entry.group || 'default',
        control = domify$1(entry.html || '<div class="entry" draggable="true"></div>'),
        container;
      attr$1(control, 'data-action', id);
      container = query('[data-group=' + escapeCSS(grouping) + ']', html);
      if (!container) {
        container = domify$1('<div class="group"></div>');
        attr$1(container, 'data-group', grouping);
        html.appendChild(container);
      }
      container.appendChild(control);
      if (entry.className) {
        addClasses$1(control, entry.className);
      }
      if (entry.title) {
        attr$1(control, 'title', entry.title);
      }
      if (entry.imageUrl) {
        image = domify$1('<img>');
        attr$1(image, 'src', entry.imageUrl);
        image.style.width = '100%';
        image.style.height = '100%';
        control.appendChild(image);
      }
    });
    classes$1(html).add('open');
    this._current = {
      entries,
      html,
      target
    };
    this._updatePosition();
    this._updateVisibility();
    this._eventBus.fire('contextPad.open', {
      current: this._current
    });
  };

  /**
   * @param {ContextPadTarget} target
   *
   * @return {HTMLElement}
   */
  ContextPad.prototype._createHtml = function (target) {
    var self = this;
    var html = domify$1('<div class="djs-context-pad"></div>');
    delegate.bind(html, entrySelector, 'click', function (event) {
      self.trigger('click', event);
    });
    delegate.bind(html, entrySelector, 'dragstart', function (event) {
      self.trigger('dragstart', event);
    });
    delegate.bind(html, entrySelector, 'mouseover', function (event) {
      self.trigger('mouseover', event);
    });
    delegate.bind(html, entrySelector, 'mouseout', function (event) {
      self.trigger('mouseout', event);
    });

    // stop propagation of mouse events
    event.bind(html, 'mousedown', function (event) {
      event.stopPropagation();
    });
    this._container.appendChild(html);
    this._eventBus.fire('contextPad.create', {
      target: target,
      pad: html
    });
    return html;
  };

  /**
   * @param {ContextPadTarget} target
   *
   * @return { { html: HTMLElement } }
   */
  ContextPad.prototype.getPad = function (target) {
    console.warn(new Error('ContextPad#getPad is deprecated and will be removed in future library versions, cf. https://github.com/bpmn-io/diagram-js/pull/888'));
    let html;
    if (this.isOpen() && targetsEqual(this._current.target, target)) {
      html = this._current.html;
    } else {
      html = this._createHtml(target);
    }
    return {
      html
    };
  };

  /**
   * Close the context pad
   */
  ContextPad.prototype.close = function () {
    if (!this.isOpen()) {
      return;
    }
    clearTimeout(this._timeout);
    this._container.innerHTML = '';
    this._eventBus.fire('contextPad.close', {
      current: this._current
    });
    this._current = null;
  };

  /**
   * Check if pad is open.
   *
   * If target is provided, check if it is opened
   * for the given target (single or multiple elements).
   *
   * @param {ContextPadTarget} [target]
   * @return {boolean}
   */
  ContextPad.prototype.isOpen = function (target) {
    var current = this._current;
    if (!current) {
      return false;
    }

    // basic no-args is open check
    if (!target) {
      return true;
    }
    var currentTarget = current.target;

    // strict handling of single vs. multi-selection
    if (isArray$6(target) !== isArray$6(currentTarget)) {
      return false;
    }
    if (isArray$6(target)) {
      return target.length === currentTarget.length && every(target, function (element) {
        return currentTarget.includes(element);
      });
    } else {
      return currentTarget === target;
    }
  };

  /**
   * Check if pad is open and not hidden.
   *
   * @return {boolean}
   */
  ContextPad.prototype.isShown = function () {
    return this.isOpen() && classes$1(this._current.html).has('open');
  };

  /**
   * Show context pad.
   */
  ContextPad.prototype.show = function () {
    if (!this.isOpen()) {
      return;
    }
    classes$1(this._current.html).add('open');
    this._updatePosition();
    this._eventBus.fire('contextPad.show', {
      current: this._current
    });
  };

  /**
   * Hide context pad.
   */
  ContextPad.prototype.hide = function () {
    if (!this.isOpen()) {
      return;
    }
    classes$1(this._current.html).remove('open');
    this._eventBus.fire('contextPad.hide', {
      current: this._current
    });
  };

  /**
   * Get context pad position.
   *
   * If target is connection context pad will be positioned at connection end.
   *
   * If multiple targets context pad will be placed at top right corner bounding
   * box.
   *
   * @param {ContextPadTarget} target
   *
   * @return {RectTRBL & { x: number, y: number }}
   */
  ContextPad.prototype._getPosition = function (target) {
    if (!isArray$6(target) && isConnection(target)) {
      var viewbox = this._canvas.viewbox();
      var lastWaypoint = getLastWaypoint(target);
      var x = lastWaypoint.x * viewbox.scale - viewbox.x * viewbox.scale,
        y = lastWaypoint.y * viewbox.scale - viewbox.y * viewbox.scale;
      return {
        left: x + CONTEXT_PAD_MARGIN * this._canvas.zoom(),
        top: y
      };
    }
    var container = this._canvas.getContainer();
    var containerBounds = container.getBoundingClientRect();
    var targetBounds = this._getTargetBounds(target);
    return {
      left: targetBounds.right - containerBounds.left + CONTEXT_PAD_MARGIN * this._canvas.zoom(),
      top: targetBounds.top - containerBounds.top
    };
  };

  /**
   * Update context pad position.
   */
  ContextPad.prototype._updatePosition = function () {
    const updateFn = () => {
      if (!this.isOpen()) {
        return;
      }
      var html = this._current.html;
      var position = this._getPosition(this._current.target);
      if ('x' in position && 'y' in position) {
        html.style.left = position.x + 'px';
        html.style.top = position.y + 'px';
      } else {
        ['top', 'right', 'bottom', 'left'].forEach(function (key) {
          if (key in position) {
            html.style[key] = position[key] + 'px';
          }
        });
      }
    };
    this._scheduler.schedule(updateFn, 'ContextPad#_updatePosition');
  };

  /**
   * Update context pad visibility. Hide if any of the target elements is hidden
   * using the `djs-element-hidden` or `djs-label-hidden` markers.
   */
  ContextPad.prototype._updateVisibility = function () {
    const updateFn = () => {
      if (!this.isOpen()) {
        return;
      }
      var self = this;
      var target = this._current.target;
      var targets = isArray$6(target) ? target : [target];
      var isHidden = targets.some(function (target) {
        return self._canvas.hasMarker(target, MARKER_HIDDEN);
      });
      if (isHidden) {
        self.hide();
      } else {
        self.show();
      }
    };
    this._scheduler.schedule(updateFn, 'ContextPad#_updateVisibility');
  };

  /**
   * Get bounding client rect of target element(s).
   *
   * @param {ContextPadTarget} target
   *
   * @returns {Rect & RectTRBL}
   */
  ContextPad.prototype._getTargetBounds = function (target) {
    var self = this;
    var elements = isArray$6(target) ? target : [target];
    var elementsGfx = elements.map(function (element) {
      return self._canvas.getGraphics(element);
    });
    return elementsGfx.reduce(function (bounds, elementGfx) {
      const elementBounds = elementGfx.getBoundingClientRect();
      bounds.top = Math.min(bounds.top, elementBounds.top);
      bounds.right = Math.max(bounds.right, elementBounds.right);
      bounds.bottom = Math.max(bounds.bottom, elementBounds.bottom);
      bounds.left = Math.min(bounds.left, elementBounds.left);
      bounds.x = bounds.left;
      bounds.y = bounds.top;
      bounds.width = bounds.right - bounds.left;
      bounds.height = bounds.bottom - bounds.top;
      return bounds;
    }, {
      top: Infinity,
      right: -Infinity,
      bottom: -Infinity,
      left: Infinity
    });
  };

  // helpers //////////

  function addClasses$1(element, classNames) {
    var classes = classes$1(element);
    classNames = isArray$6(classNames) ? classNames : classNames.split(/\s+/g);
    classNames.forEach(function (cls) {
      classes.add(cls);
    });
  }
  function getLastWaypoint(connection) {
    return connection.waypoints[connection.waypoints.length - 1];
  }

  /**
   * @param {ContextPadTarget} target
   * @param {ContextPadTarget} otherTarget
   *
   * @return {boolean}
   */
  function targetsEqual(target, otherTarget) {
    target = isArray$6(target) ? target : [target];
    otherTarget = isArray$6(otherTarget) ? otherTarget : [otherTarget];
    return target.length === otherTarget.length && every(target, function (element) {
      return otherTarget.includes(element);
    });
  }

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var DiagramContextPad = {
    __depends__: [InteractionEventsModule, SchedulerModule, OverlaysModule],
    contextPad: ['type', ContextPad]
  };

  /**
   * @typedef {import('../../model/Types').Element} Element
   *
   * @typedef {import('../../util/Types').Point} Point
   *
   * @typedef {import('../dragging/Dragging').default} Dragging
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('../modeling/Modeling').default} Modeling
   * @typedef {import('../rules/Rules').default} Rules
   */

  /**
   * @param {EventBus} eventBus
   * @param {Dragging} dragging
   * @param {Modeling} modeling
   * @param {Rules} rules
   */
  function Connect(eventBus, dragging, modeling, rules) {
    // rules

    function canConnect(source, target) {
      return rules.allowed('connection.create', {
        source: source,
        target: target
      });
    }
    function canConnectReverse(source, target) {
      return canConnect(target, source);
    }

    // event handlers

    eventBus.on('connect.hover', function (event) {
      var context = event.context,
        start = context.start,
        hover = event.hover,
        canExecute;

      // cache hover state
      context.hover = hover;
      canExecute = context.canExecute = canConnect(start, hover);

      // ignore hover
      if (isNil(canExecute)) {
        return;
      }
      if (canExecute !== false) {
        context.source = start;
        context.target = hover;
        return;
      }
      canExecute = context.canExecute = canConnectReverse(start, hover);

      // ignore hover
      if (isNil(canExecute)) {
        return;
      }
      if (canExecute !== false) {
        context.source = hover;
        context.target = start;
      }
    });
    eventBus.on(['connect.out', 'connect.cleanup'], function (event) {
      var context = event.context;
      context.hover = null;
      context.source = null;
      context.target = null;
      context.canExecute = false;
    });
    eventBus.on('connect.end', function (event) {
      var context = event.context,
        canExecute = context.canExecute,
        connectionStart = context.connectionStart,
        connectionEnd = {
          x: event.x,
          y: event.y
        },
        source = context.source,
        target = context.target;
      if (!canExecute) {
        return false;
      }
      var attrs = null,
        hints = {
          connectionStart: isReverse(context) ? connectionEnd : connectionStart,
          connectionEnd: isReverse(context) ? connectionStart : connectionEnd
        };
      if (isObject$2(canExecute)) {
        attrs = canExecute;
      }
      context.connection = modeling.connect(source, target, attrs, hints);
    });

    // API

    /**
     * Start connect operation.
     *
     * @param {MouseEvent|TouchEvent} event
     * @param {Element} start
     * @param {Point} [connectionStart]
     * @param {boolean} [autoActivate=false]
     */
    this.start = function (event, start, connectionStart, autoActivate) {
      if (!isObject$2(connectionStart)) {
        autoActivate = connectionStart;
        connectionStart = getMid(start);
      }
      dragging.init(event, 'connect', {
        autoActivate: autoActivate,
        data: {
          shape: start,
          context: {
            start: start,
            connectionStart: connectionStart
          }
        }
      });
    };
  }
  Connect.$inject = ['eventBus', 'dragging', 'modeling', 'rules'];

  // helpers //////////

  function isReverse(context) {
    var hover = context.hover,
      source = context.source,
      target = context.target;
    return hover && source && hover === source && source !== target;
  }

  /**
   * @typedef {import('didi').Injector} Injector
   *
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/EventBus').default} EventBus
   */

  var HIGH_PRIORITY$9 = 1100,
    LOW_PRIORITY$j = 900;
  var MARKER_OK$2 = 'connect-ok',
    MARKER_NOT_OK$2 = 'connect-not-ok';

  /**
   * Shows connection preview during connect.
   *
   * @param {Injector} injector
   * @param {EventBus} eventBus
   * @param {Canvas} canvas
   */
  function ConnectPreview(injector, eventBus, canvas) {
    var connectionPreview = injector.get('connectionPreview', false);
    connectionPreview && eventBus.on('connect.move', function (event) {
      var context = event.context,
        canConnect = context.canExecute,
        hover = context.hover,
        source = context.source,
        start = context.start,
        startPosition = context.startPosition,
        target = context.target,
        connectionStart = context.connectionStart || startPosition,
        connectionEnd = context.connectionEnd || {
          x: event.x,
          y: event.y
        },
        previewStart = connectionStart,
        previewEnd = connectionEnd;
      if (isReverse(context)) {
        previewStart = connectionEnd;
        previewEnd = connectionStart;
      }
      connectionPreview.drawPreview(context, canConnect, {
        source: source || start,
        target: target || hover,
        connectionStart: previewStart,
        connectionEnd: previewEnd
      });
    });
    eventBus.on('connect.hover', LOW_PRIORITY$j, function (event) {
      var context = event.context,
        hover = event.hover,
        canExecute = context.canExecute;

      // ignore hover
      if (canExecute === null) {
        return;
      }
      canvas.addMarker(hover, canExecute ? MARKER_OK$2 : MARKER_NOT_OK$2);
    });
    eventBus.on(['connect.out', 'connect.cleanup'], HIGH_PRIORITY$9, function (event) {
      var hover = event.hover;
      if (hover) {
        canvas.removeMarker(hover, MARKER_OK$2);
        canvas.removeMarker(hover, MARKER_NOT_OK$2);
      }
    });
    connectionPreview && eventBus.on('connect.cleanup', function (event) {
      connectionPreview.cleanUp(event.context);
    });
  }
  ConnectPreview.$inject = ['injector', 'eventBus', 'canvas'];

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var DiagramConnect = {
    __depends__: [SelectionModule, Rules$5, DraggingModule],
    __init__: ['connectPreview'],
    connect: ['type', Connect],
    connectPreview: ['type', ConnectPreview]
  };

  /**
   * @typedef {import('../../core/Types').ElementLike} Element
   * @typedef {import('../../core/Types').ShapeLike} Shape
   *
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/ElementRegistry').default} ElementRegistry
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('../../draw/Styles').default} Styles
   */

  const cloneIds = new IdGenerator('ps');
  var MARKER_TYPES = ['marker-start', 'marker-mid', 'marker-end'];
  var NODES_CAN_HAVE_MARKER = ['circle', 'ellipse', 'line', 'path', 'polygon', 'polyline', 'path', 'rect'];

  /**
   * Adds support for previews of moving/resizing elements.
   *
   * @param {ElementRegistry} elementRegistry
   * @param {EventBus} eventBus
   * @param {Canvas} canvas
   * @param {Styles} styles
   */
  function PreviewSupport(elementRegistry, eventBus, canvas, styles) {
    this._elementRegistry = elementRegistry;
    this._canvas = canvas;
    this._styles = styles;
  }
  PreviewSupport.$inject = ['elementRegistry', 'eventBus', 'canvas', 'styles'];

  // Markers are cleaned up with visuals, keep stub for compatibility
  // cf. https://github.com/camunda/camunda-modeler/issues/4307
  PreviewSupport.prototype.cleanUp = function () {
    console.warn('PreviewSupport#cleanUp is deprecated and will be removed in future versions. You do not need to manually clean up previews anymore. cf. https://github.com/bpmn-io/diagram-js/pull/906');
  };

  /**
   * Returns graphics of an element.
   *
   * @param {Element} element
   *
   * @return {SVGElement}
   */
  PreviewSupport.prototype.getGfx = function (element) {
    return this._elementRegistry.getGraphics(element);
  };

  /**
   * Adds a move preview of a given shape to a given SVG group.
   *
   * @param {Element} element The element to be moved.
   * @param {SVGElement} group The SVG group to add the preview to.
   * @param {SVGElement} [gfx] The optional graphical element of the element.
   * @param {string} [className="djs-dragger"] The optional class name to add to the preview.
   *
   * @return {SVGElement} The preview.
   */
  PreviewSupport.prototype.addDragger = function (element, group, gfx, className = 'djs-dragger') {
    gfx = gfx || this.getGfx(element);
    var dragger = clone$1(gfx);
    var bbox = gfx.getBoundingClientRect();
    this._cloneMarkers(getVisual(dragger), className);
    attr(dragger, this._styles.cls(className, [], {
      x: bbox.top,
      y: bbox.left
    }));
    append(group, dragger);
    attr(dragger, 'data-preview-support-element-id', element.id);
    return dragger;
  };

  /**
   * Adds a resize preview of a given shape to a given SVG group.
   *
   * @param {Shape} shape The element to be resized.
   * @param {SVGElement} group The SVG group to add the preview to.
   *
   * @return {SVGElement} The preview.
   */
  PreviewSupport.prototype.addFrame = function (shape, group) {
    var frame = create$2('rect', {
      class: 'djs-resize-overlay',
      width: shape.width,
      height: shape.height,
      x: shape.x,
      y: shape.y
    });
    append(group, frame);
    attr(frame, 'data-preview-support-element-id', shape.id);
    return frame;
  };

  /**
   * Clone all markers referenced by a node and its child nodes.
   *
   * @param {SVGElement} gfx
   * @param {string} [className="djs-dragger"]
   */
  PreviewSupport.prototype._cloneMarkers = function (gfx, className = 'djs-dragger', rootGfx = gfx) {
    var self = this;
    if (gfx.childNodes) {
      gfx.childNodes.forEach(childNode => {
        self._cloneMarkers(childNode, className, rootGfx);
      });
    }
    if (!canHaveMarker(gfx)) {
      return;
    }
    MARKER_TYPES.forEach(function (markerType) {
      if (attr(gfx, markerType)) {
        var marker = getMarker(gfx, markerType, self._canvas.getContainer());

        // Only clone marker if it is already present on the DOM
        marker && self._cloneMarker(rootGfx, gfx, marker, markerType, className);
      }
    });
  };

  /**
   * Clone marker referenced by an element.
   *
   * @param {SVGElement} gfx
   * @param {SVGElement} marker
   * @param {string} markerType
   * @param {string} [className="djs-dragger"]
   */
  PreviewSupport.prototype._cloneMarker = function (parentGfx, gfx, marker, markerType, className = 'djs-dragger') {
    // Add a random suffix to the marker ID in case the same marker is previewed multiple times
    var clonedMarkerId = [marker.id, className, cloneIds.next()].join('-');

    // reuse marker if it was part of original gfx
    var copiedMarker = query('marker#' + marker.id, parentGfx);
    parentGfx = parentGfx || this._canvas._svg;
    var clonedMarker = copiedMarker || clone$1(marker);
    clonedMarker.id = clonedMarkerId;
    classes(clonedMarker).add(className);
    var defs = query(':scope > defs', parentGfx);
    if (!defs) {
      defs = create$2('defs');
      append(parentGfx, defs);
    }
    append(defs, clonedMarker);
    var reference = idToReference(clonedMarker.id);
    attr(gfx, markerType, reference);
  };

  // helpers //////////

  /**
   * Get marker of given type referenced by node.
   *
   * @param {HTMLElement} node
   * @param {string} markerType
   * @param {HTMLElement} [parentNode]
   *
   * @param {HTMLElement}
   */
  function getMarker(node, markerType, parentNode) {
    var id = referenceToId(attr(node, markerType));
    return query('marker#' + id, parentNode || document);
  }

  /**
   * Get ID of fragment within current document from its functional IRI reference.
   * References may use single or double quotes.
   *
   * @param {string} reference
   *
   * @return {string}
   */
  function referenceToId(reference) {
    return reference.match(/url\(['"]?#([^'"]*)['"]?\)/)[1];
  }

  /**
   * Get functional IRI reference for given ID of fragment within current document.
   *
   * @param {string} id
   *
   * @return {string}
   */
  function idToReference(id) {
    return 'url(#' + id + ')';
  }

  /**
   * Check wether node type can have marker attributes.
   *
   * @param {HTMLElement} node
   *
   * @return {boolean}
   */
  function canHaveMarker(node) {
    return NODES_CAN_HAVE_MARKER.indexOf(node.nodeName) !== -1;
  }

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var PreviewSupportModule = {
    __init__: ['previewSupport'],
    previewSupport: ['type', PreviewSupport]
  };

  var MARKER_OK$1 = 'drop-ok',
    MARKER_NOT_OK$1 = 'drop-not-ok',
    MARKER_ATTACH$1 = 'attach-ok',
    MARKER_NEW_PARENT$1 = 'new-parent';

  /**
   * @typedef {import('../../core/Types').ElementLike} Element
   * @typedef {import('../../core/Types').ShapeLike} Shape
   *
   * @typedef {import('../../util/Types').Point} Point
   *
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../dragging/Dragging').default} Dragging
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('../modeling/Modeling').default} Modeling
   * @typedef {import('../rules/Rules').default} Rules
   */

  var PREFIX = 'create';
  var HIGH_PRIORITY$8 = 2000;

  /**
   * Create new elements through drag and drop.
   *
   * @param {Canvas} canvas
   * @param {Dragging} dragging
   * @param {EventBus} eventBus
   * @param {Modeling} modeling
   * @param {Rules} rules
   */
  function Create(canvas, dragging, eventBus, modeling, rules) {
    // rules //////////

    /**
     * Check wether elements can be created.
     *
     * @param {Element[]} elements
     * @param {Shape} target
     * @param {Point} position
     * @param {Element} [source]
     *
     * @return {boolean|null|Object}
     */
    function canCreate(elements, target, position, source, hints) {
      if (!target) {
        return false;
      }

      // ignore child elements and external labels
      elements = filter$2(elements, function (element) {
        var labelTarget = element.labelTarget;
        return !element.parent && !(isLabel$1(element) && elements.indexOf(labelTarget) !== -1);
      });
      var shape = find$2(elements, function (element) {
        return !isConnection(element);
      });
      var attach = false,
        connect = false,
        create = false;

      // (1) attaching single shapes
      if (isSingleShape(elements)) {
        attach = rules.allowed('shape.attach', {
          position: position,
          shape: shape,
          target: target
        });
      }
      if (!attach) {
        // (2) creating elements
        if (isSingleShape(elements)) {
          create = rules.allowed('shape.create', {
            position: position,
            shape: shape,
            source: source,
            target: target
          });
        } else {
          create = rules.allowed('elements.create', {
            elements: elements,
            position: position,
            target: target
          });
        }
      }
      var connectionTarget = hints.connectionTarget;

      // (3) appending single shapes
      if (create || attach) {
        if (shape && source) {
          connect = rules.allowed('connection.create', {
            source: connectionTarget === source ? shape : source,
            target: connectionTarget === source ? source : shape,
            hints: {
              targetParent: target,
              targetAttach: attach
            }
          });
        }
        return {
          attach: attach,
          connect: connect
        };
      }

      // ignore wether or not elements can be created
      if (create === null || attach === null) {
        return null;
      }
      return false;
    }
    function setMarker(element, marker) {
      [MARKER_ATTACH$1, MARKER_OK$1, MARKER_NOT_OK$1, MARKER_NEW_PARENT$1].forEach(function (m) {
        if (m === marker) {
          canvas.addMarker(element, m);
        } else {
          canvas.removeMarker(element, m);
        }
      });
    }

    // event handling //////////

    eventBus.on(['create.move', 'create.hover'], function (event) {
      var context = event.context,
        elements = context.elements,
        hover = event.hover,
        source = context.source,
        hints = context.hints || {};
      if (!hover) {
        context.canExecute = false;
        context.target = null;
        return;
      }
      ensureConstraints$1(event);
      var position = {
        x: event.x,
        y: event.y
      };
      var canExecute = context.canExecute = hover && canCreate(elements, hover, position, source, hints);
      if (hover && canExecute !== null) {
        context.target = hover;
        if (canExecute && canExecute.attach) {
          setMarker(hover, MARKER_ATTACH$1);
        } else {
          setMarker(hover, canExecute ? MARKER_NEW_PARENT$1 : MARKER_NOT_OK$1);
        }
      }
    });
    eventBus.on(['create.end', 'create.out', 'create.cleanup'], function (event) {
      var hover = event.hover;
      if (hover) {
        setMarker(hover, null);
      }
    });
    eventBus.on('create.end', function (event) {
      var context = event.context,
        source = context.source,
        shape = context.shape,
        elements = context.elements,
        target = context.target,
        canExecute = context.canExecute,
        attach = canExecute && canExecute.attach,
        connect = canExecute && canExecute.connect,
        hints = context.hints || {};
      if (canExecute === false || !target) {
        return false;
      }
      ensureConstraints$1(event);
      var position = {
        x: event.x,
        y: event.y
      };
      if (connect) {
        shape = modeling.appendShape(source, shape, position, target, {
          attach: attach,
          connection: connect === true ? {} : connect,
          connectionTarget: hints.connectionTarget
        });
      } else {
        elements = modeling.createElements(elements, position, target, assign$4({}, hints, {
          attach: attach
        }));

        // update shape
        shape = find$2(elements, function (element) {
          return !isConnection(element);
        });
      }

      // update elements and shape
      assign$4(context, {
        elements: elements,
        shape: shape
      });
      assign$4(event, {
        elements: elements,
        shape: shape
      });
    });
    function cancel() {
      var context = dragging.context();
      if (context && context.prefix === PREFIX) {
        dragging.cancel();
      }
    }

    // cancel on <elements.changed> that is not result of <drag.end>
    eventBus.on('create.init', function () {
      eventBus.on('elements.changed', cancel);
      eventBus.once(['create.cancel', 'create.end'], HIGH_PRIORITY$8, function () {
        eventBus.off('elements.changed', cancel);
      });
    });

    // API //////////

    /**
     * @param event
     * @param elements
     * @param {any} [context={}]
     */
    this.start = function (event, elements, context) {
      if (!isArray$6(elements)) {
        elements = [elements];
      }
      var shape = find$2(elements, function (element) {
        return !isConnection(element);
      });
      if (!shape) {
        // at least one shape is required
        return;
      }
      context = assign$4({
        elements: elements,
        hints: {},
        shape: shape
      }, context || {});

      // make sure each element has x and y
      forEach$3(elements, function (element) {
        if (!isNumber$3(element.x)) {
          element.x = 0;
        }
        if (!isNumber$3(element.y)) {
          element.y = 0;
        }
      });
      var visibleElements = filter$2(elements, function (element) {
        return !element.hidden;
      });
      var bbox = getBBox(visibleElements);

      // center elements around cursor
      forEach$3(elements, function (element) {
        if (isConnection(element)) {
          element.waypoints = map$3(element.waypoints, function (waypoint) {
            return {
              x: waypoint.x - bbox.x - bbox.width / 2,
              y: waypoint.y - bbox.y - bbox.height / 2
            };
          });
        }
        assign$4(element, {
          x: element.x - bbox.x - bbox.width / 2,
          y: element.y - bbox.y - bbox.height / 2
        });
      });
      dragging.init(event, PREFIX, {
        cursor: 'grabbing',
        autoActivate: true,
        data: {
          shape: shape,
          elements: elements,
          context: context
        }
      });
    };
  }
  Create.$inject = ['canvas', 'dragging', 'eventBus', 'modeling', 'rules'];

  // helpers //////////

  function ensureConstraints$1(event) {
    var context = event.context,
      createConstraints = context.createConstraints;
    if (!createConstraints) {
      return;
    }
    if (createConstraints.left) {
      event.x = Math.max(event.x, createConstraints.left);
    }
    if (createConstraints.right) {
      event.x = Math.min(event.x, createConstraints.right);
    }
    if (createConstraints.top) {
      event.y = Math.max(event.y, createConstraints.top);
    }
    if (createConstraints.bottom) {
      event.y = Math.min(event.y, createConstraints.bottom);
    }
  }
  function isSingleShape(elements) {
    return elements && elements.length === 1 && !isConnection(elements[0]);
  }

  /**
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('../../core/GraphicsFactory').default} GraphicsFactory
   * @typedef {import('../preview-support/PreviewSupport').default} PreviewSupport
   * @typedef {import('../../draw/Styles').default} Styles
   */

  var LOW_PRIORITY$i = 750;

  /**
   * @param {Canvas} canvas
   * @param {EventBus} eventBus
   * @param {GraphicsFactory} graphicsFactory
   * @param {PreviewSupport} previewSupport
   * @param {Styles} styles
   */
  function CreatePreview(canvas, eventBus, graphicsFactory, previewSupport, styles) {
    function createDragGroup(elements) {
      var dragGroup = create$2('g');
      attr(dragGroup, styles.cls('djs-drag-group', ['no-events']));
      var childrenGfx = create$2('g');
      elements.forEach(function (element) {
        // create graphics
        var gfx;
        if (element.hidden) {
          return;
        }
        if (element.waypoints) {
          gfx = graphicsFactory._createContainer('connection', childrenGfx);
          graphicsFactory.drawConnection(getVisual(gfx), element);
        } else {
          gfx = graphicsFactory._createContainer('shape', childrenGfx);
          graphicsFactory.drawShape(getVisual(gfx), element);
          translate$1(gfx, element.x, element.y);
        }

        // add preview
        previewSupport.addDragger(element, dragGroup, gfx);
      });
      return dragGroup;
    }
    eventBus.on('create.move', LOW_PRIORITY$i, function (event) {
      var hover = event.hover,
        context = event.context,
        elements = context.elements,
        dragGroup = context.dragGroup;

      // lazily create previews
      if (!dragGroup) {
        dragGroup = context.dragGroup = createDragGroup(elements);
      }
      var activeLayer;
      if (hover) {
        if (!dragGroup.parentNode) {
          activeLayer = canvas.getActiveLayer();
          append(activeLayer, dragGroup);
        }
        translate$1(dragGroup, event.x, event.y);
      } else {
        remove$3(dragGroup);
      }
    });
    eventBus.on('create.cleanup', function (event) {
      var context = event.context,
        dragGroup = context.dragGroup;
      if (dragGroup) {
        remove$3(dragGroup);
      }
    });
  }
  CreatePreview.$inject = ['canvas', 'eventBus', 'graphicsFactory', 'previewSupport', 'styles'];

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var DiagramCreate = {
    __depends__: [DraggingModule, PreviewSupportModule, Rules$5, SelectionModule],
    __init__: ['create', 'createPreview'],
    create: ['type', Create],
    createPreview: ['type', CreatePreview]
  };

  var n$1,
    l$1,
    u$1,
    i$2,
    o$1,
    r$2,
    f$1,
    e$2,
    c$1,
    s$1,
    h$1 = {},
    v$1 = [],
    p$1 = /acit|ex(?:s|g|n|p|$)|rph|grid|ows|mnc|ntw|ine[ch]|zoo|^ord|itera/i,
    y$1 = Array.isArray;
  function d$1(n, l) {
    for (var u in l) n[u] = l[u];
    return n;
  }
  function _$1(n) {
    var l = n.parentNode;
    l && l.removeChild(n);
  }
  function g(l, u, t) {
    var i,
      o,
      r,
      f = {};
    for (r in u) "key" == r ? i = u[r] : "ref" == r ? o = u[r] : f[r] = u[r];
    if (arguments.length > 2 && (f.children = arguments.length > 3 ? n$1.call(arguments, 2) : t), "function" == typeof l && null != l.defaultProps) for (r in l.defaultProps) void 0 === f[r] && (f[r] = l.defaultProps[r]);
    return b(l, f, i, o, null);
  }
  function b(n, t, i, o, r) {
    var f = {
      type: n,
      props: t,
      key: i,
      ref: o,
      __k: null,
      __: null,
      __b: 0,
      __e: null,
      __d: void 0,
      __c: null,
      constructor: void 0,
      __v: null == r ? ++u$1 : r,
      __i: -1,
      __u: 0
    };
    return null == r && null != l$1.vnode && l$1.vnode(f), f;
  }
  function w$1(n) {
    return n.children;
  }
  function k$1(n, l) {
    this.props = n, this.context = l;
  }
  function x$1(n, l) {
    if (null == l) return n.__ ? x$1(n.__, n.__i + 1) : null;
    for (var u; l < n.__k.length; l++) if (null != (u = n.__k[l]) && null != u.__e) return u.__e;
    return "function" == typeof n.type ? x$1(n) : null;
  }
  function C$2(n) {
    var l, u;
    if (null != (n = n.__) && null != n.__c) {
      for (n.__e = n.__c.base = null, l = 0; l < n.__k.length; l++) if (null != (u = n.__k[l]) && null != u.__e) {
        n.__e = n.__c.base = u.__e;
        break;
      }
      return C$2(n);
    }
  }
  function P(n) {
    (!n.__d && (n.__d = !0) && i$2.push(n) && !S.__r++ || o$1 !== l$1.debounceRendering) && ((o$1 = l$1.debounceRendering) || r$2)(S);
  }
  function S() {
    var n, u, t, o, r, e, c, s;
    for (i$2.sort(f$1); n = i$2.shift();) n.__d && (u = i$2.length, o = void 0, e = (r = (t = n).__v).__e, c = [], s = [], t.__P && ((o = d$1({}, r)).__v = r.__v + 1, l$1.vnode && l$1.vnode(o), O(t.__P, o, r, t.__n, void 0 !== t.__P.ownerSVGElement, 32 & r.__u ? [e] : null, c, null == e ? x$1(r) : e, !!(32 & r.__u), s), o.__v = r.__v, o.__.__k[o.__i] = o, j$1(c, o, s), o.__e != e && C$2(o)), i$2.length > u && i$2.sort(f$1));
    S.__r = 0;
  }
  function $(n, l, u, t, i, o, r, f, e, c, s) {
    var a,
      p,
      y,
      d,
      _,
      g = t && t.__k || v$1,
      b = l.length;
    for (u.__d = e, I(u, l, g), e = u.__d, a = 0; a < b; a++) null != (y = u.__k[a]) && "boolean" != typeof y && "function" != typeof y && (p = -1 === y.__i ? h$1 : g[y.__i] || h$1, y.__i = a, O(n, y, p, i, o, r, f, e, c, s), d = y.__e, y.ref && p.ref != y.ref && (p.ref && N(p.ref, null, y), s.push(y.ref, y.__c || d, y)), null == _ && null != d && (_ = d), 65536 & y.__u || p.__k === y.__k ? (e && !e.isConnected && (e = x$1(p)), e = H(y, e, n)) : "function" == typeof y.type && void 0 !== y.__d ? e = y.__d : d && (e = d.nextSibling), y.__d = void 0, y.__u &= -196609);
    u.__d = e, u.__e = _;
  }
  function I(n, l, u) {
    var t,
      i,
      o,
      r,
      f,
      e = l.length,
      c = u.length,
      s = c,
      a = 0;
    for (n.__k = [], t = 0; t < e; t++) r = t + a, null != (i = n.__k[t] = null == (i = l[t]) || "boolean" == typeof i || "function" == typeof i ? null : "string" == typeof i || "number" == typeof i || "bigint" == typeof i || i.constructor == String ? b(null, i, null, null, null) : y$1(i) ? b(w$1, {
      children: i
    }, null, null, null) : void 0 === i.constructor && i.__b > 0 ? b(i.type, i.props, i.key, i.ref ? i.ref : null, i.__v) : i) ? (i.__ = n, i.__b = n.__b + 1, f = A$1(i, u, r, s), i.__i = f, o = null, -1 !== f && (s--, (o = u[f]) && (o.__u |= 131072)), null == o || null === o.__v ? (-1 == f && a--, "function" != typeof i.type && (i.__u |= 65536)) : f !== r && (f === r + 1 ? a++ : f > r ? s > e - r ? a += f - r : a-- : f < r ? f == r - 1 && (a = f - r) : a = 0, f !== t + a && (i.__u |= 65536))) : (o = u[r]) && null == o.key && o.__e && 0 == (131072 & o.__u) && (o.__e == n.__d && (n.__d = x$1(o)), q$1(o, o, !1), u[r] = null, s--);
    if (s) for (t = 0; t < c; t++) null != (o = u[t]) && 0 == (131072 & o.__u) && (o.__e == n.__d && (n.__d = x$1(o)), q$1(o, o));
  }
  function H(n, l, u) {
    var t, i;
    if ("function" == typeof n.type) {
      for (t = n.__k, i = 0; t && i < t.length; i++) t[i] && (t[i].__ = n, l = H(t[i], l, u));
      return l;
    }
    n.__e != l && (u.insertBefore(n.__e, l || null), l = n.__e);
    do {
      l = l && l.nextSibling;
    } while (null != l && 8 === l.nodeType);
    return l;
  }
  function A$1(n, l, u, t) {
    var i = n.key,
      o = n.type,
      r = u - 1,
      f = u + 1,
      e = l[u];
    if (null === e || e && i == e.key && o === e.type && 0 == (131072 & e.__u)) return u;
    if (t > (null != e && 0 == (131072 & e.__u) ? 1 : 0)) for (; r >= 0 || f < l.length;) {
      if (r >= 0) {
        if ((e = l[r]) && 0 == (131072 & e.__u) && i == e.key && o === e.type) return r;
        r--;
      }
      if (f < l.length) {
        if ((e = l[f]) && 0 == (131072 & e.__u) && i == e.key && o === e.type) return f;
        f++;
      }
    }
    return -1;
  }
  function F$1(n, l, u) {
    "-" === l[0] ? n.setProperty(l, null == u ? "" : u) : n[l] = null == u ? "" : "number" != typeof u || p$1.test(l) ? u : u + "px";
  }
  function L(n, l, u, t, i) {
    var o;
    n: if ("style" === l) {
      if ("string" == typeof u) n.style.cssText = u;else {
        if ("string" == typeof t && (n.style.cssText = t = ""), t) for (l in t) u && l in u || F$1(n.style, l, "");
        if (u) for (l in u) t && u[l] === t[l] || F$1(n.style, l, u[l]);
      }
    } else if ("o" === l[0] && "n" === l[1]) o = l !== (l = l.replace(/(PointerCapture)$|Capture$/i, "$1")), l = l.toLowerCase() in n || "onFocusOut" === l || "onFocusIn" === l ? l.toLowerCase().slice(2) : l.slice(2), n.l || (n.l = {}), n.l[l + o] = u, u ? t ? u.u = t.u : (u.u = e$2, n.addEventListener(l, o ? s$1 : c$1, o)) : n.removeEventListener(l, o ? s$1 : c$1, o);else {
      if (i) l = l.replace(/xlink(H|:h)/, "h").replace(/sName$/, "s");else if ("width" != l && "height" != l && "href" != l && "list" != l && "form" != l && "tabIndex" != l && "download" != l && "rowSpan" != l && "colSpan" != l && "role" != l && l in n) try {
        n[l] = null == u ? "" : u;
        break n;
      } catch (n) {}
      "function" == typeof u || (null == u || !1 === u && "-" !== l[4] ? n.removeAttribute(l) : n.setAttribute(l, u));
    }
  }
  function M(n) {
    return function (u) {
      if (this.l) {
        var t = this.l[u.type + n];
        if (null == u.t) u.t = e$2++;else if (u.t < t.u) return;
        return t(l$1.event ? l$1.event(u) : u);
      }
    };
  }
  function O(n, u, t, i, o, r, f, e, c, s) {
    var a,
      h,
      v,
      p,
      _,
      g,
      b,
      m,
      x,
      C,
      P,
      S,
      I,
      H,
      T,
      A = u.type;
    if (void 0 !== u.constructor) return null;
    128 & t.__u && (c = !!(32 & t.__u), r = [e = u.__e = t.__e]), (a = l$1.__b) && a(u);
    n: if ("function" == typeof A) try {
      if (m = u.props, x = (a = A.contextType) && i[a.__c], C = a ? x ? x.props.value : a.__ : i, t.__c ? b = (h = u.__c = t.__c).__ = h.__E : ("prototype" in A && A.prototype.render ? u.__c = h = new A(m, C) : (u.__c = h = new k$1(m, C), h.constructor = A, h.render = B$1), x && x.sub(h), h.props = m, h.state || (h.state = {}), h.context = C, h.__n = i, v = h.__d = !0, h.__h = [], h._sb = []), null == h.__s && (h.__s = h.state), null != A.getDerivedStateFromProps && (h.__s == h.state && (h.__s = d$1({}, h.__s)), d$1(h.__s, A.getDerivedStateFromProps(m, h.__s))), p = h.props, _ = h.state, h.__v = u, v) null == A.getDerivedStateFromProps && null != h.componentWillMount && h.componentWillMount(), null != h.componentDidMount && h.__h.push(h.componentDidMount);else {
        if (null == A.getDerivedStateFromProps && m !== p && null != h.componentWillReceiveProps && h.componentWillReceiveProps(m, C), !h.__e && (null != h.shouldComponentUpdate && !1 === h.shouldComponentUpdate(m, h.__s, C) || u.__v === t.__v)) {
          for (u.__v !== t.__v && (h.props = m, h.state = h.__s, h.__d = !1), u.__e = t.__e, u.__k = t.__k, u.__k.forEach(function (n) {
            n && (n.__ = u);
          }), P = 0; P < h._sb.length; P++) h.__h.push(h._sb[P]);
          h._sb = [], h.__h.length && f.push(h);
          break n;
        }
        null != h.componentWillUpdate && h.componentWillUpdate(m, h.__s, C), null != h.componentDidUpdate && h.__h.push(function () {
          h.componentDidUpdate(p, _, g);
        });
      }
      if (h.context = C, h.props = m, h.__P = n, h.__e = !1, S = l$1.__r, I = 0, "prototype" in A && A.prototype.render) {
        for (h.state = h.__s, h.__d = !1, S && S(u), a = h.render(h.props, h.state, h.context), H = 0; H < h._sb.length; H++) h.__h.push(h._sb[H]);
        h._sb = [];
      } else do {
        h.__d = !1, S && S(u), a = h.render(h.props, h.state, h.context), h.state = h.__s;
      } while (h.__d && ++I < 25);
      h.state = h.__s, null != h.getChildContext && (i = d$1(d$1({}, i), h.getChildContext())), v || null == h.getSnapshotBeforeUpdate || (g = h.getSnapshotBeforeUpdate(p, _)), $(n, y$1(T = null != a && a.type === w$1 && null == a.key ? a.props.children : a) ? T : [T], u, t, i, o, r, f, e, c, s), h.base = u.__e, u.__u &= -161, h.__h.length && f.push(h), b && (h.__E = h.__ = null);
    } catch (n) {
      u.__v = null, c || null != r ? (u.__e = e, u.__u |= c ? 160 : 32, r[r.indexOf(e)] = null) : (u.__e = t.__e, u.__k = t.__k), l$1.__e(n, u, t);
    } else null == r && u.__v === t.__v ? (u.__k = t.__k, u.__e = t.__e) : u.__e = z$1(t.__e, u, t, i, o, r, f, c, s);
    (a = l$1.diffed) && a(u);
  }
  function j$1(n, u, t) {
    u.__d = void 0;
    for (var i = 0; i < t.length; i++) N(t[i], t[++i], t[++i]);
    l$1.__c && l$1.__c(u, n), n.some(function (u) {
      try {
        n = u.__h, u.__h = [], n.some(function (n) {
          n.call(u);
        });
      } catch (n) {
        l$1.__e(n, u.__v);
      }
    });
  }
  function z$1(l, u, t, i, o, r, f, e, c) {
    var s,
      a,
      v,
      p,
      d,
      g,
      b,
      m = t.props,
      w = u.props,
      k = u.type;
    if ("svg" === k && (o = !0), null != r) for (s = 0; s < r.length; s++) if ((d = r[s]) && "setAttribute" in d == !!k && (k ? d.localName === k : 3 === d.nodeType)) {
      l = d, r[s] = null;
      break;
    }
    if (null == l) {
      if (null === k) return document.createTextNode(w);
      l = o ? document.createElementNS("http://www.w3.org/2000/svg", k) : document.createElement(k, w.is && w), r = null, e = !1;
    }
    if (null === k) m === w || e && l.data === w || (l.data = w);else {
      if (r = r && n$1.call(l.childNodes), m = t.props || h$1, !e && null != r) for (m = {}, s = 0; s < l.attributes.length; s++) m[(d = l.attributes[s]).name] = d.value;
      for (s in m) d = m[s], "children" == s || ("dangerouslySetInnerHTML" == s ? v = d : "key" === s || s in w || L(l, s, null, d, o));
      for (s in w) d = w[s], "children" == s ? p = d : "dangerouslySetInnerHTML" == s ? a = d : "value" == s ? g = d : "checked" == s ? b = d : "key" === s || e && "function" != typeof d || m[s] === d || L(l, s, d, m[s], o);
      if (a) e || v && (a.__html === v.__html || a.__html === l.innerHTML) || (l.innerHTML = a.__html), u.__k = [];else if (v && (l.innerHTML = ""), $(l, y$1(p) ? p : [p], u, t, i, o && "foreignObject" !== k, r, f, r ? r[0] : t.__k && x$1(t, 0), e, c), null != r) for (s = r.length; s--;) null != r[s] && _$1(r[s]);
      e || (s = "value", void 0 !== g && (g !== l[s] || "progress" === k && !g || "option" === k && g !== m[s]) && L(l, s, g, m[s], !1), s = "checked", void 0 !== b && b !== l[s] && L(l, s, b, m[s], !1));
    }
    return l;
  }
  function N(n, u, t) {
    try {
      "function" == typeof n ? n(u) : n.current = u;
    } catch (n) {
      l$1.__e(n, t);
    }
  }
  function q$1(n, u, t) {
    var i, o;
    if (l$1.unmount && l$1.unmount(n), (i = n.ref) && (i.current && i.current !== n.__e || N(i, null, u)), null != (i = n.__c)) {
      if (i.componentWillUnmount) try {
        i.componentWillUnmount();
      } catch (n) {
        l$1.__e(n, u);
      }
      i.base = i.__P = null;
    }
    if (i = n.__k) for (o = 0; o < i.length; o++) i[o] && q$1(i[o], u, t || "function" != typeof n.type);
    t || null == n.__e || _$1(n.__e), n.__c = n.__ = n.__e = n.__d = void 0;
  }
  function B$1(n, l, u) {
    return this.constructor(n, u);
  }
  function D$1(u, t, i) {
    var o, r, f, e;
    l$1.__ && l$1.__(u, t), r = (o = "function" == typeof i) ? null : t.__k, f = [], e = [], O(t, u = (!o && i || t).__k = g(w$1, null, [u]), r || h$1, h$1, void 0 !== t.ownerSVGElement, !o && i ? [i] : r ? null : t.firstChild ? n$1.call(t.childNodes) : null, f, !o && i ? i : r ? r.__e : t.firstChild, o, e), j$1(f, u, e);
  }
  n$1 = v$1.slice, l$1 = {
    __e: function (n, l, u, t) {
      for (var i, o, r; l = l.__;) if ((i = l.__c) && !i.__) try {
        if ((o = i.constructor) && null != o.getDerivedStateFromError && (i.setState(o.getDerivedStateFromError(n)), r = i.__d), null != i.componentDidCatch && (i.componentDidCatch(n, t || {}), r = i.__d), r) return i.__E = i;
      } catch (l) {
        n = l;
      }
      throw n;
    }
  }, u$1 = 0, k$1.prototype.setState = function (n, l) {
    var u;
    u = null != this.__s && this.__s !== this.state ? this.__s : this.__s = d$1({}, this.state), "function" == typeof n && (n = n(d$1({}, u), this.props)), n && d$1(u, n), null != n && this.__v && (l && this._sb.push(l), P(this));
  }, k$1.prototype.forceUpdate = function (n) {
    this.__v && (this.__e = !0, n && this.__h.push(n), P(this));
  }, k$1.prototype.render = w$1, i$2 = [], r$2 = "function" == typeof Promise ? Promise.prototype.then.bind(Promise.resolve()) : setTimeout, f$1 = function (n, l) {
    return n.__v.__b - l.__v.__b;
  }, S.__r = 0, e$2 = 0, c$1 = M(!1), s$1 = M(!0);

  var n = function (t, s, r, e) {
      var u;
      s[0] = 0;
      for (var h = 1; h < s.length; h++) {
        var p = s[h++],
          a = s[h] ? (s[0] |= p ? 1 : 2, r[s[h++]]) : s[++h];
        3 === p ? e[0] = a : 4 === p ? e[1] = Object.assign(e[1] || {}, a) : 5 === p ? (e[1] = e[1] || {})[s[++h]] = a : 6 === p ? e[1][s[++h]] += a + "" : p ? (u = t.apply(a, n(t, a, r, ["", null])), e.push(u), a[0] ? s[0] |= 2 : (s[h - 2] = 0, s[h] = u)) : e.push(a);
      }
      return e;
    },
    t$2 = new Map();
  function e$1 (s) {
    var r = t$2.get(this);
    return r || (r = new Map(), t$2.set(this, r)), (r = n(this, r.get(s) || (r.set(s, r = function (n) {
      for (var t, s, r = 1, e = "", u = "", h = [0], p = function (n) {
          1 === r && (n || (e = e.replace(/^\s*\n\s*|\s*\n\s*$/g, ""))) ? h.push(0, n, e) : 3 === r && (n || e) ? (h.push(3, n, e), r = 2) : 2 === r && "..." === e && n ? h.push(4, n, 0) : 2 === r && e && !n ? h.push(5, 0, !0, e) : r >= 5 && ((e || !n && 5 === r) && (h.push(r, 0, e, s), r = 6), n && (h.push(r, n, 0, s), r = 6)), e = "";
        }, a = 0; a < n.length; a++) {
        a && (1 === r && p(), p(a));
        for (var l = 0; l < n[a].length; l++) t = n[a][l], 1 === r ? "<" === t ? (p(), h = [h], r = 3) : e += t : 4 === r ? "--" === e && ">" === t ? (r = 1, e = "") : e = t + e[0] : u ? t === u ? u = "" : e += t : '"' === t || "'" === t ? u = t : ">" === t ? (p(), r = 1) : r && ("=" === t ? (r = 5, s = e, e = "") : "/" === t && (r < 5 || ">" === n[a][l + 1]) ? (p(), 3 === r && (h = h[0]), r = h, (h = h[0]).push(2, 0, r), r = 0) : " " === t || "\t" === t || "\n" === t || "\r" === t ? (p(), r = 2) : e += t), 3 === r && "!--" === e && (r = 4, h = h[0]);
      }
      return p(), h;
    }(s)), r), arguments, [])).length > 1 ? r : r[0];
  }

  var m$1 = e$1.bind(g);

  var t$1,
    r$1,
    u,
    i$1,
    o = 0,
    f = [],
    c = [],
    e = l$1,
    a = e.__b,
    v = e.__r,
    l = e.diffed,
    m = e.__c,
    s = e.unmount,
    d = e.__;
  function h(n, t) {
    e.__h && e.__h(r$1, n, o || t), o = 0;
    var u = r$1.__H || (r$1.__H = {
      __: [],
      __h: []
    });
    return n >= u.__.length && u.__.push({
      __V: c
    }), u.__[n];
  }
  function p(n) {
    return o = 1, y(D, n);
  }
  function y(n, u, i) {
    var o = h(t$1++, 2);
    if (o.t = n, !o.__c && (o.__ = [D(void 0, u), function (n) {
      var t = o.__N ? o.__N[0] : o.__[0],
        r = o.t(t, n);
      t !== r && (o.__N = [r, o.__[1]], o.__c.setState({}));
    }], o.__c = r$1, !r$1.u)) {
      var f = function (n, t, r) {
        if (!o.__c.__H) return !0;
        var u = o.__c.__H.__.filter(function (n) {
          return !!n.__c;
        });
        if (u.every(function (n) {
          return !n.__N;
        })) return !c || c.call(this, n, t, r);
        var i = !1;
        return u.forEach(function (n) {
          if (n.__N) {
            var t = n.__[0];
            n.__ = n.__N, n.__N = void 0, t !== n.__[0] && (i = !0);
          }
        }), !(!i && o.__c.props === n) && (!c || c.call(this, n, t, r));
      };
      r$1.u = !0;
      var c = r$1.shouldComponentUpdate,
        e = r$1.componentWillUpdate;
      r$1.componentWillUpdate = function (n, t, r) {
        if (this.__e) {
          var u = c;
          c = void 0, f(n, t, r), c = u;
        }
        e && e.call(this, n, t, r);
      }, r$1.shouldComponentUpdate = f;
    }
    return o.__N || o.__;
  }
  function _(n, u) {
    var i = h(t$1++, 3);
    !e.__s && C$1(i.__H, u) && (i.__ = n, i.i = u, r$1.__H.__h.push(i));
  }
  function A(n, u) {
    var i = h(t$1++, 4);
    !e.__s && C$1(i.__H, u) && (i.__ = n, i.i = u, r$1.__h.push(i));
  }
  function F(n) {
    return o = 5, q(function () {
      return {
        current: n
      };
    }, []);
  }
  function q(n, r) {
    var u = h(t$1++, 7);
    return C$1(u.__H, r) ? (u.__V = n(), u.i = r, u.__h = n, u.__V) : u.__;
  }
  function x(n, t) {
    return o = 8, q(function () {
      return n;
    }, t);
  }
  function j() {
    for (var n; n = f.shift();) if (n.__P && n.__H) try {
      n.__H.__h.forEach(z), n.__H.__h.forEach(B), n.__H.__h = [];
    } catch (t) {
      n.__H.__h = [], e.__e(t, n.__v);
    }
  }
  e.__b = function (n) {
    r$1 = null, a && a(n);
  }, e.__ = function (n, t) {
    n && t.__k && t.__k.__m && (n.__m = t.__k.__m), d && d(n, t);
  }, e.__r = function (n) {
    v && v(n), t$1 = 0;
    var i = (r$1 = n.__c).__H;
    i && (u === r$1 ? (i.__h = [], r$1.__h = [], i.__.forEach(function (n) {
      n.__N && (n.__ = n.__N), n.__V = c, n.__N = n.i = void 0;
    })) : (i.__h.forEach(z), i.__h.forEach(B), i.__h = [], t$1 = 0)), u = r$1;
  }, e.diffed = function (n) {
    l && l(n);
    var t = n.__c;
    t && t.__H && (t.__H.__h.length && (1 !== f.push(t) && i$1 === e.requestAnimationFrame || ((i$1 = e.requestAnimationFrame) || w)(j)), t.__H.__.forEach(function (n) {
      n.i && (n.__H = n.i), n.__V !== c && (n.__ = n.__V), n.i = void 0, n.__V = c;
    })), u = r$1 = null;
  }, e.__c = function (n, t) {
    t.some(function (n) {
      try {
        n.__h.forEach(z), n.__h = n.__h.filter(function (n) {
          return !n.__ || B(n);
        });
      } catch (r) {
        t.some(function (n) {
          n.__h && (n.__h = []);
        }), t = [], e.__e(r, n.__v);
      }
    }), m && m(n, t);
  }, e.unmount = function (n) {
    s && s(n);
    var t,
      r = n.__c;
    r && r.__H && (r.__H.__.forEach(function (n) {
      try {
        z(n);
      } catch (n) {
        t = n;
      }
    }), r.__H = void 0, t && e.__e(t, r.__v));
  };
  var k = "function" == typeof requestAnimationFrame;
  function w(n) {
    var t,
      r = function () {
        clearTimeout(u), k && cancelAnimationFrame(t), setTimeout(n);
      },
      u = setTimeout(r, 100);
    k && (t = requestAnimationFrame(r));
  }
  function z(n) {
    var t = r$1,
      u = n.__c;
    "function" == typeof u && (n.__c = void 0, u()), r$1 = t;
  }
  function B(n) {
    var t = r$1;
    n.__c = n.__(), r$1 = t;
  }
  function C$1(n, t) {
    return !n || n.length !== t.length || t.some(function (t, r) {
      return t !== n[r];
    });
  }
  function D(n, t) {
    return "function" == typeof t ? t(n) : t;
  }

  function r(e) {
    var t,
      f,
      n = "";
    if ("string" == typeof e || "number" == typeof e) n += e;else if ("object" == typeof e) if (Array.isArray(e)) {
      var o = e.length;
      for (t = 0; t < o; t++) e[t] && (f = r(e[t])) && (n && (n += " "), n += f);
    } else for (f in e) e[f] && (n && (n += " "), n += f);
    return n;
  }
  function clsx() {
    for (var e, t, f = 0, n = "", o = arguments.length; f < o; f++) (e = arguments[f]) && (t = r(e)) && (n && (n += " "), n += t);
    return n;
  }

  /**
   * @typedef {import('./PopupMenuProvider').PopupMenuHeaderEntry} PopupMenuHeaderEntry
   */

  /**
   * Component that renders a popup menu header.
   *
   * @param {Object} props
   * @param {PopupMenuHeaderEntry[]} props.headerEntries
   * @param {PopupMenuHeaderEntry} props.selectedEntry
   * @param {(event: MouseEvent, entry: PopupMenuHeaderEntry) => void} props.onSelect
   * @param {(entry: PopupMenuHeaderEntry | null) => void} props.setSelectedEntry
   * @param {string} props.title
   */
  function PopupMenuHeader(props) {
    const {
      headerEntries,
      onSelect,
      selectedEntry,
      setSelectedEntry,
      title
    } = props;
    const groups = q(() => groupEntries$1(headerEntries), [headerEntries]);
    return m$1`
    <div class="djs-popup-header">
      <h3 class="djs-popup-title" title=${title}>${title}</h3>
      ${groups.map(group => m$1`
        <ul key=${group.id} class="djs-popup-header-group" data-header-group=${group.id}>

          ${group.entries.map(entry => m$1`
            <li key=${entry.id}>
              <${entry.action ? 'button' : 'span'}
                class=${getHeaderClasses(entry, entry === selectedEntry)}
                onClick=${event => entry.action && onSelect(event, entry)}
                title=${entry.title || entry.label}
                data-id=${entry.id}
                onMouseEnter=${() => entry.action && setSelectedEntry(entry)}
                onMouseLeave=${() => entry.action && setSelectedEntry(null)}
                onFocus=${() => entry.action && setSelectedEntry(entry)}
                onBlur=${() => entry.action && setSelectedEntry(null)}
              >
                ${entry.imageUrl && m$1`<img class="djs-popup-entry-icon" src=${entry.imageUrl} alt="" />` || entry.imageHtml && m$1`<div class="djs-popup-entry-icon" dangerouslySetInnerHTML=${{
    __html: entry.imageHtml
  }} />`}
                ${entry.label ? m$1`
                  <span class="djs-popup-label">${entry.label}</span>
                ` : null}
              </${entry.action ? 'button' : 'span'}>
            </li>
          `)}
        </ul>
      `)}
    </div>
  `;
  }

  // helpers
  function groupEntries$1(entries) {
    return entries.reduce((groups, entry) => {
      const groupId = entry.group || 'default';
      const group = groups.find(group => group.id === groupId);
      if (group) {
        group.entries.push(entry);
      } else {
        groups.push({
          id: groupId,
          entries: [entry]
        });
      }
      return groups;
    }, []);
  }
  function getHeaderClasses(entry, selected) {
    return clsx('entry', entry.className, entry.active ? 'active' : '', entry.disabled ? 'disabled' : '', selected ? 'selected' : '');
  }

  /**
   * @typedef {import('./PopupMenuProvider').PopupMenuEntry} PopupMenuEntry
   */

  /**
   * Component that renders a popup menu entry.
   *
   * @param {Object} props
   * @param {string} props.key
   * @param {PopupMenuEntry} props.entry
   * @param {boolean} props.selected
   * @param {(event: MouseEvent) => void} props.onMouseEnter
   * @param {(event: MouseEvent) => void} props.onMouseLeave
   * @param {(event: MouseEvent, entry?: PopupMenuEntry, action?: string) => void} props.onAction
   */
  function PopupMenuItem(props) {
    const {
      entry,
      selected,
      onMouseEnter,
      onMouseLeave,
      onAction
    } = props;
    return m$1`
    <li
      class=${clsx('entry', {
    selected
  })}
      data-id=${entry.id}
      title=${entry.title || entry.label}
      tabIndex="0"
      onClick=${onAction}
      onFocus=${onMouseEnter}
      onBlur=${onMouseLeave}
      onMouseEnter=${onMouseEnter}
      onMouseLeave=${onMouseLeave}
      onDragStart=${event => onAction(event, entry, 'dragstart')}
      draggable=${true}
    >
      <div class="djs-popup-entry-content">
        <span
          class=${clsx('djs-popup-entry-name', entry.className)}
        >
          ${entry.imageUrl && m$1`<img class="djs-popup-entry-icon" src=${entry.imageUrl} alt="" />` || entry.imageHtml && m$1`<div class="djs-popup-entry-icon" dangerouslySetInnerHTML=${{
    __html: entry.imageHtml
  }} />`}

          ${entry.label ? m$1`
            <span class="djs-popup-label">
              ${entry.label}
            </span>
          ` : null}
        </span>
        ${entry.description && m$1`
          <span
            class="djs-popup-entry-description"
            title=${entry.description}
          >
            ${entry.description}
          </span>
        `}
      </div>
      ${entry.documentationRef && m$1`
        <div class="djs-popup-entry-docs">
          <a
            href="${entry.documentationRef}"
            onClick=${event => event.stopPropagation()}
            title="Open element documentation"
            target="_blank"
            rel="noopener"
          >
            <svg width="12" height="12" viewBox="0 0 12 12" fill="none" xmlns="http://www.w3.org/2000/svg">
              <path fill-rule="evenodd" clip-rule="evenodd" d="M10.6368 10.6375V5.91761H11.9995V10.6382C11.9995 10.9973 11.8623 11.3141 11.5878 11.5885C11.3134 11.863 10.9966 12.0002 10.6375 12.0002H1.36266C0.982345 12.0002 0.660159 11.8681 0.396102 11.6041C0.132044 11.34 1.52588e-05 11.0178 1.52588e-05 10.6375V1.36267C1.52588e-05 0.98236 0.132044 0.660173 0.396102 0.396116C0.660159 0.132058 0.982345 2.95639e-05 1.36266 2.95639e-05H5.91624V1.36267H1.36266V10.6375H10.6368ZM12 0H7.2794L7.27873 1.36197H9.68701L3.06507 7.98391L4.01541 8.93425L10.6373 2.31231V4.72059H12V0Z" fill="#818798"/>
            </svg>
          </a>
        </div>
      `}
    </li>
  `;
  }

  /**
   * @typedef {import('./PopupMenuProvider').PopupMenuEntry} PopupMenuEntry
   */

  /**
   * Component that renders a popup menu entry list.
   *
   * @param {Object} props
   * @param {PopupMenuEntry[]} props.entries
   * @param {PopupMenuEntry} props.selectedEntry
   * @param {(entry: PopupMenuEntry | null) => void} props.setSelectedEntry
   */
  function PopupMenuList(props) {
    const {
      selectedEntry,
      setSelectedEntry,
      entries,
      ...restProps
    } = props;
    const resultsRef = F();
    const groups = q(() => groupEntries(entries), [entries]);

    // scroll to selected result
    A(() => {
      const containerEl = resultsRef.current;
      if (!containerEl) return;
      const selectedEl = containerEl.querySelector('.selected');
      if (selectedEl) {
        scrollIntoView$2(selectedEl);
      }
    }, [selectedEntry]);
    return m$1`
    <div class="djs-popup-results" ref=${resultsRef}>
      ${groups.map(group => m$1`
        ${group.name && m$1`
          <div key=${group.id} class="entry-header" title=${group.name}>
            ${group.name}
          </div>
        `}
        <ul class="djs-popup-group" data-group=${group.id}>
          ${group.entries.map(entry => m$1`
            <${PopupMenuItem}
              key=${entry.id}
              entry=${entry}
              selected=${entry === selectedEntry}
              onMouseEnter=${() => setSelectedEntry(entry)}
              onMouseLeave=${() => setSelectedEntry(null)}
              ...${restProps}
            />
          `)}
        </ul>
      `)}
    </div>
  `;
  }

  // helpers
  function groupEntries(entries) {
    const groups = [];
    const getGroup = group => groups.find(elem => group.id === elem.id);
    const containsGroup = group => !!getGroup(group);

    // legacy support for provider built for the old popUp menu
    const formatGroup = group => typeof group === 'string' ? {
      id: group
    } : group;
    entries.forEach(entry => {
      // assume a default group when none is provided
      const group = entry.group ? formatGroup(entry.group) : {
        id: 'default'
      };
      if (!containsGroup(group)) {
        groups.push({
          ...group,
          entries: [entry]
        });
      } else {
        getGroup(group).entries.push(entry);
      }
    });
    return groups;
  }

  // helpers ////////////////

  function scrollIntoView$2(el) {
    if (typeof el.scrollIntoViewIfNeeded === 'function') {
      el.scrollIntoViewIfNeeded();
    } else {
      el.scrollIntoView({
        scrollMode: 'if-needed',
        block: 'nearest'
      });
    }
  }

  /**
   * @typedef {import('./PopupMenuProvider').PopupMenuEntry} PopupMenuEntry
   * @typedef {import('./PopupMenuProvider').PopupMenuHeaderEntry} PopupMenuHeaderEntry
   * @typedef {import('./PopupMenuProvider').PopupMenuEmptyPlaceholderProvider | import('./PopupMenuProvider').PopupMenuEmptyPlaceholder} PopupMenuEmptyPlaceholder
   *
   * @typedef {import('../search/search').default} search
   *
   * @typedef {import('../../util/Types').Point} Point
   */

  /**
   * A component that renders the popup menus.
   *
   * @param {Object} props
   * @param {() => void} props.onClose
   * @param {() => void} props.onSelect
   * @param {(element: HTMLElement) => Point} props.position
   * @param {string} props.className
   * @param {PopupMenuEntry[]} props.entries
   * @param {PopupMenuHeaderEntry[]} props.headerEntries
   * @param {number} props.scale
   * @param {string} [props.title]
   * @param {boolean} [props.search]
   * @param {PopupMenuEmptyPlaceholder} [props.emptyPlaceholder]
   * @param {number} [props.width]
   * @param {search} props.searchFn
   */
  function PopupMenuComponent(props) {
    const {
      onClose,
      onSelect,
      className,
      headerEntries,
      position,
      title,
      width,
      scale,
      search,
      emptyPlaceholder,
      searchFn,
      entries: originalEntries,
      onOpened,
      onClosed
    } = props;
    const searchable = q(() => {
      if (!isDefined(search)) {
        return false;
      }
      return originalEntries.length > 5;
    }, [search, originalEntries]);
    const [value, setValue] = p('');
    const filterEntries = x((originalEntries, value) => {
      if (!searchable) {
        return originalEntries;
      }
      if (!value) {
        return originalEntries.filter(({
          rank = 0
        }) => rank >= 0);
      }
      const searchableEntries = originalEntries.filter(({
        searchable
      }) => searchable !== false);
      return searchFn(searchableEntries, value, {
        keys: ['label', 'description', 'search']
      }).map(({
        item
      }) => item);
    }, [searchable]);
    const [entries, setEntries] = p(filterEntries(originalEntries, value));
    const [selectedEntry, setSelectedEntry] = p(entries[0]);
    const updateEntries = x(newEntries => {
      // select first entry if non is selected
      if (!selectedEntry || !newEntries.includes(selectedEntry)) {
        setSelectedEntry(newEntries[0]);
      }
      setEntries(newEntries);
    }, [selectedEntry, setEntries, setSelectedEntry]);

    // filter entries on value change
    _(() => {
      updateEntries(filterEntries(originalEntries, value));
    }, [value, originalEntries]);

    // handle keyboard seleciton
    const keyboardSelect = x(direction => {
      const idx = entries.indexOf(selectedEntry);
      let nextIdx = idx + direction;
      if (nextIdx < 0) {
        nextIdx = entries.length - 1;
      }
      if (nextIdx >= entries.length) {
        nextIdx = 0;
      }
      setSelectedEntry(entries[nextIdx]);
    }, [entries, selectedEntry, setSelectedEntry]);
    const handleKeyDown = x(event => {
      if (event.key === 'Enter' && selectedEntry) {
        return onSelect(event, selectedEntry);
      }

      // ARROW_UP
      if (event.key === 'ArrowUp') {
        keyboardSelect(-1);
        return event.preventDefault();
      }

      // ARROW_DOWN
      if (event.key === 'ArrowDown') {
        keyboardSelect(1);
        return event.preventDefault();
      }
    }, [onSelect, selectedEntry, keyboardSelect]);
    const handleKey = x(event => {
      if (matches(event.target, 'input')) {
        setValue(() => event.target.value);
      }
    }, [setValue]);
    _(() => {
      onOpened();
      return () => {
        onClosed();
      };
    }, []);
    const displayHeader = q(() => title || headerEntries.length > 0, [title, headerEntries]);
    return m$1`
    <${PopupMenuWrapper}
      onClose=${onClose}
      onKeyup=${handleKey}
      onKeydown=${handleKeyDown}
      className=${className}
      position=${position}
      width=${width}
      scale=${scale}
    >
      ${displayHeader && m$1`
        <${PopupMenuHeader}
          headerEntries=${headerEntries}
          onSelect=${onSelect}
          selectedEntry=${selectedEntry}
          setSelectedEntry=${setSelectedEntry}
          title=${title}
        />
      `}
      ${originalEntries.length > 0 && m$1`
        <div class="djs-popup-body">

          ${searchable && m$1`
          <div class="djs-popup-search">
            <svg class="djs-popup-search-icon" width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg">
              <path fill-rule="evenodd" clip-rule="evenodd" d="M9.0325 8.5H9.625L13.3675 12.25L12.25 13.3675L8.5 9.625V9.0325L8.2975 8.8225C7.4425 9.5575 6.3325 10 5.125 10C2.4325 10 0.25 7.8175 0.25 5.125C0.25 2.4325 2.4325 0.25 5.125 0.25C7.8175 0.25 10 2.4325 10 5.125C10 6.3325 9.5575 7.4425 8.8225 8.2975L9.0325 8.5ZM1.75 5.125C1.75 6.9925 3.2575 8.5 5.125 8.5C6.9925 8.5 8.5 6.9925 8.5 5.125C8.5 3.2575 6.9925 1.75 5.125 1.75C3.2575 1.75 1.75 3.2575 1.75 5.125Z" fill="#22242A"/>
            </svg>
            <input type="text" spellcheck=${false} aria-label="${title}" />
          </div>
          `}

          <${PopupMenuList}
            entries=${entries}
            selectedEntry=${selectedEntry}
            setSelectedEntry=${setSelectedEntry}
            onAction=${onSelect}
          />
        </div>
      `}
    ${emptyPlaceholder && entries.length === 0 && m$1`
      <div class="djs-popup-no-results">${isFunction$2(emptyPlaceholder) ? emptyPlaceholder(value) : emptyPlaceholder}</div>
    `}
    </${PopupMenuWrapper}>
  `;
  }

  /**
   * A component that wraps the popup menu.
   *
   * @param {*} props
   */
  function PopupMenuWrapper(props) {
    const {
      onClose,
      onKeydown,
      onKeyup,
      className,
      children,
      position: positionGetter
    } = props;
    const popupRef = F();

    // initial position
    A(() => {
      if (typeof positionGetter !== 'function') {
        return;
      }
      const popupEl = popupRef.current;
      const position = positionGetter(popupEl);
      popupEl.style.left = `${position.x}px`;
      popupEl.style.top = `${position.y}px`;
    }, [popupRef.current, positionGetter]);

    // initial focus
    A(() => {
      const popupEl = popupRef.current;
      if (!popupEl) {
        return;
      }
      const inputEl = popupEl.querySelector('input');
      (inputEl || popupEl).focus();
    }, []);

    // global <Escape> / blur handlers
    _(() => {
      const handleKeyDown = event => {
        if (event.key === 'Escape') {
          event.preventDefault();
          return onClose();
        }
      };
      const handleClick = event => {
        const popup = closest(event.target, '.djs-popup', true);
        if (popup) {
          return;
        }
        return onClose();
      };
      document.documentElement.addEventListener('keydown', handleKeyDown);
      document.body.addEventListener('click', handleClick);
      return () => {
        document.documentElement.removeEventListener('keydown', handleKeyDown);
        document.body.removeEventListener('click', handleClick);
      };
    }, []);
    return m$1`
    <div
      class=${clsx('djs-popup', className)}
      style=${getPopupStyle(props)}
      onKeydown=${onKeydown}
      onKeyup=${onKeyup}
      ref=${popupRef}
      tabIndex="-1"
    >
      ${children}
    </div>
  `;
  }

  // helpers //////////////////////

  function getPopupStyle(props) {
    return {
      transform: `scale(${props.scale})`,
      width: `${props.width}px`,
      'transform-origin': 'top left'
    };
  }

  /**
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('../search/search').default} search
   *
   * @typedef {import('../../util/Types').Point} Point
   *
   * @typedef {import('./PopupMenuProvider').PopupMenuEntries} PopupMenuEntries
   * @typedef {import('./PopupMenuProvider').PopupMenuEntry} PopupMenuEntry
   * @typedef {import('./PopupMenuProvider').PopupMenuHeaderEntries} PopupMenuHeaderEntries
   * @typedef {import('./PopupMenuProvider').PopupMenuHeaderEntry} PopupMenuHeaderEntry
   * @typedef {import('./PopupMenuProvider').default} PopupMenuProvider
   *
   * @typedef {import('../../model/Types').Element} Element
   *
   * @typedef { {
   *   scale?: {
   *     min?: number;
   *     max?: number;
   *   } | boolean;
   * } } PopupMenuConfig
   *
   * @typedef {Element|Element[]} PopupMenuTarget;
   */

  var DATA_REF = 'data-id';
  var CLOSE_EVENTS = ['contextPad.close', 'canvas.viewbox.changing', 'commandStack.changed'];
  var DEFAULT_PRIORITY$6 = 1000;

  /**
   * A popup menu to show a number of actions on the canvas.
   *
   * @param {PopupMenuConfig} config
   * @param {EventBus} eventBus
   * @param {Canvas} canvas
   * @param {search} search
   */
  function PopupMenu$1(config, eventBus, canvas, search) {
    this._eventBus = eventBus;
    this._canvas = canvas;
    this._search = search;
    this._current = null;
    var scale = isDefined(config && config.scale) ? config.scale : {
      min: 1,
      max: 1
    };
    this._config = {
      scale: scale
    };
    eventBus.on('diagram.destroy', () => {
      this.close();
    });
    eventBus.on('element.changed', event => {
      const element = this.isOpen() && this._current.target;
      if (event.element === element) {
        this.refresh();
      }
    });
  }
  PopupMenu$1.$inject = ['config.popupMenu', 'eventBus', 'canvas', 'search'];
  PopupMenu$1.prototype._render = function () {
    const {
      position: _position,
      providerId: className,
      entries,
      headerEntries,
      emptyPlaceholder,
      options
    } = this._current;
    const entriesArray = Object.entries(entries).map(([key, value]) => ({
      id: key,
      ...value
    }));
    const headerEntriesArray = Object.entries(headerEntries).map(([key, value]) => ({
      id: key,
      ...value
    }));
    const position = _position && (container => this._ensureVisible(container, _position));
    const scale = this._updateScale(this._current.container);
    const onClose = result => this.close(result);
    const onSelect = (event, entry, action) => this.trigger(event, entry, action);
    D$1(m$1`
      <${PopupMenuComponent}
        onClose=${onClose}
        onSelect=${onSelect}
        position=${position}
        className=${className}
        entries=${entriesArray}
        headerEntries=${headerEntriesArray}
        emptyPlaceholder=${emptyPlaceholder}
        scale=${scale}
        onOpened=${this._onOpened.bind(this)}
        onClosed=${this._onClosed.bind(this)}
        searchFn=${this._search}
        ...${{
    ...options
  }}
      />
    `, this._current.container);
  };

  /**
   * Open the popup menu at the given position.
   *
   * @param {PopupMenuTarget} target
   * @param {string} providerId
   * @param {Point} position
   * @param {Object} [options]
   */
  PopupMenu$1.prototype.open = function (target, providerId, position, options) {
    if (!target) {
      throw new Error('target is missing');
    }
    if (!providerId) {
      throw new Error('providers for <' + providerId + '> not found');
    }
    if (!position) {
      throw new Error('position is missing');
    }
    if (this.isOpen()) {
      this.close();
    }
    const {
      entries,
      headerEntries,
      emptyPlaceholder
    } = this._getContext(target, providerId);
    this._current = {
      position,
      providerId,
      target,
      entries,
      headerEntries,
      emptyPlaceholder,
      container: this._createContainer({
        provider: providerId
      }),
      options
    };
    this._emit('open');
    this._bindAutoClose();
    this._render();
  };

  /**
   * Refresh the popup menu entries without changing the target or position.
   */
  PopupMenu$1.prototype.refresh = function () {
    if (!this.isOpen()) {
      return;
    }
    const {
      target,
      providerId
    } = this._current;
    const {
      entries,
      headerEntries,
      emptyPlaceholder
    } = this._getContext(target, providerId);
    this._current = {
      ...this._current,
      entries,
      headerEntries,
      emptyPlaceholder
    };
    this._emit('refresh');
    this._render();
  };
  PopupMenu$1.prototype._getContext = function (target, provider) {
    const providers = this._getProviders(provider);
    if (!providers || !providers.length) {
      throw new Error('provider for <' + provider + '> not found');
    }
    const entries = this._getEntries(target, providers);
    const headerEntries = this._getHeaderEntries(target, providers);
    const emptyPlaceholder = this._getEmptyPlaceholder(providers);
    return {
      entries,
      headerEntries,
      emptyPlaceholder,
      empty: !(Object.keys(entries).length || Object.keys(headerEntries).length)
    };
  };
  PopupMenu$1.prototype.close = function () {
    if (!this.isOpen()) {
      return;
    }
    this._emit('close');
    this.reset();
    this._canvas.restoreFocus();
    this._current = null;
  };
  PopupMenu$1.prototype.reset = function () {
    const container = this._current.container;
    D$1(null, container);
    remove$4(container);
  };
  PopupMenu$1.prototype._emit = function (event, payload) {
    this._eventBus.fire(`popupMenu.${event}`, payload);
  };
  PopupMenu$1.prototype._onOpened = function () {
    this._emit('opened');
  };
  PopupMenu$1.prototype._onClosed = function () {
    this._emit('closed');
  };
  PopupMenu$1.prototype._createContainer = function (config) {
    var canvas = this._canvas,
      parent = canvas.getContainer();
    const container = domify$1(`<div class="djs-popup-parent djs-scrollable" data-popup=${config.provider}></div>`);
    parent.appendChild(container);
    return container;
  };

  /**
   * Set up listener to close popup automatically on certain events.
   */
  PopupMenu$1.prototype._bindAutoClose = function () {
    this._eventBus.once(CLOSE_EVENTS, this.close, this);
  };

  /**
   * Remove the auto-closing listener.
  */
  PopupMenu$1.prototype._unbindAutoClose = function () {
    this._eventBus.off(CLOSE_EVENTS, this.close, this);
  };

  /**
   * Updates popup style.transform with respect to the config and zoom level.
   *
   * @return {number}
   */
  PopupMenu$1.prototype._updateScale = function () {
    var zoom = this._canvas.zoom();
    var scaleConfig = this._config.scale,
      minScale,
      maxScale,
      scale = zoom;
    if (scaleConfig !== true) {
      if (scaleConfig === false) {
        minScale = 1;
        maxScale = 1;
      } else {
        minScale = scaleConfig.min;
        maxScale = scaleConfig.max;
      }
      if (isDefined(minScale) && zoom < minScale) {
        scale = minScale;
      }
      if (isDefined(maxScale) && zoom > maxScale) {
        scale = maxScale;
      }
    }
    return scale;
  };
  PopupMenu$1.prototype._ensureVisible = function (container, position) {
    var documentBounds = document.documentElement.getBoundingClientRect();
    var containerBounds = container.getBoundingClientRect();
    var overAxis = {},
      left = position.x,
      top = position.y;
    if (position.x + containerBounds.width > documentBounds.width) {
      overAxis.x = true;
    }
    if (position.y + containerBounds.height > documentBounds.height) {
      overAxis.y = true;
    }
    if (overAxis.x && overAxis.y) {
      left = position.x - containerBounds.width;
      top = position.y - containerBounds.height;
    } else if (overAxis.x) {
      left = position.x - containerBounds.width;
      top = position.y;
    } else if (overAxis.y && position.y < containerBounds.height) {
      left = position.x;
      top = 10;
    } else if (overAxis.y) {
      left = position.x;
      top = position.y - containerBounds.height;
    }

    // underAxis
    if (position.y < documentBounds.top) {
      top = position.y + containerBounds.height;
    }
    return {
      x: left,
      y: top
    };
  };

  /**
   * Check whether there are no popup menu providers or provided entries for the
   * given target.
   *
   * @param {PopupMenuTarget} target
   * @param {string} providerId
   *
   * @return {boolean}
   */
  PopupMenu$1.prototype.isEmpty = function (target, providerId) {
    if (!target) {
      throw new Error('target is missing');
    }
    if (!providerId) {
      throw new Error('provider ID is missing');
    }
    const providers = this._getProviders(providerId);
    if (!providers || !providers.length) {
      return true;
    }
    return this._getContext(target, providerId).empty;
  };

  /**
   * @overlord
   *
   * Register a popup menu provider with default priority. See
   * {@link PopupMenuProvider} for examples.
   *
   * @param {string} id
   * @param {PopupMenuProvider} provider
   */

  /**
   * Register a popup menu provider with the given priority. See
   * {@link PopupMenuProvider} for examples.
   *
   * @param {string} id
   * @param {number} priority
   * @param {PopupMenuProvider} provider
   */
  PopupMenu$1.prototype.registerProvider = function (id, priority, provider) {
    if (!provider) {
      provider = priority;
      priority = DEFAULT_PRIORITY$6;
    }
    this._eventBus.on('popupMenu.getProviders.' + id, priority, function (event) {
      event.providers.push(provider);
    });
  };

  /**
   * @param {string} id
   *
   * @return {PopupMenuProvider[]}
   */
  PopupMenu$1.prototype._getProviders = function (id) {
    var event = this._eventBus.createEvent({
      type: 'popupMenu.getProviders.' + id,
      providers: []
    });
    this._eventBus.fire(event);
    return event.providers;
  };

  /**
   * @param {PopupMenuTarget} target
   * @param {PopupMenuProvider[]} providers
   *
   * @return {PopupMenuEntries}
   */
  PopupMenu$1.prototype._getEntries = function (target, providers) {
    var entries = {};
    forEach$3(providers, function (provider) {
      // handle legacy method
      if (!provider.getPopupMenuEntries) {
        forEach$3(provider.getEntries(target), function (entry) {
          var id = entry.id;
          if (!id) {
            throw new Error('entry ID is missing');
          }
          entries[id] = omit(entry, ['id']);
        });
        return;
      }
      var entriesOrUpdater = provider.getPopupMenuEntries(target);
      if (isFunction$2(entriesOrUpdater)) {
        entries = entriesOrUpdater(entries);
      } else {
        forEach$3(entriesOrUpdater, function (entry, id) {
          entries[id] = entry;
        });
      }
    });
    return entries;
  };

  /**
   * @param {PopupMenuTarget} target
   * @param {PopupMenuProvider[]} providers
   *
   * @return {PopupMenuHeaderEntries}
   */
  PopupMenu$1.prototype._getHeaderEntries = function (target, providers) {
    var entries = {};
    forEach$3(providers, function (provider) {
      // handle legacy method
      if (!provider.getPopupMenuHeaderEntries) {
        if (!provider.getHeaderEntries) {
          return;
        }
        forEach$3(provider.getHeaderEntries(target), function (entry) {
          var id = entry.id;
          if (!id) {
            throw new Error('entry ID is missing');
          }
          entries[id] = omit(entry, ['id']);
        });
        return;
      }
      var entriesOrUpdater = provider.getPopupMenuHeaderEntries(target);
      if (isFunction$2(entriesOrUpdater)) {
        entries = entriesOrUpdater(entries);
      } else {
        forEach$3(entriesOrUpdater, function (entry, id) {
          entries[id] = entry;
        });
      }
    });
    return entries;
  };
  PopupMenu$1.prototype._getEmptyPlaceholder = function (providers) {
    const provider = providers.find(provider => isFunction$2(provider.getEmptyPlaceholder));
    return provider && provider.getEmptyPlaceholder();
  };

  /**
   * Check if the popup menu is open.
   *
   * @return {boolean}
   */
  PopupMenu$1.prototype.isOpen = function () {
    return !!this._current;
  };

  /**
   * Trigger an action associated with an entry.
   *
   * @param {Event} event
   * @param {PopupMenuEntry} entry
   * @param {string} [action='click']
   *
   * @return {any}
   */
  PopupMenu$1.prototype.trigger = function (event, entry, action = 'click') {
    // silence other actions
    event.preventDefault();
    if (!entry) {
      let element = closest(event.delegateTarget || event.target, '.entry', true);
      let entryId = attr$1(element, DATA_REF);
      entry = {
        id: entryId,
        ...this._getEntry(entryId)
      };
    }
    const handler = entry.action;
    if (this._emit('trigger', {
      entry,
      event
    }) === false) {
      return;
    }
    if (isFunction$2(handler)) {
      if (action === 'click') {
        return handler(event, entry);
      }
    } else {
      if (handler[action]) {
        return handler[action](event, entry);
      }
    }
  };

  /**
   * Get the entry (entry or header entry) with the given ID.
   *
   * @param {string} entryId
   *
   * @return {PopupMenuEntry|PopupMenuHeaderEntry}
   */
  PopupMenu$1.prototype._getEntry = function (entryId) {
    var entry = this._current.entries[entryId] || this._current.headerEntries[entryId];
    if (!entry) {
      throw new Error('entry not found');
    }
    return entry;
  };

  /**
   * @typedef { {
   *   index: number;
   *   match: boolean;
   *   value: string;
   * } } Token
   *
   * @typedef {Token[]} Tokens
   */

  /**
   * @template R
   *
   * @typedef { {
   *   item: R,
   *   tokens: Record<string, Tokens>
   * } } SearchResult
   */

  /**
   * Search items by query.
   *
   * @template T
   *
   * @param {T[]} items
   * @param {string} pattern
   * @param { {
   *   keys: string[];
   * } } options
   *
   * @returns {SearchResult<T>[]}
   */
  function search(items, pattern, options) {
    const {
      keys
    } = options;
    const words = pattern.trim().toLowerCase().split(/\s+/);
    return items.flatMap(item => {
      const tokens = matchItem(item, words, keys);
      if (!tokens) {
        return [];
      }
      return {
        item,
        tokens
      };
    }).sort(createResultSorter(keys));
  }

  /**
   * Match an item and return tokens in case of a match.
   *
   * @param {Object} item
   * @param {string[]} words
   * @param {string[]} keys
   *
   * @returns {Record<string, Tokens>}
   */
  function matchItem(item, words, keys) {
    const {
      matchedWords,
      tokens
    } = keys.reduce((result, key) => {
      const string = item[key];
      const {
        tokens,
        matchedWords
      } = matchString(string, words);
      return {
        tokens: {
          ...result.tokens,
          [key]: tokens
        },
        matchedWords: {
          ...result.matchedWords,
          ...matchedWords
        }
      };
    }, {
      matchedWords: {},
      tokens: {}
    });

    // only return result if every word got matched
    if (Object.keys(matchedWords).length !== words.length) {
      return null;
    }
    return tokens;
  }

  /**
   * Get index of result in list of results.
   *
   * @param {string[]} keys
   *
   * @returns { (resultA: SearchResult, resultB: SearchResult) => number}
   */
  function createResultSorter(keys) {
    /**
     * @param {SearchResult} resultA
     * @param {SearchResult} resultB
     */
    return (resultA, resultB) => {
      for (const key of keys) {
        const tokenComparison = compareTokens(resultA.tokens[key], resultB.tokens[key]);
        if (tokenComparison !== 0) {
          return tokenComparison;
        }
        const stringComparison = compareStrings(resultA.item[key], resultB.item[key]);
        if (stringComparison !== 0) {
          return stringComparison;
        }

        // fall back to next key
        continue;
      }

      // eventually call equality
      return 0;
    };
  }

  /**
   * Compares two token arrays.
   *
   * @param {Token[]} [tokensA]
   * @param {Token[]} [tokensB]
   *
   * @returns {number}
   */
  function compareTokens(tokensA, tokensB) {
    return scoreTokens(tokensB) - scoreTokens(tokensA);
  }

  /**
   * @param { Token[] } tokens
   * @returns { number }
   */
  function scoreTokens(tokens) {
    return tokens.reduce((sum, token) => sum + scoreToken(token), 0);
  }

  /**
   * Score a token.
   *
   * @param { Token } token
   *
   * @returns { number }
   */
  function scoreToken(token) {
    if (!token.match) {
      return 0;
    }
    return token.start ? 1.37 : token.wordStart ? 1.13 : 1;
  }

  /**
   * Compares two strings.
   *
   * @param {string} [a = '']
   * @param {string} [b = '']
   *
   * @returns {number}
   */
  function compareStrings(a = '', b = '') {
    return a.localeCompare(b);
  }

  /**
   * Match a given string against a set of words,
   * and return the result.
   *
   * @param {string} string
   * @param {string[]} words
   *
   * @return { {
   *   tokens: Token[],
   *   matchedWords: Record<string, boolean>
   * } }
   */
  function matchString(string, words) {
    if (!string) {
      return {
        tokens: [],
        matchedWords: {}
      };
    }
    const tokens = [];
    const matchedWords = {};
    const regexpString = words.map(escapeRegexp).flatMap(str => ['(?<wordStart>\\b' + str + ')', str]).join('|');
    const regexp = new RegExp(regexpString, 'ig');
    let match;
    let lastIndex = 0;
    while (match = regexp.exec(string)) {
      const [value] = match;
      if (match.index > lastIndex) {
        // add previous token (NO match)
        tokens.push({
          value: string.slice(lastIndex, match.index),
          index: lastIndex
        });
      }

      // add current token (match)
      tokens.push({
        value,
        index: match.index,
        match: true,
        wordStart: !!match.groups.wordStart,
        start: match.index === 0
      });
      matchedWords[value.toLowerCase()] = true;
      lastIndex = match.index + value.length;
    }

    // add after token (NO match)
    if (lastIndex < string.length) {
      tokens.push({
        value: string.slice(lastIndex),
        index: lastIndex
      });
    }
    return {
      tokens,
      matchedWords
    };
  }
  function escapeRegexp(string) {
    return string.replace(/[/\-\\^$*+?.()|[\]{}]/g, '\\$&');
  }

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var Search = {
    search: ['value', search]
  };

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var DiagramPopupMenu = {
    __depends__: [Search],
    __init__: ['popupMenu'],
    popupMenu: ['type', PopupMenu$1]
  };

  /**
   * @typedef {import('diagram-js/lib/core/EventBus').default} EventBus
   * @typedef {import('../modeling/Modeling').default} Modeling
   *
   * @typedef {import('../../core/Types').ShapeLike} Shape
   */

  var round$3 = Math.round;

  /**
   * Service that allows replacing of elements.
   *
   * @param {Modeling} modeling
   * @param {EventBus} eventBus
   */
  function Replace$1(modeling, eventBus) {
    this._modeling = modeling;
    this._eventBus = eventBus;
  }
  Replace$1.$inject = ['modeling', 'eventBus'];

  /**
   * Replace an element.
   *
   * @param {Shape} oldElement The element to be replaced.
   * @param {Object} attrs Containing information about the new element, for
   * example the new bounds and type.
   * @param {Object} hints Custom hints that will be attached to the context. It
   * can be used to inject data that is needed in the command chain. For example
   * it could be used in eventbus.on('commandStack.shape.replace.postExecute') to
   * change shape attributes after shape creation.
   *
   * @return {Shape}
   */
  Replace$1.prototype.replaceElement = function (oldElement, attrs, hints) {
    if (oldElement.waypoints) {
      // TODO(nikku): we do not replace connections, yet
      return null;
    }
    var modeling = this._modeling;
    var eventBus = this._eventBus;
    eventBus.fire('replace.start', {
      element: oldElement,
      attrs,
      hints
    });
    var width = attrs.width || oldElement.width,
      height = attrs.height || oldElement.height,
      x = attrs.x || oldElement.x,
      y = attrs.y || oldElement.y,
      centerX = round$3(x + width / 2),
      centerY = round$3(y + height / 2);

    // modeling API requires center coordinates,
    // account for that when handling shape bounds

    var newElement = modeling.replaceShape(oldElement, assign$4({}, attrs, {
      x: centerX,
      y: centerY,
      width: width,
      height: height
    }), hints);
    eventBus.fire('replace.end', {
      element: oldElement,
      newElement,
      hints
    });
    return newElement;
  };

  function ReplaceSelectionBehavior(selection, eventBus) {
    eventBus.on('replace.end', 500, function (event) {
      const {
        newElement,
        hints = {}
      } = event;
      if (hints.select === false) {
        return;
      }
      selection.select(newElement);
    });
  }
  ReplaceSelectionBehavior.$inject = ['selection', 'eventBus'];

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var DiagramReplace = {
    __init__: ['replace', 'replaceSelectionBehavior'],
    replaceSelectionBehavior: ['type', ReplaceSelectionBehavior],
    replace: ['type', Replace$1]
  };

  /**
   * This module takes care of replacing DRD elements
   */
  function DrdReplace(drdFactory, replace, selection, modeling) {
    /**
     * Prepares a new business object for the replacement element
     * and triggers the replace operation.
     *
     * @param  {djs.model.Base} element
     * @param  {Object} target
     * @param  {Object} [hints]
     *
     * @return {djs.model.Base} the newly created element
     */
    function replaceElement(element, target, hints) {
      hints = hints || {};
      var type = target.type,
        oldBusinessObject = element.businessObject;
      var newBusinessObject = drdFactory.create(type);
      var newElement = {
        type: type,
        businessObject: newBusinessObject
      };
      newElement.width = element.width;
      newElement.height = element.height;
      newBusinessObject.name = oldBusinessObject.name;
      if (target.table) {
        var table = drdFactory.create('dmn:DecisionTable');
        table.$parent = newBusinessObject;
        var output = drdFactory.create('dmn:OutputClause');
        output.typeRef = 'string';
        output.$parent = table;
        table.output = [output];
        var input = drdFactory.create('dmn:InputClause');
        input.$parent = table;
        var inputExpression = drdFactory.create('dmn:LiteralExpression', {
          typeRef: 'string'
        });
        input.inputExpression = inputExpression;
        inputExpression.$parent = input;
        table.input = [input];
        setBoxedExpression(newBusinessObject, table, drdFactory);
      }
      if (target.expression) {
        // variable set to element name
        var literalExpression = drdFactory.create('dmn:LiteralExpression'),
          variable = drdFactory.create('dmn:InformationItem', {
            name: oldBusinessObject.name
          });
        setBoxedExpression(newBusinessObject, literalExpression, drdFactory, variable);
      }
      return replace.replaceElement(element, newElement, hints);
    }
    this.replaceElement = replaceElement;
  }
  DrdReplace.$inject = ['drdFactory', 'replace', 'selection', 'modeling'];

  // helper //////////////////////////////////////////////////////////////
  function setBoxedExpression(bo, expression, drdFactory, variable) {
    if (is$1(bo, 'dmn:Decision')) {
      bo.decisionLogic = expression;
      expression.$parent = bo;
    } else if (is$1(bo, 'dmn:BusinessKnowledgeModel')) {
      var encapsulatedLogic = drdFactory.create('dmn:FunctionDefinition', {
        body: expression
      });
      bo.encapsulatedLogic = encapsulatedLogic;
      encapsulatedLogic.$parent = bo;
      expression.$parent = encapsulatedLogic;
    }
    if (variable) {
      bo.variable = variable;
      variable.$parent = bo;
    }
  }

  var Replace = {
    __depends__: [DiagramReplace, SelectionModule],
    drdReplace: ['type', DrdReplace]
  };

  var replaceOptions = {
    DECISION: [{
      label: 'Empty',
      actionName: 'replace-with-empty-decision',
      className: 'dmn-icon-clear',
      target: {
        type: 'dmn:Decision',
        table: false,
        expression: false
      }
    }, {
      label: 'Decision table',
      actionName: 'replace-with-decision-table',
      className: 'dmn-icon-decision-table',
      target: {
        type: 'dmn:Decision',
        table: true,
        expression: false
      }
    }, {
      label: 'Literal expression',
      actionName: 'replace-with-literal-expression',
      className: 'dmn-icon-literal-expression',
      target: {
        type: 'dmn:Decision',
        table: false,
        expression: true
      }
    }],
    BKM: [{
      label: 'Empty',
      actionName: 'replace-with-empty',
      className: 'dmn-icon-clear',
      target: {
        type: 'dmn:BusinessKnowledgeModel',
        table: false,
        expression: false
      }
    }, {
      label: 'Literal Expression',
      actionName: 'replace-with-literal-expression',
      className: 'dmn-icon-literal-expression',
      target: {
        type: 'dmn:BusinessKnowledgeModel',
        table: false,
        expression: true
      }
    }]
  };

  /**
   * This module is an element agnostic replace menu provider for the popup menu.
   */
  function ReplaceMenuProvider(popupMenu, modeling, moddle, drdReplace, rules, translate) {
    this._popupMenu = popupMenu;
    this._modeling = modeling;
    this._moddle = moddle;
    this._drdReplace = drdReplace;
    this._rules = rules;
    this._translate = translate;
    this.register();
  }
  ReplaceMenuProvider.$inject = ['popupMenu', 'modeling', 'moddle', 'drdReplace', 'rules', 'translate'];

  /**
   * Register replace menu provider in the popup menu
   */
  ReplaceMenuProvider.prototype.register = function () {
    this._popupMenu.registerProvider('dmn-replace', this);
  };

  /**
   * Get all entries from replaceOptions for the given element.
   *
   * @param {djs.model.Base} element
   *
   * @return {Array<Object>} a list of menu entry items
   */
  ReplaceMenuProvider.prototype.getEntries = function (element) {
    var businessObject = element.businessObject;
    var rules = this._rules,
      options,
      boxedExpression;
    if (!rules.allowed('shape.replace', {
      element: element
    })) {
      return [];
    }
    if (is$1(businessObject, 'dmn:Decision')) {
      boxedExpression = getBoxedExpression(businessObject);
      options = filter$2(replaceOptions.DECISION, function (option) {
        var notEmpty = option.actionName === 'replace-with-empty-decision' && boxedExpression;
        var notTable = option.actionName === 'replace-with-decision-table' && !is$1(boxedExpression, 'dmn:DecisionTable');
        var notExp = option.actionName === 'replace-with-literal-expression' && !is$1(boxedExpression, 'dmn:LiteralExpression');
        return notEmpty || notTable || notExp;
      });
      return this._createEntries(element, options);
    }
    if (is$1(businessObject, 'dmn:BusinessKnowledgeModel')) {
      boxedExpression = getBoxedExpression(businessObject);
      options = filter$2(replaceOptions.BKM, function (option) {
        var notEmpty = option.actionName === 'replace-with-empty' && boxedExpression;
        var notTable = option.actionName === 'replace-with-decision-table' && !is$1(boxedExpression, 'dmn:DecisionTable');
        var notExp = option.actionName === 'replace-with-literal-expression' && !is$1(boxedExpression, 'dmn:LiteralExpression');
        return notEmpty || notTable || notExp;
      });
      return this._createEntries(element, options);
    }
    return [];
  };

  /**
   * Creates an array of menu entry objects for a given element.
   *
   * @param  {djs.model.Base} element
   * @param  {Object} replaceOptions
   *
   * @return {Array<Object>} a list of menu items
   */
  ReplaceMenuProvider.prototype._createEntries = function (element, replaceOptions) {
    var menuEntries = [];
    var self = this;
    forEach$3(replaceOptions, function (definition) {
      var entry = self._createMenuEntry(definition, element);
      menuEntries.push(entry);
    });
    return menuEntries;
  };

  /**
   * Creates and returns a single menu entry item.
   *
   * @param  {Object} definition a single replace options definition object
   * @param  {djs.model.Base} element
   * @param  {Function} [action] an action callback function which gets called when
   *                             the menu entry is being triggered.
   *
   * @return {Object} menu entry item
   */
  ReplaceMenuProvider.prototype._createMenuEntry = function (definition, element, action) {
    var replaceElement = this._drdReplace.replaceElement;
    var translate = this._translate;
    var replaceAction = function () {
      return replaceElement(element, definition.target);
    };
    action = action || replaceAction;
    var menuEntry = {
      label: translate(definition.label),
      className: definition.className,
      id: definition.actionName,
      action: action
    };
    return menuEntry;
  };
  ReplaceMenuProvider.prototype.getHeaderEntries = function (element) {
    return [];
  };

  var PopupMenu = {
    __depends__: [TranslateModule, DiagramPopupMenu, Replace],
    __init__: ['replaceMenuProvider'],
    replaceMenuProvider: ['type', ReplaceMenuProvider]
  };

  /**
  * A provider for DMN elements context pad
  */
  function ContextPadProvider(eventBus, contextPad, modeling, elementFactory, connect, create, rules, popupMenu, canvas, translate, config, injector) {
    config = config || {};
    contextPad.registerProvider(this);
    this._contextPad = contextPad;
    this._modeling = modeling;
    this._elementFactory = elementFactory;
    this._connect = connect;
    this._create = create;
    this._rules = rules;
    this._popupMenu = popupMenu;
    this._canvas = canvas;
    this._translate = translate;
    if (config.autoPlace !== false) {
      this._autoPlace = injector.get('autoPlace', false);
    }
    eventBus.on('create.end', 250, function (event) {
      var shape = event.context.shape;
      if (!hasPrimaryModifier(event)) {
        return;
      }
      var entries = contextPad.getEntries(shape);
      if (entries.replace) {
        entries.replace.action.click(event, shape);
      }
    });
  }
  ContextPadProvider.$inject = ['eventBus', 'contextPad', 'modeling', 'elementFactory', 'connect', 'create', 'rules', 'popupMenu', 'canvas', 'translate', 'config.contextPad', 'injector'];
  ContextPadProvider.prototype.getContextPadEntries = function (element) {
    var modeling = this._modeling,
      elementFactory = this._elementFactory,
      connect = this._connect,
      create = this._create,
      popupMenu = this._popupMenu,
      contextPad = this._contextPad,
      rules = this._rules,
      translate = this._translate,
      autoPlace = this._autoPlace;
    var actions = {};
    if (element.type === 'label') {
      return actions;
    }
    var businessObject = element.businessObject;
    function startConnect(event, element, autoActivate) {
      connect.start(event, element, autoActivate);
    }
    function removeElement(e) {
      modeling.removeElements([element]);
    }
    function getReplaceMenuPosition(element) {
      var Y_OFFSET = 5;
      var pad = contextPad.getPad(element).html;
      var padRect = pad.getBoundingClientRect();
      var pos = {
        x: padRect.left,
        y: padRect.bottom + Y_OFFSET
      };
      return pos;
    }

    /**
    * Create an append action
    *
    * @param {string} type
    * @param {string} className
    * @param {string} title
    * @param {Object} [options]
    *
    * @return {Object} descriptor
    */
    function appendAction(type, className, title, options) {
      function appendStart(event, element) {
        var shape = elementFactory.createShape(assign$4({
          type: type
        }, options));
        create.start(event, shape, {
          source: element,
          hints: {
            connectionTarget: element
          }
        });
      }
      var append = autoPlace ? function (event, element) {
        var shape = elementFactory.createShape(assign$4({
          type: type
        }, options));
        autoPlace.append(element, shape, {
          connectionTarget: element
        });
      } : appendStart;
      return {
        group: 'model',
        className: className,
        title: title,
        action: {
          dragstart: appendStart,
          click: append
        }
      };
    }
    if (is$1(businessObject, 'dmn:Decision')) {
      assign$4(actions, {
        'append.decision': appendAction('dmn:Decision', 'dmn-icon-decision', translate('Append decision'))
      });
    }
    if (isAny(businessObject, ['dmn:BusinessKnowledgeModel', 'dmn:Decision', 'dmn:KnowledgeSource'])) {
      assign$4(actions, {
        'append.knowledge-source': appendAction('dmn:KnowledgeSource', 'dmn-icon-knowledge-source', translate('Append knowledge source'))
      });
    }
    if (isAny(businessObject, ['dmn:BusinessKnowledgeModel', 'dmn:Decision'])) {
      assign$4(actions, {
        'append.business-knowledge-model': appendAction('dmn:BusinessKnowledgeModel', 'dmn-icon-business-knowledge', translate('Append business knowledge model'))
      });
    }
    if (isAny(businessObject, ['dmn:Decision', 'dmn:KnowledgeSource'])) {
      assign$4(actions, {
        'append.input-data': appendAction('dmn:InputData', 'dmn-icon-input-data', translate('Append input data'))
      });
    }
    if (is$1(businessObject, 'dmn:DRGElement')) {
      assign$4(actions, {
        'append.text-annotation': appendAction('dmn:TextAnnotation', 'dmn-icon-text-annotation', translate('Add text annotation')),
        'connect': {
          group: 'connect',
          className: 'dmn-icon-connection-multi',
          title: translate('Connect to other element'),
          action: {
            click: startConnect,
            dragstart: startConnect
          }
        }
      });
    }
    if (is$1(businessObject, 'dmn:TextAnnotation')) {
      assign$4(actions, {
        'connect': {
          group: 'connect',
          className: 'dmn-icon-connection-multi',
          title: translate('Connect to other element'),
          action: {
            click: startConnect,
            dragstart: startConnect
          }
        }
      });
    }
    if (!popupMenu.isEmpty(element, 'dmn-replace')) {
      // Replace menu entry
      assign$4(actions, {
        'replace': {
          group: 'edit',
          className: 'dmn-icon-screw-wrench',
          title: translate('Change type'),
          action: {
            click: function (event, element) {
              var position = assign$4(getReplaceMenuPosition(element), {
                cursor: {
                  x: event.x,
                  y: event.y
                }
              });
              popupMenu.open(element, 'dmn-replace', position);
            }
          }
        }
      });
    }

    // delete element entry, only show if allowed by rules
    var deleteAllowed = rules.allowed('elements.delete', {
      elements: [element]
    });
    if (isArray$6(deleteAllowed)) {
      // was the element returned as a deletion candidate?
      deleteAllowed = deleteAllowed[0] === element;
    }
    if (deleteAllowed) {
      assign$4(actions, {
        'delete': {
          group: 'edit',
          className: 'dmn-icon-trash',
          title: translate('Delete'),
          action: {
            click: removeElement
          }
        }
      });
    }
    return actions;
  };
  ContextPadProvider.prototype.getMultiElementContextPadEntries = function (elements) {
    var modeling = this._modeling,
      translate = this._translate;
    var actions = {};
    if (this._isDeleteAllowed(elements)) {
      assign$4(actions, {
        'delete': {
          group: 'edit',
          className: 'dmn-icon-trash',
          title: translate('Delete'),
          action: {
            click: (e, elements) => modeling.removeElements(elements.slice())
          }
        }
      });
    }
    return actions;
  };
  ContextPadProvider.prototype._isDeleteAllowed = function (elements) {
    // rules allowed can return boolean or array of elements (not reflected in type )
    var allowedOrAllowedElements = this._rules.allowed('elements.delete', {
      elements: elements
    });
    if (isArray$6(allowedOrAllowedElements)) {
      return every(elements, el => allowedOrAllowedElements.includes(el));
    }
    return allowedOrAllowedElements;
  };

  var ContextPadModule = {
    __depends__: [TranslateModule, DiagramContextPad, SelectionModule, DiagramConnect, DiagramCreate, PopupMenu],
    __init__: ['contextPadProvider'],
    contextPadProvider: ['type', ContextPadProvider]
  };

  /**
   * @typedef {import('../../model/Types').Element} Element
   * @typedef {import('../../model/Types').Connection} Connection
   * @typedef {import('../../model/Types').Shape} Shape
   *
   * @typedef {import('../../util/Types').Point} Point
   *
   * @typedef {import('didi').Injector} Injector
   *
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/ElementFactory').default} ElementFactory
   * @typedef {import('../../core/GraphicsFactory').default} GraphicsFactory
   */

  var MARKER_CONNECTION_PREVIEW = 'djs-dragger';

  /**
   * Draws connection preview. Optionally, this can use layouter and connection docking to draw
   * better looking previews.
   *
   * @param {Injector} injector
   * @param {Canvas} canvas
   * @param {GraphicsFactory} graphicsFactory
   * @param {ElementFactory} elementFactory
   */
  function ConnectionPreview(injector, canvas, graphicsFactory, elementFactory) {
    this._canvas = canvas;
    this._graphicsFactory = graphicsFactory;
    this._elementFactory = elementFactory;

    // optional components
    this._connectionDocking = injector.get('connectionDocking', false);
    this._layouter = injector.get('layouter', false);
  }
  ConnectionPreview.$inject = ['injector', 'canvas', 'graphicsFactory', 'elementFactory'];

  /**
   * Draw connection preview.
   *
   * Provide at least one of <source, connectionStart> and <target, connectionEnd> to create a preview.
   * In the clean up stage, call `connectionPreview#cleanUp` with the context to remove preview.
   *
   * @param {Object} context
   * @param {Object|boolean} canConnect
   * @param {Object} hints
   * @param {Element} [hints.source] source element
   * @param {Element} [hints.target] target element
   * @param {Point} [hints.connectionStart] connection preview start
   * @param {Point} [hints.connectionEnd] connection preview end
   * @param {Point[]} [hints.waypoints] provided waypoints for preview
   * @param {boolean} [hints.noLayout] true if preview should not be laid out
   * @param {boolean} [hints.noCropping] true if preview should not be cropped
   * @param {boolean} [hints.noNoop] true if simple connection should not be drawn
   */
  ConnectionPreview.prototype.drawPreview = function (context, canConnect, hints) {
    hints = hints || {};
    var connectionPreviewGfx = context.connectionPreviewGfx,
      getConnection = context.getConnection,
      source = hints.source,
      target = hints.target,
      waypoints = hints.waypoints,
      connectionStart = hints.connectionStart,
      connectionEnd = hints.connectionEnd,
      noLayout = hints.noLayout,
      noCropping = hints.noCropping,
      noNoop = hints.noNoop,
      connection;
    var self = this;
    if (!connectionPreviewGfx) {
      connectionPreviewGfx = context.connectionPreviewGfx = this.createConnectionPreviewGfx();
    }
    clear(connectionPreviewGfx);
    if (!getConnection) {
      getConnection = context.getConnection = cacheReturnValues(function (canConnect, source, target) {
        return self.getConnection(canConnect, source, target);
      });
    }
    if (canConnect) {
      connection = getConnection(canConnect, source, target);
    }
    if (!connection) {
      !noNoop && this.drawNoopPreview(connectionPreviewGfx, hints);
      return;
    }
    connection.waypoints = waypoints || [];

    // optional layout
    if (this._layouter && !noLayout) {
      connection.waypoints = this._layouter.layoutConnection(connection, {
        source: source,
        target: target,
        connectionStart: connectionStart,
        connectionEnd: connectionEnd,
        waypoints: hints.waypoints || connection.waypoints
      });
    }

    // fallback if no waypoints were provided nor created with layouter
    if (!connection.waypoints || !connection.waypoints.length) {
      connection.waypoints = [source ? getMid(source) : connectionStart, target ? getMid(target) : connectionEnd];
    }

    // optional cropping
    if (this._connectionDocking && (source || target) && !noCropping) {
      connection.waypoints = this._connectionDocking.getCroppedWaypoints(connection, source, target);
    }
    this._graphicsFactory.drawConnection(connectionPreviewGfx, connection, {
      stroke: 'var(--element-dragger-color)'
    });
  };

  /**
   * Draw simple connection between source and target or provided points.
   *
   * @param {SVGElement} connectionPreviewGfx container for the connection
   * @param {Object} hints
   * @param {Element} [hints.source] source element
   * @param {Element} [hints.target] target element
   * @param {Point} [hints.connectionStart] required if source is not provided
   * @param {Point} [hints.connectionEnd] required if target is not provided
   */
  ConnectionPreview.prototype.drawNoopPreview = function (connectionPreviewGfx, hints) {
    var source = hints.source,
      target = hints.target,
      start = hints.connectionStart || getMid(source),
      end = hints.connectionEnd || getMid(target);
    var waypoints = this.cropWaypoints(start, end, source, target);
    var connection = this.createNoopConnection(waypoints[0], waypoints[1]);
    append(connectionPreviewGfx, connection);
  };

  /**
   * Return cropped waypoints.
   *
   * @param {Point} start
   * @param {Point} end
   * @param {Element} source
   * @param {Element} target
   *
   * @return {Point[]}
   */
  ConnectionPreview.prototype.cropWaypoints = function (start, end, source, target) {
    var graphicsFactory = this._graphicsFactory,
      sourcePath = source && graphicsFactory.getShapePath(source),
      targetPath = target && graphicsFactory.getShapePath(target),
      connectionPath = graphicsFactory.getConnectionPath({
        waypoints: [start, end]
      });
    start = source && getElementLineIntersection(sourcePath, connectionPath, true) || start;
    end = target && getElementLineIntersection(targetPath, connectionPath, false) || end;
    return [start, end];
  };

  /**
   * Remove connection preview container if it exists.
   *
   * @param {Object} [context]
   * @param {SVGElement} [context.connectionPreviewGfx] preview container
   */
  ConnectionPreview.prototype.cleanUp = function (context) {
    if (context && context.connectionPreviewGfx) {
      remove$3(context.connectionPreviewGfx);
    }
  };

  /**
   * Get connection that connects source and target.
   *
   * @param {Object|boolean} canConnect
   *
   * @return {Connection}
   */
  ConnectionPreview.prototype.getConnection = function (canConnect) {
    var attrs = ensureConnectionAttrs(canConnect);
    return this._elementFactory.createConnection(attrs);
  };

  /**
   * Add and return preview graphics.
   *
   * @return {SVGElement}
   */
  ConnectionPreview.prototype.createConnectionPreviewGfx = function () {
    var gfx = create$2('g');
    attr(gfx, {
      pointerEvents: 'none'
    });
    classes(gfx).add(MARKER_CONNECTION_PREVIEW);
    append(this._canvas.getActiveLayer(), gfx);
    return gfx;
  };

  /**
   * Create and return simple connection.
   *
   * @param {Point} start
   * @param {Point} end
   *
   * @return {SVGElement}
   */
  ConnectionPreview.prototype.createNoopConnection = function (start, end) {
    return createLine([start, end], {
      'stroke': '#333',
      'strokeDasharray': [1],
      'strokeWidth': 2,
      'pointer-events': 'none'
    });
  };

  // helpers //////////

  /**
   * Returns function that returns cached return values referenced by stringified first argument.
   *
   * @param {Function} fn
   *
   * @return {Function}
   */
  function cacheReturnValues(fn) {
    var returnValues = {};

    /**
     * Return cached return value referenced by stringified first argument.
     *
     * @return {*}
     */
    return function (firstArgument) {
      var key = JSON.stringify(firstArgument);
      var returnValue = returnValues[key];
      if (!returnValue) {
        returnValue = returnValues[key] = fn.apply(null, arguments);
      }
      return returnValue;
    };
  }

  /**
   * Ensure connection attributes is object.
   *
   * @param {Object|boolean} canConnect
   *
   * @return {Object}
   */
  function ensureConnectionAttrs(canConnect) {
    if (isObject$2(canConnect)) {
      return canConnect;
    } else {
      return {};
    }
  }

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var ConnectPreviewModule = {
    __init__: ['connectionPreview'],
    connectionPreview: ['type', ConnectionPreview]
  };

  const SPACE_REGEX = /\s/;

  // for QName validation as per http://www.w3.org/TR/REC-xml/#NT-NameChar
  const QNAME_REGEX = /^([a-z][\w-.]*:)?[a-z_][\w-.]*$/i;

  // for ID validation as per BPMN Schema (QName - Namespace)
  const ID_REGEX = /^[a-z_][\w-.]*$/i;
  const PLACEHOLDER_REGEX = /\$\{([^}]*)\}/g;

  /**
   * Validates an ID.
   *
   * @param {ModdleElement} businessObject
   * @param {string} id
   *
   * @return {string} error message
   */
  function validateId(businessObject, id) {
    const assigned = businessObject.$model.ids.assigned(id);
    const idExists = assigned && assigned !== businessObject;
    if (!id) {
      return 'Element must have ID.';
    }
    if (idExists) {
      return 'Element must have unique ID.';
    }
    return validateIdFormat(id);
  }
  function validateIdFormat(id) {
    id = stripPlaceholders(id);
    if (containsSpace(id)) {
      return 'Id must not contain spaces.';
    }
    if (!ID_REGEX.test(id)) {
      if (QNAME_REGEX.test(id)) {
        return 'Id must not contain prefix.';
      }
      return 'Id must be a valid QName.';
    }
  }
  function containsSpace(value) {
    return SPACE_REGEX.test(value);
  }
  function stripPlaceholders(id) {
    // replace expression e.g. ${VERSION_TAG}
    // use only the content between ${}
    // for the REGEX check
    return id.replace(PLACEHOLDER_REGEX, '$1');
  }

  var DEBOUNCE_DELAY = 300;
  function DefinitionIdEdit(eventBus, modeling, canvas, definitionPropertiesView, translate) {
    this._eventBus = eventBus;
    this._modeling = modeling;
    this._canvas = canvas;
    this._definitionPropertiesView = definitionPropertiesView;
    this._translate = translate;
    eventBus.on('definitionIdView.create', function (event) {
      this._container = event.html;
      var nameElement = query('.dmn-definitions-name', this._container),
        idElement = query('.dmn-definitions-id', this._container);
      this._setup(nameElement, 'name');
      this._setup(idElement, 'id');
    }, this);
  }
  DefinitionIdEdit.$inject = ['eventBus', 'modeling', 'canvas', 'definitionPropertiesView', 'translate'];
  DefinitionIdEdit.prototype.update = function (type, newValue) {
    var element = this._canvas.getRootElement();
    var newProperties = {};
    newProperties[type] = newValue;
    if (type === 'id') {
      var errorMessage = validateId(getBusinessObject(element), newValue);
      if (errorMessage) {
        this._addErrorMessage(errorMessage);
        return;
      }
      this._clearErrorMessage();
    }
    this._modeling.updateProperties(element, newProperties);
  };
  DefinitionIdEdit.prototype._setup = function (node, type) {
    var self = this;
    node.setAttribute('contenteditable', true);
    node.addEventListener('input', debounce(function (evt) {
      var value = evt.target.value || evt.target.textContent;
      self.update(type, value.trim());
    }, DEBOUNCE_DELAY));
    node.addEventListener('keydown', function (evt) {
      if (evt.keyCode === 13) {
        node.blur();
        window.getSelection().removeAllRanges();
      }
    });
    node.addEventListener('blur', function () {
      self._clearErrorMessage();
      self._definitionPropertiesView.update();
    });
  };
  DefinitionIdEdit.prototype._addErrorMessage = function (errorMessage) {
    const errorHTML = '<span class="dmn-definitions-error-message">' + this._translate(errorMessage) + '</span>';
    var idElement = query('.dmn-definitions-id', this._container);

    // clear previous error message
    this._clearErrorMessage();

    // add current error message
    classes$1(idElement).add('dmn-definitions-error');
    idElement.parentElement.appendChild(domify$1(errorHTML));
  };
  DefinitionIdEdit.prototype._clearErrorMessage = function () {
    var idElement = query('.dmn-definitions-id', this._container);
    if (classes$1(idElement).has('dmn-definitions-error')) {
      classes$1(idElement).remove('dmn-definitions-error');
      const errorLabel = query('.dmn-definitions-error-message', this._container);
      idElement.parentNode.removeChild(errorLabel);
    }
  };

  var DefinitionPropertiesModule = {
    __depends__: [DefinitionPropertiesViewer],
    __init__: ['definitionPropertiesEdit', 'definitionPropertiesPaletteAdapter'],
    definitionPropertiesEdit: ['type', DefinitionIdEdit],
    definitionPropertiesPaletteAdapter: ['type', PaletteAdapter]
  };

  /**
   * @typedef {import('../../model/Types').Element} Element
   *
   * @typedef {import('../../util/Types').Axis} Axis
   * @typedef {import('../../util/Types').Dimension} Dimension
   * @typedef {import('../../util/Types').Rect} Rect
   *
   * @typedef {import('../modeling/Modeling').default} Modeling
   * @typedef {import('../rules/Rules').default} Rules
   *
   * @typedef { {
   *   min: number;
   *   max: number;
   * } } Range
   *
   * @typedef { {
   *   elements: Element[];
   *   range: Range;
   * } } Group
   */

  var AXIS_DIMENSIONS = {
    horizontal: ['x', 'width'],
    vertical: ['y', 'height']
  };
  var THRESHOLD = 5;

  /**
   * Groups and filters elements and then trigger even distribution.
   *
   * @param {Modeling} modeling
   * @param {Rules} rules
   */
  function DistributeElements$1(modeling, rules) {
    this._modeling = modeling;
    this._filters = [];
    this.registerFilter(function (elements) {
      var allowed = rules.allowed('elements.distribute', {
        elements: elements
      });
      if (isArray$6(allowed)) {
        return allowed;
      }
      return allowed ? elements : [];
    });
  }
  DistributeElements$1.$inject = ['modeling', 'rules'];

  /**
   * Registers filter functions that allow external parties to filter
   * out certain elements.
   *
   * @param {(distributableElements: Element[], axis: Axis, dimension: Dimension) => Element[]} filterFn
   */
  DistributeElements$1.prototype.registerFilter = function (filterFn) {
    if (typeof filterFn !== 'function') {
      throw new Error('the filter has to be a function');
    }
    this._filters.push(filterFn);
  };

  /**
   * Distributes the elements with a given orientation
   *
   * @param {Element[]} elements
   * @param {string} orientation
   */
  DistributeElements$1.prototype.trigger = function (elements, orientation) {
    var modeling = this._modeling;
    var groups, distributableElements;
    if (elements.length < 3) {
      return;
    }
    this._setOrientation(orientation);
    distributableElements = this._filterElements(elements);
    groups = this._createGroups(distributableElements);

    // nothing to distribute
    if (groups.length <= 2) {
      return;
    }
    modeling.distributeElements(groups, this._axis, this._dimension);
    return groups;
  };

  /**
   * Filters the elements with provided filters by external parties
   *
   * @param {Element[]} elements
   *
   * @return {Element[]}
   */
  DistributeElements$1.prototype._filterElements = function (elements) {
    var filters = this._filters,
      axis = this._axis,
      dimension = this._dimension,
      distributableElements = [].concat(elements);
    if (!filters.length) {
      return elements;
    }
    forEach$3(filters, function (filterFn) {
      distributableElements = filterFn(distributableElements, axis, dimension);
    });
    return distributableElements;
  };

  /**
   * Create range (min, max) groups. Also tries to group elements
   * together that share the same range.
   *
   * @example
   *
   * ```javascript
   *   const groups = [
   *     {
   *       range: { min: 100, max: 200 },
   *       elements: [ { id: 'shape1', ... } ]
   *     }
   *   ]
   * ```
   *
   * @param {Element[]} elements
   *
   * @return {Group[]}
   */
  DistributeElements$1.prototype._createGroups = function (elements) {
    var rangeGroups = [],
      self = this,
      axis = this._axis,
      dimension = this._dimension;
    if (!axis) {
      throw new Error('must have a defined "axis" and "dimension"');
    }

    // sort by 'left->right' or 'top->bottom'
    var sortedElements = sortBy(elements, axis);
    forEach$3(sortedElements, function (element, idx) {
      var elementRange = self._findRange(element, axis, dimension),
        range;
      var previous = rangeGroups[rangeGroups.length - 1];
      if (previous && self._hasIntersection(previous.range, elementRange)) {
        rangeGroups[rangeGroups.length - 1].elements.push(element);
      } else {
        range = {
          range: elementRange,
          elements: [element]
        };
        rangeGroups.push(range);
      }
    });
    return rangeGroups;
  };

  /**
   * Maps a direction to the according axis and dimension.
   *
   * @param {'horizontal' | 'vertical'} direction 'horizontal' or 'vertical'
   */
  DistributeElements$1.prototype._setOrientation = function (direction) {
    var orientation = AXIS_DIMENSIONS[direction];
    this._axis = orientation[0];
    this._dimension = orientation[1];
  };

  /**
   * Checks if the two ranges intercept each other.
   *
   * @param {Range} rangeA
   * @param {Range} rangeB
   *
   * @return {boolean}
   */
  DistributeElements$1.prototype._hasIntersection = function (rangeA, rangeB) {
    return Math.max(rangeA.min, rangeA.max) >= Math.min(rangeB.min, rangeB.max) && Math.min(rangeA.min, rangeA.max) <= Math.max(rangeB.min, rangeB.max);
  };

  /**
   * Returns the min and max values for an element
   *
   * @param {Element} element
   *
   * @return {Range}
   */
  DistributeElements$1.prototype._findRange = function (element) {
    var axis = element[this._axis],
      dimension = element[this._dimension];
    return {
      min: axis + THRESHOLD,
      max: axis + dimension - THRESHOLD
    };
  };

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var DistributeElementsModule$1 = {
    __init__: ['distributeElements'],
    distributeElements: ['type', DistributeElements$1]
  };

  /**
   * Registers element exclude filters for elements that
   * currently do not support distribution.
   */
  function DrdDistributeElements(distributeElements) {
    distributeElements.registerFilter(function (elements) {
      return filter$2(elements, function (element) {
        var cannotDistribute = isAny(element, ['dmn:AuthorityRequirement', 'dmn:InformationRequirement', 'dmn:KnowledgeRequirement', 'dmn:Association', 'dmn:TextAnnotation']);
        return !(element.labelTarget || cannotDistribute);
      });
    });
  }
  DrdDistributeElements.$inject = ['distributeElements'];

  var DistributeElementsModule = {
    __depends__: [DistributeElementsModule$1],
    __init__: ['drdDistributeElements'],
    drdDistributeElements: ['type', DrdDistributeElements]
  };

  /**
   * @typedef {import('didi').Injector} Injector
   *
   * @typedef {import('../../core/EventBus').default} EventBus
   */

  var NOT_REGISTERED_ERROR$3 = 'is not a registered action',
    IS_REGISTERED_ERROR$3 = 'is already registered';

  /**
   * An interface that provides access to modeling actions by decoupling
   * the one who requests the action to be triggered and the trigger itself.
   *
   * It's possible to add new actions by registering them with ´registerAction´
   * and likewise unregister existing ones with ´unregisterAction´.
   *
   *
   * ## Life-Cycle and configuration
   *
   * The editor actions will wait for diagram initialization before
   * registering default actions _and_ firing an `editorActions.init` event.
   *
   * Interested parties may listen to the `editorActions.init` event with
   * low priority to check, which actions got registered. Other components
   * may use the event to register their own actions via `registerAction`.
   *
   * @param {EventBus} eventBus
   * @param {Injector} injector
   */
  function EditorActions$4(eventBus, injector) {
    // initialize actions
    this._actions = {};
    var self = this;
    eventBus.on('diagram.init', function () {
      // all diagram modules got loaded; check which ones
      // are available and register the respective default actions
      self._registerDefaultActions(injector);

      // ask interested parties to register available editor
      // actions on diagram initialization
      eventBus.fire('editorActions.init', {
        editorActions: self
      });
    });
  }
  EditorActions$4.$inject = ['eventBus', 'injector'];

  /**
   * Register default actions.
   *
   * @param {Injector} injector
   */
  EditorActions$4.prototype._registerDefaultActions = function (injector) {
    // (1) retrieve optional components to integrate with

    var commandStack = injector.get('commandStack', false);
    var modeling = injector.get('modeling', false);
    var selection = injector.get('selection', false);
    var zoomScroll = injector.get('zoomScroll', false);
    var copyPaste = injector.get('copyPaste', false);
    var canvas = injector.get('canvas', false);
    var rules = injector.get('rules', false);
    var keyboardMove = injector.get('keyboardMove', false);
    var keyboardMoveSelection = injector.get('keyboardMoveSelection', false);

    // (2) check components and register actions

    if (commandStack) {
      this.register('undo', function () {
        commandStack.undo();
      });
      this.register('redo', function () {
        commandStack.redo();
      });
    }
    if (copyPaste && selection) {
      this.register('copy', function () {
        var selectedElements = selection.get();
        if (selectedElements.length) {
          return copyPaste.copy(selectedElements);
        }
      });
    }
    if (copyPaste) {
      this.register('paste', function () {
        copyPaste.paste();
      });
    }
    if (zoomScroll) {
      this.register('stepZoom', function (opts) {
        zoomScroll.stepZoom(opts.value);
      });
    }
    if (canvas) {
      this.register('zoom', function (opts) {
        canvas.zoom(opts.value);
      });
    }
    if (modeling && selection && rules) {
      this.register('removeSelection', function () {
        var selectedElements = selection.get();
        if (!selectedElements.length) {
          return;
        }
        var allowed = rules.allowed('elements.delete', {
            elements: selectedElements
          }),
          removableElements;
        if (allowed === false) {
          return;
        } else if (isArray$6(allowed)) {
          removableElements = allowed;
        } else {
          removableElements = selectedElements;
        }
        if (removableElements.length) {
          modeling.removeElements(removableElements.slice());
        }
      });
    }
    if (keyboardMove) {
      this.register('moveCanvas', function (opts) {
        keyboardMove.moveCanvas(opts);
      });
    }
    if (keyboardMoveSelection) {
      this.register('moveSelection', function (opts) {
        keyboardMoveSelection.moveSelection(opts.direction, opts.accelerated);
      });
    }
  };

  /**
   * Triggers a registered action
   *
   * @param {string} action
   * @param {Object} opts
   *
   * @return {unknown} Returns what the registered listener returns
   */
  EditorActions$4.prototype.trigger = function (action, opts) {
    if (!this._actions[action]) {
      throw error$3(action, NOT_REGISTERED_ERROR$3);
    }
    return this._actions[action](opts);
  };

  /**
   * Registers a collections of actions.
   * The key of the object will be the name of the action.
   *
   * @example
   *
   * ```javascript
   * var actions = {
   *   spaceTool: function() {
   *     spaceTool.activateSelection();
   *   },
   *   lassoTool: function() {
   *     lassoTool.activateSelection();
   *   }
   * ];
   *
   * editorActions.register(actions);
   *
   * editorActions.isRegistered('spaceTool'); // true
   * ```
   *
   * @param {Object} actions
   */
  EditorActions$4.prototype.register = function (actions, listener) {
    var self = this;
    if (typeof actions === 'string') {
      return this._registerAction(actions, listener);
    }
    forEach$3(actions, function (listener, action) {
      self._registerAction(action, listener);
    });
  };

  /**
   * Registers a listener to an action key
   *
   * @param {string} action
   * @param {Function} listener
   */
  EditorActions$4.prototype._registerAction = function (action, listener) {
    if (this.isRegistered(action)) {
      throw error$3(action, IS_REGISTERED_ERROR$3);
    }
    this._actions[action] = listener;
  };

  /**
   * Unregister an existing action
   *
   * @param {string} action
   */
  EditorActions$4.prototype.unregister = function (action) {
    if (!this.isRegistered(action)) {
      throw error$3(action, NOT_REGISTERED_ERROR$3);
    }
    this._actions[action] = undefined;
  };

  /**
   * Returns the identifiers of all currently registered editor actions
   *
   * @return {string[]}
   */
  EditorActions$4.prototype.getActions = function () {
    return Object.keys(this._actions);
  };

  /**
   * Checks wether the given action is registered
   *
   * @param {string} action
   *
   * @return {boolean}
   */
  EditorActions$4.prototype.isRegistered = function (action) {
    return !!this._actions[action];
  };
  function error$3(action, message) {
    return new Error(action + ' ' + message);
  }

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var EditorActionsModule$2 = {
    __init__: ['editorActions'],
    editorActions: ['type', EditorActions$4]
  };

  function DrdEditorActions(injector) {
    injector.invoke(EditorActions$4, this);
  }
  e$3(DrdEditorActions, EditorActions$4);
  DrdEditorActions.$inject = ['injector'];

  /**
   * Register default actions.
   *
   * @param {Injector} injector
   */
  DrdEditorActions.prototype._registerDefaultActions = function (injector) {
    // (0) invoke super method

    EditorActions$4.prototype._registerDefaultActions.call(this, injector);

    // (1) retrieve optional components to integrate with

    const canvas = injector.get('canvas', false),
      elementRegistry = injector.get('elementRegistry', false),
      selection = injector.get('selection', false),
      lassoTool = injector.get('lassoTool', false),
      handTool = injector.get('handTool', false),
      directEditing = injector.get('directEditing', false),
      distributeElements = injector.get('distributeElements', false),
      alignElements = injector.get('alignElements', false),
      searchPad = injector.get('searchPad', false);

    // (2) check components and register actions

    if (canvas && elementRegistry && selection) {
      this._registerAction('selectElements', function () {
        // select all elements except for the invisible
        // root element
        var rootElement = canvas.getRootElement();
        var elements = elementRegistry.filter(function (element) {
          return element !== rootElement;
        });
        selection.select(elements);
        return elements;
      });
    }
    if (selection && distributeElements) {
      this._registerAction('distributeElements', function (opts) {
        var currentSelection = selection.get(),
          type = opts.type;
        if (currentSelection.length > 2) {
          distributeElements.trigger(currentSelection, type);
        }
      });
    }
    if (selection && alignElements) {
      this._registerAction('alignElements', function (opts) {
        var currentSelection = selection.get(),
          type = opts.type;
        if (currentSelection.length > 1) {
          alignElements.trigger(currentSelection, type);
        }
      });
    }
    if (lassoTool) {
      this._registerAction('lassoTool', function () {
        lassoTool.toggle();
      });
    }
    if (handTool) {
      this._registerAction('handTool', function () {
        handTool.toggle();
      });
    }
    if (selection && directEditing) {
      this._registerAction('directEditing', function () {
        var currentSelection = selection.get();
        if (currentSelection.length) {
          directEditing.activate(currentSelection[0]);
        }
      });
    }
    if (selection && searchPad) {
      this._registerAction('find', function () {
        searchPad.toggle();
      });
    }
  };

  var EditorActionsModule$1 = {
    __depends__: [EditorActionsModule$2],
    editorActions: ['type', DrdEditorActions]
  };

  /**
   * Does the definitions element contain graphical information?
   *
   * @param  {ModdleElement} definitions
   *
   * @return {boolean} true, if the definitions contains graphical information
   */
  function containsDi(definitions) {
    return definitions.dmnDI && definitions.dmnDI.diagrams && definitions.dmnDI.diagrams[0];
  }

  /**
   * Generates missing DI on import.
   *
   * @param {DrdFactory} drdFactory
   * @param {ElementFactory} elementFactory
   * @param {EventBus} eventBus
   */
  function DiGenerator(drdFactory, elementFactory, eventBus, drdUpdater) {
    function createDi(definitions) {
      // retrieve or create dmnDI
      var dmnDI = definitions.dmnDI;
      if (!dmnDI) {
        dmnDI = drdFactory.create('dmndi:DMNDI');
        definitions.set('dmnDI', dmnDI);
      }
      var diagram = drdFactory.create('dmndi:DMNDiagram');
      dmnDI.set('diagrams', [diagram]);
      var index = 0;
      forEach$3(definitions.get('drgElement'), function (drgElement) {
        // generate DI for decisions only
        if (!is$1(drgElement, 'dmn:Decision')) {
          return;
        }
        var dimensions = elementFactory._getDefaultSize(drgElement);
        var di = drdFactory.createDiShape(drgElement, {
          x: 150 + index * 30,
          y: 150 + index * 30,
          width: dimensions.width,
          height: dimensions.height
        });
        drdUpdater.updateDiParent(di, diagram);
        index++;
      });
    }
    eventBus.on('import.start', ({
      definitions
    }) => {
      if (!containsDi(definitions)) {
        createDi(definitions);
      }
    });
  }
  DiGenerator.$inject = ['drdFactory', 'elementFactory', 'eventBus', 'drdUpdater'];

  var GenerateDiModule = {
    __init__: ['diGenerator'],
    diGenerator: ['type', DiGenerator]
  };

  var SPACING = 10;
  function quantize(value, quantum, fn) {
    if (!fn) {
      fn = 'round';
    }
    return Math[fn](value / quantum) * quantum;
  }

  /**
   * @typedef {import('../../core/ElementRegistry').default} ElementRegistry
   * @typedef {import('../../core/EventBus').default} EventBus
   */

  var LOWER_PRIORITY$1 = 1200;
  var LOW_PRIORITY$h = 800;

  /**
   * Basic grid snapping that covers connecting, creating, moving, resizing shapes, moving bendpoints
   * and connection segments.
   *
   * @param {ElementRegistry} elementRegistry
   * @param {EventBus} eventBus
   * @param {Object} config
   */
  function GridSnapping(elementRegistry, eventBus, config) {
    var active = !config || config.active !== false;
    this._eventBus = eventBus;
    var self = this;
    eventBus.on('diagram.init', LOW_PRIORITY$h, function () {
      self.setActive(active);
    });
    eventBus.on(['create.move', 'create.end', 'bendpoint.move.move', 'bendpoint.move.end', 'connect.move', 'connect.end', 'connectionSegment.move.move', 'connectionSegment.move.end', 'resize.move', 'resize.end', 'shape.move.move', 'shape.move.end'], LOWER_PRIORITY$1, function (event) {
      var originalEvent = event.originalEvent;
      if (!self.active || originalEvent && isCmd$5(originalEvent)) {
        return;
      }
      var context = event.context,
        gridSnappingContext = context.gridSnappingContext;
      if (!gridSnappingContext) {
        gridSnappingContext = context.gridSnappingContext = {};
      }
      ['x', 'y'].forEach(function (axis) {
        var options = {};

        // allow snapping with offset
        var snapOffset = getSnapOffset(event, axis, elementRegistry);
        if (snapOffset) {
          options.offset = snapOffset;
        }

        // allow snapping with min and max
        var snapConstraints = getSnapConstraints(event, axis);
        if (snapConstraints) {
          assign$4(options, snapConstraints);
        }
        if (!isSnapped(event, axis)) {
          self.snapEvent(event, axis, options);
        }
      });
    });
  }

  /**
   * Snap an events x or y with optional min, max and offset.
   *
   * @param {Object} event
   * @param {string} axis
   * @param {number} [options.min]
   * @param {number} [options.max]
   * @param {number} [options.offset]
   */
  GridSnapping.prototype.snapEvent = function (event, axis, options) {
    var snappedValue = this.snapValue(event[axis], options);
    setSnapped(event, axis, snappedValue);
  };

  /**
   * Expose grid spacing for third parties (i.e. extensions).
   *
   * @return {number} spacing of grid dots
   */
  GridSnapping.prototype.getGridSpacing = function () {
    return SPACING;
  };

  /**
   * Snap value with optional min, max and offset.
   *
   * @param {number} value
   * @param {Object} options
   * @param {number} [options.min]
   * @param {number} [options.max]
   * @param {number} [options.offset]
   */
  GridSnapping.prototype.snapValue = function (value, options) {
    var offset = 0;
    if (options && options.offset) {
      offset = options.offset;
    }
    value += offset;
    value = quantize(value, SPACING);
    var min, max;
    if (options && options.min) {
      min = options.min;
      if (isNumber$3(min)) {
        min = quantize(min + offset, SPACING, 'ceil');
        value = Math.max(value, min);
      }
    }
    if (options && options.max) {
      max = options.max;
      if (isNumber$3(max)) {
        max = quantize(max + offset, SPACING, 'floor');
        value = Math.min(value, max);
      }
    }
    value -= offset;
    return value;
  };
  GridSnapping.prototype.isActive = function () {
    return this.active;
  };
  GridSnapping.prototype.setActive = function (active) {
    this.active = active;
    this._eventBus.fire('gridSnapping.toggle', {
      active: active
    });
  };
  GridSnapping.prototype.toggleActive = function () {
    this.setActive(!this.active);
  };
  GridSnapping.$inject = ['elementRegistry', 'eventBus', 'config.gridSnapping'];

  // helpers //////////

  /**
   * Get minimum and maximum snap constraints.
   * Constraints are cached.
   *
   * @param {Object} event
   * @param {Object} event.context
   * @param {string} axis
   *
   * @return {boolean|Object}
   */
  function getSnapConstraints(event, axis) {
    var context = event.context,
      createConstraints = context.createConstraints,
      resizeConstraints = context.resizeConstraints || {},
      gridSnappingContext = context.gridSnappingContext,
      snapConstraints = gridSnappingContext.snapConstraints;

    // cache snap constraints
    if (snapConstraints && snapConstraints[axis]) {
      return snapConstraints[axis];
    }
    if (!snapConstraints) {
      snapConstraints = gridSnappingContext.snapConstraints = {};
    }
    if (!snapConstraints[axis]) {
      snapConstraints[axis] = {};
    }
    var direction = context.direction;

    // create
    if (createConstraints) {
      if (isHorizontal$1(axis)) {
        snapConstraints.x.min = createConstraints.left;
        snapConstraints.x.max = createConstraints.right;
      } else {
        snapConstraints.y.min = createConstraints.top;
        snapConstraints.y.max = createConstraints.bottom;
      }
    }

    // resize
    var minResizeConstraints = resizeConstraints.min,
      maxResizeConstraints = resizeConstraints.max;
    if (minResizeConstraints) {
      if (isHorizontal$1(axis)) {
        if (isWest(direction)) {
          snapConstraints.x.max = minResizeConstraints.left;
        } else {
          snapConstraints.x.min = minResizeConstraints.right;
        }
      } else {
        if (isNorth(direction)) {
          snapConstraints.y.max = minResizeConstraints.top;
        } else {
          snapConstraints.y.min = minResizeConstraints.bottom;
        }
      }
    }
    if (maxResizeConstraints) {
      if (isHorizontal$1(axis)) {
        if (isWest(direction)) {
          snapConstraints.x.min = maxResizeConstraints.left;
        } else {
          snapConstraints.x.max = maxResizeConstraints.right;
        }
      } else {
        if (isNorth(direction)) {
          snapConstraints.y.min = maxResizeConstraints.top;
        } else {
          snapConstraints.y.max = maxResizeConstraints.bottom;
        }
      }
    }
    return snapConstraints[axis];
  }

  /**
   * Get snap offset.
   * Offset is cached.
   *
   * @param {Object} event
   * @param {string} axis
   * @param {ElementRegistry} elementRegistry
   *
   * @return {number}
   */
  function getSnapOffset(event, axis, elementRegistry) {
    var context = event.context,
      shape = event.shape,
      gridSnappingContext = context.gridSnappingContext,
      snapLocation = gridSnappingContext.snapLocation,
      snapOffset = gridSnappingContext.snapOffset;

    // cache snap offset
    if (snapOffset && isNumber$3(snapOffset[axis])) {
      return snapOffset[axis];
    }
    if (!snapOffset) {
      snapOffset = gridSnappingContext.snapOffset = {};
    }
    if (!isNumber$3(snapOffset[axis])) {
      snapOffset[axis] = 0;
    }
    if (!shape) {
      return snapOffset[axis];
    }
    if (!elementRegistry.get(shape.id)) {
      if (isHorizontal$1(axis)) {
        snapOffset[axis] += shape[axis] + shape.width / 2;
      } else {
        snapOffset[axis] += shape[axis] + shape.height / 2;
      }
    }
    if (!snapLocation) {
      return snapOffset[axis];
    }
    if (axis === 'x') {
      if (/left/.test(snapLocation)) {
        snapOffset[axis] -= shape.width / 2;
      } else if (/right/.test(snapLocation)) {
        snapOffset[axis] += shape.width / 2;
      }
    } else {
      if (/top/.test(snapLocation)) {
        snapOffset[axis] -= shape.height / 2;
      } else if (/bottom/.test(snapLocation)) {
        snapOffset[axis] += shape.height / 2;
      }
    }
    return snapOffset[axis];
  }
  function isHorizontal$1(axis) {
    return axis === 'x';
  }
  function isNorth(direction) {
    return direction.indexOf('n') !== -1;
  }
  function isWest(direction) {
    return direction.indexOf('w') !== -1;
  }

  /**
   * @typedef {import('../core/Types').ElementLike} ElementLike
   * @typedef {import('../core/EventBus').default} EventBus
   * @typedef {import('./CommandStack').CommandContext} CommandContext
   *
   * @typedef {string|string[]} Events
   * @typedef { (context: CommandContext) => ElementLike[] | void } HandlerFunction
   * @typedef { (context: CommandContext) => void } ComposeHandlerFunction
   */

  var DEFAULT_PRIORITY$5 = 1000;

  /**
   * A utility that can be used to plug into the command execution for
   * extension and/or validation.
   *
   * @class
   * @constructor
   *
   * @example
   *
   * ```javascript
   * import CommandInterceptor from 'diagram-js/lib/command/CommandInterceptor';
   *
   * class CommandLogger extends CommandInterceptor {
   *   constructor(eventBus) {
   *     super(eventBus);
   *
   *   this.preExecute('shape.create', (event) => {
   *     console.log('commandStack.shape-create.preExecute', event);
   *   });
   * }
   * ```
   *
   * @param {EventBus} eventBus
   */
  function CommandInterceptor(eventBus) {
    /**
     * @type {EventBus}
     */
    this._eventBus = eventBus;
  }
  CommandInterceptor.$inject = ['eventBus'];
  function unwrapEvent(fn, that) {
    return function (event) {
      return fn.call(that || null, event.context, event.command, event);
    };
  }

  /**
   * Intercept a command during one of the phases.
   *
   * @param {Events} [events] command(s) to intercept
   * @param {string} [hook] phase to intercept
   * @param {number} [priority]
   * @param {ComposeHandlerFunction|HandlerFunction} handlerFn
   * @param {boolean} [unwrap] whether the event should be unwrapped
   * @param {any} [that]
   */
  CommandInterceptor.prototype.on = function (events, hook, priority, handlerFn, unwrap, that) {
    if (isFunction$2(hook) || isNumber$3(hook)) {
      that = unwrap;
      unwrap = handlerFn;
      handlerFn = priority;
      priority = hook;
      hook = null;
    }
    if (isFunction$2(priority)) {
      that = unwrap;
      unwrap = handlerFn;
      handlerFn = priority;
      priority = DEFAULT_PRIORITY$5;
    }
    if (isObject$2(unwrap)) {
      that = unwrap;
      unwrap = false;
    }
    if (!isFunction$2(handlerFn)) {
      throw new Error('handlerFn must be a function');
    }
    if (!isArray$6(events)) {
      events = [events];
    }
    var eventBus = this._eventBus;
    forEach$3(events, function (event) {
      // concat commandStack(.event)?(.hook)?
      var fullEvent = ['commandStack', event, hook].filter(function (e) {
        return e;
      }).join('.');
      eventBus.on(fullEvent, priority, unwrap ? unwrapEvent(handlerFn, that) : handlerFn, that);
    });
  };

  /**
   * Add a <canExecute> phase of command interceptor.
   *
   * @param {Events} [events] command(s) to intercept
   * @param {number} [priority]
   * @param {ComposeHandlerFunction|HandlerFunction} handlerFn
   * @param {boolean} [unwrap] whether the event should be unwrapped
   * @param {any} [that]
   */
  CommandInterceptor.prototype.canExecute = createHook('canExecute');

  /**
   * Add a <preExecute> phase of command interceptor.
   *
   * @param {Events} [events] command(s) to intercept
   * @param {number} [priority]
   * @param {ComposeHandlerFunction|HandlerFunction} handlerFn
   * @param {boolean} [unwrap] whether the event should be unwrapped
   * @param {any} [that]
   */
  CommandInterceptor.prototype.preExecute = createHook('preExecute');

  /**
   * Add a <preExecuted> phase of command interceptor.
   *
   * @param {Events} [events] command(s) to intercept
   * @param {number} [priority]
   * @param {ComposeHandlerFunction|HandlerFunction} handlerFn
   * @param {boolean} [unwrap] whether the event should be unwrapped
   * @param {any} [that]
   */
  CommandInterceptor.prototype.preExecuted = createHook('preExecuted');

  /**
   * Add a <execute> phase of command interceptor.
   *
   * @param {Events} [events] command(s) to intercept
   * @param {number} [priority]
   * @param {ComposeHandlerFunction|HandlerFunction} handlerFn
   * @param {boolean} [unwrap] whether the event should be unwrapped
   * @param {any} [that]
   */
  CommandInterceptor.prototype.execute = createHook('execute');

  /**
   * Add a <executed> phase of command interceptor.
   *
   * @param {Events} [events] command(s) to intercept
   * @param {number} [priority]
   * @param {ComposeHandlerFunction|HandlerFunction} handlerFn
   * @param {boolean} [unwrap] whether the event should be unwrapped
   * @param {any} [that]
   */
  CommandInterceptor.prototype.executed = createHook('executed');

  /**
   * Add a <postExecute> phase of command interceptor.
   *
   * @param {Events} [events] command(s) to intercept
   * @param {number} [priority]
   * @param {ComposeHandlerFunction|HandlerFunction} handlerFn
   * @param {boolean} [unwrap] whether the event should be unwrapped
   * @param {any} [that]
   */
  CommandInterceptor.prototype.postExecute = createHook('postExecute');

  /**
   * Add a <postExecuted> phase of command interceptor.
   *
   * @param {Events} [events] command(s) to intercept
   * @param {number} [priority]
   * @param {ComposeHandlerFunction|HandlerFunction} handlerFn
   * @param {boolean} [unwrap] whether the event should be unwrapped
   * @param {any} [that]
   */
  CommandInterceptor.prototype.postExecuted = createHook('postExecuted');

  /**
   * Add a <revert> phase of command interceptor.
   *
   * @param {Events} [events] command(s) to intercept
   * @param {number} [priority]
   * @param {ComposeHandlerFunction|HandlerFunction} handlerFn
   * @param {boolean} [unwrap] whether the event should be unwrapped
   * @param {any} [that]
   */
  CommandInterceptor.prototype.revert = createHook('revert');

  /**
   * Add a <reverted> phase of command interceptor.
   *
   * @param {Events} [events] command(s) to intercept
   * @param {number} [priority]
   * @param {ComposeHandlerFunction|HandlerFunction} handlerFn
   * @param {boolean} [unwrap] whether the event should be unwrapped
   * @param {any} [that]
   */
  CommandInterceptor.prototype.reverted = createHook('reverted');

  /*
   * Add prototype methods for each phase of command execution (e.g. execute,
   * revert).
   *
   * @param {string} hook
   *
   * @return { (
   *   events?: Events,
   *   priority?: number,
   *   handlerFn: ComposeHandlerFunction|HandlerFunction,
   *   unwrap?: boolean
   * ) => any }
   */
  function createHook(hook) {
    /**
     * @this {CommandInterceptor}
     *
     * @param {Events} [events]
     * @param {number} [priority]
     * @param {ComposeHandlerFunction|HandlerFunction} handlerFn
     * @param {boolean} [unwrap]
     * @param {any} [that]
     */
    const hookFn = function (events, priority, handlerFn, unwrap, that) {
      if (isFunction$2(events) || isNumber$3(events)) {
        that = unwrap;
        unwrap = handlerFn;
        handlerFn = priority;
        priority = events;
        events = null;
      }
      this.on(events, hook, priority, handlerFn, unwrap, that);
    };
    return hookFn;
  }

  /**
   * @typedef {import('../../../model/Types').Shape} Shape
   *
   * @typedef {import('../../../util/Types').Rect} Rect
   *
   * @typedef {import('../../../core/EventBus').default} EventBus
   * @typedef {import('../../grid-snapping/GridSnapping').default} GridSnapping
   */

  /**
   * Integrates resizing with grid snapping.
   *
   * @param {EventBus} eventBus
   * @param {GridSnapping} gridSnapping
   */
  function ResizeBehavior(eventBus, gridSnapping) {
    CommandInterceptor.call(this, eventBus);
    this._gridSnapping = gridSnapping;
    var self = this;
    this.preExecute('shape.resize', function (event) {
      var context = event.context,
        hints = context.hints || {},
        autoResize = hints.autoResize;
      if (!autoResize) {
        return;
      }
      var shape = context.shape,
        newBounds = context.newBounds;
      if (isString$4(autoResize)) {
        context.newBounds = self.snapComplex(newBounds, autoResize);
      } else {
        context.newBounds = self.snapSimple(shape, newBounds);
      }
    });
  }
  ResizeBehavior.$inject = ['eventBus', 'gridSnapping', 'modeling'];
  e$3(ResizeBehavior, CommandInterceptor);

  /**
   * Snap width and height in relation to center.
   *
   * @param {Shape} shape
   * @param {Rect} newBounds
   *
   * @return {Rect} Snapped bounds.
   */
  ResizeBehavior.prototype.snapSimple = function (shape, newBounds) {
    var gridSnapping = this._gridSnapping;
    newBounds.width = gridSnapping.snapValue(newBounds.width, {
      min: newBounds.width
    });
    newBounds.height = gridSnapping.snapValue(newBounds.height, {
      min: newBounds.height
    });
    newBounds.x = shape.x + shape.width / 2 - newBounds.width / 2;
    newBounds.y = shape.y + shape.height / 2 - newBounds.height / 2;
    return newBounds;
  };

  /**
   * Snap x, y, width and height according to given directions.
   *
   * @param {Rect} newBounds
   * @param {string} directions - Directions as {n|w|s|e}.
   *
   * @return {Rect} Snapped bounds.
   */
  ResizeBehavior.prototype.snapComplex = function (newBounds, directions) {
    if (/w|e/.test(directions)) {
      newBounds = this.snapHorizontally(newBounds, directions);
    }
    if (/n|s/.test(directions)) {
      newBounds = this.snapVertically(newBounds, directions);
    }
    return newBounds;
  };

  /**
   * Snap in one or both directions horizontally.
   *
   * @param {Rect} newBounds
   * @param {string} directions - Directions as {n|w|s|e}.
   *
   * @return {Rect} Snapped bounds.
   */
  ResizeBehavior.prototype.snapHorizontally = function (newBounds, directions) {
    var gridSnapping = this._gridSnapping,
      west = /w/.test(directions),
      east = /e/.test(directions);
    var snappedNewBounds = {};
    snappedNewBounds.width = gridSnapping.snapValue(newBounds.width, {
      min: newBounds.width
    });
    if (east) {
      // handle <we>
      if (west) {
        snappedNewBounds.x = gridSnapping.snapValue(newBounds.x, {
          max: newBounds.x
        });
        snappedNewBounds.width += gridSnapping.snapValue(newBounds.x - snappedNewBounds.x, {
          min: newBounds.x - snappedNewBounds.x
        });
      }

      // handle <e>
      else {
        newBounds.x = newBounds.x + newBounds.width - snappedNewBounds.width;
      }
    }

    // assign snapped x and width
    assign$4(newBounds, snappedNewBounds);
    return newBounds;
  };

  /**
   * Snap in one or both directions vertically.
   *
   * @param {Rect} newBounds
   * @param {string} directions - Directions as {n|w|s|e}.
   *
   * @return {Rect} Snapped bounds.
   */
  ResizeBehavior.prototype.snapVertically = function (newBounds, directions) {
    var gridSnapping = this._gridSnapping,
      north = /n/.test(directions),
      south = /s/.test(directions);
    var snappedNewBounds = {};
    snappedNewBounds.height = gridSnapping.snapValue(newBounds.height, {
      min: newBounds.height
    });
    if (north) {
      // handle <ns>
      if (south) {
        snappedNewBounds.y = gridSnapping.snapValue(newBounds.y, {
          max: newBounds.y
        });
        snappedNewBounds.height += gridSnapping.snapValue(newBounds.y - snappedNewBounds.y, {
          min: newBounds.y - snappedNewBounds.y
        });
      }

      // handle <n>
      else {
        newBounds.y = newBounds.y + newBounds.height - snappedNewBounds.height;
      }
    }

    // assign snapped y and height
    assign$4(newBounds, snappedNewBounds);
    return newBounds;
  };

  /**
   * @typedef {import('../../../core/EventBus').default} EventBus
   * @typedef {import('../../grid-snapping/GridSnapping').default} GridSnapping
   */

  var HIGH_PRIORITY$7 = 2000;

  /**
   * Integrates space tool with grid snapping.
   *
   * @param {EventBus} eventBus
   * @param {GridSnapping} gridSnapping
   */
  function SpaceToolBehavior(eventBus, gridSnapping) {
    eventBus.on(['spaceTool.move', 'spaceTool.end'], HIGH_PRIORITY$7, function (event) {
      var context = event.context;
      if (!context.initialized) {
        return;
      }
      var axis = context.axis;
      var snapped;
      if (axis === 'x') {
        // snap delta x to multiple of 10
        snapped = gridSnapping.snapValue(event.dx);
        event.x = event.x + snapped - event.dx;
        event.dx = snapped;
      } else {
        // snap delta y to multiple of 10
        snapped = gridSnapping.snapValue(event.dy);
        event.y = event.y + snapped - event.dy;
        event.dy = snapped;
      }
    });
  }
  SpaceToolBehavior.$inject = ['eventBus', 'gridSnapping'];

  var GridSnappingBehaviorModule = {
    __init__: ['gridSnappingResizeBehavior', 'gridSnappingSpaceToolBehavior'],
    gridSnappingResizeBehavior: ['type', ResizeBehavior],
    gridSnappingSpaceToolBehavior: ['type', SpaceToolBehavior]
  };

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var GridSnappingModule = {
    __depends__: [GridSnappingBehaviorModule],
    __init__: ['gridSnapping'],
    gridSnapping: ['type', GridSnapping]
  };

  /**
   * @typedef {import('../../core/EventBus').default} EventBus
   *
   * @typedef {({ keyEvent: KeyboardEvent }) => any} Listener
   */

  var KEYDOWN_EVENT = 'keyboard.keydown',
    KEYUP_EVENT = 'keyboard.keyup';
  var DEFAULT_PRIORITY$4 = 1000;
  var compatMessage$2 = 'Keyboard binding is now implicit; explicit binding to an element got removed. For more information, see https://github.com/bpmn-io/diagram-js/issues/661';

  /**
   * A keyboard abstraction that may be activated and
   * deactivated by users at will, consuming global key events
   * and triggering diagram actions.
   *
   * For keys pressed down, keyboard fires `keyboard.keydown` event.
   * The event context contains one field which is `KeyboardEvent` event.
   *
   * The implementation fires the following key events that allow
   * other components to hook into key handling:
   *
   *  - keyboard.bind
   *  - keyboard.unbind
   *  - keyboard.init
   *  - keyboard.destroy
   *
   * All events contain one field which is node.
   *
   * Specify the initial keyboard binding state via the
   * `keyboard.bind=true|false` configuration option.
   *
   * @param {Object} config
   * @param {EventTarget} [config.bindTo]
   * @param {EventBus} eventBus
   */
  function Keyboard$2(config, eventBus) {
    var self = this;
    this._config = config = config || {};
    this._eventBus = eventBus;
    this._keydownHandler = this._keydownHandler.bind(this);
    this._keyupHandler = this._keyupHandler.bind(this);

    // properly clean dom registrations
    eventBus.on('diagram.destroy', function () {
      self._fire('destroy');
      self.unbind();
    });
    if (config.bindTo) {
      console.error('unsupported configuration <keyboard.bindTo>', new Error(compatMessage$2));
    }
    var bind = config && config.bind !== false;
    eventBus.on('canvas.init', function (event) {
      self._target = event.svg;
      if (bind) {
        self.bind();
      }
      self._fire('init');
    });
  }
  Keyboard$2.$inject = ['config.keyboard', 'eventBus'];
  Keyboard$2.prototype._keydownHandler = function (event) {
    this._keyHandler(event, KEYDOWN_EVENT);
  };
  Keyboard$2.prototype._keyupHandler = function (event) {
    this._keyHandler(event, KEYUP_EVENT);
  };
  Keyboard$2.prototype._keyHandler = function (event, type) {
    var eventBusResult;
    if (this._isEventIgnored(event)) {
      return;
    }
    var context = {
      keyEvent: event
    };
    eventBusResult = this._eventBus.fire(type || KEYDOWN_EVENT, context);
    if (eventBusResult) {
      event.preventDefault();
    }
  };
  Keyboard$2.prototype._isEventIgnored = function (event) {
    return false;
  };

  /**
   * Bind keyboard events to the given DOM node.
   *
   * @param {EventTarget} node
   */
  Keyboard$2.prototype.bind = function (node) {
    if (node) {
      console.error('unsupported argument <node>', new Error(compatMessage$2));
    }

    // make sure that the keyboard is only bound once to the DOM
    this.unbind();
    node = this._node = this._target;

    // bind key events
    event.bind(node, 'keydown', this._keydownHandler);
    event.bind(node, 'keyup', this._keyupHandler);
    this._fire('bind');
  };

  /**
   * @return {EventTarget}
   */
  Keyboard$2.prototype.getBinding = function () {
    return this._node;
  };
  Keyboard$2.prototype.unbind = function () {
    var node = this._node;
    if (node) {
      this._fire('unbind');

      // unbind key events
      event.unbind(node, 'keydown', this._keydownHandler);
      event.unbind(node, 'keyup', this._keyupHandler);
    }
    this._node = null;
  };

  /**
   * @param {string} event
   */
  Keyboard$2.prototype._fire = function (event) {
    this._eventBus.fire('keyboard.' + event, {
      node: this._node
    });
  };

  /**
   * Add a listener function that is notified with `KeyboardEvent` whenever
   * the keyboard is bound and the user presses a key. If no priority is
   * provided, the default value of 1000 is used.
   *
   * @param {number} [priority]
   * @param {Listener} listener
   * @param {string} [type='keyboard.keydown']
   */
  Keyboard$2.prototype.addListener = function (priority, listener, type) {
    if (isFunction$2(priority)) {
      type = listener;
      listener = priority;
      priority = DEFAULT_PRIORITY$4;
    }
    this._eventBus.on(type || KEYDOWN_EVENT, priority, listener);
  };

  /**
   * Remove a listener function.
   *
   * @param {Listener} listener
   * @param {string} [type='keyboard.keydown']
   */
  Keyboard$2.prototype.removeListener = function (listener, type) {
    this._eventBus.off(type || KEYDOWN_EVENT, listener);
  };
  Keyboard$2.prototype.hasModifier = hasModifier;
  Keyboard$2.prototype.isCmd = isCmd$5;
  Keyboard$2.prototype.isShift = isShift$2;
  Keyboard$2.prototype.isKey = isKey;

  var LOW_PRIORITY$g = 500;

  /**
   * Adds default keyboard bindings.
   *
   * This does not pull in any features will bind only actions that
   * have previously been registered against the editorActions component.
   *
   * @param {EventBus} eventBus
   * @param {Keyboard} keyboard
   */
  function KeyboardBindings$1(eventBus, keyboard) {
    var self = this;
    eventBus.on('editorActions.init', LOW_PRIORITY$g, function (event) {
      var editorActions = event.editorActions;
      self.registerBindings(keyboard, editorActions);
    });
  }
  KeyboardBindings$1.$inject = ['eventBus', 'keyboard'];

  /**
   * Register available keyboard bindings.
   *
   * @param {Keyboard} keyboard
   * @param {EditorActions} editorActions
   */
  KeyboardBindings$1.prototype.registerBindings = function (keyboard, editorActions) {
    /**
     * Add keyboard binding if respective editor action
     * is registered.
     *
     * @param {string} action name
     * @param {Function} fn that implements the key binding
     */
    function addListener(action, fn) {
      if (editorActions.isRegistered(action)) {
        keyboard.addListener(fn);
      }
    }

    // undo
    // (CTRL|CMD) + Z
    addListener('undo', function (context) {
      var event = context.keyEvent;
      if (isUndo(event)) {
        editorActions.trigger('undo');
        return true;
      }
    });

    // redo
    // CTRL + Y
    // CMD + SHIFT + Z
    addListener('redo', function (context) {
      var event = context.keyEvent;
      if (isRedo(event)) {
        editorActions.trigger('redo');
        return true;
      }
    });

    // copy
    // CTRL/CMD + C
    addListener('copy', function (context) {
      var event = context.keyEvent;
      if (isCopy(event)) {
        editorActions.trigger('copy');
        return true;
      }
    });

    // paste
    // CTRL/CMD + V
    addListener('paste', function (context) {
      var event = context.keyEvent;
      if (isPaste(event)) {
        editorActions.trigger('paste');
        return true;
      }
    });

    // zoom in one step
    // CTRL/CMD + +
    addListener('stepZoom', function (context) {
      var event = context.keyEvent;

      // quirk: it has to be triggered by `=` as well to work on international keyboard layout
      // cf: https://github.com/bpmn-io/bpmn-js/issues/1362#issuecomment-722989754
      if (isKey(['+', 'Add', '='], event) && isCmd$5(event)) {
        editorActions.trigger('stepZoom', {
          value: 1
        });
        return true;
      }
    });

    // zoom out one step
    // CTRL + -
    addListener('stepZoom', function (context) {
      var event = context.keyEvent;
      if (isKey(['-', 'Subtract'], event) && isCmd$5(event)) {
        editorActions.trigger('stepZoom', {
          value: -1
        });
        return true;
      }
    });

    // zoom to the default level
    // CTRL + 0
    addListener('zoom', function (context) {
      var event = context.keyEvent;
      if (isKey('0', event) && isCmd$5(event)) {
        editorActions.trigger('zoom', {
          value: 1
        });
        return true;
      }
    });

    // delete selected element
    // DEL
    addListener('removeSelection', function (context) {
      var event = context.keyEvent;
      if (isKey(['Backspace', 'Delete', 'Del'], event)) {
        editorActions.trigger('removeSelection');
        return true;
      }
    });
  };

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var KeyboardModule$3 = {
    __init__: ['keyboard', 'keyboardBindings'],
    keyboard: ['type', Keyboard$2],
    keyboardBindings: ['type', KeyboardBindings$1]
  };

  /**
   * DRD specific key bindings.
   *
   * @param {Keyboard} keyboard
   * @param {EditorActions} editorActions
   */
  function DrdKeyboardBindings(injector) {
    injector.invoke(KeyboardBindings$1, this);
  }
  e$3(DrdKeyboardBindings, KeyboardBindings$1);
  DrdKeyboardBindings.$inject = ['injector'];

  /**
   * Register available keyboard bindings.
   *
   * @param {Keyboard} keyboard
   * @param {EditorActions} editorActions
   */
  DrdKeyboardBindings.prototype.registerBindings = function (keyboard, editorActions) {
    // inherit default bindings
    KeyboardBindings$1.prototype.registerBindings.call(this, keyboard, editorActions);

    /**
     * Add keyboard binding if respective editor action
     * is registered.
     *
     * @param {string} action name
     * @param {Function} fn that implements the key binding
     */
    function addListener(action, fn) {
      if (editorActions.isRegistered(action)) {
        keyboard.addListener(fn);
      }
    }

    // select all elements
    // CTRL + A
    addListener('selectElements', function (context) {
      var event = context.keyEvent;
      if (keyboard.isKey(['a', 'A'], event) && keyboard.isCmd(event)) {
        editorActions.trigger('selectElements');
        return true;
      }
    });

    // activate lasso tool
    // L
    addListener('lassoTool', function (context) {
      var event = context.keyEvent;
      if (keyboard.hasModifier(event)) {
        return;
      }
      if (keyboard.isKey(['l', 'L'], event)) {
        editorActions.trigger('lassoTool');
        return true;
      }
    });

    // activate hand tool
    // H
    addListener('handTool', function (context) {
      var event = context.keyEvent;
      if (keyboard.hasModifier(event)) {
        return;
      }
      if (keyboard.isKey(['h', 'H'], event)) {
        editorActions.trigger('handTool');
        return true;
      }
    });

    // activate direct editing
    // E
    addListener('directEditing', function (context) {
      var event = context.keyEvent;
      if (keyboard.hasModifier(event)) {
        return;
      }
      if (keyboard.isKey(['e', 'E'], event)) {
        editorActions.trigger('directEditing');
        return true;
      }
    });

    // search labels
    // CTRL + F
    addListener('find', function (context) {
      var event = context.keyEvent;
      if (keyboard.isKey(['f', 'F'], event) && keyboard.isCmd(event)) {
        editorActions.trigger('find');
        return true;
      }
    });
  };

  var KeyboardModule$2 = {
    __depends__: [KeyboardModule$3],
    __init__: ['keyboardBindings'],
    keyboardBindings: ['type', DrdKeyboardBindings]
  };

  /**
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../features/keyboard/Keyboard').default} Keyboard
   */

  var DEFAULT_CONFIG$1 = {
    moveSpeed: 50,
    moveSpeedAccelerated: 200
  };

  /**
   * A feature that allows users to move the canvas using the keyboard.
   *
   * @param {Object} config
   * @param {number} [config.moveSpeed=50]
   * @param {number} [config.moveSpeedAccelerated=200]
   * @param {Keyboard} keyboard
   * @param {Canvas} canvas
   */
  function KeyboardMove(config, keyboard, canvas) {
    var self = this;
    this._config = assign$4({}, DEFAULT_CONFIG$1, config || {});
    keyboard.addListener(arrowsListener);
    function arrowsListener(context) {
      var event = context.keyEvent,
        config = self._config;
      if (!keyboard.isCmd(event)) {
        return;
      }
      if (keyboard.isKey(['ArrowLeft', 'Left', 'ArrowUp', 'Up', 'ArrowDown', 'Down', 'ArrowRight', 'Right'], event)) {
        var speed = keyboard.isShift(event) ? config.moveSpeedAccelerated : config.moveSpeed;
        var direction;
        switch (event.key) {
          case 'ArrowLeft':
          case 'Left':
            direction = 'left';
            break;
          case 'ArrowUp':
          case 'Up':
            direction = 'up';
            break;
          case 'ArrowRight':
          case 'Right':
            direction = 'right';
            break;
          case 'ArrowDown':
          case 'Down':
            direction = 'down';
            break;
        }
        self.moveCanvas({
          speed: speed,
          direction: direction
        });
        return true;
      }
    }

    /**
     * @param {{
     *   direction: 'up' | 'down' | 'left' | 'right';
     *   speed: number;
     * }} options
     */
    this.moveCanvas = function (options) {
      var dx = 0,
        dy = 0,
        speed = options.speed;
      var actualSpeed = speed / Math.min(Math.sqrt(canvas.viewbox().scale), 1);
      switch (options.direction) {
        case 'left':
          // Left
          dx = actualSpeed;
          break;
        case 'up':
          // Up
          dy = actualSpeed;
          break;
        case 'right':
          // Right
          dx = -actualSpeed;
          break;
        case 'down':
          // Down
          dy = -actualSpeed;
          break;
      }
      canvas.scroll({
        dx: dx,
        dy: dy
      });
    };
  }
  KeyboardMove.$inject = ['config.keyboardMove', 'keyboard', 'canvas'];

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var KeyboardMoveModule = {
    __depends__: [KeyboardModule$3],
    __init__: ['keyboardMove'],
    keyboardMove: ['type', KeyboardMove]
  };

  /**
   * @typedef {import('../keyboard/Keyboard').default} Keyboard
   * @typedef {import('../modeling/Modeling').default} Modeling
   * @typedef {import('../rules/Rules').default} Rules
   * @typedef {import('../selection/Selection').default} Selection
   */

  var DEFAULT_CONFIG = {
    moveSpeed: 1,
    moveSpeedAccelerated: 10
  };
  var HIGHER_PRIORITY$4 = 1500;
  var LEFT$1 = 'left';
  var UP = 'up';
  var RIGHT$1 = 'right';
  var DOWN = 'down';
  var KEY_TO_DIRECTION = {
    ArrowLeft: LEFT$1,
    Left: LEFT$1,
    ArrowUp: UP,
    Up: UP,
    ArrowRight: RIGHT$1,
    Right: RIGHT$1,
    ArrowDown: DOWN,
    Down: DOWN
  };
  var DIRECTIONS_DELTA = {
    left: function (speed) {
      return {
        x: -speed,
        y: 0
      };
    },
    up: function (speed) {
      return {
        x: 0,
        y: -speed
      };
    },
    right: function (speed) {
      return {
        x: speed,
        y: 0
      };
    },
    down: function (speed) {
      return {
        x: 0,
        y: speed
      };
    }
  };

  /**
   * Enables to move selection with keyboard arrows.
   * Use with Shift for modified speed (default=1, with Shift=10).
   * Pressed Cmd/Ctrl turns the feature off.
   *
   * @param {Object} config
   * @param {number} [config.moveSpeed=1]
   * @param {number} [config.moveSpeedAccelerated=10]
   * @param {Keyboard} keyboard
   * @param {Modeling} modeling
   * @param {Rules} rules
   * @param {Selection} selection
   */
  function KeyboardMoveSelection(config, keyboard, modeling, rules, selection) {
    var self = this;
    this._config = assign$4({}, DEFAULT_CONFIG, config || {});
    keyboard.addListener(HIGHER_PRIORITY$4, function (event) {
      var keyEvent = event.keyEvent;
      var direction = KEY_TO_DIRECTION[keyEvent.key];
      if (!direction) {
        return;
      }
      if (keyboard.isCmd(keyEvent)) {
        return;
      }
      var accelerated = keyboard.isShift(keyEvent);
      self.moveSelection(direction, accelerated);
      return true;
    });

    /**
     * Move selected elements in the given direction,
     * optionally specifying accelerated movement.
     *
     * @param {string} direction
     * @param {boolean} [accelerated=false]
     */
    this.moveSelection = function (direction, accelerated) {
      var selectedElements = selection.get();
      if (!selectedElements.length) {
        return;
      }
      var speed = this._config[accelerated ? 'moveSpeedAccelerated' : 'moveSpeed'];
      var delta = DIRECTIONS_DELTA[direction](speed);
      var canMove = rules.allowed('elements.move', {
        shapes: selectedElements
      });
      if (canMove) {
        modeling.moveElements(selectedElements, delta);
      }
    };
  }
  KeyboardMoveSelection.$inject = ['config.keyboardMoveSelection', 'keyboard', 'modeling', 'rules', 'selection'];

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var KeyboardMoveSelectionModule = {
    __depends__: [KeyboardModule$3, SelectionModule],
    __init__: ['keyboardMoveSelection'],
    keyboardMoveSelection: ['type', KeyboardMoveSelection]
  };

  /**
   * @typedef {import('didi').Injector} Injector
   *
   * @typedef {import('../core/Types').ElementLike} ElementLike
   *
   * @typedef {import('../core/EventBus').default} EventBus
   * @typedef {import('./CommandHandler').default} CommandHandler
   *
   * @typedef { any } CommandContext
   * @typedef { {
   *   new (...args: any[]) : CommandHandler
   * } } CommandHandlerConstructor
   * @typedef { {
   *   [key: string]: CommandHandler;
   * } } CommandHandlerMap
   * @typedef { {
   *   command: string;
   *   context: any;
   *   id?: any;
   * } } CommandStackAction
   * @typedef { {
   *   actions: CommandStackAction[];
   *   dirty: ElementLike[];
   *   trigger: 'execute' | 'undo' | 'redo' | 'clear' | null;
   *   atomic?: boolean;
   * } } CurrentExecution
   */

  /**
   * A service that offers un- and redoable execution of commands.
   *
   * The command stack is responsible for executing modeling actions
   * in a un- and redoable manner. To do this it delegates the actual
   * command execution to {@link CommandHandler}s.
   *
   * Command handlers provide {@link CommandHandler#execute(ctx)} and
   * {@link CommandHandler#revert(ctx)} methods to un- and redo a command
   * identified by a command context.
   *
   *
   * ## Life-Cycle events
   *
   * In the process the command stack fires a number of life-cycle events
   * that other components to participate in the command execution.
   *
   *    * preExecute
   *    * preExecuted
   *    * execute
   *    * executed
   *    * postExecute
   *    * postExecuted
   *    * revert
   *    * reverted
   *
   * A special event is used for validating, whether a command can be
   * performed prior to its execution.
   *
   *    * canExecute
   *
   * Each of the events is fired as `commandStack.{eventName}` and
   * `commandStack.{commandName}.{eventName}`, respectively. This gives
   * components fine grained control on where to hook into.
   *
   * The event object fired transports `command`, the name of the
   * command and `context`, the command context.
   *
   *
   * ## Creating Command Handlers
   *
   * Command handlers should provide the {@link CommandHandler#execute(ctx)}
   * and {@link CommandHandler#revert(ctx)} methods to implement
   * redoing and undoing of a command.
   *
   * A command handler _must_ ensure undo is performed properly in order
   * not to break the undo chain. It must also return the shapes that
   * got changed during the `execute` and `revert` operations.
   *
   * Command handlers may execute other modeling operations (and thus
   * commands) in their `preExecute(d)` and `postExecute(d)` phases. The command
   * stack will properly group all commands together into a logical unit
   * that may be re- and undone atomically.
   *
   * Command handlers must not execute other commands from within their
   * core implementation (`execute`, `revert`).
   *
   *
   * ## Change Tracking
   *
   * During the execution of the CommandStack it will keep track of all
   * elements that have been touched during the command's execution.
   *
   * At the end of the CommandStack execution it will notify interested
   * components via an 'elements.changed' event with all the dirty
   * elements.
   *
   * The event can be picked up by components that are interested in the fact
   * that elements have been changed. One use case for this is updating
   * their graphical representation after moving / resizing or deletion.
   *
   * @see CommandHandler
   *
   * @param {EventBus} eventBus
   * @param {Injector} injector
   */
  function CommandStack$1(eventBus, injector) {
    /**
     * A map of all registered command handlers.
     *
     * @type {CommandHandlerMap}
     */
    this._handlerMap = {};

    /**
     * A stack containing all re/undoable actions on the diagram
     *
     * @type {CommandStackAction[]}
     */
    this._stack = [];

    /**
     * The current index on the stack
     *
     * @type {number}
     */
    this._stackIdx = -1;

    /**
     * Current active commandStack execution
     *
     * @type {CurrentExecution}
     */
    this._currentExecution = {
      actions: [],
      dirty: [],
      trigger: null
    };

    /**
     * @type {Injector}
     */
    this._injector = injector;

    /**
     * @type EventBus
     */
    this._eventBus = eventBus;

    /**
     * @type { number }
     */
    this._uid = 1;
    eventBus.on(['diagram.destroy', 'diagram.clear'], function () {
      this.clear(false);
    }, this);
  }
  CommandStack$1.$inject = ['eventBus', 'injector'];

  /**
   * Execute a command.
   *
   * @param {string} command The command to execute.
   * @param {CommandContext} context The context with which to execute the command.
   */
  CommandStack$1.prototype.execute = function (command, context) {
    if (!command) {
      throw new Error('command required');
    }
    this._currentExecution.trigger = 'execute';
    const action = {
      command: command,
      context: context
    };
    this._pushAction(action);
    this._internalExecute(action);
    this._popAction();
  };

  /**
   * Check whether a command can be executed.
   *
   * Implementors may hook into the mechanism on two ways:
   *
   *   * in event listeners:
   *
   *     Users may prevent the execution via an event listener.
   *     It must prevent the default action for `commandStack.(<command>.)canExecute` events.
   *
   *   * in command handlers:
   *
   *     If the method {@link CommandHandler#canExecute} is implemented in a handler
   *     it will be called to figure out whether the execution is allowed.
   *
   * @param {string} command The command to execute.
   * @param {CommandContext} context The context with which to execute the command.
   *
   * @return {boolean} Whether the command can be executed with the given context.
   */
  CommandStack$1.prototype.canExecute = function (command, context) {
    const action = {
      command: command,
      context: context
    };
    const handler = this._getHandler(command);
    let result = this._fire(command, 'canExecute', action);

    // handler#canExecute will only be called if no listener
    // decided on a result already
    if (result === undefined) {
      if (!handler) {
        return false;
      }
      if (handler.canExecute) {
        result = handler.canExecute(context);
      }
    }
    return result;
  };

  /**
   * Clear the command stack, erasing all undo / redo history.
   *
   * @param {boolean} [emit=true] Whether to fire an event. Defaults to `true`.
   */
  CommandStack$1.prototype.clear = function (emit) {
    this._stack.length = 0;
    this._stackIdx = -1;
    if (emit !== false) {
      this._fire('changed', {
        trigger: 'clear'
      });
    }
  };

  /**
   * Undo last command(s)
   */
  CommandStack$1.prototype.undo = function () {
    let action = this._getUndoAction(),
      next;
    if (action) {
      this._currentExecution.trigger = 'undo';
      this._pushAction(action);
      while (action) {
        this._internalUndo(action);
        next = this._getUndoAction();
        if (!next || next.id !== action.id) {
          break;
        }
        action = next;
      }
      this._popAction();
    }
  };

  /**
   * Redo last command(s)
   */
  CommandStack$1.prototype.redo = function () {
    let action = this._getRedoAction(),
      next;
    if (action) {
      this._currentExecution.trigger = 'redo';
      this._pushAction(action);
      while (action) {
        this._internalExecute(action, true);
        next = this._getRedoAction();
        if (!next || next.id !== action.id) {
          break;
        }
        action = next;
      }
      this._popAction();
    }
  };

  /**
   * Register a handler instance with the command stack.
   *
   * @param {string} command Command to be executed.
   * @param {CommandHandler} handler Handler to execute the command.
   */
  CommandStack$1.prototype.register = function (command, handler) {
    this._setHandler(command, handler);
  };

  /**
   * Register a handler type with the command stack  by instantiating it and
   * injecting its dependencies.
   *
   * @param {string} command Command to be executed.
   * @param {CommandHandlerConstructor} handlerCls Constructor to instantiate a {@link CommandHandler}.
   */
  CommandStack$1.prototype.registerHandler = function (command, handlerCls) {
    if (!command || !handlerCls) {
      throw new Error('command and handlerCls must be defined');
    }
    const handler = this._injector.instantiate(handlerCls);
    this.register(command, handler);
  };

  /**
   * @return {boolean}
   */
  CommandStack$1.prototype.canUndo = function () {
    return !!this._getUndoAction();
  };

  /**
   * @return {boolean}
   */
  CommandStack$1.prototype.canRedo = function () {
    return !!this._getRedoAction();
  };

  // stack access  //////////////////////

  CommandStack$1.prototype._getRedoAction = function () {
    return this._stack[this._stackIdx + 1];
  };
  CommandStack$1.prototype._getUndoAction = function () {
    return this._stack[this._stackIdx];
  };

  // internal functionality //////////////////////

  CommandStack$1.prototype._internalUndo = function (action) {
    const command = action.command,
      context = action.context;
    const handler = this._getHandler(command);

    // guard against illegal nested command stack invocations
    this._atomicDo(() => {
      this._fire(command, 'revert', action);
      if (handler.revert) {
        this._markDirty(handler.revert(context));
      }
      this._revertedAction(action);
      this._fire(command, 'reverted', action);
    });
  };
  CommandStack$1.prototype._fire = function (command, qualifier, event) {
    if (arguments.length < 3) {
      event = qualifier;
      qualifier = null;
    }
    const names = qualifier ? [command + '.' + qualifier, qualifier] : [command];
    let result;
    event = this._eventBus.createEvent(event);
    for (const name of names) {
      result = this._eventBus.fire('commandStack.' + name, event);
      if (event.cancelBubble) {
        break;
      }
    }
    return result;
  };
  CommandStack$1.prototype._createId = function () {
    return this._uid++;
  };
  CommandStack$1.prototype._atomicDo = function (fn) {
    const execution = this._currentExecution;
    execution.atomic = true;
    try {
      fn();
    } finally {
      execution.atomic = false;
    }
  };
  CommandStack$1.prototype._internalExecute = function (action, redo) {
    const command = action.command,
      context = action.context;
    const handler = this._getHandler(command);
    if (!handler) {
      throw new Error('no command handler registered for <' + command + '>');
    }
    this._pushAction(action);
    if (!redo) {
      this._fire(command, 'preExecute', action);
      if (handler.preExecute) {
        handler.preExecute(context);
      }
      this._fire(command, 'preExecuted', action);
    }

    // guard against illegal nested command stack invocations
    this._atomicDo(() => {
      this._fire(command, 'execute', action);
      if (handler.execute) {
        // actual execute + mark return results as dirty
        this._markDirty(handler.execute(context));
      }

      // log to stack
      this._executedAction(action, redo);
      this._fire(command, 'executed', action);
    });
    if (!redo) {
      this._fire(command, 'postExecute', action);
      if (handler.postExecute) {
        handler.postExecute(context);
      }
      this._fire(command, 'postExecuted', action);
    }
    this._popAction();
  };
  CommandStack$1.prototype._pushAction = function (action) {
    const execution = this._currentExecution,
      actions = execution.actions;
    const baseAction = actions[0];
    if (execution.atomic) {
      throw new Error('illegal invocation in <execute> or <revert> phase (action: ' + action.command + ')');
    }
    if (!action.id) {
      action.id = baseAction && baseAction.id || this._createId();
    }
    actions.push(action);
  };
  CommandStack$1.prototype._popAction = function () {
    const execution = this._currentExecution,
      trigger = execution.trigger,
      actions = execution.actions,
      dirty = execution.dirty;
    actions.pop();
    if (!actions.length) {
      this._eventBus.fire('elements.changed', {
        elements: uniqueBy('id', dirty.reverse())
      });
      dirty.length = 0;
      this._fire('changed', {
        trigger: trigger
      });
      execution.trigger = null;
    }
  };
  CommandStack$1.prototype._markDirty = function (elements) {
    const execution = this._currentExecution;
    if (!elements) {
      return;
    }
    elements = isArray$6(elements) ? elements : [elements];
    execution.dirty = execution.dirty.concat(elements);
  };
  CommandStack$1.prototype._executedAction = function (action, redo) {
    const stackIdx = ++this._stackIdx;
    if (!redo) {
      this._stack.splice(stackIdx, this._stack.length, action);
    }
  };
  CommandStack$1.prototype._revertedAction = function (action) {
    this._stackIdx--;
  };
  CommandStack$1.prototype._getHandler = function (command) {
    return this._handlerMap[command];
  };
  CommandStack$1.prototype._setHandler = function (command, handler) {
    if (!command || !handler) {
      throw new Error('command and handler required');
    }
    if (this._handlerMap[command]) {
      throw new Error('overriding handler for command <' + command + '>');
    }
    this._handlerMap[command] = handler;
  };

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var CommandStack = {
    commandStack: ['type', CommandStack$1]
  };

  /**
   * @typedef {import('../../core/Canvas').default} Canvas
   * @typedef {import('../../core/ElementRegistry').default} ElementRegistry
   * @typedef {import('../../core/EventBus').default} EventBus
   * @typedef {import('../../core/GraphicsFactory').default} GraphicsFactory
   */

  /**
   * Adds change support to the diagram, including
   *
   * <ul>
   *   <li>redrawing shapes and connections on change</li>
   * </ul>
   *
   * @param {EventBus} eventBus
   * @param {Canvas} canvas
   * @param {ElementRegistry} elementRegistry
   * @param {GraphicsFactory} graphicsFactory
   */
  function ChangeSupport$2(eventBus, canvas, elementRegistry, graphicsFactory) {
    // redraw shapes / connections on change

    eventBus.on('element.changed', function (event) {
      var element = event.element;

      // element might have been deleted and replaced by new element with same ID
      // thus check for parent of element except for root element
      if (element.parent || element === canvas.getRootElement()) {
        event.gfx = elementRegistry.getGraphics(element);
      }

      // shape + gfx may have been deleted
      if (!event.gfx) {
        return;
      }
      eventBus.fire(getType$1(element) + '.changed', event);
    });
    eventBus.on('elements.changed', function (event) {
      var elements = event.elements;
      elements.forEach(function (e) {
        eventBus.fire('element.changed', {
          element: e
        });
      });
      graphicsFactory.updateContainments(elements);
    });
    eventBus.on('shape.changed', function (event) {
      graphicsFactory.update('shape', event.element, event.gfx);
    });
    eventBus.on('connection.changed', function (event) {
      graphicsFactory.update('connection', event.element, event.gfx);
    });
  }
  ChangeSupport$2.$inject = ['eventBus', 'canvas', 'elementRegistry', 'graphicsFactory'];

  /**
   * @type { import('didi').ModuleDeclaration }
   */
  var DiagramChangeSupport = {
    __init__: ['changeSupport'],
    changeSupport: ['type', ChangeSupport$2]
  };

  var min$1 = Math.min,
    max$1 = Math.max;
  function preventDefault(e) {
    e.preventDefault();
  }
  function stopPropagation$1(e) {
    e.stopPropagation();
  }
  function isTextNode(node) {
    return node.nodeType === Node.TEXT_NODE;
  }
  function toArray(nodeList) {
    return [].slice.call(nodeList);
  }

  /**
   * Initializes a container for a content editable div.
   *
   * Structure:
   *
   * container
   *   parent
   *     content
   *     resize-handle
   *
   * @param {object} options
   * @param {DOMElement} options.container The DOM element to append the contentContainer to
   * @param {Function} options.keyHandler Handler for key events
   * @param {Function} options.resizeHandler Handler for resize events
   */
  function TextBox(options) {
    this.container = options.container;
    this.parent = domify$1('<div class="djs-direct-editing-parent">' + '<div class="djs-direct-editing-content" contenteditable="true"></div>' + '</div>');
    this.content = query('[contenteditable]', this.parent);
    this.keyHandler = options.keyHandler || function () {};
    this.resizeHandler = options.resizeHandler || function () {};
    this.autoResize = bind$3(this.autoResize, this);
    this.handlePaste = bind$3(this.handlePaste, this);
  }

  /**
   * Create a text box with the given position, size, style and text content
   *
   * @param {Object} bounds
   * @param {Number} bounds.x absolute x position
   * @param {Number} bounds.y absolute y position
   * @param {Number} [bounds.width] fixed width value
   * @param {Number} [bounds.height] fixed height value
   * @param {Number} [bounds.maxWidth] maximum width value
   * @param {Number} [bounds.maxHeight] maximum height value
   * @param {Number} [bounds.minWidth] minimum width value
   * @param {Number} [bounds.minHeight] minimum height value
   * @param {Object} [style]
   * @param {String} value text content
   *
   * @return {DOMElement} The created content DOM element
   */
  TextBox.prototype.create = function (bounds, style, value, options) {
    var self = this;
    var parent = this.parent,
      content = this.content,
      container = this.container;
    options = this.options = options || {};
    style = this.style = style || {};
    var parentStyle = pick$1(style, ['width', 'height', 'maxWidth', 'maxHeight', 'minWidth', 'minHeight', 'left', 'top', 'backgroundColor', 'position', 'overflow', 'border', 'wordWrap', 'textAlign', 'outline', 'transform']);
    assign$4(parent.style, {
      width: bounds.width + 'px',
      height: bounds.height + 'px',
      maxWidth: bounds.maxWidth + 'px',
      maxHeight: bounds.maxHeight + 'px',
      minWidth: bounds.minWidth + 'px',
      minHeight: bounds.minHeight + 'px',
      left: bounds.x + 'px',
      top: bounds.y + 'px',
      backgroundColor: '#ffffff',
      position: 'absolute',
      overflow: 'visible',
      border: '1px solid #ccc',
      boxSizing: 'border-box',
      wordWrap: 'normal',
      textAlign: 'center',
      outline: 'none'
    }, parentStyle);
    var contentStyle = pick$1(style, ['fontFamily', 'fontSize', 'fontWeight', 'lineHeight', 'padding', 'paddingTop', 'paddingRight', 'paddingBottom', 'paddingLeft']);
    assign$4(content.style, {
      boxSizing: 'border-box',
      width: '100%',
      outline: 'none',
      wordWrap: 'break-word'
    }, contentStyle);
    if (options.centerVertically) {
      assign$4(content.style, {
        position: 'absolute',
        top: '50%',
        transform: 'translate(0, -50%)'
      }, contentStyle);
    }
    content.innerText = value;
    event.bind(content, 'keydown', this.keyHandler);
    event.bind(content, 'mousedown', stopPropagation$1);
    event.bind(content, 'paste', self.handlePaste);
    if (options.autoResize) {
      event.bind(content, 'input', this.autoResize);
    }
    if (options.resizable) {
      this.resizable(style);
    }
    container.appendChild(parent);

    // set selection to end of text
    this.setSelection(content.lastChild, content.lastChild && content.lastChild.length);
    return parent;
  };

  /**
   * Intercept paste events to remove formatting from pasted text.
   */
  TextBox.prototype.handlePaste = function (e) {
    var options = this.options,
      style = this.style;
    e.preventDefault();
    var text;
    if (e.clipboardData) {
      // Chrome, Firefox, Safari
      text = e.clipboardData.getData('text/plain');
    } else {
      // Internet Explorer
      text = window.clipboardData.getData('Text');
    }
    this.insertText(text);
    if (options.autoResize) {
      var hasResized = this.autoResize(style);
      if (hasResized) {
        this.resizeHandler(hasResized);
      }
    }
  };
  TextBox.prototype.insertText = function (text) {
    text = normalizeEndOfLineSequences(text);

    // insertText command not supported by Internet Explorer
    var success = document.execCommand('insertText', false, text);
    if (success) {
      return;
    }
    this._insertTextIE(text);
  };
  TextBox.prototype._insertTextIE = function (text) {
    // Internet Explorer
    var range = this.getSelection(),
      startContainer = range.startContainer,
      endContainer = range.endContainer,
      startOffset = range.startOffset,
      endOffset = range.endOffset,
      commonAncestorContainer = range.commonAncestorContainer;
    var childNodesArray = toArray(commonAncestorContainer.childNodes);
    var container, offset;
    if (isTextNode(commonAncestorContainer)) {
      var containerTextContent = startContainer.textContent;
      startContainer.textContent = containerTextContent.substring(0, startOffset) + text + containerTextContent.substring(endOffset);
      container = startContainer;
      offset = startOffset + text.length;
    } else if (startContainer === this.content && endContainer === this.content) {
      var textNode = document.createTextNode(text);
      this.content.insertBefore(textNode, childNodesArray[startOffset]);
      container = textNode;
      offset = textNode.textContent.length;
    } else {
      var startContainerChildIndex = childNodesArray.indexOf(startContainer),
        endContainerChildIndex = childNodesArray.indexOf(endContainer);
      childNodesArray.forEach(function (childNode, index) {
        if (index === startContainerChildIndex) {
          childNode.textContent = startContainer.textContent.substring(0, startOffset) + text + endContainer.textContent.substring(endOffset);
        } else if (index > startContainerChildIndex && index <= endContainerChildIndex) {
          remove$4(childNode);
        }
      });
      container = startContainer;
      offset = startOffset + text.length;
    }
    if (container && offset !== undefined) {
      // is necessary in Internet Explorer
      setTimeout(function () {
        self.setSelection(container, offset);
      });
    }
  };

  /**
   * Automatically resize element vertically to fit its content.
   */
  TextBox.prototype.autoResize = function () {
    var parent = this.parent,
      content = this.content;
    var fontSize = parseInt(this.style.fontSize) || 12;
    if (content.scrollHeight > parent.offsetHeight || content.scrollHeight < parent.offsetHeight - fontSize) {
      var bounds = parent.getBoundingClientRect();
      var height = content.scrollHeight;
      parent.style.height = height + 'px';
      this.resizeHandler({
        width: bounds.width,
        height: bounds.height,
        dx: 0,
        dy: height - bounds.height
      });
    }
  };

  /**
   * Make an element resizable by adding a resize handle.
   */
  TextBox.prototype.resizable = function () {
    var self = this;
    var parent = this.parent,
      resizeHandle = this.resizeHandle;
    var minWidth = parseInt(this.style.minWidth) || 0,
      minHeight = parseInt(this.style.minHeight) || 0,
      maxWidth = parseInt(this.style.maxWidth) || Infinity,
      maxHeight = parseInt(this.style.maxHeight) || Infinity;
    if (!resizeHandle) {
      resizeHandle = this.resizeHandle = domify$1('<div class="djs-direct-editing-resize-handle"></div>');
      var startX, startY, startWidth, startHeight;
      var onMouseDown = function (e) {
        preventDefault(e);
        stopPropagation$1(e);
        startX = e.clientX;
        startY = e.clientY;
        var bounds = parent.getBoundingClientRect();
        startWidth = bounds.width;
        startHeight = bounds.height;
        event.bind(document, 'mousemove', onMouseMove);
        event.bind(document, 'mouseup', onMouseUp);
      };
      var onMouseMove = function (e) {
        preventDefault(e);
        stopPropagation$1(e);
        var newWidth = min$1(max$1(startWidth + e.clientX - startX, minWidth), maxWidth);
        var newHeight = min$1(max$1(startHeight + e.clientY - startY, minHeight), maxHeight);
        parent.style.width = newWidth + 'px';
        parent.style.height = newHeight + 'px';
        self.resizeHandler({
          width: startWidth,
          height: startHeight,
          dx: e.clientX - startX,
          dy: e.clientY - startY
        });
      };
      var onMouseUp = function (e) {
        preventDefault(e);
        stopPropagation$1(e);
        event.unbind(document, 'mousemove', onMouseMove, false);
        event.unbind(document, 'mouseup', onMouseUp, false);
      };
      event.bind(resizeHandle, 'mousedown', onMouseDown);
    }
    assign$4(resizeHandle.style, {
      position: 'absolute',
      bottom: '0px',
      right: '0px',
      cursor: 'nwse-resize',
      width: '0',
      height: '0',
      borderTop: (parseInt(this.style.fontSize) / 4 || 3) + 'px solid transparent',
      borderRight: (parseInt(this.style.fontSize) / 4 || 3) + 'px solid #ccc',
      borderBottom: (parseInt(this.style.fontSize) / 4 || 3) + 'px solid #ccc',
      borderLeft: (parseInt(this.style.fontSize) / 4 || 3) + 'px solid transparent'
    });
    parent.appendChild(resizeHandle);
  };

  /**
   * Clear content and style of the textbox, unbind listeners and
   * reset CSS style.
   */
  TextBox.prototype.destroy = function () {
    var parent = this.parent,
      content = this.content,
      resizeHandle = this.resizeHandle;

    // clear content
    content.innerText = '';

    // clear styles
    parent.removeAttribute('style');
    content.removeAttribute('style');
    event.unbind(content, 'keydown', this.keyHandler);
    event.unbind(content, 'mousedown', stopPropagation$1);
    event.unbind(content, 'input', this.autoResize);
    event.unbind(content, 'paste', this.handlePaste);
    if (resizeHandle) {
      resizeHandle.removeAttribute('style');
      remove$4(resizeHandle);
    }
    remove$4(parent);
  };
  TextBox.prototype.getValue = function () {
    return this.content.innerText.trim();
  };
  TextBox.prototype.getSelection = function () {
    var selection = window.getSelection(),
      range = selection.getRangeAt(0);
    return range;
  };
  TextBox.prototype.setSelection = function (container, offset) {
    var range = document.createRange();
    if (container === null) {
      range.selectNodeContents(this.content);
    } else {
      range.setStart(container, offset);
      range.setEnd(container, offset);
    }
    var selection = window.getSelection();
    selection.removeAllRanges();
    selection.addRange(range);
  };

  // helpers //////////

  function normalizeEndOfLineSequences(string) {
    return string.replace(/\r\n|\r|\n/g, '\n');
  }

  /**
   * A direct editing component that allows users
   * to edit an elements text directly in the diagram
   *
   * @param {EventBus} eventBus the event bus
   * @param {Canvas} canvas the canvas
   */
  function DirectEditing(eventBus, canvas) {
    this._eventBus = eventBus;
    this._canvas = canvas;
    this._providers = [];
    this._textbox = new TextBox({
      container: canvas.getContainer(),
      keyHandler: bind$3(this._handleKey, this),
      resizeHandler: bind$3(this._handleResize, this)
    });
  }
  DirectEditing.$inject = ['eventBus', 'canvas'];

  /**
   * Register a direct editing provider

   * @param {Object} provider the provider, must expose an #activate(element) method that returns
   *                          an activation context ({ bounds: {x, y, width, height }, text }) if
   *                          direct editing is available for the given element.
   *                          Additionally the provider must expose a #update(element, value) method
   *                          to receive direct editing updates.
   */
  DirectEditing.prototype.registerProvider = function (provider) {
    this._providers.push(provider);
  };

  /**
   * Returns true if direct editing is currently active
   *
   * @param {djs.model.Base} [element]
   *
   * @return {boolean}
   */
  DirectEditing.prototype.isActive = function (element) {
    return !!(this._active && (!element || this._active.element === element));
  };

  /**
   * Cancel direct editing, if it is currently active
   */
  DirectEditing.prototype.cancel = function () {
    if (!this._active) {
      return;
    }
    this._fire('cancel');
    this.close();
  };
  DirectEditing.prototype._fire = function (event, context) {
    this._eventBus.fire('directEditing.' + event, context || {
      active: this._active
    });
  };
  DirectEditing.prototype.close = function () {
    this._textbox.destroy();
    this._fire('deactivate');
    this._active = null;
    this.resizable = undefined;

    // restoreFocus API is available from diagram-js@15.0.0
    this._canvas.restoreFocus && this._canvas.restoreFocus();
  };
  DirectEditing.prototype.complete = function () {
    var active = this._active;
    if (!active) {
      return;
    }
    var containerBounds,
      previousBounds = active.context.bounds,
      newBounds = this.$textbox.getBoundingClientRect(),
      newText = this.getValue(),
      previousText = active.context.text;
    if (newText !== previousText || newBounds.height !== previousBounds.height || newBounds.width !== previousBounds.width) {
      containerBounds = this._textbox.container.getBoundingClientRect();
      active.provider.update(active.element, newText, active.context.text, {
        x: newBounds.left - containerBounds.left,
        y: newBounds.top - containerBounds.top,
        width: newBounds.width,
        height: newBounds.height
      });
    }
    this._fire('complete');
    this.close();
  };
  DirectEditing.prototype.getValue = function () {
    return this._textbox.getValue();
  };
  DirectEditing.prototype._handleKey = function (e) {
    // stop bubble
    e.stopPropagation();
    var key = e.keyCode || e.charCode;

    // ESC
    if (key === 27) {
      e.preventDefault();
      return this.cancel();
    }

    // Enter
    if (key === 13 && !e.shiftKey) {
      e.preventDefault();
      return this.complete();
    }
  };
  DirectEditing.prototype._handleResize = function (event) {
    this._fire('resize', event);
  };

  /**
   * Activate direct editing on the given element
   *
   * @param {Object} ElementDescriptor the descriptor for a shape or connection
   * @return {Boolean} true if the activation was possible
   */
  DirectEditing.prototype.activate = function (element) {
    if (this.isActive()) {
      this.cancel();
    }

    // the direct editing context
    var context;
    var provider = find$2(this._providers, function (p) {
      return (context = p.activate(element)) ? p : null;
    });

    // check if activation took place
    if (context) {
      this.$textbox = this._textbox.create(context.bounds, context.style, context.text, context.options);
      this._active = {
        element: element,
        context: context,
        provider: provider
      };
      if (context.options && context.options.resizable) {
        this.resizable = true;
      }
      this._fire('activate');
    }
    return !!context;
  };

  var DiagramDirectEditing = {
    __depends__: [InteractionEventsModule],
    __init__: ['directEditing'],
    directEditing: ['type', DirectEditing]
  };

  var MARKER_LABEL_HIDDEN = 'djs-label-hidden';
  function LabelEditingProvider(canvas, directEditing, eventBus, modeling, textRenderer) {
    this._canvas = canvas;
    this._modeling = modeling;
    this._textRenderer = textRenderer;
    this._eventBus = eventBus;
    directEditing.registerProvider(this);

    // listen to dblclick on non-root elements
    eventBus.on('element.dblclick', function (event) {
      directEditing.activate(event.element);
    });

    // complete on followup canvas operation
    eventBus.on(['autoPlace.start', 'canvas.viewbox.changing', 'drag.init', 'drillDown.click', 'element.mousedown', 'popupMenu.open', 'selection.changed'], function () {
      directEditing.complete();
    });

    // cancel on command stack changes
    eventBus.on(['commandStack.changed'], function () {
      directEditing.cancel();
    });
    eventBus.on('create.end', 500, function (e) {
      var element = e.shape;
      if (is$1(element, 'dmn:Decision') || is$1(element, 'dmn:InputData') || is$1(element, 'dmn:BusinessKnowledgeModel') || is$1(element, 'dmn:KnowledgeSource') || is$1(element, 'dmn:TextAnnotation')) {
        directEditing.activate(element);
      }
    });
    eventBus.on('autoPlace.end', 500, function (event) {
      directEditing.activate(event.shape);
    });
    this._eventBus.on(['directEditing.complete', 'directEditing.cancel'], function (context) {
      var activeProvider = context.active;
      if (activeProvider) {
        var element = activeProvider.element.label || activeProvider.element;
        canvas.removeMarker(element, MARKER_LABEL_HIDDEN);
      }
    });
  }
  LabelEditingProvider.$inject = ['canvas', 'directEditing', 'eventBus', 'modeling', 'textRenderer'];

  /**
   * Activate direct editing for drgs and text annotations.
   *
   * @param  {djs.model.Base} element
   *
   * @return {Object} an object with properties bounds (position and size) and text
   */
  LabelEditingProvider.prototype.activate = function (element) {
    var text = getLabel(element);
    if (!isDefined(text)) {
      return;
    }
    var context = {
      text: text
    };
    var editingBBox = this.getEditingBBox(element);
    assign$4(context, editingBBox);
    var options = {};
    var style = context.style || {};

    // DRG elements
    if (is$1(element, 'dmn:DRGElement')) {
      assign$4(options, {
        centerVertically: true
      });
      assign$4(style, {
        backgroundColor: null,
        border: null
      });
    }

    // text annotations
    if (is$1(element, 'dmn:TextAnnotation')) {
      assign$4(options, {
        resizable: true
      });
    }
    assign$4(context, {
      options: options,
      style: style
    });
    this._canvas.addMarker(element, MARKER_LABEL_HIDDEN);
    return context;
  };

  /**
   * Get the editing bounding box based on the element's size and position
   *
   * @param  {djs.model.Base} element
   *
   * @return {Object}
   *         an object containing information about position and
   *         size (fixed or minimum and/or maximum)
   */
  LabelEditingProvider.prototype.getEditingBBox = function (element) {
    var canvas = this._canvas;
    var target = element.label || element;
    var bbox = canvas.getAbsoluteBBox(target);

    // default position
    var bounds = {
      x: bbox.x,
      y: bbox.y
    };
    var zoom = canvas.zoom();
    var defaultStyle = this._textRenderer.getDefaultStyle();

    // take zoom into account
    var defaultFontSize = defaultStyle.fontSize * zoom,
      defaultLineHeight = defaultStyle.lineHeight;
    var style = {
      fontFamily: this._textRenderer.getDefaultStyle().fontFamily,
      fontWeight: this._textRenderer.getDefaultStyle().fontWeight
    };

    // DRG elements
    if (is$1(element, 'dmn:DRGElement')) {
      assign$4(bounds, {
        width: bbox.width,
        height: bbox.height
      });
      assign$4(style, {
        fontSize: defaultFontSize + 'px',
        lineHeight: defaultLineHeight,
        paddingTop: 7 * zoom + 'px',
        paddingBottom: 7 * zoom + 'px',
        paddingLeft: 5 * zoom + 'px',
        paddingRight: 5 * zoom + 'px'
      });
    }

    // text annotations
    if (is$1(element, 'dmn:TextAnnotation')) {
      assign$4(bounds, {
        width: bbox.width,
        height: bbox.height,
        minWidth: 30 * zoom,
        minHeight: 10 * zoom
      });
      assign$4(style, {
        textAlign: 'left',
        paddingTop: 5 * zoom + 'px',
        paddingBottom: 7 * zoom + 'px',
        paddingLeft: 7 * zoom + 'px',
        paddingRight: 5 * zoom + 'px',
        fontSize: defaultFontSize + 'px',
        lineHeight: defaultLineHeight
      });
    }
    return {
      bounds: bounds,
      style: style
    };
  };
  LabelEditingProvider.prototype.update = function (element, newLabel, activeContextText, bounds) {
    var newBounds, bbox;
    if (is$1(element, 'dmn:TextAnnotation')) {
      bbox = this._canvas.getAbsoluteBBox(element);
      newBounds = {
        x: element.x,
        y: element.y,
        width: element.width / bbox.width * bounds.width,
        height: element.height / bbox.height * bounds.height
      };
    }
    if (isEmptyText(newLabel)) {
      newLabel = null;
    }
    this._modeling.updateLabel(element, newLabel, newBounds);
  };

  // helpers //////////

  function isEmptyText(label) {
    return !label || !label.trim();
  }

  var LabelEditingModule = {
    __depends__: [CommandStack, DiagramChangeSupport, DiagramDirectEditing],
    __init__: ['labelEditingProvider'],
    labelEditingProvider: ['type', LabelEditingProvider]
  };

  /**
   * Creates DMN-specific refs for new connection.
   *
   * @param {DrdFactory} drdFactory
   * @param {Injector} injector
   */
  function CreateConnectionBehavior(drdFactory, injector) {
    injector.invoke(CommandInterceptor, this);
    this.preExecute('connection.create', function (context) {
      var connection = context.connection,
        connectionBo = connection.businessObject,
        source = context.source,
        target = context.target,
        elementRef,
        sourceRef,
        targetRef;
      if (is$1(connection, 'dmn:Association')) {
        sourceRef = connectionBo.sourceRef = drdFactory.create('dmn:DMNElementReference', {
          href: '#' + source.id
        });
        sourceRef.$parent = connectionBo;
        targetRef = connectionBo.targetRef = drdFactory.create('dmn:DMNElementReference', {
          href: '#' + target.id
        });
        targetRef.$parent = connectionBo;
      } else {
        elementRef = connectionBo['required' + getRequirementType(source)] = drdFactory.create('dmn:DMNElementReference', {
          href: '#' + source.id
        });
        elementRef.$parent = connectionBo;
      }
    }, true);
  }
  CreateConnectionBehavior.$inject = ['drdFactory', 'injector'];
  e$3(CreateConnectionBehavior, CommandInterceptor);

  // helpers //////////

  function getRequirementType(source) {
    if (is$1(source, 'dmn:BusinessKnowledgeModel')) {
      return 'Knowledge';
    } else if (is$1(source, 'dmn:Decision')) {
      return 'Decision';
    } else if (is$1(source, 'dmn:InputData')) {
      return 'Input';
    } else if (is$1(source, 'dmn:KnowledgeSource')) {
      return 'Authority';
    }
  }

  var LOW_PRIORITY$f = 500;
  function LayoutConnectionBehavior(injector, layouter, modeling, rules) {
    injector.invoke(CommandInterceptor, this);

    // specify connection start and end on connection create
    this.preExecute(['connection.create', 'connection.reconnect'], function (context) {
      var connection = context.connection,
        source = context.newSource || context.source,
        target = context.newTarget || context.target;
      if (is$1(connection, 'dmn:InformationRequirement') && !rules.allowed('connection.connect', {
        connection: connection,
        source: source,
        target: target
      })) {
        return;
      }
      if (!is$1(connection, 'dmn:InformationRequirement')) {
        return;
      }
      var orientation = getOrientation(source, target);
      if (!context.hints) {
        context.hints = {};
      }
      assign$4(context.hints, getConnectionHints(source, target, orientation));
    }, true);

    /**
     * Update incoming information requirements.
     *
     * @param {djs.model.Shape} target
     * @param {Array<djs.model.Connection>} [informationRequirements]
     * @param {string} [orientation]
     */
    function updateInformationRequirements(target, informationRequirements, orientation) {
      // (1) get information requirements
      if (!informationRequirements) {
        informationRequirements = target.incoming.filter(function (incoming) {
          return is$1(incoming, 'dmn:InformationRequirement');
        });
      }
      var incomingInformationRequirementsByOrientation = {};

      // (2) get information requirements per orientation
      if (orientation) {
        incomingInformationRequirementsByOrientation[orientation] = informationRequirements;
      } else {
        incomingInformationRequirementsByOrientation = getInformationRequirementsByOrientation(target, informationRequirements);
      }

      // (3) update information requirements per orientation
      forEach$3(incomingInformationRequirementsByOrientation, function (informationRequirements, orientation) {
        // (3.1) sort information requirements
        informationRequirements = sortInformationRequirements(informationRequirements, orientation);

        // (3.2) get new connection start and end
        var connectionStartEnd = getConnectionsStartEnd(informationRequirements, target, orientation);

        // (3.3) update information requirements
        informationRequirements.forEach((informationRequirement, index) => {
          var connectionStart = connectionStartEnd[index].start,
            connectionEnd = connectionStartEnd[index].end;
          var waypoints = layouter.layoutConnection(informationRequirement, {
            connectionStart: connectionStart,
            connectionEnd: connectionEnd
          });
          modeling.updateWaypoints(informationRequirement, waypoints);
        });
      });
    }

    // update information requirements on connection create and delete
    // update information requirements of new target on connection reconnect
    this.postExecuted(['connection.create', 'connection.delete', 'connection.reconnect'], function (context) {
      var connection = context.connection,
        source = connection.source || context.source,
        target = connection.target || context.target;
      if (!is$1(connection, 'dmn:InformationRequirement')) {
        return;
      }
      var orientation = getOrientation(source, target);

      // update all information requirements with same orientation
      var informationRequirements = target.incoming.filter(incoming => {
        var incomingOrientation = getOrientation(incoming.source, incoming.target);
        return is$1(incoming, 'dmn:InformationRequirement') && isSameOrientation(incomingOrientation, orientation);
      });
      if (!informationRequirements.length) {
        return;
      }
      updateInformationRequirements(target, informationRequirements, orientation);
    }, true);

    // update information requirements of old target on connection reconnect
    this.preExecute('connection.reconnect', function (context) {
      var connection = context.connection,
        source = connection.source,
        target = connection.target;
      if (!is$1(connection, 'dmn:InformationRequirement')) {
        return;
      }
      var orientation = getOrientation(source, target);

      // update all information requirements with same orientation except reconnected
      var informationRequirements = target.incoming.filter(incoming => {
        var incomingOrientation = getOrientation(incoming.source, incoming.target);
        return incoming !== connection && is$1(incoming, 'dmn:InformationRequirement') && isSameOrientation(incomingOrientation, orientation);
      });
      if (!informationRequirements.length) {
        return;
      }
      updateInformationRequirements(target, informationRequirements, orientation);
    }, true);

    // update information requirements on elements move
    this.postExecuted('elements.move', LOW_PRIORITY$f, function (context) {
      var shapes = context.shapes,
        closure = context.closure,
        enclosedConnections = closure.enclosedConnections;
      shapes.forEach(function (shape) {
        if (!isAny(shape, ['dmn:Decision', 'dmn:InputData'])) {
          return;
        }

        // (1) update incoming information requirements
        var incomingInformationRequirements = shape.incoming.filter(function (incoming) {
          return is$1(incoming, 'dmn:InformationRequirement') && !enclosedConnections[incoming.id];
        });
        if (incomingInformationRequirements.length) {
          updateInformationRequirements(shape, incomingInformationRequirements);
        }

        // (2) update outgoing information requirements
        shape.outgoing.forEach(function (outgoing) {
          if (!is$1(outgoing, 'dmn:InformationRequirement') || enclosedConnections[outgoing.id]) {
            return;
          }
          updateInformationRequirements(outgoing.target);
        });
      });
    }, true);
  }
  LayoutConnectionBehavior.$inject = ['injector', 'layouter', 'modeling', 'rules'];
  e$3(LayoutConnectionBehavior, CommandInterceptor);

  // helpers //////////

  function getConnectionHints(source, target, orientation) {
    var connectionStart = getMid(source),
      connectionEnd = getMid(target);
    if (orientation.includes('bottom')) {
      connectionStart.y = source.y;
      connectionEnd.y = target.y + target.height;
    } else if (orientation.includes('top')) {
      connectionStart.y = source.y + source.height;
      connectionEnd.y = target.y;
    } else if (orientation.includes('right')) {
      connectionStart.x = source.x;
      connectionEnd.x = target.x + target.width;
    } else {
      connectionStart.x = source.x + source.width;
      connectionEnd.x = target.x;
    }
    return {
      connectionStart: connectionStart,
      connectionEnd: connectionEnd
    };
  }

  /**
   * Get connections start and end based on number of information requirements and
   * orientation.
   *
   * @param {Array<djs.model.Connection>} informationRequirements
   * @param {djs.model.Shape} target
   * @param {string} orientation
   *
   * @returns {Array<Object>}
   */
  function getConnectionsStartEnd(informationRequirements, target, orientation) {
    return informationRequirements.map(function (informationRequirement, index) {
      var source = informationRequirement.source,
        sourceMid = getMid(source),
        sourceTrbl = asTRBL(source),
        targetTrbl = asTRBL(target);
      var length = informationRequirements.length;
      if (orientation.includes('bottom')) {
        return {
          start: {
            x: sourceMid.x,
            y: sourceTrbl.top
          },
          end: {
            x: targetTrbl.left + target.width / (length + 1) * (index + 1),
            y: targetTrbl.bottom
          }
        };
      } else if (orientation.includes('top')) {
        return {
          start: {
            x: sourceMid.x,
            y: sourceTrbl.bottom
          },
          end: {
            x: targetTrbl.left + target.width / (length + 1) * (index + 1),
            y: targetTrbl.top
          }
        };
      } else if (orientation.includes('right')) {
        return {
          start: {
            x: sourceTrbl.left,
            y: sourceMid.y
          },
          end: {
            x: targetTrbl.right,
            y: targetTrbl.top + target.height / (length + 1) * (index + 1)
          }
        };
      } else {
        return {
          start: {
            x: sourceTrbl.right,
            y: sourceMid.y
          },
          end: {
            x: targetTrbl.left,
            y: targetTrbl.top + target.height / (length + 1) * (index + 1)
          }
        };
      }
    });
  }

  /**
   * Get information requirements by orientation.
   *
   * @param {djs.model.shape} target
   * @param {Array<djs.model.Connection>} informationRequirements
   *
   * @returns {Object}
   */
  function getInformationRequirementsByOrientation(target, informationRequirements) {
    var incomingInformationRequirementsByOrientation = {};
    informationRequirements.forEach(function (incoming) {
      var orientation = getOrientation(incoming.source, target).split('-').shift();
      if (!incomingInformationRequirementsByOrientation[orientation]) {
        incomingInformationRequirementsByOrientation[orientation] = [];
      }
      incomingInformationRequirementsByOrientation[orientation].push(incoming);
    });
    return incomingInformationRequirementsByOrientation;
  }
  function isSameOrientation(orientationA, orientationB) {
    return orientationA && orientationB && orientationA.split('-').shift() === orientationB.split('-').shift();
  }
  function sortInformationRequirements(informationRequirements, orientation) {
    var axis;
    if (orientation.includes('top') || orientation.includes('bottom')) {
      axis = 'x';
    } else {
      axis = 'y';
    }
    return informationRequirements.sort(function (a, b) {
      return getMid(a.source)[axis] - getMid(b.source)[axis];
    });
  }

  function ReplaceConnectionBehavior(injector, modeling, rules) {
    injector.invoke(CommandInterceptor, this);
    this.preExecute('connection.reconnect', function (context) {
      var connection = context.connection,
        source = context.newSource || connection.source,
        target = context.newTarget || connection.target,
        waypoints = connection.waypoints.slice();
      var allowed = rules.allowed('connection.reconnect', {
        connection: connection,
        source: source,
        target: target
      });
      if (!allowed || allowed.type === connection.type) {
        return;
      }
      context.connection = modeling.connect(source, target, {
        type: allowed.type,
        waypoints: waypoints
      });
      modeling.removeConnection(connection);
    }, true);
  }
  e$3(ReplaceConnectionBehavior, CommandInterceptor);
  ReplaceConnectionBehavior.$inject = ['injector', 'modeling', 'rules'];

  /**
   * Defines the behaviour of what happens to the elements inside a container
   * that morphs into another DRD element
   */
  function ReplaceElementBehaviour(eventBus, modeling, selection) {
    CommandInterceptor.call(this, eventBus);
    this._modeling = modeling;
    this.postExecuted(['shape.replace'], 1500, function (e) {
      var context = e.context,
        oldShape = context.oldShape,
        newShape = context.newShape;
      modeling.unclaimId(oldShape.businessObject.id, oldShape.businessObject);
      modeling.updateProperties(newShape, {
        id: oldShape.id
      });
      selection.select(newShape);
    });
  }
  e$3(ReplaceElementBehaviour, CommandInterceptor);
  ReplaceElementBehaviour.$inject = ['eventBus', 'modeling', 'selection'];

  const ID$2 = 'id';
  class IdChangeBehavior extends CommandInterceptor {
    constructor(eventBus) {
      super(eventBus);
      this.executed('element.updateProperties', this.updateIds.bind(this));
    }
    updateIds({
      context
    }) {
      const {
        element,
        oldProperties,
        properties
      } = context;
      const bo = getBusinessObject(element);
      if (this.shouldSkipUpdate(bo, oldProperties, properties)) {
        return;
      }
      const definitions = getDefinitions$2(bo);
      const drgElements = definitions.get('drgElement');
      drgElements.forEach(drgElement => {
        updateElementReferences(drgElement, oldProperties.id, properties.id);
      });
      const artifacts = definitions.get('artifact');
      artifacts.forEach(artifact => {
        updateAssociationReferences(artifact, oldProperties.id, properties.id);
      });
    }
    shouldSkipUpdate(bo, oldProperties, newProperties) {
      return !isIdChange$2(oldProperties, newProperties) || !is$1(bo, 'dmn:DRGElement') && !is$1(bo, 'dmn:TextAnnotation');
    }
  }
  IdChangeBehavior.$inject = ['eventBus'];

  // helpers //////////////////////

  function isIdChange$2(oldProperties, properties) {
    return ID$2 in oldProperties && ID$2 in properties;
  }

  /**
   * Walk up the tree until at the root to get to dmn:Definitions.
   *
   * @param {ModdleElement} element
   */
  function getDefinitions$2(element) {
    let definitions = element;
    while (!is$1(definitions, 'dmn:Definitions')) {
      definitions = definitions.$parent;
    }
    return definitions;
  }
  function updateElementReferences(element, oldId, id) {
    const handlers = {
      authorityRequirement: () => {
        element.authorityRequirement.forEach(authorityRequirement => {
          const {
            requiredAuthority,
            requiredDecision,
            requiredInput
          } = authorityRequirement;
          if (requiredAuthority && requiredAuthority.href === `#${oldId}`) {
            requiredAuthority.href = `#${id}`;
          }
          if (requiredDecision && requiredDecision.href === `#${oldId}`) {
            requiredDecision.href = `#${id}`;
          }
          if (requiredInput && requiredInput.href === `#${oldId}`) {
            requiredInput.href = `#${id}`;
          }
        });
      },
      informationRequirement: () => {
        element.informationRequirement.forEach(informationRequirement => {
          const {
            requiredDecision,
            requiredInput
          } = informationRequirement;
          if (requiredDecision && requiredDecision.href === `#${oldId}`) {
            requiredDecision.href = `#${id}`;
          }
          if (requiredInput && requiredInput.href === `#${oldId}`) {
            requiredInput.href = `#${id}`;
          }
        });
      },
      knowledgeRequirement: () => {
        element.knowledgeRequirement.forEach(knowledgeRequirement => {
          const {
            requiredKnowledge
          } = knowledgeRequirement;
          if (requiredKnowledge && requiredKnowledge.href === `#${oldId}`) {
            requiredKnowledge.href = `#${id}`;
          }
        });
      }
    };
    forEach$3(handlers, (handler, key) => {
      if (element[key]) {
        handler();
      }
    });
  }
  function updateAssociationReferences(element, oldId, id) {
    const handlers = {
      sourceRef: () => {
        const {
          sourceRef
        } = element;
        if (sourceRef.href === `#${oldId}`) {
          sourceRef.href = `#${id}`;
        }
      },
      targetRef: () => {
        const {
          targetRef
        } = element;
        if (targetRef.href === `#${oldId}`) {
          targetRef.href = `#${id}`;
        }
      }
    };
    forEach$3(handlers, (handler, key) => {
      if (element[key]) {
        handler();
      }
    });
  }

  let NameChangeBehavior$1 = class NameChangeBehavior extends CommandInterceptor {
    static $inject = ['eventBus', 'modeling'];
    constructor(eventBus, modeling) {
      super(eventBus);
      this._modeling = modeling;
      this.postExecuted('element.updateProperties', this.updateVariableFromElement);
      this.postExecuted('element.updateLabel', this.updateVariableFromLabel);
    }
    updateVariableFromLabel = ({
      context
    }) => {
      const {
        element,
        newLabel
      } = context;
      const bo = getBusinessObject(element),
        variable = bo.variable;
      if (!variable) {
        return;
      }
      this._modeling.updateProperties(variable, {
        name: newLabel
      });
    };
    updateVariableFromElement = ({
      context
    }) => {
      const {
        element,
        properties
      } = context;
      const bo = getBusinessObject(element);
      if (!bo.variable) {
        return;
      }
      if (!(is$1(element, 'dmn:Decision') || is$1(element, 'dmn:BusinessKnowledgeModel'))) {
        return;
      }
      if (!this.isNameChanged(properties)) {
        return;
      }
      if (this.isVariable(element)) {
        return;
      } else if (!this.shouldSyncVariable(element)) {
        this.syncElementVariableChange(bo);
      }
    };
    isNameChanged(properties) {
      return 'name' in properties;
    }
    isVariable(element) {
      const parent = getParent$1(element);
      return is$1(element, 'dmn:InformationItem') && parent && parent.get('variable') === element;
    }
    shouldSyncVariable(element) {
      const bo = getBusinessObject(element),
        variable = bo.get('variable');
      return variable && bo.name === variable.name;
    }
    syncElementVariableChange(businessObject) {
      const name = businessObject.get('name');
      const variable = businessObject.variable;
      this._modeling.updateProperties(variable, {
        name
      });
    }
  };

  // helpers //////////////////////

  function getParent$1(element) {
    return element.$parent;
  }

  var ModelingBehavior = {
    __init__: ['createConnectionBehavior', 'idChangeBehavior', 'nameChangeBehavior', 'layoutConnectionBehavior', 'replaceConnectionBehavior', 'replaceElementBehavior'],
    createConnectionBehavior: ['type', CreateConnectionBehavior],
    idChangeBehavior: ['type', IdChangeBehavior],
    nameChangeBehavior: ['type', NameChangeBehavior$1],
    layoutConnectionBehavior: ['type', LayoutConnectionBehavior],
    replaceConnectionBehavior: ['type', ReplaceConnectionBehavior],
    replaceElementBehavior: ['type', ReplaceElementBehaviour]
  };

  /**
   * @typedef {import('../../core/EventBus').default} EventBus
   */

  /**
   * A basic provider that may be extended to implement modeling rules.
   *
   * Extensions should implement the init method to actually add their custom
   * modeling checks. Checks may be added via the #addRule(action, fn) method.
   *
   * @class
   *
   * @param {EventBus} eventBus
   */
  function RuleProvider(eventBus) {
    CommandInterceptor.call(this, eventBus);
    this.init();
  }
  RuleProvider.$inject = ['eventBus'];
  e$3(RuleProvider, CommandInterceptor);

  /**
   * Adds a modeling rule for the given action, implemented through
   * a callback function.
   *
   * The callback receives a modeling specific action context
   * to perform its check. It must return `false` to disallow the
   * action from happening or `true` to allow the action. Usually returing
   * `null` denotes that a particular interaction shall be ignored.
   * By returning nothing or `undefined` you pass evaluation to lower
   * priority rules.
   *
   * @example
   *
   * ```javascript
   * ResizableRules.prototype.init = function() {
   *
   *   \/**
   *    * Return `true`, `false` or nothing to denote
   *    * _allowed_, _not allowed_ and _continue evaluating_.
   *    *\/
   *   this.addRule('shape.resize', function(context) {
   *
   *     var shape = context.shape;
   *
   *     if (!context.newBounds) {
   *       // check general resizability
   *       if (!shape.resizable) {
   *         return false;
   *       }
   *
   *       // not returning anything (read: undefined)
   *       // will continue the evaluation of other rules
   *       // (with lower priority)
   *       return;
   *     } else {
   *       // element must have minimum size of 10*10 points
   *       return context.newBounds.width > 10 && context.newBounds.height > 10;
   *     }
   *   });
   * };
   * ```
   *
   * @param {string|string[]} actions the identifier for the modeling action to check
   * @param {number} [priority] the priority at which this rule is being applied
   * @param {(any) => any} fn the callback function that performs the actual check
   */
  RuleProvider.prototype.addRule = function (actions, priority, fn) {
    var self = this;
    if (typeof actions === 'string') {
      actions = [actions];
    }
    actions.forEach(function (action) {
      self.canExecute(action, priority, function (context, action, event) {
        return fn(context);
      }, true);
    });
  };

  /**
   * Implement this method to add new rules during provider initialization.
   */
  RuleProvider.prototype.init = function () {};

  /**
   * DRD modeling rules.
   */
  function DrdRules(injector) {
    injector.invoke(RuleProvider, this);
  }
  e$3(DrdRules, RuleProvider);
  DrdRules.$inject = ['injector'];
  DrdRules.prototype.init = function () {
    this.addRule('connection.create', function (context) {
      var source = context.source,
        target = context.target;
      return canConnect(source, target);
    });
    this.addRule('connection.reconnect', function (context) {
      context.connection;
        var source = context.source,
        target = context.target;
      return canConnect(source, target);
    });
    this.addRule('connection.updateWaypoints', function (context) {
      const connection = context.connection;
      return {
        type: connection.type,
        businessObject: connection.businessObject
      };
    });
    this.addRule('elements.move', function (context) {
      var target = context.target,
        shapes = context.shapes;
        context.position;
      return canMove(shapes, target);
    });
    this.addRule('shape.create', function (context) {
      var shape = context.shape,
        target = context.target;
      return canCreate(shape, target);
    });
    this.addRule('shape.resize', function (context) {
      var shape = context.shape;
      return is$1(shape, 'dmn:TextAnnotation');
    });
  };
  DrdRules.prototype.canConnect = canConnect;
  DrdRules.prototype.canCreate = canCreate;
  DrdRules.prototype.canMove = canMove;
  function canConnect(source, target) {
    if (!source || isLabel(source) || !target || isLabel(target)) {
      return null;
    }
    if (source === target) {
      return false;
    }
    if (is$1(source, 'dmn:BusinessKnowledgeModel') && isAny(target, ['dmn:BusinessKnowledgeModel', 'dmn:Decision'])) {
      return {
        type: 'dmn:KnowledgeRequirement'
      };
    }
    if (is$1(source, 'dmn:Decision')) {
      if (is$1(target, 'dmn:Decision')) {
        return {
          type: 'dmn:InformationRequirement'
        };
      }
      if (is$1(target, 'dmn:KnowledgeSource')) {
        return {
          type: 'dmn:AuthorityRequirement'
        };
      }
    }
    if (is$1(source, 'dmn:Definitions') || is$1(target, 'dmn:Definitions')) {
      return false;
    }
    if (is$1(source, 'dmn:InputData')) {
      if (is$1(target, 'dmn:Decision')) {
        return {
          type: 'dmn:InformationRequirement'
        };
      }
      if (is$1(target, 'dmn:KnowledgeSource')) {
        return {
          type: 'dmn:AuthorityRequirement'
        };
      }
    }
    if (is$1(source, 'dmn:KnowledgeSource') && isAny(target, ['dmn:BusinessKnowledgeModel', 'dmn:Decision', 'dmn:KnowledgeSource'])) {
      return {
        type: 'dmn:AuthorityRequirement'
      };
    }
    if (is$1(source, 'dmn:TextAnnotation') && !is$1(target, 'dmn:TextAnnotation') || !is$1(source, 'dmn:TextAnnotation') && is$1(target, 'dmn:TextAnnotation')) {
      return {
        type: 'dmn:Association'
      };
    }
    return false;
  }
  function canCreate(shape, target) {
    return isAny(shape, ['dmn:BusinessKnowledgeModel', 'dmn:Decision', 'dmn:InputData', 'dmn:KnowledgeSource', 'dmn:TextAnnotation']) && is$1(target, 'dmn:Definitions');
  }
  function canMove(elements, target) {
    if (!isArray$6(elements)) {
      elements = [elements];
    }

    // allow default move check to start move operation
    if (!target) {
      return true;
    }
    if (every(elements, function (element) {
      return isAny(element, ['dmn:BusinessKnowledgeModel', 'dmn:Decision', 'dmn:InputData', 'dmn:KnowledgeSource', 'dmn:TextAnnotation', 'dmn:InformationRequirement', 'dmn:AuthorityRequirement', 'dmn:KnowledgeRequirement', 'dmn:Association']);
    }) && is$1(target, 'dmn:Definitions')) {
      return true;
    }
    return false;
  }
  function isLabel(element) {
    return !!element.labelTarget;
  }

  var Rules$4 = {
    __depends__: [Rules$5],
    __init__: ['drdRules'],
    drdRules: ['type', DrdRules]
  };

  function DrdFactory(moddle) {
    this._model = moddle;
  }
  DrdFactory.$inject = ['moddle'];
  DrdFactory.prototype._needsId = function (element) {
    return isAny(element, ['dmn:Artifact', 'dmn:DMNElement', 'dmn:DRGElement', 'dmndi:DMNDiagram', 'dmndi:DMNDiagramElement']);
  };
  DrdFactory.prototype._ensureId = function (element) {
    var prefix = (element.$type || '').replace(/^[^:]*:/g, '') + '_';
    if (!element.id && this._needsId(element)) {
      element.id = this._model.ids.nextPrefixed(prefix, element);
    }
  };
  DrdFactory.prototype.create = function (type, attrs) {
    var element = this._model.create(type, attrs || {});
    this._ensureId(element);
    return element;
  };
  DrdFactory.prototype.createDiShape = function (semantic, bounds, attrs) {
    return this.create('dmndi:DMNShape', assign$4({
      dmnElementRef: semantic,
      bounds: this.createDiBounds(bounds)
    }, attrs));
  };
  DrdFactory.prototype.createDiBounds = function (bounds) {
    return this.create('dc:Bounds', bounds);
  };
  DrdFactory.prototype.createDiEdge = function (semantic, waypoints, attrs) {
    return this.create('dmndi:DMNEdge', {
      dmnElementRef: semantic,
      waypoint: this.createDiWaypoints(waypoints)
    }, attrs);
  };
  DrdFactory.prototype.createDiWaypoints = function (waypoints) {
    var self = this;
    return waypoints.map(function (waypoint) {
      return self.createDiWaypoint(waypoint);
    });
  };
  DrdFactory.prototype.createDiWaypoint = function (waypoint) {
    return this.create('dc:Point', pick$1(waypoint, ['x', 'y']));
  };
  DrdFactory.prototype.createExtensionElements = function () {
    return this.create('dmn:ExtensionElements', {
      values: []
    });
  };

  /**
   * Update DMN 1.3 information.
   */
  function DrdUpdater(connectionDocking, definitionPropertiesView, drdFactory, drdRules, injector) {
    injector.invoke(CommandInterceptor, this);
    this._definitionPropertiesView = definitionPropertiesView;
    this._drdFactory = drdFactory;
    this._drdRules = drdRules;
    var self = this;
    function cropConnection(context) {
      var connection = context.connection,
        cropped = context.cropped;
      if (!cropped) {
        connection.waypoints = connectionDocking.getCroppedWaypoints(connection);
        context.cropped = true;
      }
    }
    this.executed(['connection.create', 'connection.layout'], cropConnection, true);
    this.reverted(['connection.layout'], function (context) {
      delete context.cropped;
    }, true);
    function updateParent(context) {
      var connection = context.connection,
        parent = context.parent,
        shape = context.shape;
      if (connection && !is$1(connection, 'dmn:Association')) {
        parent = connection.target;
      }
      self.updateParent(shape || connection, parent);
    }
    function reverseUpdateParent(context) {
      var connection = context.connection,
        shape = context.shape;
      var oldParent = context.parent || context.newParent;
      if (connection && !is$1(connection, 'dmn:Association')) {
        oldParent = connection.target;
      }
      self.updateParent(shape || connection, oldParent);
    }
    this.executed(['connection.create', 'connection.delete', 'connection.move', 'shape.create', 'shape.delete'], updateParent, true);
    this.reverted(['connection.create', 'connection.delete', 'connection.move', 'shape.create', 'shape.delete'], reverseUpdateParent, true);
    function updateBounds(context) {
      var shape = context.shape;
      if (!(is$1(shape, 'dmn:DRGElement') || is$1(shape, 'dmn:TextAnnotation'))) {
        return;
      }
      self.updateBounds(shape);
    }
    this.executed(['shape.create', 'shape.move', 'shape.resize'], updateBounds, true);
    this.reverted(['shape.create', 'shape.move', 'shape.resize'], updateBounds, true);
    function updateConnectionWaypoints(context) {
      self.updateConnectionWaypoints(context);
    }
    this.executed(['connection.create', 'connection.layout', 'connection.move', 'connection.updateWaypoints'], updateConnectionWaypoints, true);
    this.reverted(['connection.create', 'connection.layout', 'connection.move', 'connection.updateWaypoints'], updateConnectionWaypoints, true);
    this.executed('connection.create', function (context) {
      var connection = context.connection,
        connectionBo = connection.businessObject,
        target = context.target,
        targetBo = target.businessObject;
      if (is$1(connection, 'dmn:Association')) {
        updateParent(context);
      } else {
        // parent is target
        self.updateSemanticParent(connectionBo, targetBo);
      }
    }, true);
    this.reverted('connection.create', function (context) {
      reverseUpdateParent(context);
    }, true);
    this.executed('connection.reconnect', function (context) {
      var connection = context.connection,
        connectionBo = connection.businessObject,
        newTarget = context.newTarget,
        newTargetBo = newTarget.businessObject;
      if (is$1(connectionBo, 'dmn:Association')) {
        return;
      }
      self.updateSemanticParent(connectionBo, newTargetBo);
    }, true);
    this.reverted('connection.reconnect', function (context) {
      var connection = context.connection,
        connectionBo = connection.businessObject,
        oldTarget = context.oldTarget,
        oldTargetBo = oldTarget.businessObject;
      if (is$1(connectionBo, 'dmn:Association')) {
        return;
      }
      self.updateSemanticParent(connectionBo, oldTargetBo);
    }, true);
    this.executed('element.updateProperties', function (context) {
      definitionPropertiesView.update();
    }, true);
    this.reverted('element.updateProperties', function (context) {
      definitionPropertiesView.update();
    }, true);
  }
  e$3(DrdUpdater, CommandInterceptor);
  DrdUpdater.$inject = ['connectionDocking', 'definitionPropertiesView', 'drdFactory', 'drdRules', 'injector'];
  DrdUpdater.prototype.updateBounds = function (shape) {
    var businessObject = shape.businessObject,
      bounds = businessObject.di.bounds;

    // update bounds
    assign$4(bounds, {
      x: shape.x,
      y: shape.y,
      width: shape.width,
      height: shape.height
    });
  };
  DrdUpdater.prototype.updateConnectionWaypoints = function (context) {
    var drdFactory = this._drdFactory;
    var connection = context.connection,
      businessObject = connection.businessObject,
      edge = businessObject.di;
    edge.waypoint = drdFactory.createDiWaypoints(connection.waypoints).map(function (waypoint) {
      waypoint.$parent = edge;
      return waypoint;
    });
  };
  DrdUpdater.prototype.updateParent = function (element, oldParent) {
    var parent = element.parent;
    if (!is$1(element, 'dmn:DRGElement') && !is$1(element, 'dmn:Artifact')) {
      parent = oldParent;
    }
    var businessObject = element.businessObject,
      parentBo = parent && parent.businessObject;
    this.updateSemanticParent(businessObject, parentBo);
    this.updateDiParent(businessObject.di, parentBo && parentBo.di);
  };
  DrdUpdater.prototype.updateSemanticParent = function (businessObject, parent) {
    var children, containment;
    if (businessObject.$parent === parent) {
      return;
    }
    if (is$1(businessObject, 'dmn:DRGElement')) {
      containment = 'drgElement';
    } else if (is$1(businessObject, 'dmn:Artifact')) {
      containment = 'artifact';
    } else if (is$1(businessObject, 'dmn:InformationRequirement')) {
      containment = 'informationRequirement';
    } else if (is$1(businessObject, 'dmn:AuthorityRequirement')) {
      containment = 'authorityRequirement';
    } else if (is$1(businessObject, 'dmn:KnowledgeRequirement')) {
      containment = 'knowledgeRequirement';
    }
    if (businessObject.$parent) {
      // remove from old parent
      children = businessObject.$parent.get(containment);
      remove$2(children, businessObject);
    }
    if (parent) {
      // add to new parent
      children = parent.get(containment);
      if (children) {
        children.push(businessObject);
        businessObject.$parent = parent;
      }
    } else {
      businessObject.$parent = null;
    }
  };
  DrdUpdater.prototype.updateDiParent = function (di, parentDi) {
    if (di.$parent === parentDi) {
      return;
    }
    if (isAny(di, ['dmndi:DMNEdge', 'dmndi:DMNShape'])) {
      var diagram = parentDi || di;
      while (!is$1(diagram, 'dmndi:DMNDiagram')) {
        diagram = diagram.$parent;
      }
      var diagramElements = diagram.get('diagramElements');
      if (parentDi) {
        di.$parent = diagram;
        add$2(diagramElements, di);
      } else {
        di.$parent = null;
        remove$2(diagramElements, di);
      }
    } else {
      throw new Error('unsupported');
    }
  };

  /**
   * @typedef {import('../../../core/Canvas').default} Canvas
   * @typedef {import('../Modeling').default} Modeling
   */

  /**
   * A handler that align elements in a certain way.
   *
   * @param {Modeling} modeling
   * @param {Canvas} canvas
   */
  function AlignElements(modeling, canvas) {
    this._modeling = modeling;
    this._canvas = canvas;
  }
  AlignElements.$inject = ['modeling', 'canvas'];
  AlignElements.prototype.preExecute = function (context) {
    var modeling = this._modeling;
    var elements = context.elements,
      alignment = context.alignment;
    forEach$3(elements, function (element) {
      var delta = {
        x: 0,
        y: 0
      };
      if (isDefined(alignment.left)) {
        delta.x = alignment.left - element.x;
      } else if (isDefined(alignment.right)) {
        delta.x = alignment.right - element.width - element.x;
      } else if (isDefined(alignment.center)) {
        delta.x = alignment.center - Math.round(element.width / 2) - element.x;
      } else if (isDefined(alignment.top)) {
        delta.y = alignment.top - element.y;
      } else if (isDefined(alignment.bottom)) {
        delta.y = alignment.bottom - element.height - element.y;
      } else if (isDefined(alignment.middle)) {
        delta.y = alignment.middle - Math.round(element.height / 2) - element.y;
      }
      modeling.moveElements([element], delta, element.parent);
    });
  };
  AlignElements.prototype.postExecute = function (context) {};

  /**
   * @typedef {import('../../../model/Types').Element} Element
   * @typedef {import('../../../model/Types').Parent} Parent
   * @typedef {import('../../../model/Types').Shape} Shape
   *
   * @typedef {import('../../../util/Types').Point} Point
   *
   * @typedef {import('../Modeling').default} Modeling
   */

  /**
   * A handler that implements reversible appending of shapes
   * to a source shape.
   *
   * @param {Modeling} modeling
   */
  function AppendShapeHandler(modeling) {
    this._modeling = modeling;
  }
  AppendShapeHandler.$inject = ['modeling'];

  // api //////////////////////

  /**
   * Creates a new shape.
   *
   * @param {Object} context
   * @param {Partial<Shape>} context.shape The new shape.
   * @param {Element} context.source The element to which to append the new shape to.
   * @param {Parent} context.parent The parent.
   * @param {Point} context.position The position at which to create the new shape.
   */
  AppendShapeHandler.prototype.preExecute = function (context) {
    var source = context.source;
    if (!source) {
      throw new Error('source required');
    }
    var target = context.target || source.parent,
      shape = context.shape,
      hints = context.hints || {};
    shape = context.shape = this._modeling.createShape(shape, context.position, target, {
      attach: hints.attach
    });
    context.shape = shape;
  };
  AppendShapeHandler.prototype.postExecute = function (context) {
    var hints = context.hints || {};
    if (!existsConnection(context.source, context.shape)) {
      // create connection
      if (hints.connectionTarget === context.source) {
        this._modeling.connect(context.shape, context.source, context.connection);
      } else {
        this._modeling.connect(context.source, context.shape, context.connection);
      }
    }
  };
  function existsConnection(source, target) {
    return some(source.outgoing, function (c) {
      return c.target === target;
    });
  }

  /**
   * @typedef {import('../../../model/Types').Element} Element
   * @typedef {import('../../../model/Types').Shape} Shape
   *
   * @typedef {import('../../../util/Types').Point} Point
   *
   * @typedef {import('../Modeling').ModelingHints} ModelingHints
   *
   * @typedef {import('../../../core/Canvas').default} Canvas
   * @typedef {import('../../../layout/BaseLayouter').default} Layouter
   */

  /**
   * @param {Canvas} canvas
   * @param {Layouter} layouter
   */
  function CreateConnectionHandler(canvas, layouter) {
    this._canvas = canvas;
    this._layouter = layouter;
  }
  CreateConnectionHandler.$inject = ['canvas', 'layouter'];

  // api //////////////////////

  /**
   * Creates a new connection between two elements.
   *
   * @param {Object} context
   * @param {Element} context.source The source.
   * @param {Element} context.target The target.
   * @param {Shape} context.parent The parent.
   * @param {number} [context.parentIndex] The optional index at which to add the
   * connection to the parent's children.
   * @param {ModelingHints} [context.hints] The optional hints.
   */
  CreateConnectionHandler.prototype.execute = function (context) {
    var connection = context.connection,
      source = context.source,
      target = context.target,
      parent = context.parent,
      parentIndex = context.parentIndex,
      hints = context.hints;
    if (!source || !target) {
      throw new Error('source and target required');
    }
    if (!parent) {
      throw new Error('parent required');
    }
    connection.source = source;
    connection.target = target;
    if (!connection.waypoints) {
      connection.waypoints = this._layouter.layoutConnection(connection, hints);
    }

    // add connection
    this._canvas.addConnection(connection, parent, parentIndex);
    return connection;
  };
  CreateConnectionHandler.prototype.revert = function (context) {
    var connection = context.connection;
    this._canvas.removeConnection(connection);
    connection.source = null;
    connection.target = null;
    return connection;
  };

  /**
   * @typedef {import('../Modeling').default} Modeling
   */

  var round$2 = Math.round;

  /**
   * @param {Modeling} modeling
   */
  function CreateElementsHandler(modeling) {
    this._modeling = modeling;
  }
  CreateElementsHandler.$inject = ['modeling'];
  CreateElementsHandler.prototype.preExecute = function (context) {
    var elements = context.elements,
      parent = context.parent,
      parentIndex = context.parentIndex,
      position = context.position,
      hints = context.hints;
    var modeling = this._modeling;

    // make sure each element has x and y
    forEach$3(elements, function (element) {
      if (!isNumber$3(element.x)) {
        element.x = 0;
      }
      if (!isNumber$3(element.y)) {
        element.y = 0;
      }
    });
    var visibleElements = filter$2(elements, function (element) {
      return !element.hidden;
    });
    var bbox = getBBox(visibleElements);

    // center elements around position
    forEach$3(elements, function (element) {
      if (isConnection(element)) {
        element.waypoints = map$3(element.waypoints, function (waypoint) {
          return {
            x: round$2(waypoint.x - bbox.x - bbox.width / 2 + position.x),
            y: round$2(waypoint.y - bbox.y - bbox.height / 2 + position.y)
          };
        });
      }
      assign$4(element, {
        x: round$2(element.x - bbox.x - bbox.width / 2 + position.x),
        y: round$2(element.y - bbox.y - bbox.height / 2 + position.y)
      });
    });
    var parents = getParents(elements);
    var cache = {};
    forEach$3(elements, function (element) {
      if (isConnection(element)) {
        cache[element.id] = isNumber$3(parentIndex) ? modeling.createConnection(cache[element.source.id], cache[element.target.id], parentIndex, element, element.parent || parent, hints) : modeling.createConnection(cache[element.source.id], cache[element.target.id], element, element.parent || parent, hints);
        return;
      }
      var createShapeHints = assign$4({}, hints);
      if (parents.indexOf(element) === -1) {
        createShapeHints.autoResize = false;
      }
      if (isLabel$1(element)) {
        createShapeHints = omit(createShapeHints, ['attach']);
      }
      cache[element.id] = isNumber$3(parentIndex) ? modeling.createShape(element, pick$1(element, ['x', 'y', 'width', 'height']), element.parent || parent, parentIndex, createShapeHints) : modeling.createShape(element, pick$1(element, ['x', 'y', 'width', 'height']), element.parent || parent, createShapeHints);
    });
    context.elements = values$1(cache);
  };

  /**
   * @typedef {import('../../../model/Types').Element} Element
   * @typedef {import('../../../util/Types').Point} Point
   *
   * @typedef {import('../../../core/Canvas').default} Canvas
   */

  var round$1 = Math.round;

  /**
   * A handler that implements reversible addition of shapes.
   *
   * @param {Canvas} canvas
   */
  function CreateShapeHandler(canvas) {
    this._canvas = canvas;
  }
  CreateShapeHandler.$inject = ['canvas'];

  // api //////////////////////

  /**
   * Appends a shape to a target shape
   *
   * @param {Object} context
   * @param {Element} context.parent The parent.
   * @param {Point} context.position The position at which to create the new shape.
   * @param {number} [context.parentIndex] The optional index at which to add the
   * shape to the parent's children.
   */
  CreateShapeHandler.prototype.execute = function (context) {
    var shape = context.shape,
      positionOrBounds = context.position,
      parent = context.parent,
      parentIndex = context.parentIndex;
    if (!parent) {
      throw new Error('parent required');
    }
    if (!positionOrBounds) {
      throw new Error('position required');
    }

    // (1) add at event center position _or_ at given bounds
    if (positionOrBounds.width !== undefined) {
      assign$4(shape, positionOrBounds);
    } else {
      assign$4(shape, {
        x: positionOrBounds.x - round$1(shape.width / 2),
        y: positionOrBounds.y - round$1(shape.height / 2)
      });
    }

    // (2) add to canvas
    this._canvas.addShape(shape, parent, parentIndex);
    return shape;
  };

  /**
   * Undo append by removing the shape
   */
  CreateShapeHandler.prototype.revert = function (context) {
    var shape = context.shape;

    // (3) remove form canvas
    this._canvas.removeShape(shape);
    return shape;
  };

  /**
   * @typedef {import('../../../core/Canvas').default} Canvas
   *
   * @typedef {import('../../../model/Types').Element} Element
   * @typedef {import('../../../model/Types').Parent} Parent
   * @typedef {import('../../../model/Types').Shape} Shape
   * @typedef {import('../../../util/Types').Point} Point
   */

  /**
   * A handler that attaches a label to a given target shape.
   *
   * @param {Canvas} canvas
   */
  function CreateLabelHandler(canvas) {
    CreateShapeHandler.call(this, canvas);
  }
  e$3(CreateLabelHandler, CreateShapeHandler);
  CreateLabelHandler.$inject = ['canvas'];

  // api //////////////////////

  var originalExecute = CreateShapeHandler.prototype.execute;

  /**
   * Append label to element.
   *
   * @param { {
   *   parent: Parent;
   *   position: Point;
   *   shape: Shape;
   *   target: Element;
   * } } context
   */
  CreateLabelHandler.prototype.execute = function (context) {
    var label = context.shape;
    ensureValidDimensions(label);
    label.labelTarget = context.labelTarget;
    return originalExecute.call(this, context);
  };
  var originalRevert = CreateShapeHandler.prototype.revert;

  /**
   * Revert appending by removing label.
   */
  CreateLabelHandler.prototype.revert = function (context) {
    context.shape.labelTarget = null;
    return originalRevert.call(this, context);
  };

  // helpers //////////////////////

  function ensureValidDimensions(label) {
    // make sure a label has valid { width, height } dimensions
    ['width', 'height'].forEach(function (prop) {
      if (typeof label[prop] === 'undefined') {
        label[prop] = 0;
      }
    });
  }

  /**
   * Remove from the beginning of a collection until it is empty.
   *
   * This is a null-safe operation that ensures elements
   * are being removed from the given collection until the
   * collection is empty.
   *
   * The implementation deals with the fact that a remove operation
   * may touch, i.e. remove multiple elements in the collection
   * at a time.
   *
   * @param {Object[]} [collection]
   * @param {(element: Object) => void} removeFn
   *
   * @return {Object[]} the cleared collection
   */
  function saveClear(collection, removeFn) {
    if (typeof removeFn !== 'function') {
      throw new Error('removeFn iterator must be a function');
    }
    if (!collection) {
      return;
    }
    var e;
    while (e = collection[0]) {
      removeFn(e);
    }
    return collection;
  }

  /**
   * @typedef {import('../../../core/Canvas').default} Canvas
   * @typedef {import('../Modeling').default} Modeling
   */

  /**
   * A handler that implements reversible deletion of Connections.
   */
  function DeleteConnectionHandler(canvas, modeling) {
    this._canvas = canvas;
    this._modeling = modeling;
  }
  DeleteConnectionHandler.$inject = ['canvas', 'modeling'];

  /**
   * - Remove connections
   */
  DeleteConnectionHandler.prototype.preExecute = function (context) {
    var modeling = this._modeling;
    var connection = context.connection;

    // remove connections
    saveClear(connection.incoming, function (connection) {
      // To make sure that the connection isn't removed twice
      // For example if a container is removed
      modeling.removeConnection(connection, {
        nested: true
      });
    });
    saveClear(connection.outgoing, function (connection) {
      modeling.removeConnection(connection, {
        nested: true
      });
    });
  };
  DeleteConnectionHandler.prototype.execute = function (context) {
    var connection = context.connection,
      parent = connection.parent;
    context.parent = parent;

    // remember containment
    context.parentIndex = indexOf(parent.children, connection);
    context.source = connection.source;
    context.target = connection.target;
    this._canvas.removeConnection(connection);
    connection.source = null;
    connection.target = null;
    return connection;
  };

  /**
   * Command revert implementation.
   */
  DeleteConnectionHandler.prototype.revert = function (context) {
    var connection = context.connection,
      parent = context.parent,
      parentIndex = context.parentIndex;
    connection.source = context.source;
    connection.target = context.target;

    // restore containment
    add$2(parent.children, connection, parentIndex);
    this._canvas.addConnection(connection, parent);
    return connection;
  };

  /**
   * @typedef {import('../../../core/ElementRegistry').default} ElementRegistry
   * @typedef {import('../Modeling').default} Modeling
   */

  /**
   * @param {Modeling} modeling
   * @param {ElementRegistry} elementRegistry
   */
  function DeleteElementsHandler(modeling, elementRegistry) {
    this._modeling = modeling;
    this._elementRegistry = elementRegistry;
  }
  DeleteElementsHandler.$inject = ['modeling', 'elementRegistry'];
  DeleteElementsHandler.prototype.postExecute = function (context) {
    var modeling = this._modeling,
      elementRegistry = this._elementRegistry,
      elements = context.elements;
    forEach$3(elements, function (element) {
      // element may have been removed with previous
      // remove operations already (e.g. in case of nesting)
      if (!elementRegistry.get(element.id)) {
        return;
      }
      if (element.waypoints) {
        modeling.removeConnection(element);
      } else {
        modeling.removeShape(element);
      }
    });
  };

  /**
   * @typedef {import('../../../core/Canvas').default} Canvas
   * @typedef {import('../Modeling').default} Modeling
   */

  /**
   * A handler that implements reversible deletion of shapes.
   *
   * @param {Canvas} canvas
   * @param {Modeling} modeling
   */
  function DeleteShapeHandler(canvas, modeling) {
    this._canvas = canvas;
    this._modeling = modeling;
  }
  DeleteShapeHandler.$inject = ['canvas', 'modeling'];

  /**
   * - Remove connections
   * - Remove all direct children
   */
  DeleteShapeHandler.prototype.preExecute = function (context) {
    var modeling = this._modeling;
    var shape = context.shape;

    // remove connections
    saveClear(shape.incoming, function (connection) {
      // To make sure that the connection isn't removed twice
      // For example if a container is removed
      modeling.removeConnection(connection, {
        nested: true
      });
    });
    saveClear(shape.outgoing, function (connection) {
      modeling.removeConnection(connection, {
        nested: true
      });
    });

    // remove child shapes and connections
    saveClear(shape.children, function (child) {
      if (isConnection(child)) {
        modeling.removeConnection(child, {
          nested: true
        });
      } else {
        modeling.removeShape(child, {
          nested: true
        });
      }
    });
  };

  /**
   * Remove shape and remember the parent
   */
  DeleteShapeHandler.prototype.execute = function (context) {
    var canvas = this._canvas;
    var shape = context.shape,
      oldParent = shape.parent;
    context.oldParent = oldParent;

    // remove containment
    context.oldParentIndex = indexOf(oldParent.children, shape);

    // remove shape
    canvas.removeShape(shape);
    return shape;
  };

  /**
   * Command revert implementation
   */
  DeleteShapeHandler.prototype.revert = function (context) {
    var canvas = this._canvas;
    var shape = context.shape,
      oldParent = context.oldParent,
      oldParentIndex = context.oldParentIndex;

    // restore containment
    add$2(oldParent.children, shape, oldParentIndex);
    canvas.addShape(shape, oldParent);
    return shape;
  };

  /**
   * @typedef {import('../Modeling').default} Modeling
   */

  /**
   * A handler that distributes elements evenly.
   *
   * @param {Modeling} modeling
   */
  function DistributeElements(modeling) {
    this._modeling = modeling;
  }
  DistributeElements.$inject = ['modeling'];
  var OFF_AXIS = {
    x: 'y',
    y: 'x'
  };
  DistributeElements.prototype.preExecute = function (context) {
    var modeling = this._modeling;
    var groups = context.groups,
      axis = context.axis,
      dimension = context.dimension;
    function updateRange(group, element) {
      group.range.min = Math.min(element[axis], group.range.min);
      group.range.max = Math.max(element[axis] + element[dimension], group.range.max);
    }
    function center(element) {
      return element[axis] + element[dimension] / 2;
    }
    function lastIdx(arr) {
      return arr.length - 1;
    }
    function rangeDiff(range) {
      return range.max - range.min;
    }
    function centerElement(refCenter, element) {
      var delta = {
        y: 0
      };
      delta[axis] = refCenter - center(element);
      if (delta[axis]) {
        delta[OFF_AXIS[axis]] = 0;
        modeling.moveElements([element], delta, element.parent);
      }
    }
    var firstGroup = groups[0],
      lastGroupIdx = lastIdx(groups),
      lastGroup = groups[lastGroupIdx];
    var margin,
      spaceInBetween,
      groupsSize = 0; // the size of each range

    forEach$3(groups, function (group, idx) {
      var sortedElements, refElem, refCenter;
      if (group.elements.length < 2) {
        if (idx && idx !== groups.length - 1) {
          updateRange(group, group.elements[0]);
          groupsSize += rangeDiff(group.range);
        }
        return;
      }
      sortedElements = sortBy(group.elements, axis);
      refElem = sortedElements[0];
      if (idx === lastGroupIdx) {
        refElem = sortedElements[lastIdx(sortedElements)];
      }
      refCenter = center(refElem);

      // wanna update the ranges after the shapes have been centered
      group.range = null;
      forEach$3(sortedElements, function (element) {
        centerElement(refCenter, element);
        if (group.range === null) {
          group.range = {
            min: element[axis],
            max: element[axis] + element[dimension]
          };
          return;
        }

        // update group's range after centering the range elements
        updateRange(group, element);
      });
      if (idx && idx !== groups.length - 1) {
        groupsSize += rangeDiff(group.range);
      }
    });
    spaceInBetween = Math.abs(lastGroup.range.min - firstGroup.range.max);
    margin = Math.round((spaceInBetween - groupsSize) / (groups.length - 1));
    if (margin < groups.length - 1) {
      return;
    }
    forEach$3(groups, function (group, groupIdx) {
      var delta = {},
        prevGroup;
      if (group === firstGroup || group === lastGroup) {
        return;
      }
      prevGroup = groups[groupIdx - 1];
      group.range.max = 0;
      forEach$3(group.elements, function (element, idx) {
        delta[OFF_AXIS[axis]] = 0;
        delta[axis] = prevGroup.range.max - element[axis] + margin;
        if (group.range.min !== element[axis]) {
          delta[axis] += element[axis] - group.range.min;
        }
        if (delta[axis]) {
          modeling.moveElements([element], delta, element.parent);
        }
        group.range.max = Math.max(element[axis] + element[dimension], idx ? group.range.max : 0);
      });
    });
  };
  DistributeElements.prototype.postExecute = function (context) {};

  /**
   * @typedef {import('../../../core/Canvas').default} Canvas
   * @typedef {import('../../../layout/BaseLayouter').default} Layouter
   */

  /**
   * A handler that implements reversible moving of shapes.
   *
   * @param {Layouter} layouter
   * @param {Canvas} canvas
   */
  function LayoutConnectionHandler(layouter, canvas) {
    this._layouter = layouter;
    this._canvas = canvas;
  }
  LayoutConnectionHandler.$inject = ['layouter', 'canvas'];
  LayoutConnectionHandler.prototype.execute = function (context) {
    var connection = context.connection;
    var oldWaypoints = connection.waypoints;
    assign$4(context, {
      oldWaypoints: oldWaypoints
    });
    connection.waypoints = this._layouter.layoutConnection(connection, context.hints);
    return connection;
  };
  LayoutConnectionHandler.prototype.revert = function (context) {
    var connection = context.connection;
    connection.waypoints = context.oldWaypoints;
    return connection;
  };

  /**
   * A handler that implements reversible moving of connections.
   *
   * The handler differs from the layout connection handler in a sense
   * that it preserves the connection layout.
   */
  function MoveConnectionHandler() {}
  MoveConnectionHandler.prototype.execute = function (context) {
    var connection = context.connection,
      delta = context.delta;
    var newParent = context.newParent || connection.parent,
      newParentIndex = context.newParentIndex,
      oldParent = connection.parent;

    // save old parent in context
    context.oldParent = oldParent;
    context.oldParentIndex = remove$2(oldParent.children, connection);

    // add to new parent at position
    add$2(newParent.children, connection, newParentIndex);

    // update parent
    connection.parent = newParent;

    // update waypoint positions
    forEach$3(connection.waypoints, function (p) {
      p.x += delta.x;
      p.y += delta.y;
      if (p.original) {
        p.original.x += delta.x;
        p.original.y += delta.y;
      }
    });
    return connection;
  };
  MoveConnectionHandler.prototype.revert = function (context) {
    var connection = context.connection,
      newParent = connection.parent,
      oldParent = context.oldParent,
      oldParentIndex = context.oldParentIndex,
      delta = context.delta;

    // remove from newParent
    remove$2(newParent.children, connection);

    // restore previous location in old parent
    add$2(oldParent.children, connection, oldParentIndex);

    // restore parent
    connection.parent = oldParent;

    // revert to old waypoint positions
    forEach$3(connection.waypoints, function (p) {
      p.x -= delta.x;
      p.y -= delta.y;
      if (p.original) {
        p.original.x -= delta.x;
        p.original.y -= delta.y;
      }
    });
    return connection;
  };

  /**
   * @typedef {import('../model/Types').Shape} Shape
   *
   * @typedef {import('../util/Types').Point} Point
   * @typedef {import('../util/Types').Rect} Rect
   */

  /**
   * Calculates the absolute point relative to the new element's position.
   *
   * @param {Point} point [absolute]
   * @param {Rect} oldBounds
   * @param {Rect} newBounds
   *
   * @return {Point} point [absolute]
   */
  function getNewAttachPoint(point, oldBounds, newBounds) {
    var oldCenter = center(oldBounds),
      newCenter = center(newBounds),
      oldDelta = delta(point, oldCenter);
    var newDelta = {
      x: oldDelta.x * (newBounds.width / oldBounds.width),
      y: oldDelta.y * (newBounds.height / oldBounds.height)
    };
    return roundPoint({
      x: newCenter.x + newDelta.x,
      y: newCenter.y + newDelta.y
    });
  }

  /**
   * @typedef {import('../../../../core/Types').ConnectionLike} Connection
   * @typedef {import('../../../../core/Types').ShapeLike} Shape
   *
   * @typedef {import('../../../../util/Types').Point} Point
   * @typedef {import('../../../../util/Types').Rect} Rect
   */

  /**
   * @param {Connection} connection
   * @param {Shape} shape
   * @param {Rect} oldBounds
   * @return {Point}
   */
  function getResizedSourceAnchor(connection, shape, oldBounds) {
    var waypoints = safeGetWaypoints(connection),
      waypointsInsideNewBounds = getWaypointsInsideBounds(waypoints, shape),
      oldAnchor = waypoints[0];

    // new anchor is the last waypoint enclosed be resized source
    if (waypointsInsideNewBounds.length) {
      return waypointsInsideNewBounds[waypointsInsideNewBounds.length - 1];
    }
    return getNewAttachPoint(oldAnchor.original || oldAnchor, oldBounds, shape);
  }
  function getResizedTargetAnchor(connection, shape, oldBounds) {
    var waypoints = safeGetWaypoints(connection),
      waypointsInsideNewBounds = getWaypointsInsideBounds(waypoints, shape),
      oldAnchor = waypoints[waypoints.length - 1];

    // new anchor is the first waypoint enclosed be resized target
    if (waypointsInsideNewBounds.length) {
      return waypointsInsideNewBounds[0];
    }
    return getNewAttachPoint(oldAnchor.original || oldAnchor, oldBounds, shape);
  }
  function getMovedSourceAnchor(connection, source, moveDelta) {
    var waypoints = safeGetWaypoints(connection),
      oldBounds = subtract(source, moveDelta),
      oldAnchor = waypoints[0];
    return getNewAttachPoint(oldAnchor.original || oldAnchor, oldBounds, source);
  }
  function getMovedTargetAnchor(connection, target, moveDelta) {
    var waypoints = safeGetWaypoints(connection),
      oldBounds = subtract(target, moveDelta),
      oldAnchor = waypoints[waypoints.length - 1];
    return getNewAttachPoint(oldAnchor.original || oldAnchor, oldBounds, target);
  }

  // helpers //////////////////////

  function subtract(bounds, delta) {
    return {
      x: bounds.x - delta.x,
      y: bounds.y - delta.y,
      width: bounds.width,
      height: bounds.height
    };
  }

  /**
   * Return waypoints of given connection; throw if non exists (should not happen!!).
   *
   * @param {Connection} connection
   *
   * @return {Point[]}
   */
  function safeGetWaypoints(connection) {
    var waypoints = connection.waypoints;
    if (!waypoints.length) {
      throw new Error('connection#' + connection.id + ': no waypoints');
    }
    return waypoints;
  }
  function getWaypointsInsideBounds(waypoints, bounds) {
    var originalWaypoints = map$3(waypoints, getOriginal);
    return filter$2(originalWaypoints, function (waypoint) {
      return isInsideBounds(waypoint, bounds);
    });
  }

  /**
   * Checks if point is inside bounds, incl. edges.
   *
   * @param {Point} point
   * @param {Rect} bounds
   */
  function isInsideBounds(point, bounds) {
    return getOrientation(bounds, point, 1) === 'intersect';
  }
  function getOriginal(point) {
    return point.original || point;
  }

  /**
   * @typedef {import('../../../../model/Types').Connection} Connection
   * @typedef {import('../../../../model/Types').Element} Element
   * @typedef {import('../../../../model/Types').Shape} Shape
   */

  function MoveClosure() {
    /**
     * @type {Record<string, Shape>}
     */
    this.allShapes = {};

    /**
     * @type {Record<string, Connection>}
     */
    this.allConnections = {};

    /**
     * @type {Record<string, Element>}
     */
    this.enclosedElements = {};

    /**
     * @type {Record<string, Connection>}
     */
    this.enclosedConnections = {};

    /**
     * @type {Record<string, Element>}
     */
    this.topLevel = {};
  }

  /**
   * @param {Element} element
   * @param {boolean} [isTopLevel]
   *
   * @return {MoveClosure}
   */
  MoveClosure.prototype.add = function (element, isTopLevel) {
    return this.addAll([element], isTopLevel);
  };

  /**
   * @param {Element[]} elements
   * @param {boolean} [isTopLevel]
   *
   * @return {MoveClosure}
   */
  MoveClosure.prototype.addAll = function (elements, isTopLevel) {
    var newClosure = getClosure(elements, !!isTopLevel, this);
    assign$4(this, newClosure);
    return this;
  };

  /**
   * @typedef {import('../../../../core/Types').ElementLike} Element
   * @typedef {import('../../../../core/Types').ShapeLike} Shape
   *
   * @typedef {import('../../../../util/Types').Point} Point
   *
   * @typedef {import('../../Modeling').default} Modeling
   */

  /**
   * A helper that is able to carry out serialized move
   * operations on multiple elements.
   *
   * @param {Modeling} modeling
   */
  function MoveHelper(modeling) {
    this._modeling = modeling;
  }

  /**
   * Move the specified elements and all children by the given delta.
   *
   * This moves all enclosed connections, too and layouts all affected
   * external connections.
   *
   * @template {Element} T
   *
   * @param {T[]} elements
   * @param {Point} delta
   * @param {Shape} newParent The new parent of all elements that are not nested.
   *
   * @return {T[]}
   */
  MoveHelper.prototype.moveRecursive = function (elements, delta, newParent) {
    if (!elements) {
      return [];
    } else {
      return this.moveClosure(this.getClosure(elements), delta, newParent);
    }
  };

  /**
   * Move the given closure of elmements.
   *
   * @param {Object} closure
   * @param {Point} delta
   * @param {Shape} [newParent]
   * @param {Shape} [newHost]
   */
  MoveHelper.prototype.moveClosure = function (closure, delta, newParent, newHost, primaryShape) {
    var modeling = this._modeling;
    var allShapes = closure.allShapes,
      allConnections = closure.allConnections,
      enclosedConnections = closure.enclosedConnections,
      topLevel = closure.topLevel,
      keepParent = false;
    if (primaryShape && primaryShape.parent === newParent) {
      keepParent = true;
    }

    // move all shapes
    forEach$3(allShapes, function (shape) {
      // move the element according to the given delta
      modeling.moveShape(shape, delta, topLevel[shape.id] && !keepParent && newParent, {
        recurse: false,
        layout: false
      });
    });

    // move all child connections / layout external connections
    forEach$3(allConnections, function (c) {
      var sourceMoved = !!allShapes[c.source.id],
        targetMoved = !!allShapes[c.target.id];
      if (enclosedConnections[c.id] && sourceMoved && targetMoved) {
        modeling.moveConnection(c, delta, topLevel[c.id] && !keepParent && newParent);
      } else {
        modeling.layoutConnection(c, {
          connectionStart: sourceMoved && getMovedSourceAnchor(c, c.source, delta),
          connectionEnd: targetMoved && getMovedTargetAnchor(c, c.target, delta)
        });
      }
    });
  };

  /**
   * Returns the closure for the selected elements
   *
   * @param {Element[]} elements
   *
   * @return {MoveClosure}
   */
  MoveHelper.prototype.getClosure = function (elements) {
    return new MoveClosure().addAll(elements, true);
  };

  /**
   * @typedef {import('../Modeling').default} Modeling
   */

  /**
   * A handler that implements reversible moving of shapes.
   *
   * @param {Modeling} modeling
   */
  function MoveElementsHandler(modeling) {
    this._helper = new MoveHelper(modeling);
  }
  MoveElementsHandler.$inject = ['modeling'];
  MoveElementsHandler.prototype.preExecute = function (context) {
    context.closure = this._helper.getClosure(context.shapes);
  };
  MoveElementsHandler.prototype.postExecute = function (context) {
    var hints = context.hints,
      primaryShape;
    if (hints && hints.primaryShape) {
      primaryShape = hints.primaryShape;
      hints.oldParent = primaryShape.parent;
    }
    this._helper.moveClosure(context.closure, context.delta, context.newParent, context.newHost, primaryShape);
  };

  /**
   * @typedef {import('../Modeling').default} Modeling
   */

  /**
   * A handler that implements reversible moving of shapes.
   *
   * @param {Modeling} modeling
   */
  function MoveShapeHandler(modeling) {
    this._modeling = modeling;
    this._helper = new MoveHelper(modeling);
  }
  MoveShapeHandler.$inject = ['modeling'];
  MoveShapeHandler.prototype.execute = function (context) {
    var shape = context.shape,
      delta = context.delta,
      newParent = context.newParent || shape.parent,
      newParentIndex = context.newParentIndex,
      oldParent = shape.parent;
    context.oldBounds = pick$1(shape, ['x', 'y', 'width', 'height']);

    // save old parent in context
    context.oldParent = oldParent;
    context.oldParentIndex = remove$2(oldParent.children, shape);

    // add to new parent at position
    add$2(newParent.children, shape, newParentIndex);

    // update shape parent + position
    assign$4(shape, {
      parent: newParent,
      x: shape.x + delta.x,
      y: shape.y + delta.y
    });
    return shape;
  };
  MoveShapeHandler.prototype.postExecute = function (context) {
    var shape = context.shape,
      delta = context.delta,
      hints = context.hints;
    var modeling = this._modeling;
    if (hints.layout !== false) {
      forEach$3(shape.incoming, function (c) {
        modeling.layoutConnection(c, {
          connectionEnd: getMovedTargetAnchor(c, shape, delta)
        });
      });
      forEach$3(shape.outgoing, function (c) {
        modeling.layoutConnection(c, {
          connectionStart: getMovedSourceAnchor(c, shape, delta)
        });
      });
    }
    if (hints.recurse !== false) {
      this.moveChildren(context);
    }
  };
  MoveShapeHandler.prototype.revert = function (context) {
    var shape = context.shape,
      oldParent = context.oldParent,
      oldParentIndex = context.oldParentIndex,
      delta = context.delta;

    // restore previous location in old parent
    add$2(oldParent.children, shape, oldParentIndex);

    // revert to old position and parent
    assign$4(shape, {
      parent: oldParent,
      x: shape.x - delta.x,
      y: shape.y - delta.y
    });
    return shape;
  };
  MoveShapeHandler.prototype.moveChildren = function (context) {
    var delta = context.delta,
      shape = context.shape;
    this._helper.moveRecursive(shape.children, delta, null);
  };
  MoveShapeHandler.prototype.getNewParent = function (context) {
    return context.newParent || context.shape.parent;
  };

  /**
   * @typedef {import('../Modeling').default} Modeling
   */

  /**
   * Reconnect connection handler.
   *
   * @param {Modeling} modeling
   */
  function ReconnectConnectionHandler(modeling) {
    this._modeling = modeling;
  }
  ReconnectConnectionHandler.$inject = ['modeling'];
  ReconnectConnectionHandler.prototype.execute = function (context) {
    var newSource = context.newSource,
      newTarget = context.newTarget,
      connection = context.connection,
      dockingOrPoints = context.dockingOrPoints;
    if (!newSource && !newTarget) {
      throw new Error('newSource or newTarget required');
    }
    if (isArray$6(dockingOrPoints)) {
      context.oldWaypoints = connection.waypoints;
      connection.waypoints = dockingOrPoints;
    }
    if (newSource) {
      context.oldSource = connection.source;
      connection.source = newSource;
    }
    if (newTarget) {
      context.oldTarget = connection.target;
      connection.target = newTarget;
    }
    return connection;
  };
  ReconnectConnectionHandler.prototype.postExecute = function (context) {
    var connection = context.connection,
      newSource = context.newSource,
      newTarget = context.newTarget,
      dockingOrPoints = context.dockingOrPoints,
      hints = context.hints || {};
    var layoutConnectionHints = {};
    if (hints.connectionStart) {
      layoutConnectionHints.connectionStart = hints.connectionStart;
    }
    if (hints.connectionEnd) {
      layoutConnectionHints.connectionEnd = hints.connectionEnd;
    }
    if (hints.layoutConnection === false) {
      return;
    }
    if (newSource && (!newTarget || hints.docking === 'source')) {
      layoutConnectionHints.connectionStart = layoutConnectionHints.connectionStart || getDocking(isArray$6(dockingOrPoints) ? dockingOrPoints[0] : dockingOrPoints);
    }
    if (newTarget && (!newSource || hints.docking === 'target')) {
      layoutConnectionHints.connectionEnd = layoutConnectionHints.connectionEnd || getDocking(isArray$6(dockingOrPoints) ? dockingOrPoints[dockingOrPoints.length - 1] : dockingOrPoints);
    }
    if (hints.newWaypoints) {
      layoutConnectionHints.waypoints = hints.newWaypoints;
    }
    this._modeling.layoutConnection(connection, layoutConnectionHints);
  };
  ReconnectConnectionHandler.prototype.revert = function (context) {
    var oldSource = context.oldSource,
      oldTarget = context.oldTarget,
      oldWaypoints = context.oldWaypoints,
      connection = context.connection;
    if (oldSource) {
      connection.source = oldSource;
    }
    if (oldTarget) {
      connection.target = oldTarget;
    }
    if (oldWaypoints) {
      connection.waypoints = oldWaypoints;
    }
    return connection;
  };

  // helpers //////////

  function getDocking(point) {
    return point.original || point;
  }

  /**
   * @typedef {import('../../model/Types').Shape} Shape
   *
   * @typedef {import('../Modeling').default} Modeling
   * @typedef {import('../../rules/Rules').default} Rules
   */

  /**
   * Replace shape by adding new shape and removing old shape. Incoming and outgoing connections will
   * be kept if possible.
   *
   * @class
   * @constructor
   *
   * @param {Modeling} modeling
   * @param {Rules} rules
   */
  function ReplaceShapeHandler(modeling, rules) {
    this._modeling = modeling;
    this._rules = rules;
  }
  ReplaceShapeHandler.$inject = ['modeling', 'rules'];

  /**
   * Add new shape.
   *
   * @param {Object} context
   * @param {Shape} context.oldShape
   * @param {Object} context.newData
   * @param {string} context.newData.type
   * @param {number} context.newData.x
   * @param {number} context.newData.y
   * @param {Object} [context.hints]
   */
  ReplaceShapeHandler.prototype.preExecute = function (context) {
    var self = this,
      modeling = this._modeling,
      rules = this._rules;
    var oldShape = context.oldShape,
      newData = context.newData,
      hints = context.hints || {},
      newShape;
    function canReconnect(source, target, connection) {
      return rules.allowed('connection.reconnect', {
        connection: connection,
        source: source,
        target: target
      });
    }

    // (1) add new shape at given position
    var position = {
      x: newData.x,
      y: newData.y
    };
    var oldBounds = {
      x: oldShape.x,
      y: oldShape.y,
      width: oldShape.width,
      height: oldShape.height
    };
    newShape = context.newShape = context.newShape || self.createShape(newData, position, oldShape.parent, hints);

    // (2) update host
    if (oldShape.host) {
      modeling.updateAttachment(newShape, oldShape.host);
    }

    // (3) adopt all children from old shape
    var children;
    if (hints.moveChildren !== false) {
      children = oldShape.children.slice();
      modeling.moveElements(children, {
        x: 0,
        y: 0
      }, newShape, hints);
    }

    // (4) reconnect connections to new shape if possible
    var incoming = oldShape.incoming.slice(),
      outgoing = oldShape.outgoing.slice();
    forEach$3(incoming, function (connection) {
      var source = connection.source,
        allowed = canReconnect(source, newShape, connection);
      if (allowed) {
        self.reconnectEnd(connection, newShape, getResizedTargetAnchor(connection, newShape, oldBounds), hints);
      }
    });
    forEach$3(outgoing, function (connection) {
      var target = connection.target,
        allowed = canReconnect(newShape, target, connection);
      if (allowed) {
        self.reconnectStart(connection, newShape, getResizedSourceAnchor(connection, newShape, oldBounds), hints);
      }
    });
  };

  /**
   * Remove old shape.
   */
  ReplaceShapeHandler.prototype.postExecute = function (context) {
    var oldShape = context.oldShape;
    this._modeling.removeShape(oldShape);
  };
  ReplaceShapeHandler.prototype.execute = function (context) {};
  ReplaceShapeHandler.prototype.revert = function (context) {};
  ReplaceShapeHandler.prototype.createShape = function (shape, position, target, hints) {
    return this._modeling.createShape(shape, position, target, hints);
  };
  ReplaceShapeHandler.prototype.reconnectStart = function (connection, newSource, dockingPoint, hints) {
    this._modeling.reconnectStart(connection, newSource, dockingPoint, hints);
  };
  ReplaceShapeHandler.prototype.reconnectEnd = function (connection, newTarget, dockingPoint, hints) {
    this._modeling.reconnectEnd(connection, newTarget, dockingPoint, hints);
  };

  /**
   * @typedef {import('../Modeling').default} Modeling
   */

  /**
   * A handler that implements reversible resizing of shapes.
   *
   * @param {Modeling} modeling
   */
  function ResizeShapeHandler(modeling) {
    this._modeling = modeling;
  }
  ResizeShapeHandler.$inject = ['modeling'];

  /**
   * {
   *   shape: {....}
   *   newBounds: {
   *     width:  20,
   *     height: 40,
   *     x:       5,
   *     y:      10
   *   }
   *
   * }
   */
  ResizeShapeHandler.prototype.execute = function (context) {
    var shape = context.shape,
      newBounds = context.newBounds,
      minBounds = context.minBounds;
    if (newBounds.x === undefined || newBounds.y === undefined || newBounds.width === undefined || newBounds.height === undefined) {
      throw new Error('newBounds must have {x, y, width, height} properties');
    }
    if (minBounds && (newBounds.width < minBounds.width || newBounds.height < minBounds.height)) {
      throw new Error('width and height cannot be less than minimum height and width');
    } else if (!minBounds && newBounds.width < 10 || newBounds.height < 10) {
      throw new Error('width and height cannot be less than 10px');
    }

    // save old bbox in context
    context.oldBounds = {
      width: shape.width,
      height: shape.height,
      x: shape.x,
      y: shape.y
    };

    // update shape
    assign$4(shape, {
      width: newBounds.width,
      height: newBounds.height,
      x: newBounds.x,
      y: newBounds.y
    });
    return shape;
  };
  ResizeShapeHandler.prototype.postExecute = function (context) {
    var modeling = this._modeling;
    var shape = context.shape,
      oldBounds = context.oldBounds,
      hints = context.hints || {};
    if (hints.layout === false) {
      return;
    }
    forEach$3(shape.incoming, function (c) {
      modeling.layoutConnection(c, {
        connectionEnd: getResizedTargetAnchor(c, shape, oldBounds)
      });
    });
    forEach$3(shape.outgoing, function (c) {
      modeling.layoutConnection(c, {
        connectionStart: getResizedSourceAnchor(c, shape, oldBounds)
      });
    });
  };
  ResizeShapeHandler.prototype.revert = function (context) {
    var shape = context.shape,
      oldBounds = context.oldBounds;

    // restore previous bbox
    assign$4(shape, {
      width: oldBounds.width,
      height: oldBounds.height,
      x: oldBounds.x,
      y: oldBounds.y
    });
    return shape;
  };

  /**
   * @typedef {import('../../core/Types').ConnectionLike} Connection
   * @typedef {import('../../core/Types').ShapeLike} Shape
   *
   * @typedef {import('../../util/Types').Axis} Axis
   * @typedef {import('../../util/Types').Direction} Direction
   * @typedef {import('../../util/Types').Point} Point
   * @typedef {import('../../util/Types').Rect} Rect
   */


  /**
   * Returns connections whose waypoints are to be updated. Waypoints are to be updated if start
   * or end is to be moved or resized.
   *
   * @param {Array<Shape>} movingShapes
   * @param {Array<Shape>} resizingShapes
   *
   * @return {Array<Connection>}
   */
  function getWaypointsUpdatingConnections(movingShapes, resizingShapes) {
    var waypointsUpdatingConnections = [];
    forEach$3(movingShapes.concat(resizingShapes), function (shape) {
      var incoming = shape.incoming,
        outgoing = shape.outgoing;
      forEach$3(incoming.concat(outgoing), function (connection) {
        var source = connection.source,
          target = connection.target;
        if (includes$2(movingShapes, source) || includes$2(movingShapes, target) || includes$2(resizingShapes, source) || includes$2(resizingShapes, target)) {
          if (!includes$2(waypointsUpdatingConnections, connection)) {
            waypointsUpdatingConnections.push(connection);
          }
        }
      });
    });
    return waypointsUpdatingConnections;
  }
  function includes$2(array, item) {
    return array.indexOf(item) !== -1;
  }

  /**
   * Resize bounds.
   *
   * @param {Rect} bounds
   * @param {Direction} direction
   * @param {Point} delta
   *
   * @return {Rect}
   */
  function resizeBounds$1(bounds, direction, delta) {
    var x = bounds.x,
      y = bounds.y,
      width = bounds.width,
      height = bounds.height,
      dx = delta.x,
      dy = delta.y;
    switch (direction) {
      case 'n':
        return {
          x: x,
          y: y + dy,
          width: width,
          height: height - dy
        };
      case 's':
        return {
          x: x,
          y: y,
          width: width,
          height: height + dy
        };
      case 'w':
        return {
          x: x + dx,
          y: y,
          width: width - dx,
          height: height
        };
      case 'e':
        return {
          x: x,
          y: y,
          width: width + dx,
          height: height
        };
      default:
        throw new Error('unknown direction: ' + direction);
    }
  }

  /**
   * @typedef {import('../Modeling').default} Modeling
   */

  /**
   * Add or remove space by moving and resizing shapes and updating connection waypoints.
   *
   * @param {Modeling} modeling
   */
  function SpaceToolHandler(modeling) {
    this._modeling = modeling;
  }
  SpaceToolHandler.$inject = ['modeling'];
  SpaceToolHandler.prototype.preExecute = function (context) {
    var delta = context.delta,
      direction = context.direction,
      movingShapes = context.movingShapes,
      resizingShapes = context.resizingShapes,
      start = context.start,
      oldBounds = {};

    // (1) move shapes
    this.moveShapes(movingShapes, delta);

    // (2a) save old bounds of resized shapes
    forEach$3(resizingShapes, function (shape) {
      oldBounds[shape.id] = getBounds(shape);
    });

    // (2b) resize shapes
    this.resizeShapes(resizingShapes, delta, direction);

    // (3) update connection waypoints
    this.updateConnectionWaypoints(getWaypointsUpdatingConnections(movingShapes, resizingShapes), delta, direction, start, movingShapes, resizingShapes, oldBounds);
  };
  SpaceToolHandler.prototype.execute = function () {};
  SpaceToolHandler.prototype.revert = function () {};
  SpaceToolHandler.prototype.moveShapes = function (shapes, delta) {
    var self = this;
    forEach$3(shapes, function (element) {
      self._modeling.moveShape(element, delta, null, {
        autoResize: false,
        layout: false,
        recurse: false
      });
    });
  };
  SpaceToolHandler.prototype.resizeShapes = function (shapes, delta, direction) {
    var self = this;
    forEach$3(shapes, function (shape) {
      var newBounds = resizeBounds$1(shape, direction, delta);
      self._modeling.resizeShape(shape, newBounds, null, {
        attachSupport: false,
        autoResize: false,
        layout: false
      });
    });
  };

  /**
   * Update connections waypoints according to the rules:
   *   1. Both source and target are moved/resized => move waypoints by the delta
   *   2. Only one of source and target is moved/resized => re-layout connection with moved start/end
   */
  SpaceToolHandler.prototype.updateConnectionWaypoints = function (connections, delta, direction, start, movingShapes, resizingShapes, oldBounds) {
    var self = this,
      affectedShapes = movingShapes.concat(resizingShapes);
    forEach$3(connections, function (connection) {
      var source = connection.source,
        target = connection.target,
        waypoints = copyWaypoints(connection),
        axis = getAxisFromDirection(direction),
        layoutHints = {};
      if (includes$1(affectedShapes, source) && includes$1(affectedShapes, target)) {
        // move waypoints
        waypoints = map$3(waypoints, function (waypoint) {
          if (shouldMoveWaypoint(waypoint, start, direction)) {
            // move waypoint
            waypoint[axis] = waypoint[axis] + delta[axis];
          }
          if (waypoint.original && shouldMoveWaypoint(waypoint.original, start, direction)) {
            // move waypoint original
            waypoint.original[axis] = waypoint.original[axis] + delta[axis];
          }
          return waypoint;
        });
        self._modeling.updateWaypoints(connection, waypoints, {
          labelBehavior: false
        });
      } else if (includes$1(affectedShapes, source) || includes$1(affectedShapes, target)) {
        // re-layout connection with moved start/end
        if (includes$1(movingShapes, source)) {
          layoutHints.connectionStart = getMovedSourceAnchor(connection, source, delta);
        } else if (includes$1(movingShapes, target)) {
          layoutHints.connectionEnd = getMovedTargetAnchor(connection, target, delta);
        } else if (includes$1(resizingShapes, source)) {
          layoutHints.connectionStart = getResizedSourceAnchor(connection, source, oldBounds[source.id]);
        } else if (includes$1(resizingShapes, target)) {
          layoutHints.connectionEnd = getResizedTargetAnchor(connection, target, oldBounds[target.id]);
        }
        self._modeling.layoutConnection(connection, layoutHints);
      }
    });
  };

  // helpers //////////

  function copyWaypoint(waypoint) {
    return assign$4({}, waypoint);
  }
  function copyWaypoints(connection) {
    return map$3(connection.waypoints, function (waypoint) {
      waypoint = copyWaypoint(waypoint);
      if (waypoint.original) {
        waypoint.original = copyWaypoint(waypoint.original);
      }
      return waypoint;
    });
  }
  function getAxisFromDirection(direction) {
    switch (direction) {
      case 'n':
        return 'y';
      case 'w':
        return 'x';
      case 's':
        return 'y';
      case 'e':
        return 'x';
    }
  }
  function shouldMoveWaypoint(waypoint, start, direction) {
    var relevantAxis = getAxisFromDirection(direction);
    if (/e|s/.test(direction)) {
      return waypoint[relevantAxis] > start;
    } else if (/n|w/.test(direction)) {
      return waypoint[relevantAxis] < start;
    }
  }
  function includes$1(array, item) {
    return array.indexOf(item) !== -1;
  }
  function getBounds(shape) {
    return {
      x: shape.x,
      y: shape.y,
      height: shape.height,
      width: shape.width
    };
  }

  /**
   * @typedef {import('../../model/Types').Shape} Shape
   *
   * @typedef {import('../Modeling').default} Modeling
   */

  /**
   * A handler that toggles the collapsed state of an element
   * and the visibility of all its children.
   *
   * @param {Modeling} modeling
   */
  function ToggleShapeCollapseHandler(modeling) {
    this._modeling = modeling;
  }
  ToggleShapeCollapseHandler.$inject = ['modeling'];
  ToggleShapeCollapseHandler.prototype.execute = function (context) {
    var shape = context.shape,
      children = shape.children;

    // recursively remember previous visibility of children
    context.oldChildrenVisibility = getElementsVisibilityRecursive(children);

    // toggle state
    shape.collapsed = !shape.collapsed;

    // recursively hide/show children
    var result = setHiddenRecursive(children, shape.collapsed);
    return [shape].concat(result);
  };
  ToggleShapeCollapseHandler.prototype.revert = function (context) {
    var shape = context.shape,
      oldChildrenVisibility = context.oldChildrenVisibility;
    var children = shape.children;

    // recursively set old visability of children
    var result = restoreVisibilityRecursive(children, oldChildrenVisibility);

    // retoggle state
    shape.collapsed = !shape.collapsed;
    return [shape].concat(result);
  };

  // helpers //////////////////////

  /**
   * Return a map { elementId -> hiddenState}.
   *
   * @param {Shape[]} elements
   *
   * @return {Object}
   */
  function getElementsVisibilityRecursive(elements) {
    var result = {};
    forEach$3(elements, function (element) {
      result[element.id] = element.hidden;
      if (element.children) {
        result = assign$4({}, result, getElementsVisibilityRecursive(element.children));
      }
    });
    return result;
  }
  function setHiddenRecursive(elements, newHidden) {
    var result = [];
    forEach$3(elements, function (element) {
      element.hidden = newHidden;
      result = result.concat(element);
      if (element.children) {
        result = result.concat(setHiddenRecursive(element.children, element.collapsed || newHidden));
      }
    });
    return result;
  }
  function restoreVisibilityRecursive(elements, lastState) {
    var result = [];
    forEach$3(elements, function (element) {
      element.hidden = lastState[element.id];
      result = result.concat(element);
      if (element.children) {
        result = result.concat(restoreVisibilityRecursive(element.children, lastState));
      }
    });
    return result;
  }

  /**
   * @typedef {import('../Modeling').default} Modeling
   */

  /**
   * A handler that implements reversible attaching/detaching of shapes.
   *
   * @param {Modeling} modeling
   */
  function UpdateAttachmentHandler(modeling) {
    this._modeling = modeling;
  }
  UpdateAttachmentHandler.$inject = ['modeling'];
  UpdateAttachmentHandler.prototype.execute = function (context) {
    var shape = context.shape,
      newHost = context.newHost,
      oldHost = shape.host;

    // (0) detach from old host
    context.oldHost = oldHost;
    context.attacherIdx = removeAttacher(oldHost, shape);

    // (1) attach to new host
    addAttacher(newHost, shape);

    // (2) update host
    shape.host = newHost;
    return shape;
  };
  UpdateAttachmentHandler.prototype.revert = function (context) {
    var shape = context.shape,
      newHost = context.newHost,
      oldHost = context.oldHost,
      attacherIdx = context.attacherIdx;

    // (2) update host
    shape.host = oldHost;

    // (1) attach to new host
    removeAttacher(newHost, shape);

    // (0) detach from old host
    addAttacher(oldHost, shape, attacherIdx);
    return shape;
  };
  function removeAttacher(host, attacher) {
    // remove attacher from host
    return remove$2(host && host.attachers, attacher);
  }
  function addAttacher(host, attacher, idx) {
    if (!host) {
      return;
    }
    var attachers = host.attachers;
    if (!attachers) {
      host.attachers = attachers = [];
    }
    add$2(attachers, attacher, idx);
  }

  function UpdateWaypointsHandler() {}
  UpdateWaypointsHandler.prototype.execute = function (context) {
    var connection = context.connection,
      newWaypoints = context.newWaypoints;
    context.oldWaypoints = connection.waypoints;
    connection.waypoints = newWaypoints;
    return connection;
  };
  UpdateWaypointsHandler.prototype.revert = function (context) {
    var connection = context.connection,
      oldWaypoints = context.oldWaypoints;
    connection.waypoints = oldWaypoints;
    return connection;
  };

  /**
   * @typedef {import('../../model/Types').Element} Element
   * @typedef {import('../../model/Types').Connection} Connection
   * @typedef {import('../../model/Types').Parent} Parent
   * @typedef {import('../../model/Types').Shape} Shape
   * @typedef {import('../../model/Types').Label} Label
   *
   * @typedef {import('../../command/CommandStack').default} CommandStack
   * @typedef {import('../../core/ElementFactory').default} ElementFactory
   * @typedef {import('../../core/EventBus').default} EventBus
   *
   * @typedef {import('../../command/CommandStack').CommandHandlerConstructor} CommandHandlerConstructor
   *
   * @typedef {import('../../util/Types').Dimensions} Dimensions
   * @typedef {import('../../util/Types').Direction} Direction
   * @typedef {import('../../util/Types').Point} Point
   * @typedef {import('../../util/Types').Rect} Rect
   *
   * @typedef { 'x' | 'y' } ModelingDistributeAxis
   *
   * @typedef { 'width' | 'height' } ModelingDistributeDimension
   *
   * @typedef { {
   *   bottom?: number;
   *   center?: number;
   *   left?: number;
   *   middle?: number;
   *   right?: number;
   *   top?: number;
   * } } ModelingAlignAlignment
   *
   * @typedef { {
   *   [key: string]: any;
   * } } ModelingHints
   *
   * @typedef { {
   *   attach?: boolean;
   * } & ModelingHints } ModelingMoveElementsHints
   *
   * @typedef { {
   *   attach?: boolean;
   * } & ModelingHints } ModelingCreateShapeHints
   */

  /**
   * @template {Element} U
   *
   * @typedef { {
   *   elements: U[],
   *   range: {
   *     min: number;
   *     max: number;
   *   } }
   * } ModelingDistributeGroup
   */

  /**
   * The basic modeling entry point.
   *
   * @template {Connection} [T=Connection]
   * @template {Element} [U=Element]
   * @template {Label} [V=Label]
   * @template {Parent} [W=Parent]
   * @template {Shape} [X=Shape]
   *
   * @param {EventBus} eventBus
   * @param {ElementFactory} elementFactory
   * @param {CommandStack} commandStack
   */
  function Modeling$5(eventBus, elementFactory, commandStack) {
    this._eventBus = eventBus;
    this._elementFactory = elementFactory;
    this._commandStack = commandStack;
    var self = this;
    eventBus.on('diagram.init', function () {
      // register modeling handlers
      self.registerHandlers(commandStack);
    });
  }
  Modeling$5.$inject = ['eventBus', 'elementFactory', 'commandStack'];

  /**
   * Get a map of all command handlers.
   *
   * @return {Map<string, CommandHandlerConstructor>}
   */
  Modeling$5.prototype.getHandlers = function () {
    return {
      'shape.append': AppendShapeHandler,
      'shape.create': CreateShapeHandler,
      'shape.delete': DeleteShapeHandler,
      'shape.move': MoveShapeHandler,
      'shape.resize': ResizeShapeHandler,
      'shape.replace': ReplaceShapeHandler,
      'shape.toggleCollapse': ToggleShapeCollapseHandler,
      'spaceTool': SpaceToolHandler,
      'label.create': CreateLabelHandler,
      'connection.create': CreateConnectionHandler,
      'connection.delete': DeleteConnectionHandler,
      'connection.move': MoveConnectionHandler,
      'connection.layout': LayoutConnectionHandler,
      'connection.updateWaypoints': UpdateWaypointsHandler,
      'connection.reconnect': ReconnectConnectionHandler,
      'elements.create': CreateElementsHandler,
      'elements.move': MoveElementsHandler,
      'elements.delete': DeleteElementsHandler,
      'elements.distribute': DistributeElements,
      'elements.align': AlignElements,
      'element.updateAttachment': UpdateAttachmentHandler
    };
  };

  /**
   * Register handlers with the command stack
   *
   * @param {CommandStack} commandStack
   */
  Modeling$5.prototype.registerHandlers = function (commandStack) {
    forEach$3(this.getHandlers(), function (handler, id) {
      commandStack.registerHandler(id, handler);
    });
  };

  /**
   * Move a shape by the given delta and optionally to a new parent.
   *
   * @param {X} shape
   * @param {Point} delta
   * @param {W} [newParent]
   * @param {number} [newParentIndex]
   * @param {ModelingHints} [hints]
   */
  Modeling$5.prototype.moveShape = function (shape, delta, newParent, newParentIndex, hints) {
    if (typeof newParentIndex === 'object') {
      hints = newParentIndex;
      newParentIndex = null;
    }
    var context = {
      shape: shape,
      delta: delta,
      newParent: newParent,
      newParentIndex: newParentIndex,
      hints: hints || {}
    };
    this._commandStack.execute('shape.move', context);
  };

  /**
   * Update the attachment of a shape.
   *
   * @param {X} shape
   * @param {X} [newHost=undefined]
   */
  Modeling$5.prototype.updateAttachment = function (shape, newHost) {
    var context = {
      shape: shape,
      newHost: newHost
    };
    this._commandStack.execute('element.updateAttachment', context);
  };

  /**
   * Move elements by a given delta and optionally to a new parent.
   *
   * @param {U[]} shapes
   * @param {Point} delta
   * @param {W} [target]
   * @param {ModelingMoveElementsHints} [hints]
   */
  Modeling$5.prototype.moveElements = function (shapes, delta, target, hints) {
    hints = hints || {};
    var attach = hints.attach;
    var newParent = target,
      newHost;
    if (attach === true) {
      newHost = target;
      newParent = target.parent;
    } else if (attach === false) {
      newHost = null;
    }
    var context = {
      shapes: shapes,
      delta: delta,
      newParent: newParent,
      newHost: newHost,
      hints: hints
    };
    this._commandStack.execute('elements.move', context);
  };

  /**
   * Move a shape by the given delta and optionally to a new parent.
   *
   * @param {T} connection
   * @param {Point} delta
   * @param {W} [newParent]
   * @param {number} [newParentIndex]
   * @param {ModelingHints} [hints]
   */
  Modeling$5.prototype.moveConnection = function (connection, delta, newParent, newParentIndex, hints) {
    if (typeof newParentIndex === 'object') {
      hints = newParentIndex;
      newParentIndex = undefined;
    }
    var context = {
      connection: connection,
      delta: delta,
      newParent: newParent,
      newParentIndex: newParentIndex,
      hints: hints || {}
    };
    this._commandStack.execute('connection.move', context);
  };

  /**
   * Layout a connection.
   *
   * @param {T} connection
   * @param {ModelingHints} [hints]
   */
  Modeling$5.prototype.layoutConnection = function (connection, hints) {
    var context = {
      connection: connection,
      hints: hints || {}
    };
    this._commandStack.execute('connection.layout', context);
  };

  /**
   * Create a connection.
   *
   * @overlord
   *
   * @param {U} source
   * @param {U} target
   * @param {Partial<T>} connection
   * @param {W} parent
   * @param {ModelingHints} [hints]
   *
   * @return {T}
   */

  /**
   * Create a connection.
   *
   * @param {U} source
   * @param {U} target
   * @param {number} parentIndex
   * @param {Partial<T>} connection
   * @param {W} parent
   * @param {ModelingHints} [hints]
   *
   * @return {T}
   */
  Modeling$5.prototype.createConnection = function (source, target, parentIndex, connection, parent, hints) {
    if (typeof parentIndex === 'object') {
      hints = parent;
      parent = connection;
      connection = parentIndex;
      parentIndex = undefined;
    }
    connection = this._create('connection', connection);
    var context = {
      source: source,
      target: target,
      parent: parent,
      parentIndex: parentIndex,
      connection: connection,
      hints: hints
    };
    this._commandStack.execute('connection.create', context);
    return context.connection;
  };

  /**
   * Create a shape.
   *
   * @overlord
   *
   * @param {Partial<X>} shape
   * @param {Point} position
   * @param {W} target
   * @param {ModelingCreateShapeHints} [hints]
   *
   * @return {X}
   */

  /**
   * Create a shape.
   *
   * @param {Partial<X>} shape
   * @param {Point} position
   * @param {W} target
   * @param {number} parentIndex
   * @param {ModelingCreateShapeHints} [hints]
   *
   * @return {X}
   */
  Modeling$5.prototype.createShape = function (shape, position, target, parentIndex, hints) {
    if (typeof parentIndex !== 'number') {
      hints = parentIndex;
      parentIndex = undefined;
    }
    hints = hints || {};
    var attach = hints.attach,
      parent,
      host;
    shape = this._create('shape', shape);
    if (attach) {
      parent = target.parent;
      host = target;
    } else {
      parent = target;
    }
    var context = {
      position: position,
      shape: shape,
      parent: parent,
      parentIndex: parentIndex,
      host: host,
      hints: hints
    };
    this._commandStack.execute('shape.create', context);
    return context.shape;
  };

  /**
   * Create elements.
   *
   * @param {Partial<U>[]} elements
   * @param {Point} position
   * @param {W} parent
   * @param {number} [parentIndex]
   * @param {ModelingHints} [hints]
   *
   * @return {U[]}
   */
  Modeling$5.prototype.createElements = function (elements, position, parent, parentIndex, hints) {
    if (!isArray$6(elements)) {
      elements = [elements];
    }
    if (typeof parentIndex !== 'number') {
      hints = parentIndex;
      parentIndex = undefined;
    }
    hints = hints || {};
    var context = {
      position: position,
      elements: elements,
      parent: parent,
      parentIndex: parentIndex,
      hints: hints
    };
    this._commandStack.execute('elements.create', context);
    return context.elements;
  };

  /**
   * Create a label.
   *
   * @param {U} labelTarget
   * @param {Point} position
   * @param {Partial<V>} label
   * @param {W} [parent]
   *
   * @return {V}
   */
  Modeling$5.prototype.createLabel = function (labelTarget, position, label, parent) {
    label = this._create('label', label);
    var context = {
      labelTarget: labelTarget,
      position: position,
      parent: parent || labelTarget.parent,
      shape: label
    };
    this._commandStack.execute('label.create', context);
    return context.shape;
  };

  /**
   * Create and connect a shape to a source.
   *
   * @param {U} source
   * @param {Partial<X>} shape
   * @param {Point} position
   * @param {W} target
   * @param {ModelingHints} [hints]
   *
   * @return {X}
   */
  Modeling$5.prototype.appendShape = function (source, shape, position, target, hints) {
    hints = hints || {};
    shape = this._create('shape', shape);
    var context = {
      source: source,
      position: position,
      target: target,
      shape: shape,
      connection: hints.connection,
      connectionParent: hints.connectionParent,
      hints: hints
    };
    this._commandStack.execute('shape.append', context);
    return context.shape;
  };

  /**
   * Remove elements.
   *
   * @param {U[]} elements
   */
  Modeling$5.prototype.removeElements = function (elements) {
    var context = {
      elements: elements
    };
    this._commandStack.execute('elements.delete', context);
  };

  /**
   * Distribute elements along a given axis.
   *
   * @param {ModelingDistributeGroup<U>[]} groups
   * @param {ModelingDistributeAxis} axis
   * @param {ModelingDistributeDimension} dimension
   */
  Modeling$5.prototype.distributeElements = function (groups, axis, dimension) {
    var context = {
      groups: groups,
      axis: axis,
      dimension: dimension
    };
    this._commandStack.execute('elements.distribute', context);
  };

  /**
   * Remove a shape.
   *
   * @param {X} shape
   * @param {ModelingHints} [hints]
   */
  Modeling$5.prototype.removeShape = function (shape, hints) {
    var context = {
      shape: shape,
      hints: hints || {}
    };
    this._commandStack.execute('shape.delete', context);
  };

  /**
   * Remove a connection.
   *
   * @param {T} connection
   * @param {ModelingHints} [hints]
   */
  Modeling$5.prototype.removeConnection = function (connection, hints) {
    var context = {
      connection: connection,
      hints: hints || {}
    };
    this._commandStack.execute('connection.delete', context);
  };

  /**
   * Replace a shape.
   *
   * @param {X} oldShape
   * @param {Partial<X>} newShape
   * @param {ModelingHints} [hints]
   *
   * @return {X}
   */
  Modeling$5.prototype.replaceShape = function (oldShape, newShape, hints) {
    var context = {
      oldShape: oldShape,
      newData: newShape,
      hints: hints || {}
    };
    this._commandStack.execute('shape.replace', context);
    return context.newShape;
  };

  /**
   * Align elements.
   *
   * @param {U[]} elements
   * @param {ModelingAlignAlignment} alignment
   */
  Modeling$5.prototype.alignElements = function (elements, alignment) {
    var context = {
      elements: elements,
      alignment: alignment
    };
    this._commandStack.execute('elements.align', context);
  };

  /**
   * Resize a shape.
   *
   * @param {X} shape
   * @param {Rect} newBounds
   * @param {Dimensions} [minBounds]
   * @param {ModelingHints} [hints]
   */
  Modeling$5.prototype.resizeShape = function (shape, newBounds, minBounds, hints) {
    var context = {
      shape: shape,
      newBounds: newBounds,
      minBounds: minBounds,
      hints: hints
    };
    this._commandStack.execute('shape.resize', context);
  };

  /**
   * Create space along an horizontally or vertically.
   *
   * @param {X[]} movingShapes
   * @param {X[]} resizingShapes
   * @param {Point} delta
   * @param {Direction} direction
   * @param {number} start
   */
  Modeling$5.prototype.createSpace = function (movingShapes, resizingShapes, delta, direction, start) {
    var context = {
      delta: delta,
      direction: direction,
      movingShapes: movingShapes,
      resizingShapes: resizingShapes,
      start: start
    };
    this._commandStack.execute('spaceTool', context);
  };

  /**
   * Update a connetions waypoints.
   *
   * @param {T} connection
   * @param {Point[]} newWaypoints
   * @param {ModelingHints} [hints]
   */
  Modeling$5.prototype.updateWaypoints = function (connection, newWaypoints, hints) {
    var context = {
      connection: connection,
      newWaypoints: newWaypoints,
      hints: hints || {}
    };
    this._commandStack.execute('connection.updateWaypoints', context);
  };

  /**
   * Reconnect a connections source and/or target.
   *
   * @param {T} connection
   * @param {U} source
   * @param {U} target
   * @param {Point|Point[]} dockingOrPoints
   * @param {ModelingHints} [hints]
   */
  Modeling$5.prototype.reconnect = function (connection, source, target, dockingOrPoints, hints) {
    var context = {
      connection: connection,
      newSource: source,
      newTarget: target,
      dockingOrPoints: dockingOrPoints,
      hints: hints || {}
    };
    this._commandStack.execute('connection.reconnect', context);
  };

  /**
   * Reconnect a connections source.
   *
   * @param {T} connection
   * @param {U} newSource
   * @param {Point|Point[]} dockingOrPoints
   * @param {ModelingHints} [hints]
   */
  Modeling$5.prototype.reconnectStart = function (connection, newSource, dockingOrPoints, hints) {
    if (!hints) {
      hints = {};
    }
    this.reconnect(connection, newSource, connection.target, dockingOrPoints, assign$4(hints, {
      docking: 'source'
    }));
  };

  /**
   * Reconnect a connections target.
   *
   * @param {T} connection
   * @param {U} newTarget
   * @param {Point|Point[]} dockingOrPoints
   * @param {ModelingHints} [hints]
   */
  Modeling$5.prototype.reconnectEnd = function (connection, newTarget, dockingOrPoints, hints) {
    if (!hints) {
      hints = {};
    }
    this.reconnect(connection, connection.source, newTarget, dockingOrPoints, assign$4(hints, {
      docking: 'target'
    }));
  };

  /**
   * Connect two elements.
   *
   * @param {U} source
   * @param {U} target
   * @param {Partial<T>} [attrs]
   * @param {ModelingHints} [hints]
   *
   * @return {T}
   */
  Modeling$5.prototype.connect = function (source, target, attrs, hints) {
    return this.createConnection(source, target, attrs || {}, source.parent, hints);
  };
  Modeling$5.prototype._create = function (type, attrs) {
    if (isModelElement(attrs)) {
      return attrs;
    } else {
      return this._elementFactory.create(type, attrs);
    }
  };

  /**
   * Collapse or expand a shape.
   *
   * @param {X} shape
   * @param {ModelingHints} [hints]
   */
  Modeling$5.prototype.toggleCollapse = function (shape, hints) {
    var context = {
      shape: shape,
      hints: hints || {}
    };
    this._commandStack.execute('shape.toggleCollapse', context);
  };

  function IdClaimHandler$1(moddle) {
    this._moddle = moddle;
  }
  IdClaimHandler$1.$inject = ['moddle'];
  IdClaimHandler$1.prototype.execute = function (context) {
    var ids = this._moddle.ids,
      id = context.id,
      element = context.element,
      claiming = context.claiming;
    if (claiming) {
      ids.claim(id, element);
    } else {
      ids.unclaim(id);
    }
  };

  /**
   * Command revert implementation.
   */
  IdClaimHandler$1.prototype.revert = function (context) {
    var ids = this._moddle.ids,
      id = context.id,
      element = context.element,
      claiming = context.claiming;
    if (claiming) {
      ids.unclaim(id);
    } else {
      ids.claim(id, element);
    }
  };

  /**
   * A handler that updates the name of a DMN element.
   */
  function UpdateLabelHandler() {
    /**
     * Set the label and return the changed elements.
     *
     * Element parameter can be label itself or connection (i.e. sequence flow).
     *
     * @param {djs.model.Base} element
     * @param {string} text
     */
    function setText(element, text) {
      // external label if present
      var label = element.label || element;
      var labelTarget = element.labelTarget || element;
      setLabel(label, text, labelTarget !== label);
      return [label, labelTarget];
    }
    function execute(ctx) {
      ctx.oldLabel = getLabel(ctx.element);
      return setText(ctx.element, ctx.newLabel);
    }
    function revert(ctx) {
      return setText(ctx.element, ctx.oldLabel);
    }

    // API

    this.execute = execute;
    this.revert = revert;
  }

  const NAME = 'name',
    ID$1 = 'id';

  /**
   * A handler that implements a DMN property update.
   *
   * This should be used to set simple properties on elements with
   * an underlying DMN business object.
   *
   * Use respective diagram-js provided handlers if you would
   * like to perform automated modeling.
   */
  function UpdatePropertiesHandler(elementRegistry, moddle) {
    this._elementRegistry = elementRegistry;
    this._moddle = moddle;
  }
  UpdatePropertiesHandler.$inject = ['elementRegistry', 'moddle'];

  /**
   * Updates a DMN element with a list of new properties
   *
   * @param {Object} context
   * @param {djs.model.Base} context.element the element to update
   * @param {Object} context.properties a list of properties to set on the element's
   *                                    businessObject (the DMN model element)
   *
   * @return {Array<djs.model.Base>} the updated element
   */
  UpdatePropertiesHandler.prototype.execute = function (context) {
    const {
        element,
        properties
      } = context,
      changed = [element];
    if (!element) {
      throw new Error('element required');
    }
    const elementRegistry = this._elementRegistry,
      ids = this._moddle.ids;
    const businessObject = getBusinessObject(element),
      oldProperties = context.oldProperties || getProperties(businessObject, keys(properties));
    if (isIdChange$1(properties, businessObject)) {
      ids.unclaim(businessObject[ID$1]);
      elementRegistry.updateId(element, properties[ID$1]);
      ids.claim(properties[ID$1], businessObject);
    }
    if (NAME in properties && element.label) {
      changed.push(element.label);
    }

    // update properties
    setProperties(businessObject, properties);

    // store old values
    context.oldProperties = oldProperties;
    context.changed = changed;

    // indicate changed on objects affected by the update
    return changed;
  };

  /**
   * Reverts the update on a DMN elements properties.
   *
   * @param  {Object} context
   *
   * @return {djs.model.Base} the updated element
   */
  UpdatePropertiesHandler.prototype.revert = function (context) {
    const {
      element,
      properties,
      oldProperties
    } = context;
    const businessObject = getBusinessObject(element);
    const elementRegistry = this._elementRegistry,
      ids = this._moddle.ids;

    // update properties
    setProperties(businessObject, oldProperties);
    if (isIdChange$1(properties, businessObject)) {
      ids.unclaim(properties[ID$1]);
      elementRegistry.updateId(element, oldProperties[ID$1]);
      ids.claim(oldProperties[ID$1], businessObject);
    }
    return context.changed;
  };
  function isIdChange$1(properties, businessObject) {
    return ID$1 in properties && properties[ID$1] !== businessObject[ID$1];
  }
  function getProperties(businessObject, propertyNames) {
    return reduce(propertyNames, function (result, key) {
      result[key] = businessObject.get(key);
      return result;
    }, {});
  }
  function setProperties(businessObject, properties) {
    forEach$3(properties, function (value, key) {
      businessObject.set(key, value);
    });
  }

  function UpdateModdlePropertiesHandler(elementRegistry) {
    this._elementRegistry = elementRegistry;
  }
  UpdateModdlePropertiesHandler.$inject = ['elementRegistry'];
  UpdateModdlePropertiesHandler.prototype.execute = function (context) {
    var element = context.element,
      moddleElement = context.moddleElement,
      properties = context.properties;
    if (!moddleElement) {
      throw new Error('<moddleElement> required');
    }

    // TODO(nikku): we need to ensure that ID properties
    // are properly registered / unregistered via
    // this._moddle.ids.assigned(id)
    var changed = context.changed || [element];
    var oldProperties = context.oldProperties || getModdleProperties(moddleElement, keys(properties));
    setModdleProperties(moddleElement, properties);
    context.oldProperties = oldProperties;
    context.changed = changed;
    return changed;
  };
  UpdateModdlePropertiesHandler.prototype.revert = function (context) {
    var oldProperties = context.oldProperties,
      moddleElement = context.moddleElement,
      changed = context.changed;
    setModdleProperties(moddleElement, oldProperties);
    return changed;
  };

  // helpers /////////////////

  function getModdleProperties(moddleElement, propertyNames) {
    return reduce(propertyNames, function (result, key) {
      result[key] = moddleElement.get(key);
      return result;
    }, {});
  }
  function setModdleProperties(moddleElement, properties) {
    forEach$3(properties, function (value, key) {
      moddleElement.set(key, value);
    });
  }

  /**
   * DMN modeling.
   *
   * @param {Canvas} canvas
   * @param {CommandStack} commandStack
   * @param {DrdRules} drdRules
   * @param {ElementFactory} elementFactory
   * @param {EventBus} eventBus
   */
  function Modeling$4(canvas, drdRules, injector) {
    this._canvas = canvas;
    this._drdRules = drdRules;
    injector.invoke(Modeling$5, this);
  }
  e$3(Modeling$4, Modeling$5);
  Modeling$4.$inject = ['canvas', 'drdRules', 'injector'];
  Modeling$4.prototype.claimId = function (id, moddleElement) {
    this._commandStack.execute('id.updateClaim', {
      id: id,
      element: moddleElement,
      claiming: true
    });
  };
  Modeling$4.prototype.connect = function (source, target, attrs, hints) {
    var drdRules = this._drdRules,
      rootElement = this._canvas.getRootElement();
    if (!attrs) {
      attrs = drdRules.canConnect(source, target) || {
        type: 'dmn:Association'
      };
    }
    return this.createConnection(source, target, attrs, rootElement, hints);
  };
  Modeling$4.prototype.getHandlers = function () {
    var handlers = Modeling$5.prototype.getHandlers.call(this);
    handlers['id.updateClaim'] = IdClaimHandler$1;
    handlers['element.updateLabel'] = UpdateLabelHandler;
    handlers['element.updateProperties'] = UpdatePropertiesHandler;
    handlers['element.updateModdleProperties'] = UpdateModdlePropertiesHandler;
    return handlers;
  };
  Modeling$4.prototype.unclaimId = function (id, moddleElement) {
    this._commandStack.execute('id.updateClaim', {
      id: id,
      element: moddleElement
    });
  };
  Modeling$4.prototype.updateModdleProperties = function (element, moddleElement, properties) {
    this._commandStack.execute('element.updateModdleProperties', {
      element: element,
      moddleElement: moddleElement,
      properties: properties
    });
  };
  Modeling$4.prototype.updateProperties = function (element, properties) {
    this._commandStack.execute('element.updateProperties', {
      element: element,
      properties: properties
    });
  };
  Modeling$4.prototype.updateLabel = function (element, newLabel, newBounds, hints) {
    this._commandStack.execute('element.updateLabel', {
      element: element,
      newLabel: newLabel,
      newBounds: newBounds,
      hints: hints || {}
    });
  };

  /**
   * @typedef {import('../core/Types').ElementLike} Element
   * @typedef {import('../core/Types').ConnectionLike} Connection
   *
   * @typedef {import('../util').Point} Point
   *
   * @typedef { {
   *   connectionStart?: Point;
   *   connectionEnd?: Point;
   *   source?: Element;
   *   target?: Element;
   * } } LayoutConnectionHints
   */


  /**
   * A base connection layouter implementation
   * that layouts the connection by directly connecting
   * mid(source) + mid(target).
   */
  function BaseLayouter() {}

  /**
   * Return the new layouted waypoints for the given connection.
   *
   * The connection passed is still unchanged; you may figure out about
   * the new connection start / end via the layout hints provided.
   *
   * @param {Connection} connection
   * @param {LayoutConnectionHints} [hints]
   *
   * @return {Point[]} The waypoints of the laid out connection.
   */
  BaseLayouter.prototype.layoutConnection = function (connection, hints) {
    hints = hints || {};
    return [hints.connectionStart || getMid(hints.source || connection.source), hints.connectionEnd || getMid(hints.target || connection.target)];
  };

  var ADDITIONAL_WAYPOINT_DISTANCE = 20;
  function DrdLayouter(connectionDocking) {
    this._connectionDocking = connectionDocking;
  }
  e$3(DrdLayouter, BaseLayouter);
  DrdLayouter.$inject = ['connectionDocking'];
  DrdLayouter.prototype.layoutConnection = function (connection, hints) {
    var connectionDocking = this._connectionDocking;
    if (!hints) {
      hints = {};
    }
    var source = hints.source || connection.source,
      target = hints.target || connection.target,
      waypoints = hints.waypoints || connection.waypoints || [],
      connectionStart = hints.connectionStart,
      connectionEnd = hints.connectionEnd,
      orientation = getOrientation(source, target);
    if (!connectionStart) {
      connectionStart = getConnectionDocking(waypoints[0], source);
    }
    if (!connectionEnd) {
      connectionEnd = getConnectionDocking(waypoints[waypoints.length - 1], target);
    }
    if (is$1(connection, 'dmn:InformationRequirement')) {
      waypoints = [connectionStart, connectionEnd];
      var croppedWaypoints = connectionDocking.getCroppedWaypoints(assign$4({}, connection, {
        waypoints: waypoints
      }), source, target);
      connectionEnd = croppedWaypoints.pop();
      var additionalWaypoint = {
        x: connectionEnd.x,
        y: connectionEnd.y
      };
      if (orientation.includes('bottom')) {
        additionalWaypoint.y += ADDITIONAL_WAYPOINT_DISTANCE;
      } else if (orientation.includes('top')) {
        additionalWaypoint.y -= ADDITIONAL_WAYPOINT_DISTANCE;
      } else if (orientation.includes('right')) {
        additionalWaypoint.x += ADDITIONAL_WAYPOINT_DISTANCE;
      } else {
        additionalWaypoint.x -= ADDITIONAL_WAYPOINT_DISTANCE;
      }
      waypoints = croppedWaypoints.concat([additionalWaypoint, connectionEnd]);
      return waypoints;
    }
    return [connectionStart, connectionEnd];
  };
  function getConnectionDocking(point, shape) {
    return point ? point.original || point : getMid(shape);
  }

  /**
   * @typedef {import('../core/ElementRegistry').default} ElementRegistry
   * @typedef {import('../core/GraphicsFactory').default} GraphicsFactory
   */

  function dockingToPoint(docking) {
    // use the dockings actual point and
    // retain the original docking
    return assign$4({
      original: docking.point.original || docking.point
    }, docking.actual);
  }

  /**
   * A {@link ConnectionDocking} that crops connection waypoints based on
   * the path(s) of the connection source and target.
   *
   * @param {ElementRegistry} elementRegistry
   * @param {GraphicsFactory} graphicsFactory
   */
  function CroppingConnectionDocking(elementRegistry, graphicsFactory) {
    this._elementRegistry = elementRegistry;
    this._graphicsFactory = graphicsFactory;
  }
  CroppingConnectionDocking.$inject = ['elementRegistry', 'graphicsFactory'];

  /**
   * @inheritDoc ConnectionDocking#getCroppedWaypoints
   */
  CroppingConnectionDocking.prototype.getCroppedWaypoints = function (connection, source, targe